/*--------------------------------------------------------------------------+
$Id: XMLWriterTest.java 26283 2010-02-18 11:18:57Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.commons.xml;

import java.io.ByteArrayOutputStream;

import junit.framework.TestCase;
import edu.tum.cs.commons.string.StringUtils;

/**
 * Test for {@link XMLWriter}.
 * 
 * @author Florian Deissenboeck
 * @author $Author: juergens $
 * @version $Rev: 26283 $
 * @levd.rating GREEN Hash: 4E7A347F2EFC64AFB162BE7A6043A9A5
 */
public class XMLWriterTest extends TestCase {

	/** Stream to write to. */
	private ByteArrayOutputStream stream;

	/** Writer under test. */
	private XMLWriter<Elements, Attributes> writer;

	/** Create new writer. */
	@Override
	public void setUp() {
		stream = new ByteArrayOutputStream();
		writer = new XMLWriter<Elements, Attributes>(stream,
				new XMLResolver<Elements, Attributes>(Attributes.class));
	}

	/** Test method. */
	public void testAttributeOutsideElement() {

		try {
			writer.addAttribute(Attributes.attribute1, "test1");
		} catch (XMLWriterException e) {
			assertEquals(EXMLWriterExceptionType.ATTRIBUTE_OUTSIDE_ELEMENT, e
					.getType());
		}
	}

	/** Test method. */
	public void testComment() {
		writer.addComment("test");
		
		print();

		assertEquals("<!-- test -->", normalize());
	}

	/** Test method. */
	public void testCommentBetweenElements() {
		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);
		writer.addComment("test");
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);

		assertEquals("<element1 />\n<!-- test -->\n<element2 />", normalize());
	}

	/** Test method. */
	public void testCommentInElement() {
		writer.openElement(Elements.element1);
		writer.addComment("test");
		writer.closeElement(Elements.element1);

		assertEquals("<element1>\n  <!-- test -->\n</element1>", normalize());
	}

	/** Test method. */
	public void testNewLineBetweenElements() {
		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);
		writer.addNewLine();
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);

		assertEquals("<element1 />\n\n<element2 />", normalize());
	}

	/** Test method. */
	public void testCommentAtBeginningOfDocument() {
		writer.addComment("test");
		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);

		assertEquals("<!-- test -->\n<element1 />", normalize());
	}

	/** Test method. */
	public void testNewLineInElement() {
		writer.openElement(Elements.element1);
		writer.addNewLine();
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals("<element1>\n\n  <element2 />\n</element1>", normalize());
	}

	/** Test method. */
	public void testDuplicateAttribute() {
		writer.openElement(Elements.element1);
		writer.addAttribute(Attributes.attribute1, "test1");

		try {
			writer.addAttribute(Attributes.attribute1, "test1");
		} catch (XMLWriterException e) {
			assertEquals(EXMLWriterExceptionType.DUPLICATE_ATTRIBUTE, e
					.getType());
		}
	}

	/** Test method. */
	public void testEmptyElement() {
		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);

		assertEquals("<element1 />", normalize());
	}

	/** Test method. */
	public void testEscapedTextElement() {
		writer.openElement(Elements.element1);
		writer.addText("test<test2>test3&test4");
		writer.closeElement(Elements.element1);

		assertEquals("<element1>test&lt;test2&gt;test3&amp;test4</element1>",
				normalize());
	}

	/** Test method. */
	public void testMultiLineTextElement() {
		writer.openElement(Elements.element1);
		writer.addText("test" + StringUtils.CR + "test2");
		writer.closeElement(Elements.element1);

		assertEquals("<element1>test\ntest2</element1>", normalize());
	}

	/** Test method. */
	public void testNestedComment() {
		writer.openElement(Elements.element1);
		writer.addComment("test");
		writer.closeElement(Elements.element1);

		assertEquals("<element1>\n  <!-- test -->\n</element1>", normalize());
	}

	/** Test method. */
	public void testNestedElements() {
		writer.openElement(Elements.element1);
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals("<element1>\n  <element2 />\n</element1>", normalize());
	}

	/** Test method. */
	public void testNestedElementsWithAttributes() {
		writer.openElement(Elements.element1);
		writer.addAttribute(Attributes.attribute1, "test1");
		writer.openElement(Elements.element2);
		writer.addAttribute(Attributes.attribute1, "test1");
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals(
				"<element1 attribute1=\"test1\">\n  <element2 attribute1=\"test1\" />\n</element1>",
				normalize());
	}

	/** Test method. */
	public void testNestedMultiLineTextElement() {
		writer.openElement(Elements.element1);
		writer.openElement(Elements.element2);
		writer.addText("test" + StringUtils.CR + "test2");
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals(
				"<element1>\n  <element2>test\ntest2</element2>\n</element1>",
				normalize());
	}

	/** Test method. */
	public void testNestedSequentialElements() {
		writer.openElement(Elements.element1);
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals("<element1>\n  <element2 />\n  <element2 />\n</element1>",
				normalize());
	}

	/** Test method. */
	public void testNestedTextElement() {
		writer.openElement(Elements.element1);
		writer.openElement(Elements.element2);
		writer.addText("test");
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals("<element1>\n  <element2>test</element2>\n</element1>",
				normalize());
	}

	/** Test method. */
	public void testElementInText() {
		writer.openElement(Elements.element1);
		writer.addText("test");
		writer.openElement(Elements.element2);
		writer.addText("test2");
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals("<element1>test<element2>test2</element2>\n</element1>",
				normalize());
	}

	/** Test method. */
	public void testAddHeader() {

		writer.addHeader("1.0", "ISO-8859-1");

		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);

		assertEquals(
				"<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n<element1 />",
				normalize());
	}

	/** Test method. */
	public void testOneElementWithAttributes() {
		writer.openElement(Elements.element1);
		writer.addAttribute(Attributes.attribute1, "test1");
		writer.addAttribute(Attributes.attribute2, "test2");
		writer.closeElement(Elements.element1);

		assertEquals("<element1 attribute1=\"test1\" attribute2=\"test2\" />",
				normalize());
	}

	/** Test method. */
	public void testAttributesWithEscapes() {
		writer.openElement(Elements.element1);
		writer.addAttribute(Attributes.attribute1, "a&<>\"");
		writer.closeElement(Elements.element1);

		assertEquals("<element1 attribute1=\"a&amp;&lt;&gt;&quot;\" />",
				normalize());
	}

	/** Test method. */
	public void testCData() {
		writer.openElement(Elements.element1);
		writer.addCDataSection("test <element>");
		writer.closeElement(Elements.element1);

		assertEquals("<element1><![CDATA[test <element>]]></element1>",
				normalize());
	}

	/** Test method. */
	public void testCDataEscpaing() {
		writer.openElement(Elements.element1);

		try {
			writer.addCDataSection("test <element> test ]]>");
		} catch (XMLWriterException e) {
			assertEquals(
					EXMLWriterExceptionType.CDATA_CONTAINS_CDATA_CLOSING_TAG, e
							.getType());
		}
	}

	/** Test method. */
	public void testCDataWithinText() {
		writer.openElement(Elements.element1);
		writer.addText("test");
		writer.addCDataSection("test <element>");
		writer.addText("test");
		writer.closeElement(Elements.element1);

		assertEquals("<element1>test<![CDATA[test <element>]]>test</element1>",
				normalize());
	}

	/** Test method. */
	public void testSequentialComment() {
		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);
		writer.addComment("test");
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);

		assertEquals("<element1 />\n<!-- test -->\n<element2 />", normalize());
	}

	/** Test method. */
	public void testSequentialElements() {
		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);

		assertEquals("<element1 />\n<element2 />", normalize());
	}

	/** Test method. */
	public void testTextElement() {
		writer.openElement(Elements.element1);
		writer.addText("test");
		writer.closeElement(Elements.element1);

		assertEquals("<element1>test</element1>", normalize());
	}

	/** Test method. */
	public void testAddPublicDocTypeDefintion() {
		writer.addPublicDocTypeDefintion(Elements.element1, "x", "y");

		assertEquals("<!DOCTYPE element1 PUBLIC \"x\" \"y\">", normalize());
	}

	/** Test method for {@link XMLWriter#openElement(Enum)} */
	public void testOpenElement() {
		writer.openElement(Elements.element1, Attributes.attribute1, "test1",
				Attributes.attribute2, "test2");
		writer.closeElement(Elements.element1);
		assertEquals("<element1 attribute1=\"test1\" attribute2=\"test2\" />",
				normalize());
	}

	/** Test method for {@link XMLWriter#openElement(Enum)} */
	public void testOpenElementWithOddNumberOfAttributes() {
		try {
			writer.openElement(Elements.element1, Attributes.attribute1,
					"test1", Attributes.attribute2);
			fail();
		} catch (XMLWriterException e) {
			assertEquals(EXMLWriterExceptionType.ODD_NUMBER_OF_ARGUMENTS, e
					.getType());
		}
	}

	/** Test method for {@link XMLWriter#openElement(Enum)} */
	public void testOpenElementWithIllegalAttribute() {
		try {
			writer.openElement(Elements.element1, Attributes.attribute1,
					"test1", new Integer(5), "test2");
			fail();
		} catch (XMLWriterException e) {
			assertEquals(EXMLWriterExceptionType.ILLEGAL_ATTRIBUTE_TYPE, e
					.getType());
		}
	}

	/** Test method. */
	public void testUnclosedElement() {
		writer.openElement(Elements.element1);

		try {
			writer.closeElement(Elements.element2);
			fail();
		} catch (XMLWriterException e) {
			assertEquals(EXMLWriterExceptionType.UNCLOSED_ELEMENT, e.getType());
		}
	}

	/** Test method. */
	public void testRemainingOpenElementWhenClosingWriter() {
		writer.openElement(Elements.element1);

		try {
			writer.close();
			fail();
		} catch (XMLWriterException e) {
			assertEquals(EXMLWriterExceptionType.UNCLOSED_ELEMENT, e.getType());
		}
	}

	/** Test method. */
	public void testSuppressLineBreaks() {

		writer.setSuppressLineBreaks(true);

		writer.openElement(Elements.element1);
		writer.closeElement(Elements.element1);
		writer.openElement(Elements.element2);
		writer.closeElement(Elements.element2);

		writer.openElement(Elements.element1);
		writer.addText("test");
		writer.openElement(Elements.element2);
		writer.addText("test2");
		writer.closeElement(Elements.element2);
		writer.closeElement(Elements.element1);

		assertEquals(
				"<element1 /><element2 /><element1>test<element2>test2</element2></element1>",
				normalize());

	}

	/** Test method. */
	public void testSuppressLineBreaks2() {

		writer.setSuppressLineBreaks(true);

	}

	/**
	 * Use this method to create a test string that can be copied from the
	 * console directly to the assert statement.
	 * <p>
	 * We only need this for creating new tests.
	 */
	private String createTestString() {
		String string = stream.toString();

		string = StringUtils.replaceLineBreaks(string);
		string = string.replace("\"", "\\\"");
		return string;
	}

	/**
	 * Replace line breaks in current stream by '\n' and return it. We need this
	 * to define asserts in platform-independent manner.
	 */
	private String normalize() {
		writer.flush();
		String string = stream.toString();
		string = StringUtils.replaceLineBreaks(string, "\n");
		return string;
	}

	/**
	 * Print current stream and an representation ready for copy&paste.
	 * <p>
	 * We only need this for creating new tests.
	 */
	private void print() {
		writer.flush();
		System.out.println(stream);
		System.out.println(createTestString());
	}

	/** Test enumeration for attributes. */
	private enum Attributes {
		/** */
		attribute1,
		/** */
		attribute2
	}

	/** Test enumeration for element. */
	private enum Elements {
		/** */
		element1,
		/** */
		element2
	}

}