/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import java.awt.GraphicsConfiguration;

import javax.swing.JFrame;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;

/**
 * i18n version of javax.swing.JFrame
 *
 * @author Jean-Hugues de Raigniac
 */
public class I18NJFrame extends JFrame implements LocaleListener {

    /** i18n bundle key */
    private String i18nKey = "misc.error";

    /**
     * Override javax.swing.JFrame.JFrame ().
     */
    public I18NJFrame () {
        this("", (GraphicsConfiguration) null);
    }

    public I18NJFrame (GraphicsConfiguration gc) {
        this("", gc);
    }

    /**
     * Override javax.swing.JFrame.JFrame (java.lang.String).
     *
     * @param i18nKey i18n bundle key
     */
    public I18NJFrame (String i18nKey) {
        this(i18nKey, (GraphicsConfiguration) null);
    }

    public I18NJFrame (String i18nKey, GraphicsConfiguration gc) {
        super(i18nKey, gc);
        setTitle(i18nKey);
        setRootPane(new I18NJRootPane());
    }

    /**
     * Override java.awt.Frame.setTitle (java.lang.String).
     *
     * @param i18nKey i18n bundle key
     */
    public void setTitle (String i18nKey) {
        this.i18nKey = i18nKey;
        if (i18nKey != null && ! i18nKey.equals("")) {
            super.setTitle(Translator.checkValue(i18nKey, this));
        }
    }

    /**
     * Propagate the LocaleEvent to the components of this JFrame.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {

        ((I18NJRootPane) getRootPane()).localeChanged(event);
        setTitle(i18nKey);
    }
}
