/* ************************************************************************
 * Copyright (C) 2020-2023 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell cop-
 * ies of the Software, and to permit persons to whom the Software is furnished
 * to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IM-
 * PLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNE-
 * CTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * ************************************************************************ */

#pragma once

/*!\file
 *  This file interfaces with our Fortran BLAS interface.
 */

/*
 * ============================================================================
 *     Fortran functions
 * ============================================================================
 */
extern "C" {

/* ==========
 *    L1
 * ========== */

// scal
rocblas_status
    rocblas_sscal_fortran(rocblas_handle handle, int n, const float* alpha, float* x, int incx);

rocblas_status
    rocblas_dscal_fortran(rocblas_handle handle, int n, const double* alpha, double* x, int incx);

rocblas_status rocblas_cscal_fortran(rocblas_handle               handle,
                                     int                          n,
                                     const rocblas_float_complex* alpha,
                                     rocblas_float_complex*       x,
                                     int                          incx);

rocblas_status rocblas_zscal_fortran(rocblas_handle                handle,
                                     int                           n,
                                     const rocblas_double_complex* alpha,
                                     rocblas_double_complex*       x,
                                     int                           incx);

rocblas_status rocblas_csscal_fortran(
    rocblas_handle handle, int n, const float* alpha, rocblas_float_complex* x, int incx);

rocblas_status rocblas_zdscal_fortran(
    rocblas_handle handle, int n, const double* alpha, rocblas_double_complex* x, int incx);

// scal_batched
rocblas_status rocblas_sscal_batched_fortran(
    rocblas_handle handle, int n, const float* alpha, float* const x[], int incx, int batch_count);

rocblas_status rocblas_dscal_batched_fortran(rocblas_handle handle,
                                             int            n,
                                             const double*  alpha,
                                             double* const  x[],
                                             int            incx,
                                             int            batch_count);

rocblas_status rocblas_cscal_batched_fortran(rocblas_handle               handle,
                                             int                          n,
                                             const rocblas_float_complex* alpha,
                                             rocblas_float_complex* const x[],
                                             int                          incx,
                                             int                          batch_count);

rocblas_status rocblas_zscal_batched_fortran(rocblas_handle                handle,
                                             int                           n,
                                             const rocblas_double_complex* alpha,
                                             rocblas_double_complex* const x[],
                                             int                           incx,
                                             int                           batch_count);

rocblas_status rocblas_csscal_batched_fortran(rocblas_handle               handle,
                                              int                          n,
                                              const float*                 alpha,
                                              rocblas_float_complex* const x[],
                                              int                          incx,
                                              int                          batch_count);

rocblas_status rocblas_zdscal_batched_fortran(rocblas_handle                handle,
                                              int                           n,
                                              const double*                 alpha,
                                              rocblas_double_complex* const x[],
                                              int                           incx,
                                              int                           batch_count);

// scal_strided_batched
rocblas_status rocblas_sscal_strided_batched_fortran(rocblas_handle handle,
                                                     int            n,
                                                     const float*   alpha,
                                                     float*         x,
                                                     int            incx,
                                                     rocblas_stride stride_x,
                                                     int            batch_count);

rocblas_status rocblas_dscal_strided_batched_fortran(rocblas_handle handle,
                                                     int            n,
                                                     const double*  alpha,
                                                     double*        x,
                                                     int            incx,
                                                     rocblas_stride stride_x,
                                                     int            batch_count);

rocblas_status rocblas_cscal_strided_batched_fortran(rocblas_handle               handle,
                                                     int                          n,
                                                     const rocblas_float_complex* alpha,
                                                     rocblas_float_complex*       x,
                                                     int                          incx,
                                                     rocblas_stride               stride_x,
                                                     int                          batch_count);

rocblas_status rocblas_zscal_strided_batched_fortran(rocblas_handle                handle,
                                                     int                           n,
                                                     const rocblas_double_complex* alpha,
                                                     rocblas_double_complex*       x,
                                                     int                           incx,
                                                     rocblas_stride                stride_x,
                                                     int                           batch_count);

rocblas_status rocblas_csscal_strided_batched_fortran(rocblas_handle         handle,
                                                      int                    n,
                                                      const float*           alpha,
                                                      rocblas_float_complex* x,
                                                      int                    incx,
                                                      rocblas_stride         stride_x,
                                                      int                    batch_count);

rocblas_status rocblas_zdscal_strided_batched_fortran(rocblas_handle          handle,
                                                      int                     n,
                                                      const double*           alpha,
                                                      rocblas_double_complex* x,
                                                      int                     incx,
                                                      rocblas_stride          stride_x,
                                                      int                     batch_count);

// copy
rocblas_status rocblas_scopy_fortran(rocblas_handle handle,
                                     rocblas_int    n,
                                     const float*   x,
                                     rocblas_int    incx,
                                     float*         y,
                                     rocblas_int    incy);

rocblas_status rocblas_dcopy_fortran(rocblas_handle handle,
                                     rocblas_int    n,
                                     const double*  x,
                                     rocblas_int    incx,
                                     double*        y,
                                     rocblas_int    incy);

rocblas_status rocblas_ccopy_fortran(rocblas_handle               handle,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zcopy_fortran(rocblas_handle                handle,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// copy_batched
rocblas_status rocblas_scopy_batched_fortran(rocblas_handle     handle,
                                             rocblas_int        n,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dcopy_batched_fortran(rocblas_handle      handle,
                                             rocblas_int         n,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ccopy_batched_fortran(rocblas_handle                     handle,
                                             rocblas_int                        n,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zcopy_batched_fortran(rocblas_handle                      handle,
                                             rocblas_int                         n,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// copy_strided_batched
rocblas_status rocblas_scopy_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     float*         y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dcopy_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     double*        y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_ccopy_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zcopy_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_int                   batch_count);

// dot
rocblas_status rocblas_sdot_fortran(rocblas_handle handle,
                                    rocblas_int    n,
                                    const float*   x,
                                    rocblas_int    incx,
                                    const float*   y,
                                    rocblas_int    incy,
                                    float*         result);

rocblas_status rocblas_ddot_fortran(rocblas_handle handle,
                                    rocblas_int    n,
                                    const double*  x,
                                    rocblas_int    incx,
                                    const double*  y,
                                    rocblas_int    incy,
                                    double*        result);

rocblas_status rocblas_hdot_fortran(rocblas_handle      handle,
                                    rocblas_int         n,
                                    const rocblas_half* x,
                                    rocblas_int         incx,
                                    const rocblas_half* y,
                                    rocblas_int         incy,
                                    rocblas_half*       result);

rocblas_status rocblas_bfdot_fortran(rocblas_handle          handle,
                                     rocblas_int             n,
                                     const rocblas_bfloat16* x,
                                     rocblas_int             incx,
                                     const rocblas_bfloat16* y,
                                     rocblas_int             incy,
                                     rocblas_bfloat16*       result);

rocblas_status rocblas_cdotu_fortran(rocblas_handle               handle,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       result);

rocblas_status rocblas_zdotu_fortran(rocblas_handle                handle,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       result);

rocblas_status rocblas_cdotc_fortran(rocblas_handle               handle,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       result);

rocblas_status rocblas_zdotc_fortran(rocblas_handle                handle,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       result);

// dot_batched
rocblas_status rocblas_sdot_batched_fortran(rocblas_handle     handle,
                                            rocblas_int        n,
                                            const float* const x[],
                                            rocblas_int        incx,
                                            const float* const y[],
                                            rocblas_int        incy,
                                            rocblas_int        batch_count,
                                            float*             result);

rocblas_status rocblas_ddot_batched_fortran(rocblas_handle      handle,
                                            rocblas_int         n,
                                            const double* const x[],
                                            rocblas_int         incx,
                                            const double* const y[],
                                            rocblas_int         incy,
                                            rocblas_int         batch_count,
                                            double*             result);

rocblas_status rocblas_hdot_batched_fortran(rocblas_handle            handle,
                                            rocblas_int               n,
                                            const rocblas_half* const x[],
                                            rocblas_int               incx,
                                            const rocblas_half* const y[],
                                            rocblas_int               incy,
                                            rocblas_int               batch_count,
                                            rocblas_half*             result);

rocblas_status rocblas_bfdot_batched_fortran(rocblas_handle                handle,
                                             rocblas_int                   n,
                                             const rocblas_bfloat16* const x[],
                                             rocblas_int                   incx,
                                             const rocblas_bfloat16* const y[],
                                             rocblas_int                   incy,
                                             rocblas_int                   batch_count,
                                             rocblas_bfloat16*             result);

rocblas_status rocblas_cdotu_batched_fortran(rocblas_handle                     handle,
                                             rocblas_int                        n,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count,
                                             rocblas_float_complex*             result);

rocblas_status rocblas_zdotu_batched_fortran(rocblas_handle                      handle,
                                             rocblas_int                         n,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count,
                                             rocblas_double_complex*             result);

rocblas_status rocblas_cdotc_batched_fortran(rocblas_handle                     handle,
                                             rocblas_int                        n,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count,
                                             rocblas_float_complex*             result);

rocblas_status rocblas_zdotc_batched_fortran(rocblas_handle                      handle,
                                             rocblas_int                         n,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count,
                                             rocblas_double_complex*             result);

// dot_strided_batched
rocblas_status rocblas_sdot_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_int    n,
                                                    const float*   x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stridex,
                                                    const float*   y,
                                                    rocblas_int    incy,
                                                    rocblas_stride stridey,
                                                    rocblas_int    batch_count,
                                                    float*         result);

rocblas_status rocblas_ddot_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_int    n,
                                                    const double*  x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stridex,
                                                    const double*  y,
                                                    rocblas_int    incy,
                                                    rocblas_stride stridey,
                                                    rocblas_int    batch_count,
                                                    double*        result);

rocblas_status rocblas_hdot_strided_batched_fortran(rocblas_handle      handle,
                                                    rocblas_int         n,
                                                    const rocblas_half* x,
                                                    rocblas_int         incx,
                                                    rocblas_stride      stridex,
                                                    const rocblas_half* y,
                                                    rocblas_int         incy,
                                                    rocblas_stride      stridey,
                                                    rocblas_int         batch_count,
                                                    rocblas_half*       result);

rocblas_status rocblas_bfdot_strided_batched_fortran(rocblas_handle          handle,
                                                     rocblas_int             n,
                                                     const rocblas_bfloat16* x,
                                                     rocblas_int             incx,
                                                     rocblas_stride          stridex,
                                                     const rocblas_bfloat16* y,
                                                     rocblas_int             incy,
                                                     rocblas_stride          stridey,
                                                     rocblas_int             batch_count,
                                                     rocblas_bfloat16*       result);

rocblas_status rocblas_cdotu_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_int                  batch_count,
                                                     rocblas_float_complex*       result);

rocblas_status rocblas_zdotu_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_int                   batch_count,
                                                     rocblas_double_complex*       result);

rocblas_status rocblas_cdotc_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_int                  batch_count,
                                                     rocblas_float_complex*       result);

rocblas_status rocblas_zdotc_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_int                   batch_count,
                                                     rocblas_double_complex*       result);

// swap
rocblas_status rocblas_sswap_fortran(
    rocblas_handle handle, rocblas_int n, float* x, rocblas_int incx, float* y, rocblas_int incy);

rocblas_status rocblas_dswap_fortran(
    rocblas_handle handle, rocblas_int n, double* x, rocblas_int incx, double* y, rocblas_int incy);

rocblas_status rocblas_cswap_fortran(rocblas_handle         handle,
                                     rocblas_int            n,
                                     rocblas_float_complex* x,
                                     rocblas_int            incx,
                                     rocblas_float_complex* y,
                                     rocblas_int            incy);

rocblas_status rocblas_zswap_fortran(rocblas_handle          handle,
                                     rocblas_int             n,
                                     rocblas_double_complex* x,
                                     rocblas_int             incx,
                                     rocblas_double_complex* y,
                                     rocblas_int             incy);

// swap_batched
rocblas_status rocblas_sswap_batched_fortran(rocblas_handle handle,
                                             rocblas_int    n,
                                             float* const   x[],
                                             rocblas_int    incx,
                                             float* const   y[],
                                             rocblas_int    incy,
                                             rocblas_int    batch_count);

rocblas_status rocblas_dswap_batched_fortran(rocblas_handle handle,
                                             rocblas_int    n,
                                             double* const  x[],
                                             rocblas_int    incx,
                                             double* const  y[],
                                             rocblas_int    incy,
                                             rocblas_int    batch_count);

rocblas_status rocblas_cswap_batched_fortran(rocblas_handle               handle,
                                             rocblas_int                  n,
                                             rocblas_float_complex* const x[],
                                             rocblas_int                  incx,
                                             rocblas_float_complex* const y[],
                                             rocblas_int                  incy,
                                             rocblas_int                  batch_count);

rocblas_status rocblas_zswap_batched_fortran(rocblas_handle                handle,
                                             rocblas_int                   n,
                                             rocblas_double_complex* const x[],
                                             rocblas_int                   incx,
                                             rocblas_double_complex* const y[],
                                             rocblas_int                   incy,
                                             rocblas_int                   batch_count);

// swap_strided_batched
rocblas_status rocblas_sswap_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     float*         x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     float*         y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dswap_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     double*        x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     double*        y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_cswap_strided_batched_fortran(rocblas_handle         handle,
                                                     rocblas_int            n,
                                                     rocblas_float_complex* x,
                                                     rocblas_int            incx,
                                                     rocblas_stride         stridex,
                                                     rocblas_float_complex* y,
                                                     rocblas_int            incy,
                                                     rocblas_stride         stridey,
                                                     rocblas_int            batch_count);

rocblas_status rocblas_zswap_strided_batched_fortran(rocblas_handle          handle,
                                                     rocblas_int             n,
                                                     rocblas_double_complex* x,
                                                     rocblas_int             incx,
                                                     rocblas_stride          stridex,
                                                     rocblas_double_complex* y,
                                                     rocblas_int             incy,
                                                     rocblas_stride          stridey,
                                                     rocblas_int             batch_count);

// axpy
rocblas_status rocblas_haxpy_fortran(rocblas_handle      handle,
                                     const int           N,
                                     const rocblas_half* alpha,
                                     const rocblas_half* x,
                                     const int           incx,
                                     rocblas_half*       y,
                                     const int           incy);

rocblas_status rocblas_saxpy_fortran(rocblas_handle handle,
                                     const int      N,
                                     const float*   alpha,
                                     const float*   x,
                                     const int      incx,
                                     float*         y,
                                     const int      incy);

rocblas_status rocblas_daxpy_fortran(rocblas_handle handle,
                                     const int      N,
                                     const double*  alpha,
                                     const double*  x,
                                     const int      incx,
                                     double*        y,
                                     const int      incy);

rocblas_status rocblas_caxpy_fortran(rocblas_handle               handle,
                                     const int                    N,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* x,
                                     const int                    incx,
                                     rocblas_float_complex*       y,
                                     const int                    incy);

rocblas_status rocblas_zaxpy_fortran(rocblas_handle                handle,
                                     const int                     N,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* x,
                                     const int                     incx,
                                     rocblas_double_complex*       y,
                                     const int                     incy);

// axpy_batched
rocblas_status rocblas_haxpy_batched_fortran(rocblas_handle            handle,
                                             const int                 N,
                                             const rocblas_half*       alpha,
                                             const rocblas_half* const x[],
                                             const int                 incx,
                                             rocblas_half* const       y[],
                                             const int                 incy,
                                             const int                 batch_count);

rocblas_status rocblas_saxpy_batched_fortran(rocblas_handle     handle,
                                             const int          N,
                                             const float*       alpha,
                                             const float* const x[],
                                             const int          incx,
                                             float* const       y[],
                                             const int          incy,
                                             const int          batch_count);

rocblas_status rocblas_saxpy_batched_fortran(rocblas_handle     handle,
                                             const int          N,
                                             const float*       alpha,
                                             const float* const x[],
                                             const int          incx,
                                             float* const       y[],
                                             const int          incy,
                                             const int          batch_count);

rocblas_status rocblas_daxpy_batched_fortran(rocblas_handle      handle,
                                             const int           N,
                                             const double*       alpha,
                                             const double* const x[],
                                             const int           incx,
                                             double* const       y[],
                                             const int           incy,
                                             const int           batch_count);

rocblas_status rocblas_caxpy_batched_fortran(rocblas_handle                     handle,
                                             const int                          N,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const x[],
                                             const int                          incx,
                                             rocblas_float_complex* const       y[],
                                             const int                          incy,
                                             const int                          batch_count);

rocblas_status rocblas_zaxpy_batched_fortran(rocblas_handle                      handle,
                                             const int                           N,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const x[],
                                             const int                           incx,
                                             rocblas_double_complex* const       y[],
                                             const int                           incy,
                                             const int                           batch_count);

// axpy_strided_batched
rocblas_status rocblas_haxpy_strided_batched_fortran(rocblas_handle       handle,
                                                     const int            N,
                                                     const rocblas_half*  alpha,
                                                     const rocblas_half*  x,
                                                     const int            incx,
                                                     const rocblas_stride stride_x,
                                                     rocblas_half*        y,
                                                     const int            incy,
                                                     const rocblas_stride stride_y,
                                                     const int            batch_count);

rocblas_status rocblas_saxpy_strided_batched_fortran(rocblas_handle       handle,
                                                     const int            N,
                                                     const float*         alpha,
                                                     const float*         x,
                                                     const int            incx,
                                                     const rocblas_stride stride_x,
                                                     float*               y,
                                                     const int            incy,
                                                     const rocblas_stride stride_y,
                                                     const int            batch_count);

rocblas_status rocblas_daxpy_strided_batched_fortran(rocblas_handle       handle,
                                                     const int            N,
                                                     const double*        alpha,
                                                     const double*        x,
                                                     const int            incx,
                                                     const rocblas_stride stride_x,
                                                     double*              y,
                                                     const int            incy,
                                                     const rocblas_stride stride_y,
                                                     const int            batch_count);

rocblas_status rocblas_caxpy_strided_batched_fortran(rocblas_handle               handle,
                                                     const int                    N,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* x,
                                                     const int                    incx,
                                                     const rocblas_stride         stride_x,
                                                     rocblas_float_complex*       y,
                                                     const int                    incy,
                                                     const rocblas_stride         stride_y,
                                                     const int                    batch_count);

rocblas_status rocblas_zaxpy_strided_batched_fortran(rocblas_handle                handle,
                                                     const int                     N,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* x,
                                                     const int                     incx,
                                                     const rocblas_stride          stride_x,
                                                     rocblas_double_complex*       y,
                                                     const int                     incy,
                                                     const rocblas_stride          stride_y,
                                                     const int                     batch_count);

// asum
rocblas_status rocblas_sasum_fortran(
    rocblas_handle handle, rocblas_int n, const float* x, rocblas_int incx, float* result);

rocblas_status rocblas_dasum_fortran(
    rocblas_handle handle, rocblas_int n, const double* x, rocblas_int incx, double* result);

rocblas_status rocblas_scasum_fortran(rocblas_handle               handle,
                                      rocblas_int                  n,
                                      const rocblas_float_complex* x,
                                      rocblas_int                  incx,
                                      float*                       result);

rocblas_status rocblas_dzasum_fortran(rocblas_handle                handle,
                                      rocblas_int                   n,
                                      const rocblas_double_complex* x,
                                      rocblas_int                   incx,
                                      double*                       result);

// asum_batched
rocblas_status rocblas_sasum_batched_fortran(rocblas_handle     handle,
                                             rocblas_int        n,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             rocblas_int        batch_count,
                                             float*             results);

rocblas_status rocblas_dasum_batched_fortran(rocblas_handle      handle,
                                             rocblas_int         n,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             rocblas_int         batch_count,
                                             double*             results);

rocblas_status rocblas_scasum_batched_fortran(rocblas_handle                     handle,
                                              rocblas_int                        n,
                                              const rocblas_float_complex* const x[],
                                              rocblas_int                        incx,
                                              rocblas_int                        batch_count,
                                              float*                             results);

rocblas_status rocblas_dzasum_batched_fortran(rocblas_handle                      handle,
                                              rocblas_int                         n,
                                              const rocblas_double_complex* const x[],
                                              rocblas_int                         incx,
                                              rocblas_int                         batch_count,
                                              double*                             results);

// asum_strided_batched
rocblas_status rocblas_sasum_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     rocblas_int    batch_count,
                                                     float*         results);

rocblas_status rocblas_dasum_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     rocblas_int    batch_count,
                                                     double*        results);

rocblas_status rocblas_scasum_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_int                  n,
                                                      const rocblas_float_complex* x,
                                                      rocblas_int                  incx,
                                                      rocblas_stride               stridex,
                                                      rocblas_int                  batch_count,
                                                      float*                       results);

rocblas_status rocblas_dzasum_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_int                   n,
                                                      const rocblas_double_complex* x,
                                                      rocblas_int                   incx,
                                                      rocblas_stride                stridex,
                                                      rocblas_int                   batch_count,
                                                      double*                       results);

// nrm2
rocblas_status rocblas_snrm2_fortran(
    rocblas_handle handle, rocblas_int n, const float* x, rocblas_int incx, float* result);

rocblas_status rocblas_dnrm2_fortran(
    rocblas_handle handle, rocblas_int n, const double* x, rocblas_int incx, double* result);

rocblas_status rocblas_scnrm2_fortran(rocblas_handle               handle,
                                      rocblas_int                  n,
                                      const rocblas_float_complex* x,
                                      rocblas_int                  incx,
                                      float*                       result);

rocblas_status rocblas_dznrm2_fortran(rocblas_handle                handle,
                                      rocblas_int                   n,
                                      const rocblas_double_complex* x,
                                      rocblas_int                   incx,
                                      double*                       result);

// nrm2_batched
rocblas_status rocblas_snrm2_batched_fortran(rocblas_handle     handle,
                                             rocblas_int        n,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             rocblas_int        batch_count,
                                             float*             results);

rocblas_status rocblas_dnrm2_batched_fortran(rocblas_handle      handle,
                                             rocblas_int         n,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             rocblas_int         batch_count,
                                             double*             results);

rocblas_status rocblas_scnrm2_batched_fortran(rocblas_handle                     handle,
                                              rocblas_int                        n,
                                              const rocblas_float_complex* const x[],
                                              rocblas_int                        incx,
                                              rocblas_int                        batch_count,
                                              float*                             results);

rocblas_status rocblas_dznrm2_batched_fortran(rocblas_handle                      handle,
                                              rocblas_int                         n,
                                              const rocblas_double_complex* const x[],
                                              rocblas_int                         incx,
                                              rocblas_int                         batch_count,
                                              double*                             results);

// nrm2_strided_batched
rocblas_status rocblas_snrm2_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     rocblas_int    batch_count,
                                                     float*         results);

rocblas_status rocblas_dnrm2_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     rocblas_int    batch_count,
                                                     double*        results);

rocblas_status rocblas_scnrm2_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_int                  n,
                                                      const rocblas_float_complex* x,
                                                      rocblas_int                  incx,
                                                      rocblas_stride               stridex,
                                                      rocblas_int                  batch_count,
                                                      float*                       results);

rocblas_status rocblas_dznrm2_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_int                   n,
                                                      const rocblas_double_complex* x,
                                                      rocblas_int                   incx,
                                                      rocblas_stride                stridex,
                                                      rocblas_int                   batch_count,
                                                      double*                       results);

// amax
rocblas_status rocblas_isamax_fortran(
    rocblas_handle handle, rocblas_int n, const float* x, rocblas_int incx, rocblas_int* result);

rocblas_status rocblas_idamax_fortran(
    rocblas_handle handle, rocblas_int n, const double* x, rocblas_int incx, rocblas_int* result);

rocblas_status rocblas_icamax_fortran(rocblas_handle               handle,
                                      rocblas_int                  n,
                                      const rocblas_float_complex* x,
                                      rocblas_int                  incx,
                                      rocblas_int*                 result);

rocblas_status rocblas_izamax_fortran(rocblas_handle                handle,
                                      rocblas_int                   n,
                                      const rocblas_double_complex* x,
                                      rocblas_int                   incx,
                                      rocblas_int*                  result);

// amax_batched
rocblas_status rocblas_isamax_batched_fortran(rocblas_handle     handle,
                                              rocblas_int        n,
                                              const float* const x[],
                                              rocblas_int        incx,
                                              rocblas_int        batch_count,
                                              rocblas_int*       result);

rocblas_status rocblas_idamax_batched_fortran(rocblas_handle      handle,
                                              rocblas_int         n,
                                              const double* const x[],
                                              rocblas_int         incx,
                                              rocblas_int         batch_count,
                                              rocblas_int*        result);

rocblas_status rocblas_icamax_batched_fortran(rocblas_handle                     handle,
                                              rocblas_int                        n,
                                              const rocblas_float_complex* const x[],
                                              rocblas_int                        incx,
                                              rocblas_int                        batch_count,
                                              rocblas_int*                       result);

rocblas_status rocblas_izamax_batched_fortran(rocblas_handle                      handle,
                                              rocblas_int                         n,
                                              const rocblas_double_complex* const x[],
                                              rocblas_int                         incx,
                                              rocblas_int                         batch_count,
                                              rocblas_int*                        result);

// amax_strided_batched
rocblas_status rocblas_isamax_strided_batched_fortran(rocblas_handle handle,
                                                      rocblas_int    n,
                                                      const float*   x,
                                                      rocblas_int    incx,
                                                      rocblas_stride stridex,
                                                      rocblas_int    batch_count,
                                                      rocblas_int*   result);

rocblas_status rocblas_idamax_strided_batched_fortran(rocblas_handle handle,
                                                      rocblas_int    n,
                                                      const double*  x,
                                                      rocblas_int    incx,
                                                      rocblas_stride stridex,
                                                      rocblas_int    batch_count,
                                                      rocblas_int*   result);

rocblas_status rocblas_icamax_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_int                  n,
                                                      const rocblas_float_complex* x,
                                                      rocblas_int                  incx,
                                                      rocblas_stride               stridex,
                                                      rocblas_int                  batch_count,
                                                      rocblas_int*                 result);

rocblas_status rocblas_izamax_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_int                   n,
                                                      const rocblas_double_complex* x,
                                                      rocblas_int                   incx,
                                                      rocblas_stride                stridex,
                                                      rocblas_int                   batch_count,
                                                      rocblas_int*                  result);

// amin
rocblas_status rocblas_isamin_fortran(
    rocblas_handle handle, rocblas_int n, const float* x, rocblas_int incx, rocblas_int* result);

rocblas_status rocblas_idamin_fortran(
    rocblas_handle handle, rocblas_int n, const double* x, rocblas_int incx, rocblas_int* result);

rocblas_status rocblas_icamin_fortran(rocblas_handle               handle,
                                      rocblas_int                  n,
                                      const rocblas_float_complex* x,
                                      rocblas_int                  incx,
                                      rocblas_int*                 result);

rocblas_status rocblas_izamin_fortran(rocblas_handle                handle,
                                      rocblas_int                   n,
                                      const rocblas_double_complex* x,
                                      rocblas_int                   incx,
                                      rocblas_int*                  result);

// amin_batched
rocblas_status rocblas_isamin_batched_fortran(rocblas_handle     handle,
                                              rocblas_int        n,
                                              const float* const x[],
                                              rocblas_int        incx,
                                              rocblas_int        batch_count,
                                              rocblas_int*       result);

rocblas_status rocblas_idamin_batched_fortran(rocblas_handle      handle,
                                              rocblas_int         n,
                                              const double* const x[],
                                              rocblas_int         incx,
                                              rocblas_int         batch_count,
                                              rocblas_int*        result);

rocblas_status rocblas_icamin_batched_fortran(rocblas_handle                     handle,
                                              rocblas_int                        n,
                                              const rocblas_float_complex* const x[],
                                              rocblas_int                        incx,
                                              rocblas_int                        batch_count,
                                              rocblas_int*                       result);

rocblas_status rocblas_izamin_batched_fortran(rocblas_handle                      handle,
                                              rocblas_int                         n,
                                              const rocblas_double_complex* const x[],
                                              rocblas_int                         incx,
                                              rocblas_int                         batch_count,
                                              rocblas_int*                        result);

// amin_strided_batched
rocblas_status rocblas_isamin_strided_batched_fortran(rocblas_handle handle,
                                                      rocblas_int    n,
                                                      const float*   x,
                                                      rocblas_int    incx,
                                                      rocblas_stride stridex,
                                                      rocblas_int    batch_count,
                                                      rocblas_int*   result);

rocblas_status rocblas_idamin_strided_batched_fortran(rocblas_handle handle,
                                                      rocblas_int    n,
                                                      const double*  x,
                                                      rocblas_int    incx,
                                                      rocblas_stride stridex,
                                                      rocblas_int    batch_count,
                                                      rocblas_int*   result);

rocblas_status rocblas_icamin_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_int                  n,
                                                      const rocblas_float_complex* x,
                                                      rocblas_int                  incx,
                                                      rocblas_stride               stridex,
                                                      rocblas_int                  batch_count,
                                                      rocblas_int*                 result);

rocblas_status rocblas_izamin_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_int                   n,
                                                      const rocblas_double_complex* x,
                                                      rocblas_int                   incx,
                                                      rocblas_stride                stridex,
                                                      rocblas_int                   batch_count,
                                                      rocblas_int*                  result);

// rot
rocblas_status rocblas_srot_fortran(rocblas_handle handle,
                                    rocblas_int    n,
                                    float*         x,
                                    rocblas_int    incx,
                                    float*         y,
                                    rocblas_int    incy,
                                    const float*   c,
                                    const float*   s);

rocblas_status rocblas_drot_fortran(rocblas_handle handle,
                                    rocblas_int    n,
                                    double*        x,
                                    rocblas_int    incx,
                                    double*        y,
                                    rocblas_int    incy,
                                    const double*  c,
                                    const double*  s);

rocblas_status rocblas_crot_fortran(rocblas_handle               handle,
                                    rocblas_int                  n,
                                    rocblas_float_complex*       x,
                                    rocblas_int                  incx,
                                    rocblas_float_complex*       y,
                                    rocblas_int                  incy,
                                    const float*                 c,
                                    const rocblas_float_complex* s);

rocblas_status rocblas_csrot_fortran(rocblas_handle         handle,
                                     rocblas_int            n,
                                     rocblas_float_complex* x,
                                     rocblas_int            incx,
                                     rocblas_float_complex* y,
                                     rocblas_int            incy,
                                     const float*           c,
                                     const float*           s);

rocblas_status rocblas_zrot_fortran(rocblas_handle                handle,
                                    rocblas_int                   n,
                                    rocblas_double_complex*       x,
                                    rocblas_int                   incx,
                                    rocblas_double_complex*       y,
                                    rocblas_int                   incy,
                                    const double*                 c,
                                    const rocblas_double_complex* s);

rocblas_status rocblas_zdrot_fortran(rocblas_handle          handle,
                                     rocblas_int             n,
                                     rocblas_double_complex* x,
                                     rocblas_int             incx,
                                     rocblas_double_complex* y,
                                     rocblas_int             incy,
                                     const double*           c,
                                     const double*           s);

// rot_batched
rocblas_status rocblas_srot_batched_fortran(rocblas_handle handle,
                                            rocblas_int    n,
                                            float* const   x[],
                                            rocblas_int    incx,
                                            float* const   y[],
                                            rocblas_int    incy,
                                            const float*   c,
                                            const float*   s,
                                            rocblas_int    batch_count);

rocblas_status rocblas_drot_batched_fortran(rocblas_handle handle,
                                            rocblas_int    n,
                                            double* const  x[],
                                            rocblas_int    incx,
                                            double* const  y[],
                                            rocblas_int    incy,
                                            const double*  c,
                                            const double*  s,
                                            rocblas_int    batch_count);

rocblas_status rocblas_crot_batched_fortran(rocblas_handle               handle,
                                            rocblas_int                  n,
                                            rocblas_float_complex* const x[],
                                            rocblas_int                  incx,
                                            rocblas_float_complex* const y[],
                                            rocblas_int                  incy,
                                            const float*                 c,
                                            const rocblas_float_complex* s,
                                            rocblas_int                  batch_count);

rocblas_status rocblas_csrot_batched_fortran(rocblas_handle               handle,
                                             rocblas_int                  n,
                                             rocblas_float_complex* const x[],
                                             rocblas_int                  incx,
                                             rocblas_float_complex* const y[],
                                             rocblas_int                  incy,
                                             const float*                 c,
                                             const float*                 s,
                                             rocblas_int                  batch_count);

rocblas_status rocblas_zrot_batched_fortran(rocblas_handle                handle,
                                            rocblas_int                   n,
                                            rocblas_double_complex* const x[],
                                            rocblas_int                   incx,
                                            rocblas_double_complex* const y[],
                                            rocblas_int                   incy,
                                            const double*                 c,
                                            const rocblas_double_complex* s,
                                            rocblas_int                   batch_count);

rocblas_status rocblas_zdrot_batched_fortran(rocblas_handle                handle,
                                             rocblas_int                   n,
                                             rocblas_double_complex* const x[],
                                             rocblas_int                   incx,
                                             rocblas_double_complex* const y[],
                                             rocblas_int                   incy,
                                             const double*                 c,
                                             const double*                 s,
                                             rocblas_int                   batch_count);

// rot_strided_batched
rocblas_status rocblas_srot_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_int    n,
                                                    float*         x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stride_x,
                                                    float*         y,
                                                    rocblas_int    incy,
                                                    rocblas_stride stride_y,
                                                    const float*   c,
                                                    const float*   s,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_drot_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_int    n,
                                                    double*        x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stride_x,
                                                    double*        y,
                                                    rocblas_int    incy,
                                                    rocblas_stride stride_y,
                                                    const double*  c,
                                                    const double*  s,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_crot_strided_batched_fortran(rocblas_handle               handle,
                                                    rocblas_int                  n,
                                                    rocblas_float_complex*       x,
                                                    rocblas_int                  incx,
                                                    rocblas_stride               stride_x,
                                                    rocblas_float_complex*       y,
                                                    rocblas_int                  incy,
                                                    rocblas_stride               stride_y,
                                                    const float*                 c,
                                                    const rocblas_float_complex* s,
                                                    rocblas_int                  batch_count);

rocblas_status rocblas_csrot_strided_batched_fortran(rocblas_handle         handle,
                                                     rocblas_int            n,
                                                     rocblas_float_complex* x,
                                                     rocblas_int            incx,
                                                     rocblas_stride         stride_x,
                                                     rocblas_float_complex* y,
                                                     rocblas_int            incy,
                                                     rocblas_stride         stride_y,
                                                     const float*           c,
                                                     const float*           s,
                                                     rocblas_int            batch_count);

rocblas_status rocblas_zrot_strided_batched_fortran(rocblas_handle                handle,
                                                    rocblas_int                   n,
                                                    rocblas_double_complex*       x,
                                                    rocblas_int                   incx,
                                                    rocblas_stride                stride_x,
                                                    rocblas_double_complex*       y,
                                                    rocblas_int                   incy,
                                                    rocblas_stride                stride_y,
                                                    const double*                 c,
                                                    const rocblas_double_complex* s,
                                                    rocblas_int                   batch_count);

rocblas_status rocblas_zdrot_strided_batched_fortran(rocblas_handle          handle,
                                                     rocblas_int             n,
                                                     rocblas_double_complex* x,
                                                     rocblas_int             incx,
                                                     rocblas_stride          stride_x,
                                                     rocblas_double_complex* y,
                                                     rocblas_int             incy,
                                                     rocblas_stride          stride_y,
                                                     const double*           c,
                                                     const double*           s,
                                                     rocblas_int             batch_count);

// rotg
rocblas_status rocblas_srotg_fortran(rocblas_handle handle, float* a, float* b, float* c, float* s);

rocblas_status
    rocblas_drotg_fortran(rocblas_handle handle, double* a, double* b, double* c, double* s);

rocblas_status rocblas_crotg_fortran(rocblas_handle         handle,
                                     rocblas_float_complex* a,
                                     rocblas_float_complex* b,
                                     float*                 c,
                                     rocblas_float_complex* s);

rocblas_status rocblas_zrotg_fortran(rocblas_handle          handle,
                                     rocblas_double_complex* a,
                                     rocblas_double_complex* b,
                                     double*                 c,
                                     rocblas_double_complex* s);

// rotg_batched
rocblas_status rocblas_srotg_batched_fortran(rocblas_handle handle,
                                             float* const   a[],
                                             float* const   b[],
                                             float* const   c[],
                                             float* const   s[],
                                             rocblas_int    batch_count);

rocblas_status rocblas_drotg_batched_fortran(rocblas_handle handle,
                                             double* const  a[],
                                             double* const  b[],
                                             double* const  c[],
                                             double* const  s[],
                                             rocblas_int    batch_count);

rocblas_status rocblas_crotg_batched_fortran(rocblas_handle               handle,
                                             rocblas_float_complex* const a[],
                                             rocblas_float_complex* const b[],
                                             float* const                 c[],
                                             rocblas_float_complex* const s[],
                                             rocblas_int                  batch_count);

rocblas_status rocblas_zrotg_batched_fortran(rocblas_handle                handle,
                                             rocblas_double_complex* const a[],
                                             rocblas_double_complex* const b[],
                                             double* const                 c[],
                                             rocblas_double_complex* const s[],
                                             rocblas_int                   batch_count);

// rotg_strided_batched
rocblas_status rocblas_srotg_strided_batched_fortran(rocblas_handle handle,
                                                     float*         a,
                                                     rocblas_stride stride_a,
                                                     float*         b,
                                                     rocblas_stride stride_b,
                                                     float*         c,
                                                     rocblas_stride stride_c,
                                                     float*         s,
                                                     rocblas_stride stride_s,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_drotg_strided_batched_fortran(rocblas_handle handle,
                                                     double*        a,
                                                     rocblas_stride stride_a,
                                                     double*        b,
                                                     rocblas_stride stride_b,
                                                     double*        c,
                                                     rocblas_stride stride_c,
                                                     double*        s,
                                                     rocblas_stride stride_s,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_crotg_strided_batched_fortran(rocblas_handle         handle,
                                                     rocblas_float_complex* a,
                                                     rocblas_stride         stride_a,
                                                     rocblas_float_complex* b,
                                                     rocblas_stride         stride_b,
                                                     float*                 c,
                                                     rocblas_stride         stride_c,
                                                     rocblas_float_complex* s,
                                                     rocblas_stride         stride_s,
                                                     rocblas_int            batch_count);

rocblas_status rocblas_zrotg_strided_batched_fortran(rocblas_handle          handle,
                                                     rocblas_double_complex* a,
                                                     rocblas_stride          stride_a,
                                                     rocblas_double_complex* b,
                                                     rocblas_stride          stride_b,
                                                     double*                 c,
                                                     rocblas_stride          stride_c,
                                                     rocblas_double_complex* s,
                                                     rocblas_stride          stride_s,
                                                     rocblas_int             batch_count);

// rotm
rocblas_status rocblas_srotm_fortran(rocblas_handle handle,
                                     rocblas_int    n,
                                     float*         x,
                                     rocblas_int    incx,
                                     float*         y,
                                     rocblas_int    incy,
                                     const float*   param);

rocblas_status rocblas_drotm_fortran(rocblas_handle handle,
                                     rocblas_int    n,
                                     double*        x,
                                     rocblas_int    incx,
                                     double*        y,
                                     rocblas_int    incy,
                                     const double*  param);

// rotm_batched
rocblas_status rocblas_srotm_batched_fortran(rocblas_handle     handle,
                                             rocblas_int        n,
                                             float* const       x[],
                                             rocblas_int        incx,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             const float* const param[],
                                             rocblas_int        batch_count);

rocblas_status rocblas_drotm_batched_fortran(rocblas_handle      handle,
                                             rocblas_int         n,
                                             double* const       x[],
                                             rocblas_int         incx,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             const double* const param[],
                                             rocblas_int         batch_count);

// rotm_strided_batched
rocblas_status rocblas_srotm_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     float*         x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stride_x,
                                                     float*         y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stride_y,
                                                     const float*   param,
                                                     rocblas_stride stride_param,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_drotm_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_int    n,
                                                     double*        x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stride_x,
                                                     double*        y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stride_y,
                                                     const double*  param,
                                                     rocblas_stride stride_param,
                                                     rocblas_int    batch_count);

// rotmg
rocblas_status rocblas_srotmg_fortran(
    rocblas_handle handle, float* d1, float* d2, float* x1, const float* y1, float* param);

rocblas_status rocblas_drotmg_fortran(
    rocblas_handle handle, double* d1, double* d2, double* x1, const double* y1, double* param);

// rotmg_batched
rocblas_status rocblas_srotmg_batched_fortran(rocblas_handle     handle,
                                              float* const       d1[],
                                              float* const       d2[],
                                              float* const       x1[],
                                              const float* const y1[],
                                              float* const       param[],
                                              rocblas_int        batch_count);

rocblas_status rocblas_drotmg_batched_fortran(rocblas_handle      handle,
                                              double* const       d1[],
                                              double* const       d2[],
                                              double* const       x1[],
                                              const double* const y1[],
                                              double* const       param[],
                                              rocblas_int         batch_count);

// rotmg_strided_batched
rocblas_status rocblas_srotmg_strided_batched_fortran(rocblas_handle handle,
                                                      float*         d1,
                                                      rocblas_stride stride_d1,
                                                      float*         d2,
                                                      rocblas_stride stride_d2,
                                                      float*         x1,
                                                      rocblas_stride stride_x1,
                                                      const float*   y1,
                                                      rocblas_stride stride_y1,
                                                      float*         param,
                                                      rocblas_stride stride_param,
                                                      rocblas_int    batch_count);

rocblas_status rocblas_drotmg_strided_batched_fortran(rocblas_handle handle,
                                                      double*        d1,
                                                      rocblas_stride stride_d1,
                                                      double*        d2,
                                                      rocblas_stride stride_d2,
                                                      double*        x1,
                                                      rocblas_stride stride_x1,
                                                      const double*  y1,
                                                      rocblas_stride stride_y1,
                                                      double*        param,
                                                      rocblas_stride stride_param,
                                                      rocblas_int    batch_count);

/* ==========
 *    L2
 * ========== */

// gbmv
rocblas_status rocblas_sgbmv_fortran(rocblas_handle    handle,
                                     rocblas_operation trans,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     rocblas_int       kl,
                                     rocblas_int       ku,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     const float*      x,
                                     rocblas_int       incx,
                                     const float*      beta,
                                     float*            y,
                                     rocblas_int       incy);

rocblas_status rocblas_dgbmv_fortran(rocblas_handle    handle,
                                     rocblas_operation trans,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     rocblas_int       kl,
                                     rocblas_int       ku,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     const double*     x,
                                     rocblas_int       incx,
                                     const double*     beta,
                                     double*           y,
                                     rocblas_int       incy);

rocblas_status rocblas_cgbmv_fortran(rocblas_handle               handle,
                                     rocblas_operation            trans,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     rocblas_int                  kl,
                                     rocblas_int                  ku,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zgbmv_fortran(rocblas_handle                handle,
                                     rocblas_operation             trans,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     rocblas_int                   kl,
                                     rocblas_int                   ku,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// gbmv_batched
rocblas_status rocblas_sgbmv_batched_fortran(rocblas_handle     handle,
                                             rocblas_operation  trans,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             rocblas_int        kl,
                                             rocblas_int        ku,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float*       beta,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dgbmv_batched_fortran(rocblas_handle      handle,
                                             rocblas_operation   trans,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             rocblas_int         kl,
                                             rocblas_int         ku,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double*       beta,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             rocblas_int         batch_count);

rocblas_status rocblas_cgbmv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_operation                  trans,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             rocblas_int                        kl,
                                             rocblas_int                        ku,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zgbmv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_operation                   trans,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             rocblas_int                         kl,
                                             rocblas_int                         ku,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// gbmv_strided_batched
rocblas_status rocblas_sgbmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation trans,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     rocblas_int       kl,
                                                     rocblas_int       ku,
                                                     const float*      alpha,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     const float*      x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     const float*      beta,
                                                     float*            y,
                                                     rocblas_int       incy,
                                                     rocblas_stride    stride_y,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dgbmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation trans,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     rocblas_int       kl,
                                                     rocblas_int       ku,
                                                     const double*     alpha,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     const double*     x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     const double*     beta,
                                                     double*           y,
                                                     rocblas_int       incy,
                                                     rocblas_stride    stride_y,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_cgbmv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_operation            trans,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     rocblas_int                  kl,
                                                     rocblas_int                  ku,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stride_y,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zgbmv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_operation             trans,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     rocblas_int                   kl,
                                                     rocblas_int                   ku,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stride_y,
                                                     rocblas_int                   batch_count);

// gemv
rocblas_status rocblas_sgemv_fortran(rocblas_handle    handle,
                                     rocblas_operation trans,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     const float*      x,
                                     rocblas_int       incx,
                                     const float*      beta,
                                     float*            y,
                                     rocblas_int       incy);

rocblas_status rocblas_dgemv_fortran(rocblas_handle    handle,
                                     rocblas_operation trans,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     const double*     x,
                                     rocblas_int       incx,
                                     const double*     beta,
                                     double*           y,
                                     rocblas_int       incy);

rocblas_status rocblas_cgemv_fortran(rocblas_handle               handle,
                                     rocblas_operation            trans,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zgemv_fortran(rocblas_handle                handle,
                                     rocblas_operation             trans,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// gemv_batched
rocblas_status rocblas_sgemv_batched_fortran(rocblas_handle     handle,
                                             rocblas_operation  trans,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float*       beta,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dgemv_batched_fortran(rocblas_handle      handle,
                                             rocblas_operation   trans,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double*       beta,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             rocblas_int         batch_count);

rocblas_status rocblas_cgemv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_operation                  trans,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zgemv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_operation                   trans,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// gemv_strided_batched
rocblas_status rocblas_sgemv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation transA,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const float*      alpha,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    strideA,
                                                     const float*      x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stridex,
                                                     const float*      beta,
                                                     float*            y,
                                                     rocblas_int       incy,
                                                     rocblas_stride    stridey,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dgemv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation transA,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const double*     alpha,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    strideA,
                                                     const double*     x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stridex,
                                                     const double*     beta,
                                                     double*           y,
                                                     rocblas_int       incy,
                                                     rocblas_stride    stridey,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_cgemv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_operation            transA,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               strideA,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zgemv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_operation             transA,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                strideA,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_int                   batch_count);

// hbmv
rocblas_status rocblas_chbmv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     rocblas_int                  k,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zhbmv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     rocblas_int                   k,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// hbmv_batched
rocblas_status rocblas_chbmv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             rocblas_int                        k,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zhbmv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             rocblas_int                         k,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// hbmv_strided_batched
rocblas_status rocblas_chbmv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     rocblas_int                  k,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stride_y,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zhbmv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     rocblas_int                   k,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stride_y,
                                                     rocblas_int                   batch_count);

// hemv
rocblas_status rocblas_chemv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zhemv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// hemv_batched
rocblas_status rocblas_chemv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zhemv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// hemv_strided_batched
rocblas_status rocblas_chemv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stride_y,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zhemv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stride_y,
                                                     rocblas_int                   batch_count);

// her
rocblas_status rocblas_cher_fortran(rocblas_handle               handle,
                                    rocblas_fill                 uplo,
                                    rocblas_int                  n,
                                    const float*                 alpha,
                                    const rocblas_float_complex* x,
                                    rocblas_int                  incx,
                                    rocblas_float_complex*       A,
                                    rocblas_int                  lda);

rocblas_status rocblas_zher_fortran(rocblas_handle                handle,
                                    rocblas_fill                  uplo,
                                    rocblas_int                   n,
                                    const double*                 alpha,
                                    const rocblas_double_complex* x,
                                    rocblas_int                   incx,
                                    rocblas_double_complex*       A,
                                    rocblas_int                   lda);

// her_batched
rocblas_status rocblas_cher_batched_fortran(rocblas_handle                     handle,
                                            rocblas_fill                       uplo,
                                            rocblas_int                        n,
                                            const float*                       alpha,
                                            const rocblas_float_complex* const x[],
                                            rocblas_int                        incx,
                                            rocblas_float_complex* const       A[],
                                            rocblas_int                        lda,
                                            rocblas_int                        batch_count);

rocblas_status rocblas_zher_batched_fortran(rocblas_handle                      handle,
                                            rocblas_fill                        uplo,
                                            rocblas_int                         n,
                                            const double*                       alpha,
                                            const rocblas_double_complex* const x[],
                                            rocblas_int                         incx,
                                            rocblas_double_complex* const       A[],
                                            rocblas_int                         lda,
                                            rocblas_int                         batch_count);

// her_strided_batched
rocblas_status rocblas_cher_strided_batched_fortran(rocblas_handle               handle,
                                                    rocblas_fill                 uplo,
                                                    rocblas_int                  n,
                                                    const float*                 alpha,
                                                    const rocblas_float_complex* x,
                                                    rocblas_int                  incx,
                                                    rocblas_stride               stride_x,
                                                    rocblas_float_complex*       A,
                                                    rocblas_int                  lda,
                                                    rocblas_stride               stride_A,
                                                    rocblas_int                  batch_count);

rocblas_status rocblas_zher_strided_batched_fortran(rocblas_handle                handle,
                                                    rocblas_fill                  uplo,
                                                    rocblas_int                   n,
                                                    const double*                 alpha,
                                                    const rocblas_double_complex* x,
                                                    rocblas_int                   incx,
                                                    rocblas_stride                stride_x,
                                                    rocblas_double_complex*       A,
                                                    rocblas_int                   lda,
                                                    rocblas_stride                stride_A,
                                                    rocblas_int                   batch_count);

// her2
rocblas_status rocblas_cher2_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       A,
                                     rocblas_int                  lda);

rocblas_status rocblas_zher2_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       A,
                                     rocblas_int                   lda);

// her2_batched
rocblas_status rocblas_cher2_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_float_complex* const       A[],
                                             rocblas_int                        lda,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zher2_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_double_complex* const       A[],
                                             rocblas_int                         lda,
                                             rocblas_int                         batch_count);

// her2_strided_batched
rocblas_status rocblas_cher2_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stride_y,
                                                     rocblas_float_complex*       A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zher2_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stride_y,
                                                     rocblas_double_complex*       A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     rocblas_int                   batch_count);

// hpmv
rocblas_status rocblas_chpmv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* AP,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zhpmv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* AP,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// hpmv_batched
rocblas_status rocblas_chpmv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const AP[],
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zhpmv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const AP[],
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// hpmv_strided_batched
rocblas_status rocblas_chpmv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* AP,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stride_y,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zhpmv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* AP,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stride_y,
                                                     rocblas_int                   batch_count);

// hpr
rocblas_status rocblas_chpr_fortran(rocblas_handle               handle,
                                    rocblas_fill                 uplo,
                                    rocblas_int                  n,
                                    const float*                 alpha,
                                    const rocblas_float_complex* x,
                                    rocblas_int                  incx,
                                    rocblas_float_complex*       AP);

rocblas_status rocblas_zhpr_fortran(rocblas_handle                handle,
                                    rocblas_fill                  uplo,
                                    rocblas_int                   n,
                                    const double*                 alpha,
                                    const rocblas_double_complex* x,
                                    rocblas_int                   incx,
                                    rocblas_double_complex*       AP);

// hpr_batched
rocblas_status rocblas_chpr_batched_fortran(rocblas_handle                     handle,
                                            rocblas_fill                       uplo,
                                            rocblas_int                        n,
                                            const float*                       alpha,
                                            const rocblas_float_complex* const x[],
                                            rocblas_int                        incx,
                                            rocblas_float_complex* const       AP[],
                                            rocblas_int                        batch_count);

rocblas_status rocblas_zhpr_batched_fortran(rocblas_handle                      handle,
                                            rocblas_fill                        uplo,
                                            rocblas_int                         n,
                                            const double*                       alpha,
                                            const rocblas_double_complex* const x[],
                                            rocblas_int                         incx,
                                            rocblas_double_complex* const       AP[],
                                            rocblas_int                         batch_count);

// hpr_strided_batched
rocblas_status rocblas_chpr_strided_batched_fortran(rocblas_handle               handle,
                                                    rocblas_fill                 uplo,
                                                    rocblas_int                  n,
                                                    const float*                 alpha,
                                                    const rocblas_float_complex* x,
                                                    rocblas_int                  incx,
                                                    rocblas_stride               stride_x,
                                                    rocblas_float_complex*       AP,
                                                    rocblas_stride               stride_A,
                                                    rocblas_int                  batch_count);

rocblas_status rocblas_zhpr_strided_batched_fortran(rocblas_handle                handle,
                                                    rocblas_fill                  uplo,
                                                    rocblas_int                   n,
                                                    const double*                 alpha,
                                                    const rocblas_double_complex* x,
                                                    rocblas_int                   incx,
                                                    rocblas_stride                stride_x,
                                                    rocblas_double_complex*       AP,
                                                    rocblas_stride                stride_A,
                                                    rocblas_int                   batch_count);

// hpr2
rocblas_status rocblas_chpr2_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       AP);

rocblas_status rocblas_zhpr2_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       AP);

// hpr2_batched
rocblas_status rocblas_chpr2_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_float_complex* const       AP[],
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zhpr2_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_double_complex* const       AP[],
                                             rocblas_int                         batch_count);

// hpr2_strided_batched
rocblas_status rocblas_chpr2_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stride_y,
                                                     rocblas_float_complex*       AP,
                                                     rocblas_stride               stride_A,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zhpr2_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stride_y,
                                                     rocblas_double_complex*       AP,
                                                     rocblas_stride                stride_A,
                                                     rocblas_int                   batch_count);

// trmv
rocblas_status rocblas_strmv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     const float*      A,
                                     rocblas_int       lda,
                                     float*            x,
                                     rocblas_int       incx);

rocblas_status rocblas_dtrmv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     const double*     A,
                                     rocblas_int       lda,
                                     double*           x,
                                     rocblas_int       incx);

rocblas_status rocblas_ctrmv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  m,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     rocblas_float_complex*       x,
                                     rocblas_int                  incx);

rocblas_status rocblas_ztrmv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   m,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     rocblas_double_complex*       x,
                                     rocblas_int                   incx);

// trmv_batched
rocblas_status rocblas_strmv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_diagonal    diag,
                                             rocblas_int         m,
                                             const float* const* A,
                                             rocblas_int         lda,
                                             float* const*       x,
                                             rocblas_int         incx,
                                             rocblas_int         batch_count);

rocblas_status rocblas_dtrmv_batched_fortran(rocblas_handle       handle,
                                             rocblas_fill         uplo,
                                             rocblas_operation    transA,
                                             rocblas_diagonal     diag,
                                             rocblas_int          m,
                                             const double* const* A,
                                             rocblas_int          lda,
                                             double* const*       x,
                                             rocblas_int          incx,
                                             rocblas_int          batch_count);

rocblas_status rocblas_ctrmv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         m,
                                             const rocblas_float_complex* const* A,
                                             rocblas_int                         lda,
                                             rocblas_float_complex* const*       x,
                                             rocblas_int                         incx,
                                             rocblas_int                         batch_count);

rocblas_status rocblas_ztrmv_batched_fortran(rocblas_handle                       handle,
                                             rocblas_fill                         uplo,
                                             rocblas_operation                    transA,
                                             rocblas_diagonal                     diag,
                                             rocblas_int                          m,
                                             const rocblas_double_complex* const* A,
                                             rocblas_int                          lda,
                                             rocblas_double_complex* const*       x,
                                             rocblas_int                          incx,
                                             rocblas_int                          batch_count);

// trmv_strided_batched
rocblas_status rocblas_strmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stridea,
                                                     float*            x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stridex,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtrmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stridea,
                                                     double*           x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stridex,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctrmv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  m,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stridea,
                                                     rocblas_float_complex*       x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztrmv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   m,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stridea,
                                                     rocblas_double_complex*       x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     rocblas_int                   batch_count);

// tpmv
rocblas_status rocblas_stpmv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     const float*      A,
                                     float*            x,
                                     rocblas_int       incx);

rocblas_status rocblas_dtpmv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     const double*     A,
                                     double*           x,
                                     rocblas_int       incx);

rocblas_status rocblas_ctpmv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  m,
                                     const rocblas_float_complex* A,
                                     rocblas_float_complex*       x,
                                     rocblas_int                  incx);

rocblas_status rocblas_ztpmv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   m,
                                     const rocblas_double_complex* A,
                                     rocblas_double_complex*       x,
                                     rocblas_int                   incx);

// tpmv_batched
rocblas_status rocblas_stpmv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_diagonal    diag,
                                             rocblas_int         m,
                                             const float* const* A,
                                             float* const*       x,
                                             rocblas_int         incx,
                                             rocblas_int         batch_count);

rocblas_status rocblas_dtpmv_batched_fortran(rocblas_handle       handle,
                                             rocblas_fill         uplo,
                                             rocblas_operation    transA,
                                             rocblas_diagonal     diag,
                                             rocblas_int          m,
                                             const double* const* A,
                                             double* const*       x,
                                             rocblas_int          incx,
                                             rocblas_int          batch_count);

rocblas_status rocblas_ctpmv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         m,
                                             const rocblas_float_complex* const* A,
                                             rocblas_float_complex* const*       x,
                                             rocblas_int                         incx,
                                             rocblas_int                         batch_count);

rocblas_status rocblas_ztpmv_batched_fortran(rocblas_handle                       handle,
                                             rocblas_fill                         uplo,
                                             rocblas_operation                    transA,
                                             rocblas_diagonal                     diag,
                                             rocblas_int                          m,
                                             const rocblas_double_complex* const* A,
                                             rocblas_double_complex* const*       x,
                                             rocblas_int                          incx,
                                             rocblas_int                          batch_count);

// tpmv_strided_batched
rocblas_status rocblas_stpmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     const float*      A,
                                                     rocblas_stride    stridea,
                                                     float*            x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stridex,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtpmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     const double*     A,
                                                     rocblas_stride    stridea,
                                                     double*           x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stridex,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctpmv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  m,
                                                     const rocblas_float_complex* A,
                                                     rocblas_stride               stridea,
                                                     rocblas_float_complex*       x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztpmv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   m,
                                                     const rocblas_double_complex* A,
                                                     rocblas_stride                stridea,
                                                     rocblas_double_complex*       x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     rocblas_int                   batch_count);

// tbmv
rocblas_status rocblas_stbmv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation trans,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     rocblas_int       k,
                                     const float*      A,
                                     rocblas_int       lda,
                                     float*            x,
                                     rocblas_int       incx);

rocblas_status rocblas_dtbmv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation trans,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     rocblas_int       k,
                                     const double*     A,
                                     rocblas_int       lda,
                                     double*           x,
                                     rocblas_int       incx);

rocblas_status rocblas_ctbmv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            trans,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  m,
                                     rocblas_int                  k,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     rocblas_float_complex*       x,
                                     rocblas_int                  incx);

rocblas_status rocblas_ztbmv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             trans,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   m,
                                     rocblas_int                   k,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     rocblas_double_complex*       x,
                                     rocblas_int                   incx);

// tbmv_batched
rocblas_status rocblas_stbmv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_operation  trans,
                                             rocblas_diagonal   diag,
                                             rocblas_int        m,
                                             rocblas_int        k,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             float* const       x[],
                                             rocblas_int        incx,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dtbmv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   trans,
                                             rocblas_diagonal    diag,
                                             rocblas_int         m,
                                             rocblas_int         k,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             double* const       x[],
                                             rocblas_int         incx,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ctbmv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  trans,
                                             rocblas_diagonal                   diag,
                                             rocblas_int                        m,
                                             rocblas_int                        k,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        da,
                                             rocblas_float_complex* const       x[],
                                             rocblas_int                        incx,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_ztbmv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   trans,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         m,
                                             rocblas_int                         k,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             rocblas_double_complex* const       x[],
                                             rocblas_int                         incx,
                                             rocblas_int                         batch_count);

// tbmv_strided_batched
rocblas_status rocblas_stbmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation trans,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     rocblas_int       k,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     float*            x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtbmv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation trans,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     rocblas_int       k,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     double*           x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctbmv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            trans,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  m,
                                                     rocblas_int                  k,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     rocblas_float_complex*       x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztbmv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             trans,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   m,
                                                     rocblas_int                   k,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     rocblas_double_complex*       x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     rocblas_int                   batch_count);

// tbsv
rocblas_status rocblas_stbsv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       n,
                                     rocblas_int       k,
                                     const float*      A,
                                     rocblas_int       lda,
                                     float*            x,
                                     rocblas_int       incx);

rocblas_status rocblas_dtbsv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       n,
                                     rocblas_int       k,
                                     const double*     A,
                                     rocblas_int       lda,
                                     double*           x,
                                     rocblas_int       incx);

rocblas_status rocblas_ctbsv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  n,
                                     rocblas_int                  k,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     rocblas_float_complex*       x,
                                     rocblas_int                  incx);

rocblas_status rocblas_ztbsv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   n,
                                     rocblas_int                   k,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     rocblas_double_complex*       x,
                                     rocblas_int                   incx);

// tbsv_batched
rocblas_status rocblas_stbsv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_operation  transA,
                                             rocblas_diagonal   diag,
                                             rocblas_int        n,
                                             rocblas_int        k,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             float* const       x[],
                                             rocblas_int        incx,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dtbsv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_diagonal    diag,
                                             rocblas_int         n,
                                             rocblas_int         k,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             double* const       x[],
                                             rocblas_int         incx,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ctbsv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transA,
                                             rocblas_diagonal                   diag,
                                             rocblas_int                        n,
                                             rocblas_int                        k,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             rocblas_float_complex* const       x[],
                                             rocblas_int                        incx,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_ztbsv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         n,
                                             rocblas_int                         k,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             rocblas_double_complex* const       x[],
                                             rocblas_int                         incx,
                                             rocblas_int                         batch_count);

// tbsv_strided_batched
rocblas_status rocblas_stbsv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       n,
                                                     rocblas_int       k,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     float*            x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtbsv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       n,
                                                     rocblas_int       k,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     double*           x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctbsv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  n,
                                                     rocblas_int                  k,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     rocblas_float_complex*       x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztbsv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   n,
                                                     rocblas_int                   k,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     rocblas_double_complex*       x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     rocblas_int                   batch_count);

// trsv
rocblas_status rocblas_strsv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     const float*      A,
                                     rocblas_int       lda,
                                     float*            x,
                                     rocblas_int       incx);

rocblas_status rocblas_dtrsv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     const double*     A,
                                     rocblas_int       lda,
                                     double*           x,
                                     rocblas_int       incx);

rocblas_status rocblas_ctrsv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  m,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     rocblas_float_complex*       x,
                                     rocblas_int                  incx);

rocblas_status rocblas_ztrsv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   m,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     rocblas_double_complex*       x,
                                     rocblas_int                   incx);

// trsv_batched
rocblas_status rocblas_strsv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_operation  transA,
                                             rocblas_diagonal   diag,
                                             rocblas_int        m,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             float* const       x[],
                                             rocblas_int        incx,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dtrsv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_diagonal    diag,
                                             rocblas_int         m,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             double* const       x[],
                                             rocblas_int         incx,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ctrsv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transA,
                                             rocblas_diagonal                   diag,
                                             rocblas_int                        m,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             rocblas_float_complex* const       x[],
                                             rocblas_int                        incx,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_ztrsv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         m,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             rocblas_double_complex* const       x[],
                                             rocblas_int                         incx,
                                             rocblas_int                         batch_count);

// trsv_strided_batched
rocblas_status rocblas_strsv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     float*            x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtrsv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     double*           x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctrsv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  m,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     rocblas_float_complex*       x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztrsv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   m,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     rocblas_double_complex*       x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     rocblas_int                   batch_count);

// tpsv
rocblas_status rocblas_stpsv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       n,
                                     const float*      AP,
                                     float*            x,
                                     rocblas_int       incx);

rocblas_status rocblas_dtpsv_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       n,
                                     const double*     AP,
                                     double*           x,
                                     rocblas_int       incx);

rocblas_status rocblas_ctpsv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* AP,
                                     rocblas_float_complex*       x,
                                     rocblas_int                  incx);

rocblas_status rocblas_ztpsv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* AP,
                                     rocblas_double_complex*       x,
                                     rocblas_int                   incx);

// tpsv_batched
rocblas_status rocblas_stpsv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_operation  transA,
                                             rocblas_diagonal   diag,
                                             rocblas_int        n,
                                             const float* const AP[],
                                             float* const       x[],
                                             rocblas_int        incx,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dtpsv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_diagonal    diag,
                                             rocblas_int         n,
                                             const double* const AP[],
                                             double* const       x[],
                                             rocblas_int         incx,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ctpsv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transA,
                                             rocblas_diagonal                   diag,
                                             rocblas_int                        n,
                                             const rocblas_float_complex* const AP[],
                                             rocblas_float_complex* const       x[],
                                             rocblas_int                        incx,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_ztpsv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         n,
                                             const rocblas_double_complex* const AP[],
                                             rocblas_double_complex* const       x[],
                                             rocblas_int                         incx,
                                             rocblas_int                         batch_count);

// tpsv_strided_batched
rocblas_status rocblas_stpsv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       n,
                                                     const float*      AP,
                                                     rocblas_stride    stride_A,
                                                     float*            x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtpsv_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       n,
                                                     const double*     AP,
                                                     rocblas_stride    stride_A,
                                                     double*           x,
                                                     rocblas_int       incx,
                                                     rocblas_stride    stride_x,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctpsv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* AP,
                                                     rocblas_stride               stride_A,
                                                     rocblas_float_complex*       x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztpsv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* AP,
                                                     rocblas_stride                stride_A,
                                                     rocblas_double_complex*       x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     rocblas_int                   batch_count);

// symv
rocblas_status rocblas_ssymv_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const float*   alpha,
                                     const float*   A,
                                     rocblas_int    lda,
                                     const float*   x,
                                     rocblas_int    incx,
                                     const float*   beta,
                                     float*         y,
                                     rocblas_int    incy);

rocblas_status rocblas_dsymv_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const double*  alpha,
                                     const double*  A,
                                     rocblas_int    lda,
                                     const double*  x,
                                     rocblas_int    incx,
                                     const double*  beta,
                                     double*        y,
                                     rocblas_int    incy);

rocblas_status rocblas_csymv_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       y,
                                     rocblas_int                  incy);

rocblas_status rocblas_zsymv_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       y,
                                     rocblas_int                   incy);

// symv_batched
rocblas_status rocblas_ssymv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float*       beta,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dsymv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double*       beta,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             rocblas_int         batch_count);

rocblas_status rocblas_csymv_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       y[],
                                             rocblas_int                        incy,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zsymv_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       y[],
                                             rocblas_int                         incy,
                                             rocblas_int                         batch_count);

// symv_strided_batched
rocblas_status rocblas_ssymv_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const float*   alpha,
                                                     const float*   A,
                                                     rocblas_int    lda,
                                                     rocblas_stride strideA,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const float*   beta,
                                                     float*         y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dsymv_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const double*  alpha,
                                                     const double*  A,
                                                     rocblas_int    lda,
                                                     rocblas_stride strideA,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const double*  beta,
                                                     double*        y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_csymv_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               strideA,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zsymv_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                strideA,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_int                   batch_count);

// spmv
rocblas_status rocblas_sspmv_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const float*   alpha,
                                     const float*   A,
                                     const float*   x,
                                     rocblas_int    incx,
                                     const float*   beta,
                                     float*         y,
                                     rocblas_int    incy);

rocblas_status rocblas_dspmv_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const double*  alpha,
                                     const double*  A,
                                     const double*  x,
                                     rocblas_int    incx,
                                     const double*  beta,
                                     double*        y,
                                     rocblas_int    incy);

// spmv_batched
rocblas_status rocblas_sspmv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const A[],
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float*       beta,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dspmv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const A[],
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double*       beta,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             rocblas_int         batch_count);

// spmv_strided_batched
rocblas_status rocblas_sspmv_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const float*   alpha,
                                                     const float*   A,
                                                     rocblas_stride strideA,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const float*   beta,
                                                     float*         y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dspmv_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const double*  alpha,
                                                     const double*  A,
                                                     rocblas_stride strideA,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const double*  beta,
                                                     double*        y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

// sbmv
rocblas_status rocblas_ssbmv_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     rocblas_int    k,
                                     const float*   alpha,
                                     const float*   A,
                                     rocblas_int    lda,
                                     const float*   x,
                                     rocblas_int    incx,
                                     const float*   beta,
                                     float*         y,
                                     rocblas_int    incy);

rocblas_status rocblas_dsbmv_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     rocblas_int    k,
                                     const double*  alpha,
                                     const double*  A,
                                     rocblas_int    lda,
                                     const double*  x,
                                     rocblas_int    incx,
                                     const double*  beta,
                                     double*        y,
                                     rocblas_int    incy);

// sbmv_batched
rocblas_status rocblas_dsbmv_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_int         n,
                                             rocblas_int         k,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double*       beta,
                                             double* const       y[],
                                             rocblas_int         incy,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ssbmv_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_int        n,
                                             rocblas_int        k,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float*       beta,
                                             float* const       y[],
                                             rocblas_int        incy,
                                             rocblas_int        batch_count);

// sbmv_strided_batched
rocblas_status rocblas_ssbmv_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     rocblas_int    k,
                                                     const float*   alpha,
                                                     const float*   A,
                                                     rocblas_int    lda,
                                                     rocblas_stride strideA,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const float*   beta,
                                                     float*         y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dsbmv_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     rocblas_int    k,
                                                     const double*  alpha,
                                                     const double*  A,
                                                     rocblas_int    lda,
                                                     rocblas_stride strideA,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const double*  beta,
                                                     double*        y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     rocblas_int    batch_count);

// ger
rocblas_status rocblas_sger_fortran(rocblas_handle handle,
                                    rocblas_int    m,
                                    rocblas_int    n,
                                    const float*   alpha,
                                    const float*   x,
                                    rocblas_int    incx,
                                    const float*   y,
                                    rocblas_int    incy,
                                    float*         A,
                                    rocblas_int    lda);

rocblas_status rocblas_dger_fortran(rocblas_handle handle,
                                    rocblas_int    m,
                                    rocblas_int    n,
                                    const double*  alpha,
                                    const double*  x,
                                    rocblas_int    incx,
                                    const double*  y,
                                    rocblas_int    incy,
                                    double*        A,
                                    rocblas_int    lda);

rocblas_status rocblas_cgeru_fortran(rocblas_handle               handle,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       A,
                                     rocblas_int                  lda);

rocblas_status rocblas_zgeru_fortran(rocblas_handle                handle,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       A,
                                     rocblas_int                   lda);

rocblas_status rocblas_cgerc_fortran(rocblas_handle               handle,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       A,
                                     rocblas_int                  lda);

rocblas_status rocblas_zgerc_fortran(rocblas_handle                handle,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       A,
                                     rocblas_int                   lda);

// ger_batched
rocblas_status rocblas_sger_batched_fortran(rocblas_handle     handle,
                                            rocblas_int        m,
                                            rocblas_int        n,
                                            const float*       alpha,
                                            const float* const x[],
                                            rocblas_int        incx,
                                            const float* const y[],
                                            rocblas_int        incy,
                                            float* const       A[],
                                            rocblas_int        lda,
                                            rocblas_int        batch_count);

rocblas_status rocblas_dger_batched_fortran(rocblas_handle      handle,
                                            rocblas_int         m,
                                            rocblas_int         n,
                                            const double*       alpha,
                                            const double* const x[],
                                            rocblas_int         incx,
                                            const double* const y[],
                                            rocblas_int         incy,
                                            double* const       A[],
                                            rocblas_int         lda,
                                            rocblas_int         batch_count);

rocblas_status rocblas_cgeru_batched_fortran(rocblas_handle                     handle,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_float_complex* const       A[],
                                             rocblas_int                        lda,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zgeru_batched_fortran(rocblas_handle                      handle,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_double_complex* const       A[],
                                             rocblas_int                         lda,
                                             rocblas_int                         batch_count);

rocblas_status rocblas_cgerc_batched_fortran(rocblas_handle                     handle,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_float_complex* const       A[],
                                             rocblas_int                        lda,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zgerc_batched_fortran(rocblas_handle                      handle,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_double_complex* const       A[],
                                             rocblas_int                         lda,
                                             rocblas_int                         batch_count);

// ger_strided_batched
rocblas_status rocblas_sger_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_int    m,
                                                    rocblas_int    n,
                                                    const float*   alpha,
                                                    const float*   x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stridex,
                                                    const float*   y,
                                                    rocblas_int    incy,
                                                    rocblas_stride stridey,
                                                    float*         A,
                                                    rocblas_int    lda,
                                                    rocblas_stride strideA,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_dger_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_int    m,
                                                    rocblas_int    n,
                                                    const double*  alpha,
                                                    const double*  x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stridex,
                                                    const double*  y,
                                                    rocblas_int    incy,
                                                    rocblas_stride stridey,
                                                    double*        A,
                                                    rocblas_int    lda,
                                                    rocblas_stride strideA,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_cgeru_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_float_complex*       A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               strideA,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zgeru_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_double_complex*       A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                strideA,
                                                     rocblas_int                   batch_count);

rocblas_status rocblas_cgerc_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_float_complex*       A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               strideA,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zgerc_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_double_complex*       A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                strideA,
                                                     rocblas_int                   batch_count);

// spr
rocblas_status rocblas_sspr_fortran(rocblas_handle handle,
                                    rocblas_fill   uplo,
                                    rocblas_int    n,
                                    const float*   alpha,
                                    const float*   x,
                                    rocblas_int    incx,
                                    float*         AP);

rocblas_status rocblas_dspr_fortran(rocblas_handle handle,
                                    rocblas_fill   uplo,
                                    rocblas_int    n,
                                    const double*  alpha,
                                    const double*  x,
                                    rocblas_int    incx,
                                    double*        AP);

rocblas_status rocblas_cspr_fortran(rocblas_handle               handle,
                                    rocblas_fill                 uplo,
                                    rocblas_int                  n,
                                    const rocblas_float_complex* alpha,
                                    const rocblas_float_complex* x,
                                    rocblas_int                  incx,
                                    rocblas_float_complex*       AP);

rocblas_status rocblas_zspr_fortran(rocblas_handle                handle,
                                    rocblas_fill                  uplo,
                                    rocblas_int                   n,
                                    const rocblas_double_complex* alpha,
                                    const rocblas_double_complex* x,
                                    rocblas_int                   incx,
                                    rocblas_double_complex*       AP);

// spr_batched
rocblas_status rocblas_sspr_batched_fortran(rocblas_handle     handle,
                                            rocblas_fill       uplo,
                                            rocblas_int        n,
                                            const float*       alpha,
                                            const float* const x[],
                                            rocblas_int        incx,
                                            float* const       AP[],
                                            rocblas_int        batch_count);

rocblas_status rocblas_dspr_batched_fortran(rocblas_handle      handle,
                                            rocblas_fill        uplo,
                                            rocblas_int         n,
                                            const double*       alpha,
                                            const double* const x[],
                                            rocblas_int         incx,
                                            double* const       AP[],
                                            rocblas_int         batch_count);

rocblas_status rocblas_cspr_batched_fortran(rocblas_handle                     handle,
                                            rocblas_fill                       uplo,
                                            rocblas_int                        n,
                                            const rocblas_float_complex*       alpha,
                                            const rocblas_float_complex* const x[],
                                            rocblas_int                        incx,
                                            rocblas_float_complex* const       AP[],
                                            rocblas_int                        batch_count);

rocblas_status rocblas_zspr_batched_fortran(rocblas_handle                      handle,
                                            rocblas_fill                        uplo,
                                            rocblas_int                         n,
                                            const rocblas_double_complex*       alpha,
                                            const rocblas_double_complex* const x[],
                                            rocblas_int                         incx,
                                            rocblas_double_complex* const       AP[],
                                            rocblas_int                         batch_count);

// spr_strided_batched
rocblas_status rocblas_sspr_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_fill   uplo,
                                                    rocblas_int    n,
                                                    const float*   alpha,
                                                    const float*   x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stride_x,
                                                    float*         AP,
                                                    rocblas_stride stride_A,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_dspr_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_fill   uplo,
                                                    rocblas_int    n,
                                                    const double*  alpha,
                                                    const double*  x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stride_x,
                                                    double*        AP,
                                                    rocblas_stride stride_A,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_cspr_strided_batched_fortran(rocblas_handle               handle,
                                                    rocblas_fill                 uplo,
                                                    rocblas_int                  n,
                                                    const rocblas_float_complex* alpha,
                                                    const rocblas_float_complex* x,
                                                    rocblas_int                  incx,
                                                    rocblas_stride               stride_x,
                                                    rocblas_float_complex*       AP,
                                                    rocblas_stride               stride_A,
                                                    rocblas_int                  batch_count);

rocblas_status rocblas_zspr_strided_batched_fortran(rocblas_handle                handle,
                                                    rocblas_fill                  uplo,
                                                    rocblas_int                   n,
                                                    const rocblas_double_complex* alpha,
                                                    const rocblas_double_complex* x,
                                                    rocblas_int                   incx,
                                                    rocblas_stride                stride_x,
                                                    rocblas_double_complex*       AP,
                                                    rocblas_stride                stride_A,
                                                    rocblas_int                   batch_count);

// spr2
rocblas_status rocblas_sspr2_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const float*   alpha,
                                     const float*   x,
                                     rocblas_int    incx,
                                     const float*   y,
                                     rocblas_int    incy,
                                     float*         AP);

rocblas_status rocblas_dspr2_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const double*  alpha,
                                     const double*  x,
                                     rocblas_int    incx,
                                     const double*  y,
                                     rocblas_int    incy,
                                     double*        AP);

// spr2_batched
rocblas_status rocblas_sspr2_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float* const y[],
                                             rocblas_int        incy,
                                             float* const       AP[],
                                             rocblas_int        batch_count);

rocblas_status rocblas_dspr2_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double* const y[],
                                             rocblas_int         incy,
                                             double* const       AP[],
                                             rocblas_int         batch_count);

// spr2_strided_batched
rocblas_status rocblas_sspr2_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const float*   alpha,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stride_x,
                                                     const float*   y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stride_y,
                                                     float*         AP,
                                                     rocblas_stride stride_A,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dspr2_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const double*  alpha,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stride_x,
                                                     const double*  y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stride_y,
                                                     double*        AP,
                                                     rocblas_stride stride_A,
                                                     rocblas_int    batch_count);

// syr
rocblas_status rocblas_ssyr_fortran(rocblas_handle handle,
                                    rocblas_fill   uplo,
                                    rocblas_int    n,
                                    const float*   alpha,
                                    const float*   x,
                                    rocblas_int    incx,
                                    float*         A,
                                    rocblas_int    lda);

rocblas_status rocblas_dsyr_fortran(rocblas_handle handle,
                                    rocblas_fill   uplo,
                                    rocblas_int    n,
                                    const double*  alpha,
                                    const double*  x,
                                    rocblas_int    incx,
                                    double*        A,
                                    rocblas_int    lda);

rocblas_status rocblas_csyr_fortran(rocblas_handle               handle,
                                    rocblas_fill                 uplo,
                                    rocblas_int                  n,
                                    const rocblas_float_complex* alpha,
                                    const rocblas_float_complex* x,
                                    rocblas_int                  incx,
                                    rocblas_float_complex*       A,
                                    rocblas_int                  lda);

rocblas_status rocblas_zsyr_fortran(rocblas_handle                handle,
                                    rocblas_fill                  uplo,
                                    rocblas_int                   n,
                                    const rocblas_double_complex* alpha,
                                    const rocblas_double_complex* x,
                                    rocblas_int                   incx,
                                    rocblas_double_complex*       A,
                                    rocblas_int                   lda);

// syr_batched
rocblas_status rocblas_ssyr_batched_fortran(rocblas_handle     handle,
                                            rocblas_fill       uplo,
                                            rocblas_int        n,
                                            const float*       alpha,
                                            const float* const x[],
                                            rocblas_int        incx,
                                            float* const       A[],
                                            rocblas_int        lda,
                                            rocblas_int        batch_count);

rocblas_status rocblas_dsyr_batched_fortran(rocblas_handle      handle,
                                            rocblas_fill        uplo,
                                            rocblas_int         n,
                                            const double*       alpha,
                                            const double* const x[],
                                            rocblas_int         incx,
                                            double* const       A[],
                                            rocblas_int         lda,
                                            rocblas_int         batch_count);

rocblas_status rocblas_csyr_batched_fortran(rocblas_handle                     handle,
                                            rocblas_fill                       uplo,
                                            rocblas_int                        n,
                                            const rocblas_float_complex*       alpha,
                                            const rocblas_float_complex* const x[],
                                            rocblas_int                        incx,
                                            rocblas_float_complex* const       A[],
                                            rocblas_int                        lda,
                                            rocblas_int                        batch_count);

rocblas_status rocblas_zsyr_batched_fortran(rocblas_handle                      handle,
                                            rocblas_fill                        uplo,
                                            rocblas_int                         n,
                                            const rocblas_double_complex*       alpha,
                                            const rocblas_double_complex* const x[],
                                            rocblas_int                         incx,
                                            rocblas_double_complex* const       A[],
                                            rocblas_int                         lda,
                                            rocblas_int                         batch_count);

// syr_strided_batched
rocblas_status rocblas_ssyr_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_fill   uplo,
                                                    rocblas_int    n,
                                                    const float*   alpha,
                                                    const float*   x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stridex,
                                                    float*         A,
                                                    rocblas_int    lda,
                                                    rocblas_stride strideA,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_dsyr_strided_batched_fortran(rocblas_handle handle,
                                                    rocblas_fill   uplo,
                                                    rocblas_int    n,
                                                    const double*  alpha,
                                                    const double*  x,
                                                    rocblas_int    incx,
                                                    rocblas_stride stridex,
                                                    double*        A,
                                                    rocblas_int    lda,
                                                    rocblas_stride strideA,
                                                    rocblas_int    batch_count);

rocblas_status rocblas_csyr_strided_batched_fortran(rocblas_handle               handle,
                                                    rocblas_fill                 uplo,
                                                    rocblas_int                  n,
                                                    const rocblas_float_complex* alpha,
                                                    const rocblas_float_complex* x,
                                                    rocblas_int                  incx,
                                                    rocblas_stride               stridex,
                                                    rocblas_float_complex*       A,
                                                    rocblas_int                  lda,
                                                    rocblas_stride               strideA,
                                                    rocblas_int                  batch_count);

rocblas_status rocblas_zsyr_strided_batched_fortran(rocblas_handle                handle,
                                                    rocblas_fill                  uplo,
                                                    rocblas_int                   n,
                                                    const rocblas_double_complex* alpha,
                                                    const rocblas_double_complex* x,
                                                    rocblas_int                   incx,
                                                    rocblas_stride                stridex,
                                                    rocblas_double_complex*       A,
                                                    rocblas_int                   lda,
                                                    rocblas_stride                strideA,
                                                    rocblas_int                   batch_count);

// syr2
rocblas_status rocblas_ssyr2_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const float*   alpha,
                                     const float*   x,
                                     rocblas_int    incx,
                                     const float*   y,
                                     rocblas_int    incy,
                                     float*         A,
                                     rocblas_int    lda);

rocblas_status rocblas_dsyr2_fortran(rocblas_handle handle,
                                     rocblas_fill   uplo,
                                     rocblas_int    n,
                                     const double*  alpha,
                                     const double*  x,
                                     rocblas_int    incx,
                                     const double*  y,
                                     rocblas_int    incy,
                                     double*        A,
                                     rocblas_int    lda);

rocblas_status rocblas_csyr2_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     const rocblas_float_complex* y,
                                     rocblas_int                  incy,
                                     rocblas_float_complex*       A,
                                     rocblas_int                  lda);

rocblas_status rocblas_zsyr2_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     const rocblas_double_complex* y,
                                     rocblas_int                   incy,
                                     rocblas_double_complex*       A,
                                     rocblas_int                   lda);

// syr2_batched
rocblas_status rocblas_ssyr2_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             const float* const y[],
                                             rocblas_int        incy,
                                             float* const       A[],
                                             rocblas_int        lda,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dsyr2_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             const double* const y[],
                                             rocblas_int         incy,
                                             double* const       A[],
                                             rocblas_int         lda,
                                             rocblas_int         batch_count);

rocblas_status rocblas_csyr2_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             const rocblas_float_complex* const y[],
                                             rocblas_int                        incy,
                                             rocblas_float_complex* const       A[],
                                             rocblas_int                        lda,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zsyr2_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             const rocblas_double_complex* const y[],
                                             rocblas_int                         incy,
                                             rocblas_double_complex* const       A[],
                                             rocblas_int                         lda,
                                             rocblas_int                         batch_count);

// syr2_strided_batched
rocblas_status rocblas_ssyr2_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const float*   alpha,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const float*   y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     float*         A,
                                                     rocblas_int    lda,
                                                     rocblas_stride strideA,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dsyr2_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    n,
                                                     const double*  alpha,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stridex,
                                                     const double*  y,
                                                     rocblas_int    incy,
                                                     rocblas_stride stridey,
                                                     double*        A,
                                                     rocblas_int    lda,
                                                     rocblas_stride strideA,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_csyr2_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stridex,
                                                     const rocblas_float_complex* y,
                                                     rocblas_int                  incy,
                                                     rocblas_stride               stridey,
                                                     rocblas_float_complex*       A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               strideA,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zsyr2_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stridex,
                                                     const rocblas_double_complex* y,
                                                     rocblas_int                   incy,
                                                     rocblas_stride                stridey,
                                                     rocblas_double_complex*       A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                strideA,
                                                     rocblas_int                   batch_count);

/* ==========
 *    L3
 * ========== */

// hemm
rocblas_status rocblas_chemm_fortran(rocblas_handle               handle,
                                     rocblas_side                 side,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  n,
                                     rocblas_int                  k,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* B,
                                     rocblas_int                  ldb,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zhemm_fortran(rocblas_handle                handle,
                                     rocblas_side                  side,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   n,
                                     rocblas_int                   k,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* B,
                                     rocblas_int                   ldb,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// hemm_batched
rocblas_status rocblas_chemm_batched_fortran(rocblas_handle                     handle,
                                             rocblas_side                       side,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const B[],
                                             rocblas_int                        ldb,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zhemm_batched_fortran(rocblas_handle                      handle,
                                             rocblas_side                        side,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const B[],
                                             rocblas_int                         ldb,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// hemm_strided_batched
rocblas_status rocblas_chemm_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_side                 side,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* B,
                                                     rocblas_int                  ldb,
                                                     rocblas_stride               stride_B,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_C,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zhemm_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_side                  side,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* B,
                                                     rocblas_int                   ldb,
                                                     rocblas_stride                stride_B,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_C,
                                                     rocblas_int                   batch_count);

// herk
rocblas_status rocblas_cherk_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_int                  n,
                                     rocblas_int                  k,
                                     const float*                 alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const float*                 beta,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zherk_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_int                   n,
                                     rocblas_int                   k,
                                     const double*                 alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const double*                 beta,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// herk_batched
rocblas_status rocblas_cherk_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transA,
                                             rocblas_int                        n,
                                             rocblas_int                        k,
                                             const float*                       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const float*                       beta,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zherk_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_int                         n,
                                             rocblas_int                         k,
                                             const double*                       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const double*                       beta,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// herk_strided_batched
rocblas_status rocblas_cherk_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_int                  n,
                                                     rocblas_int                  k,
                                                     const float*                 alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const float*                 beta,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_C,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zherk_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_int                   n,
                                                     rocblas_int                   k,
                                                     const double*                 alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const double*                 beta,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_C,
                                                     rocblas_int                   batch_count);

// her2k
rocblas_status rocblas_cher2k_fortran(rocblas_handle               handle,
                                      rocblas_fill                 uplo,
                                      rocblas_operation            trans,
                                      rocblas_int                  n,
                                      rocblas_int                  k,
                                      const rocblas_float_complex* alpha,
                                      const rocblas_float_complex* A,
                                      rocblas_int                  lda,
                                      const rocblas_float_complex* B,
                                      rocblas_int                  ldb,
                                      const float*                 beta,
                                      rocblas_float_complex*       C,
                                      rocblas_int                  ldc);

rocblas_status rocblas_zher2k_fortran(rocblas_handle                handle,
                                      rocblas_fill                  uplo,
                                      rocblas_operation             trans,
                                      rocblas_int                   n,
                                      rocblas_int                   k,
                                      const rocblas_double_complex* alpha,
                                      const rocblas_double_complex* A,
                                      rocblas_int                   lda,
                                      const rocblas_double_complex* B,
                                      rocblas_int                   ldb,
                                      const double*                 beta,
                                      rocblas_double_complex*       C,
                                      rocblas_int                   ldc);

// her2k_batched
rocblas_status rocblas_cher2k_batched_fortran(rocblas_handle                     handle,
                                              rocblas_fill                       uplo,
                                              rocblas_operation                  trans,
                                              rocblas_int                        n,
                                              rocblas_int                        k,
                                              const rocblas_float_complex*       alpha,
                                              const rocblas_float_complex* const A[],
                                              rocblas_int                        lda,
                                              const rocblas_float_complex* const B[],
                                              rocblas_int                        ldb,
                                              const float*                       beta,
                                              rocblas_float_complex* const       C[],
                                              rocblas_int                        ldc,
                                              rocblas_int                        batch_count);

rocblas_status rocblas_zher2k_batched_fortran(rocblas_handle                      handle,
                                              rocblas_fill                        uplo,
                                              rocblas_operation                   trans,
                                              rocblas_int                         n,
                                              rocblas_int                         k,
                                              const rocblas_double_complex*       alpha,
                                              const rocblas_double_complex* const A[],
                                              rocblas_int                         lda,
                                              const rocblas_double_complex* const B[],
                                              rocblas_int                         ldb,
                                              const double*                       beta,
                                              rocblas_double_complex* const       C[],
                                              rocblas_int                         ldc,
                                              rocblas_int                         batch_count);

// her2k_strided_batched
rocblas_status rocblas_cher2k_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_fill                 uplo,
                                                      rocblas_operation            trans,
                                                      rocblas_int                  n,
                                                      rocblas_int                  k,
                                                      const rocblas_float_complex* alpha,
                                                      const rocblas_float_complex* A,
                                                      rocblas_int                  lda,
                                                      rocblas_stride               stride_A,
                                                      const rocblas_float_complex* B,
                                                      rocblas_int                  ldb,
                                                      rocblas_stride               stride_B,
                                                      const float*                 beta,
                                                      rocblas_float_complex*       C,
                                                      rocblas_int                  ldc,
                                                      rocblas_stride               stride_C,
                                                      rocblas_int                  batch_count);

rocblas_status rocblas_zher2k_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_fill                  uplo,
                                                      rocblas_operation             trans,
                                                      rocblas_int                   n,
                                                      rocblas_int                   k,
                                                      const rocblas_double_complex* alpha,
                                                      const rocblas_double_complex* A,
                                                      rocblas_int                   lda,
                                                      rocblas_stride                stride_A,
                                                      const rocblas_double_complex* B,
                                                      rocblas_int                   ldb,
                                                      rocblas_stride                stride_B,
                                                      const double*                 beta,
                                                      rocblas_double_complex*       C,
                                                      rocblas_int                   ldc,
                                                      rocblas_stride                stride_C,
                                                      rocblas_int                   batch_count);

// herkx
rocblas_status rocblas_cherkx_fortran(rocblas_handle               handle,
                                      rocblas_fill                 uplo,
                                      rocblas_operation            trans,
                                      rocblas_int                  n,
                                      rocblas_int                  k,
                                      const rocblas_float_complex* alpha,
                                      const rocblas_float_complex* A,
                                      rocblas_int                  lda,
                                      const rocblas_float_complex* B,
                                      rocblas_int                  ldb,
                                      const float*                 beta,
                                      rocblas_float_complex*       C,
                                      rocblas_int                  ldc);

rocblas_status rocblas_zherkx_fortran(rocblas_handle                handle,
                                      rocblas_fill                  uplo,
                                      rocblas_operation             trans,
                                      rocblas_int                   n,
                                      rocblas_int                   k,
                                      const rocblas_double_complex* alpha,
                                      const rocblas_double_complex* A,
                                      rocblas_int                   lda,
                                      const rocblas_double_complex* B,
                                      rocblas_int                   ldb,
                                      const double*                 beta,
                                      rocblas_double_complex*       C,
                                      rocblas_int                   ldc);

// herkx_batched
rocblas_status rocblas_cherkx_batched_fortran(rocblas_handle                     handle,
                                              rocblas_fill                       uplo,
                                              rocblas_operation                  trans,
                                              rocblas_int                        n,
                                              rocblas_int                        k,
                                              const rocblas_float_complex*       alpha,
                                              const rocblas_float_complex* const A[],
                                              rocblas_int                        lda,
                                              const rocblas_float_complex* const B[],
                                              rocblas_int                        ldb,
                                              const float*                       beta,
                                              rocblas_float_complex* const       C[],
                                              rocblas_int                        ldc,
                                              rocblas_int                        batch_count);

rocblas_status rocblas_zherkx_batched_fortran(rocblas_handle                      handle,
                                              rocblas_fill                        uplo,
                                              rocblas_operation                   trans,
                                              rocblas_int                         n,
                                              rocblas_int                         k,
                                              const rocblas_double_complex*       alpha,
                                              const rocblas_double_complex* const A[],
                                              rocblas_int                         lda,
                                              const rocblas_double_complex* const B[],
                                              rocblas_int                         ldb,
                                              const double*                       beta,
                                              rocblas_double_complex* const       C[],
                                              rocblas_int                         ldc,
                                              rocblas_int                         batch_count);

// herkx_strided_batched
rocblas_status rocblas_cherkx_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_fill                 uplo,
                                                      rocblas_operation            trans,
                                                      rocblas_int                  n,
                                                      rocblas_int                  k,
                                                      const rocblas_float_complex* alpha,
                                                      const rocblas_float_complex* A,
                                                      rocblas_int                  lda,
                                                      rocblas_stride               stride_A,
                                                      const rocblas_float_complex* B,
                                                      rocblas_int                  ldb,
                                                      rocblas_stride               stride_B,
                                                      const float*                 beta,
                                                      rocblas_float_complex*       C,
                                                      rocblas_int                  ldc,
                                                      rocblas_stride               stride_C,
                                                      rocblas_int                  batch_count);

rocblas_status rocblas_zherkx_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_fill                  uplo,
                                                      rocblas_operation             trans,
                                                      rocblas_int                   n,
                                                      rocblas_int                   k,
                                                      const rocblas_double_complex* alpha,
                                                      const rocblas_double_complex* A,
                                                      rocblas_int                   lda,
                                                      rocblas_stride                stride_A,
                                                      const rocblas_double_complex* B,
                                                      rocblas_int                   ldb,
                                                      rocblas_stride                stride_B,
                                                      const double*                 beta,
                                                      rocblas_double_complex*       C,
                                                      rocblas_int                   ldc,
                                                      rocblas_stride                stride_C,
                                                      rocblas_int                   batch_count);

// symm
rocblas_status rocblas_ssymm_fortran(rocblas_handle handle,
                                     rocblas_side   side,
                                     rocblas_fill   uplo,
                                     rocblas_int    m,
                                     rocblas_int    n,
                                     const float*   alpha,
                                     const float*   A,
                                     rocblas_int    lda,
                                     const float*   B,
                                     rocblas_int    ldb,
                                     const float*   beta,
                                     float*         C,
                                     rocblas_int    ldc);

rocblas_status rocblas_dsymm_fortran(rocblas_handle handle,
                                     rocblas_side   side,
                                     rocblas_fill   uplo,
                                     rocblas_int    m,
                                     rocblas_int    n,
                                     const double*  alpha,
                                     const double*  A,
                                     rocblas_int    lda,
                                     const double*  B,
                                     rocblas_int    ldb,
                                     const double*  beta,
                                     double*        C,
                                     rocblas_int    ldc);

rocblas_status rocblas_csymm_fortran(rocblas_handle               handle,
                                     rocblas_side                 side,
                                     rocblas_fill                 uplo,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* B,
                                     rocblas_int                  ldb,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zsymm_fortran(rocblas_handle                handle,
                                     rocblas_side                  side,
                                     rocblas_fill                  uplo,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* B,
                                     rocblas_int                   ldb,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// symm_batched
rocblas_status rocblas_ssymm_batched_fortran(rocblas_handle     handle,
                                             rocblas_side       side,
                                             rocblas_fill       uplo,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const B[],
                                             rocblas_int        ldb,
                                             const float*       beta,
                                             float* const       C[],
                                             rocblas_int        ldc,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dsymm_batched_fortran(rocblas_handle      handle,
                                             rocblas_side        side,
                                             rocblas_fill        uplo,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const B[],
                                             rocblas_int         ldb,
                                             const double*       beta,
                                             double* const       C[],
                                             rocblas_int         ldc,
                                             rocblas_int         batch_count);

rocblas_status rocblas_csymm_batched_fortran(rocblas_handle                     handle,
                                             rocblas_side                       side,
                                             rocblas_fill                       uplo,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const B[],
                                             rocblas_int                        ldb,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zsymm_batched_fortran(rocblas_handle                      handle,
                                             rocblas_side                        side,
                                             rocblas_fill                        uplo,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const B[],
                                             rocblas_int                         ldb,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// symm_strided_batched
rocblas_status rocblas_ssymm_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_side   side,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    m,
                                                     rocblas_int    n,
                                                     const float*   alpha,
                                                     const float*   A,
                                                     rocblas_int    lda,
                                                     rocblas_stride stride_A,
                                                     const float*   B,
                                                     rocblas_int    ldb,
                                                     rocblas_stride stride_B,
                                                     const float*   beta,
                                                     float*         C,
                                                     rocblas_int    ldc,
                                                     rocblas_stride stride_C,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_dsymm_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_side   side,
                                                     rocblas_fill   uplo,
                                                     rocblas_int    m,
                                                     rocblas_int    n,
                                                     const double*  alpha,
                                                     const double*  A,
                                                     rocblas_int    lda,
                                                     rocblas_stride stride_A,
                                                     const double*  B,
                                                     rocblas_int    ldb,
                                                     rocblas_stride stride_B,
                                                     const double*  beta,
                                                     double*        C,
                                                     rocblas_int    ldc,
                                                     rocblas_stride stride_C,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_csymm_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_side                 side,
                                                     rocblas_fill                 uplo,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* B,
                                                     rocblas_int                  ldb,
                                                     rocblas_stride               stride_B,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_C,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zsymm_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_side                  side,
                                                     rocblas_fill                  uplo,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* B,
                                                     rocblas_int                   ldb,
                                                     rocblas_stride                stride_B,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_C,
                                                     rocblas_int                   batch_count);

// syrk
rocblas_status rocblas_ssyrk_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_int       n,
                                     rocblas_int       k,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     const float*      beta,
                                     float*            C,
                                     rocblas_int       ldc);

rocblas_status rocblas_dsyrk_fortran(rocblas_handle    handle,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_int       n,
                                     rocblas_int       k,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     const double*     beta,
                                     double*           C,
                                     rocblas_int       ldc);

rocblas_status rocblas_csyrk_fortran(rocblas_handle               handle,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_int                  n,
                                     rocblas_int                  k,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zsyrk_fortran(rocblas_handle                handle,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_int                   n,
                                     rocblas_int                   k,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// syrk_batched
rocblas_status rocblas_ssyrk_batched_fortran(rocblas_handle     handle,
                                             rocblas_fill       uplo,
                                             rocblas_operation  transA,
                                             rocblas_int        n,
                                             rocblas_int        k,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float*       beta,
                                             float* const       C[],
                                             rocblas_int        ldc,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dsyrk_batched_fortran(rocblas_handle      handle,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_int         n,
                                             rocblas_int         k,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double*       beta,
                                             double* const       C[],
                                             rocblas_int         ldc,
                                             rocblas_int         batch_count);

rocblas_status rocblas_csyrk_batched_fortran(rocblas_handle                     handle,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transA,
                                             rocblas_int                        n,
                                             rocblas_int                        k,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zsyrk_batched_fortran(rocblas_handle                      handle,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_int                         n,
                                             rocblas_int                         k,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// syrk_strided_batched
rocblas_status rocblas_ssyrk_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_int       n,
                                                     rocblas_int       k,
                                                     const float*      alpha,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     const float*      beta,
                                                     float*            C,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_C,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dsyrk_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_int       n,
                                                     rocblas_int       k,
                                                     const double*     alpha,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_A,
                                                     const double*     beta,
                                                     double*           C,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_C,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_csyrk_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_int                  n,
                                                     rocblas_int                  k,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_A,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_C,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zsyrk_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_int                   n,
                                                     rocblas_int                   k,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_A,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_C,
                                                     rocblas_int                   batch_count);

// syr2k
rocblas_status rocblas_ssyr2k_fortran(rocblas_handle    handle,
                                      rocblas_fill      uplo,
                                      rocblas_operation trans,
                                      rocblas_int       n,
                                      rocblas_int       k,
                                      const float*      alpha,
                                      const float*      A,
                                      rocblas_int       lda,
                                      const float*      B,
                                      rocblas_int       ldb,
                                      const float*      beta,
                                      float*            C,
                                      rocblas_int       ldc);

rocblas_status rocblas_dsyr2k_fortran(rocblas_handle    handle,
                                      rocblas_fill      uplo,
                                      rocblas_operation trans,
                                      rocblas_int       n,
                                      rocblas_int       k,
                                      const double*     alpha,
                                      const double*     A,
                                      rocblas_int       lda,
                                      const double*     B,
                                      rocblas_int       ldb,
                                      const double*     beta,
                                      double*           C,
                                      rocblas_int       ldc);

rocblas_status rocblas_csyr2k_fortran(rocblas_handle               handle,
                                      rocblas_fill                 uplo,
                                      rocblas_operation            trans,
                                      rocblas_int                  n,
                                      rocblas_int                  k,
                                      const rocblas_float_complex* alpha,
                                      const rocblas_float_complex* A,
                                      rocblas_int                  lda,
                                      const rocblas_float_complex* B,
                                      rocblas_int                  ldb,
                                      const rocblas_float_complex* beta,
                                      rocblas_float_complex*       C,
                                      rocblas_int                  ldc);

rocblas_status rocblas_zsyr2k_fortran(rocblas_handle                handle,
                                      rocblas_fill                  uplo,
                                      rocblas_operation             trans,
                                      rocblas_int                   n,
                                      rocblas_int                   k,
                                      const rocblas_double_complex* alpha,
                                      const rocblas_double_complex* A,
                                      rocblas_int                   lda,
                                      const rocblas_double_complex* B,
                                      rocblas_int                   ldb,
                                      const rocblas_double_complex* beta,
                                      rocblas_double_complex*       C,
                                      rocblas_int                   ldc);

// syr2k_batched
rocblas_status rocblas_ssyr2k_batched_fortran(rocblas_handle     handle,
                                              rocblas_fill       uplo,
                                              rocblas_operation  trans,
                                              rocblas_int        n,
                                              rocblas_int        k,
                                              const float*       alpha,
                                              const float* const A[],
                                              rocblas_int        lda,
                                              const float* const B[],
                                              rocblas_int        ldb,
                                              const float*       beta,
                                              float* const       C[],
                                              rocblas_int        ldc,
                                              rocblas_int        batch_count);

rocblas_status rocblas_dsyr2k_batched_fortran(rocblas_handle      handle,
                                              rocblas_fill        uplo,
                                              rocblas_operation   trans,
                                              rocblas_int         n,
                                              rocblas_int         k,
                                              const double*       alpha,
                                              const double* const A[],
                                              rocblas_int         lda,
                                              const double* const B[],
                                              rocblas_int         ldb,
                                              const double*       beta,
                                              double* const       C[],
                                              rocblas_int         ldc,
                                              rocblas_int         batch_count);

rocblas_status rocblas_csyr2k_batched_fortran(rocblas_handle                     handle,
                                              rocblas_fill                       uplo,
                                              rocblas_operation                  trans,
                                              rocblas_int                        n,
                                              rocblas_int                        k,
                                              const rocblas_float_complex*       alpha,
                                              const rocblas_float_complex* const A[],
                                              rocblas_int                        lda,
                                              const rocblas_float_complex* const B[],
                                              rocblas_int                        ldb,
                                              const rocblas_float_complex*       beta,
                                              rocblas_float_complex* const       C[],
                                              rocblas_int                        ldc,
                                              rocblas_int                        batch_count);

rocblas_status rocblas_zsyr2k_batched_fortran(rocblas_handle                      handle,
                                              rocblas_fill                        uplo,
                                              rocblas_operation                   trans,
                                              rocblas_int                         n,
                                              rocblas_int                         k,
                                              const rocblas_double_complex*       alpha,
                                              const rocblas_double_complex* const A[],
                                              rocblas_int                         lda,
                                              const rocblas_double_complex* const B[],
                                              rocblas_int                         ldb,
                                              const rocblas_double_complex*       beta,
                                              rocblas_double_complex* const       C[],
                                              rocblas_int                         ldc,
                                              rocblas_int                         batch_count);

// syr2k_strided_batched
rocblas_status rocblas_ssyr2k_strided_batched_fortran(rocblas_handle    handle,
                                                      rocblas_fill      uplo,
                                                      rocblas_operation trans,
                                                      rocblas_int       n,
                                                      rocblas_int       k,
                                                      const float*      alpha,
                                                      const float*      A,
                                                      rocblas_int       lda,
                                                      rocblas_stride    stride_A,
                                                      const float*      B,
                                                      rocblas_int       ldb,
                                                      rocblas_stride    stride_B,
                                                      const float*      beta,
                                                      float*            C,
                                                      rocblas_int       ldc,
                                                      rocblas_stride    stride_C,
                                                      rocblas_int       batch_count);

rocblas_status rocblas_dsyr2k_strided_batched_fortran(rocblas_handle    handle,
                                                      rocblas_fill      uplo,
                                                      rocblas_operation trans,
                                                      rocblas_int       n,
                                                      rocblas_int       k,
                                                      const double*     alpha,
                                                      const double*     A,
                                                      rocblas_int       lda,
                                                      rocblas_stride    stride_A,
                                                      const double*     B,
                                                      rocblas_int       ldb,
                                                      rocblas_stride    stride_B,
                                                      const double*     beta,
                                                      double*           C,
                                                      rocblas_int       ldc,
                                                      rocblas_stride    stride_C,
                                                      rocblas_int       batch_count);

rocblas_status rocblas_csyr2k_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_fill                 uplo,
                                                      rocblas_operation            trans,
                                                      rocblas_int                  n,
                                                      rocblas_int                  k,
                                                      const rocblas_float_complex* alpha,
                                                      const rocblas_float_complex* A,
                                                      rocblas_int                  lda,
                                                      rocblas_stride               stride_A,
                                                      const rocblas_float_complex* B,
                                                      rocblas_int                  ldb,
                                                      rocblas_stride               stride_B,
                                                      const rocblas_float_complex* beta,
                                                      rocblas_float_complex*       C,
                                                      rocblas_int                  ldc,
                                                      rocblas_stride               stride_C,
                                                      rocblas_int                  batch_count);

rocblas_status rocblas_zsyr2k_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_fill                  uplo,
                                                      rocblas_operation             trans,
                                                      rocblas_int                   n,
                                                      rocblas_int                   k,
                                                      const rocblas_double_complex* alpha,
                                                      const rocblas_double_complex* A,
                                                      rocblas_int                   lda,
                                                      rocblas_stride                stride_A,
                                                      const rocblas_double_complex* B,
                                                      rocblas_int                   ldb,
                                                      rocblas_stride                stride_B,
                                                      const rocblas_double_complex* beta,
                                                      rocblas_double_complex*       C,
                                                      rocblas_int                   ldc,
                                                      rocblas_stride                stride_C,
                                                      rocblas_int                   batch_count);

// syrkx
rocblas_status rocblas_ssyrkx_fortran(rocblas_handle    handle,
                                      rocblas_fill      uplo,
                                      rocblas_operation trans,
                                      rocblas_int       n,
                                      rocblas_int       k,
                                      const float*      alpha,
                                      const float*      A,
                                      rocblas_int       lda,
                                      const float*      B,
                                      rocblas_int       ldb,
                                      const float*      beta,
                                      float*            C,
                                      rocblas_int       ldc);

rocblas_status rocblas_dsyrkx_fortran(rocblas_handle    handle,
                                      rocblas_fill      uplo,
                                      rocblas_operation trans,
                                      rocblas_int       n,
                                      rocblas_int       k,
                                      const double*     alpha,
                                      const double*     A,
                                      rocblas_int       lda,
                                      const double*     B,
                                      rocblas_int       ldb,
                                      const double*     beta,
                                      double*           C,
                                      rocblas_int       ldc);

rocblas_status rocblas_csyrkx_fortran(rocblas_handle               handle,
                                      rocblas_fill                 uplo,
                                      rocblas_operation            trans,
                                      rocblas_int                  n,
                                      rocblas_int                  k,
                                      const rocblas_float_complex* alpha,
                                      const rocblas_float_complex* A,
                                      rocblas_int                  lda,
                                      const rocblas_float_complex* B,
                                      rocblas_int                  ldb,
                                      const rocblas_float_complex* beta,
                                      rocblas_float_complex*       C,
                                      rocblas_int                  ldc);

rocblas_status rocblas_zsyrkx_fortran(rocblas_handle                handle,
                                      rocblas_fill                  uplo,
                                      rocblas_operation             trans,
                                      rocblas_int                   n,
                                      rocblas_int                   k,
                                      const rocblas_double_complex* alpha,
                                      const rocblas_double_complex* A,
                                      rocblas_int                   lda,
                                      const rocblas_double_complex* B,
                                      rocblas_int                   ldb,
                                      const rocblas_double_complex* beta,
                                      rocblas_double_complex*       C,
                                      rocblas_int                   ldc);

// syrkx_batched
rocblas_status rocblas_ssyrkx_batched_fortran(rocblas_handle     handle,
                                              rocblas_fill       uplo,
                                              rocblas_operation  trans,
                                              rocblas_int        n,
                                              rocblas_int        k,
                                              const float*       alpha,
                                              const float* const A[],
                                              rocblas_int        lda,
                                              const float* const B[],
                                              rocblas_int        ldb,
                                              const float*       beta,
                                              float* const       C[],
                                              rocblas_int        ldc,
                                              rocblas_int        batch_count);

rocblas_status rocblas_dsyrkx_batched_fortran(rocblas_handle      handle,
                                              rocblas_fill        uplo,
                                              rocblas_operation   trans,
                                              rocblas_int         n,
                                              rocblas_int         k,
                                              const double*       alpha,
                                              const double* const A[],
                                              rocblas_int         lda,
                                              const double* const B[],
                                              rocblas_int         ldb,
                                              const double*       beta,
                                              double* const       C[],
                                              rocblas_int         ldc,
                                              rocblas_int         batch_count);

rocblas_status rocblas_csyrkx_batched_fortran(rocblas_handle                     handle,
                                              rocblas_fill                       uplo,
                                              rocblas_operation                  trans,
                                              rocblas_int                        n,
                                              rocblas_int                        k,
                                              const rocblas_float_complex*       alpha,
                                              const rocblas_float_complex* const A[],
                                              rocblas_int                        lda,
                                              const rocblas_float_complex* const B[],
                                              rocblas_int                        ldb,
                                              const rocblas_float_complex*       beta,
                                              rocblas_float_complex* const       C[],
                                              rocblas_int                        ldc,
                                              rocblas_int                        batch_count);

rocblas_status rocblas_zsyrkx_batched_fortran(rocblas_handle                      handle,
                                              rocblas_fill                        uplo,
                                              rocblas_operation                   trans,
                                              rocblas_int                         n,
                                              rocblas_int                         k,
                                              const rocblas_double_complex*       alpha,
                                              const rocblas_double_complex* const A[],
                                              rocblas_int                         lda,
                                              const rocblas_double_complex* const B[],
                                              rocblas_int                         ldb,
                                              const rocblas_double_complex*       beta,
                                              rocblas_double_complex* const       C[],
                                              rocblas_int                         ldc,
                                              rocblas_int                         batch_count);

// syrkx_strided_batched
rocblas_status rocblas_ssyrkx_strided_batched_fortran(rocblas_handle    handle,
                                                      rocblas_fill      uplo,
                                                      rocblas_operation trans,
                                                      rocblas_int       n,
                                                      rocblas_int       k,
                                                      const float*      alpha,
                                                      const float*      A,
                                                      rocblas_int       lda,
                                                      rocblas_stride    stride_A,
                                                      const float*      B,
                                                      rocblas_int       ldb,
                                                      rocblas_stride    stride_B,
                                                      const float*      beta,
                                                      float*            C,
                                                      rocblas_int       ldc,
                                                      rocblas_stride    stride_C,
                                                      rocblas_int       batch_count);

rocblas_status rocblas_dsyrkx_strided_batched_fortran(rocblas_handle    handle,
                                                      rocblas_fill      uplo,
                                                      rocblas_operation trans,
                                                      rocblas_int       n,
                                                      rocblas_int       k,
                                                      const double*     alpha,
                                                      const double*     A,
                                                      rocblas_int       lda,
                                                      rocblas_stride    stride_A,
                                                      const double*     B,
                                                      rocblas_int       ldb,
                                                      rocblas_stride    stride_B,
                                                      const double*     beta,
                                                      double*           C,
                                                      rocblas_int       ldc,
                                                      rocblas_stride    stride_C,
                                                      rocblas_int       batch_count);

rocblas_status rocblas_csyrkx_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_fill                 uplo,
                                                      rocblas_operation            trans,
                                                      rocblas_int                  n,
                                                      rocblas_int                  k,
                                                      const rocblas_float_complex* alpha,
                                                      const rocblas_float_complex* A,
                                                      rocblas_int                  lda,
                                                      rocblas_stride               stride_A,
                                                      const rocblas_float_complex* B,
                                                      rocblas_int                  ldb,
                                                      rocblas_stride               stride_B,
                                                      const rocblas_float_complex* beta,
                                                      rocblas_float_complex*       C,
                                                      rocblas_int                  ldc,
                                                      rocblas_stride               stride_C,
                                                      rocblas_int                  batch_count);

rocblas_status rocblas_zsyrkx_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_fill                  uplo,
                                                      rocblas_operation             trans,
                                                      rocblas_int                   n,
                                                      rocblas_int                   k,
                                                      const rocblas_double_complex* alpha,
                                                      const rocblas_double_complex* A,
                                                      rocblas_int                   lda,
                                                      rocblas_stride                stride_A,
                                                      const rocblas_double_complex* B,
                                                      rocblas_int                   ldb,
                                                      rocblas_stride                stride_B,
                                                      const rocblas_double_complex* beta,
                                                      rocblas_double_complex*       C,
                                                      rocblas_int                   ldc,
                                                      rocblas_stride                stride_C,
                                                      rocblas_int                   batch_count);

// trmm
rocblas_status rocblas_strmm_fortran(rocblas_handle    handle,
                                     rocblas_side      side,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     float*            B,
                                     rocblas_int       ldb);

rocblas_status rocblas_dtrmm_fortran(rocblas_handle    handle,
                                     rocblas_side      side,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     double*           B,
                                     rocblas_int       ldb);

rocblas_status rocblas_ctrmm_fortran(rocblas_handle               handle,
                                     rocblas_side                 side,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     rocblas_float_complex*       B,
                                     rocblas_int                  ldb);

rocblas_status rocblas_ztrmm_fortran(rocblas_handle                handle,
                                     rocblas_side                  side,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     rocblas_double_complex*       B,
                                     rocblas_int                   ldb);

// trmm_batched
rocblas_status rocblas_strmm_batched_fortran(rocblas_handle     handle,
                                             rocblas_side       side,
                                             rocblas_fill       uplo,
                                             rocblas_operation  transa,
                                             rocblas_diagonal   diag,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const a[],
                                             rocblas_int        lda,
                                             float* const       c[],
                                             rocblas_int        ldc,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dtrmm_batched_fortran(rocblas_handle      handle,
                                             rocblas_side        side,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transa,
                                             rocblas_diagonal    diag,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const a[],
                                             rocblas_int         lda,
                                             double* const       c[],
                                             rocblas_int         ldc,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ctrmm_batched_fortran(rocblas_handle                     handle,
                                             rocblas_side                       side,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transa,
                                             rocblas_diagonal                   diag,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const a[],
                                             rocblas_int                        lda,
                                             rocblas_float_complex* const       c[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_ztrmm_batched_fortran(rocblas_handle                      handle,
                                             rocblas_side                        side,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transa,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const a[],
                                             rocblas_int                         lda,
                                             rocblas_double_complex* const       c[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// trmm_strided_batched
rocblas_status rocblas_strmm_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_side      side,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transa,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const float*      alpha,
                                                     const float*      a,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     float*            c,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_c,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtrmm_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_side      side,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transa,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const double*     alpha,
                                                     const double*     a,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     double*           c,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_c,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctrmm_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_side                 side,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transa,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* a,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_a,
                                                     rocblas_float_complex*       c,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_c,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztrmm_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_side                  side,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transa,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* a,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_a,
                                                     rocblas_double_complex*       c,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_c,
                                                     rocblas_int                   batch_count);

// trmm_outofplace
rocblas_status rocblas_strmm_outofplace_fortran(rocblas_handle    handle,
                                                rocblas_side      side,
                                                rocblas_fill      uplo,
                                                rocblas_operation transA,
                                                rocblas_diagonal  diag,
                                                rocblas_int       m,
                                                rocblas_int       n,
                                                const float*      alpha,
                                                const float*      A,
                                                rocblas_int       lda,
                                                const float*      B,
                                                rocblas_int       ldb,
                                                float*            C,
                                                rocblas_int       ldc);

rocblas_status rocblas_dtrmm_outofplace_fortran(rocblas_handle    handle,
                                                rocblas_side      side,
                                                rocblas_fill      uplo,
                                                rocblas_operation transA,
                                                rocblas_diagonal  diag,
                                                rocblas_int       m,
                                                rocblas_int       n,
                                                const double*     alpha,
                                                const double*     A,
                                                rocblas_int       lda,
                                                const double*     B,
                                                rocblas_int       ldb,
                                                double*           C,
                                                rocblas_int       ldc);

rocblas_status rocblas_ctrmm_outofplace_fortran(rocblas_handle               handle,
                                                rocblas_side                 side,
                                                rocblas_fill                 uplo,
                                                rocblas_operation            transA,
                                                rocblas_diagonal             diag,
                                                rocblas_int                  m,
                                                rocblas_int                  n,
                                                const rocblas_float_complex* alpha,
                                                const rocblas_float_complex* A,
                                                rocblas_int                  lda,
                                                const rocblas_float_complex* B,
                                                rocblas_int                  ldb,
                                                rocblas_float_complex*       C,
                                                rocblas_int                  ldc);

rocblas_status rocblas_ztrmm_outofplace_fortran(rocblas_handle                handle,
                                                rocblas_side                  side,
                                                rocblas_fill                  uplo,
                                                rocblas_operation             transA,
                                                rocblas_diagonal              diag,
                                                rocblas_int                   m,
                                                rocblas_int                   n,
                                                const rocblas_double_complex* alpha,
                                                const rocblas_double_complex* A,
                                                rocblas_int                   lda,
                                                const rocblas_double_complex* B,
                                                rocblas_int                   ldb,
                                                rocblas_double_complex*       C,
                                                rocblas_int                   ldc);

// trmm_outofplace_batched
rocblas_status rocblas_strmm_outofplace_batched_fortran(rocblas_handle     handle,
                                                        rocblas_side       side,
                                                        rocblas_fill       uplo,
                                                        rocblas_operation  transa,
                                                        rocblas_diagonal   diag,
                                                        rocblas_int        m,
                                                        rocblas_int        n,
                                                        const float*       alpha,
                                                        const float* const a[],
                                                        rocblas_int        lda,
                                                        const float* const b[],
                                                        rocblas_int        ldb,
                                                        float* const       c[],
                                                        rocblas_int        ldc,
                                                        rocblas_int        batch_count);

rocblas_status rocblas_dtrmm_outofplace_batched_fortran(rocblas_handle      handle,
                                                        rocblas_side        side,
                                                        rocblas_fill        uplo,
                                                        rocblas_operation   transa,
                                                        rocblas_diagonal    diag,
                                                        rocblas_int         m,
                                                        rocblas_int         n,
                                                        const double*       alpha,
                                                        const double* const a[],
                                                        rocblas_int         lda,
                                                        const double* const b[],
                                                        rocblas_int         ldb,
                                                        double* const       c[],
                                                        rocblas_int         ldc,
                                                        rocblas_int         batch_count);

rocblas_status rocblas_ctrmm_outofplace_batched_fortran(rocblas_handle                     handle,
                                                        rocblas_side                       side,
                                                        rocblas_fill                       uplo,
                                                        rocblas_operation                  transa,
                                                        rocblas_diagonal                   diag,
                                                        rocblas_int                        m,
                                                        rocblas_int                        n,
                                                        const rocblas_float_complex*       alpha,
                                                        const rocblas_float_complex* const a[],
                                                        rocblas_int                        lda,
                                                        const rocblas_float_complex* const b[],
                                                        rocblas_int                        ldb,
                                                        rocblas_float_complex* const       c[],
                                                        rocblas_int                        ldc,
                                                        rocblas_int batch_count);

rocblas_status rocblas_ztrmm_outofplace_batched_fortran(rocblas_handle                      handle,
                                                        rocblas_side                        side,
                                                        rocblas_fill                        uplo,
                                                        rocblas_operation                   transa,
                                                        rocblas_diagonal                    diag,
                                                        rocblas_int                         m,
                                                        rocblas_int                         n,
                                                        const rocblas_double_complex*       alpha,
                                                        const rocblas_double_complex* const a[],
                                                        rocblas_int                         lda,
                                                        const rocblas_double_complex* const b[],
                                                        rocblas_int                         ldb,
                                                        rocblas_double_complex* const       c[],
                                                        rocblas_int                         ldc,
                                                        rocblas_int batch_count);

// trmm_outofplace_strided_batched
rocblas_status rocblas_strmm_outofplace_strided_batched_fortran(rocblas_handle    handle,
                                                                rocblas_side      side,
                                                                rocblas_fill      uplo,
                                                                rocblas_operation transa,
                                                                rocblas_diagonal  diag,
                                                                rocblas_int       m,
                                                                rocblas_int       n,
                                                                const float*      alpha,
                                                                const float*      a,
                                                                rocblas_int       lda,
                                                                rocblas_stride    stride_a,
                                                                const float*      b,
                                                                rocblas_int       ldb,
                                                                rocblas_stride    stride_b,
                                                                float*            c,
                                                                rocblas_int       ldc,
                                                                rocblas_stride    stride_c,
                                                                rocblas_int       batch_count);

rocblas_status rocblas_dtrmm_outofplace_strided_batched_fortran(rocblas_handle    handle,
                                                                rocblas_side      side,
                                                                rocblas_fill      uplo,
                                                                rocblas_operation transa,
                                                                rocblas_diagonal  diag,
                                                                rocblas_int       m,
                                                                rocblas_int       n,
                                                                const double*     alpha,
                                                                const double*     a,
                                                                rocblas_int       lda,
                                                                rocblas_stride    stride_a,
                                                                const double*     b,
                                                                rocblas_int       ldb,
                                                                rocblas_stride    stride_b,
                                                                double*           c,
                                                                rocblas_int       ldc,
                                                                rocblas_stride    stride_c,
                                                                rocblas_int       batch_count);

rocblas_status rocblas_ctrmm_outofplace_strided_batched_fortran(rocblas_handle               handle,
                                                                rocblas_side                 side,
                                                                rocblas_fill                 uplo,
                                                                rocblas_operation            transa,
                                                                rocblas_diagonal             diag,
                                                                rocblas_int                  m,
                                                                rocblas_int                  n,
                                                                const rocblas_float_complex* alpha,
                                                                const rocblas_float_complex* a,
                                                                rocblas_int                  lda,
                                                                rocblas_stride stride_a,
                                                                const rocblas_float_complex* b,
                                                                rocblas_int                  ldb,
                                                                rocblas_stride         stride_b,
                                                                rocblas_float_complex* c,
                                                                rocblas_int            ldc,
                                                                rocblas_stride         stride_c,
                                                                rocblas_int            batch_count);

rocblas_status rocblas_ztrmm_outofplace_strided_batched_fortran(rocblas_handle    handle,
                                                                rocblas_side      side,
                                                                rocblas_fill      uplo,
                                                                rocblas_operation transa,
                                                                rocblas_diagonal  diag,
                                                                rocblas_int       m,
                                                                rocblas_int       n,
                                                                const rocblas_double_complex* alpha,
                                                                const rocblas_double_complex* a,
                                                                rocblas_int                   lda,
                                                                rocblas_stride stride_a,
                                                                const rocblas_double_complex* b,
                                                                rocblas_int                   ldb,
                                                                rocblas_stride          stride_b,
                                                                rocblas_double_complex* c,
                                                                rocblas_int             ldc,
                                                                rocblas_stride          stride_c,
                                                                rocblas_int batch_count);

// trtri
rocblas_status rocblas_strtri_fortran(rocblas_handle   handle,
                                      rocblas_fill     uplo,
                                      rocblas_diagonal diag,
                                      rocblas_int      n,
                                      const float*     A,
                                      rocblas_int      lda,
                                      float*           invA,
                                      rocblas_int      ldinvA);

rocblas_status rocblas_dtrtri_fortran(rocblas_handle   handle,
                                      rocblas_fill     uplo,
                                      rocblas_diagonal diag,
                                      rocblas_int      n,
                                      const double*    A,
                                      rocblas_int      lda,
                                      double*          invA,
                                      rocblas_int      ldinvA);

rocblas_status rocblas_ctrtri_fortran(rocblas_handle               handle,
                                      rocblas_fill                 uplo,
                                      rocblas_diagonal             diag,
                                      rocblas_int                  n,
                                      const rocblas_float_complex* A,
                                      rocblas_int                  lda,
                                      rocblas_float_complex*       invA,
                                      rocblas_int                  ldinvA);

rocblas_status rocblas_ztrtri_fortran(rocblas_handle                handle,
                                      rocblas_fill                  uplo,
                                      rocblas_diagonal              diag,
                                      rocblas_int                   n,
                                      const rocblas_double_complex* A,
                                      rocblas_int                   lda,
                                      rocblas_double_complex*       invA,
                                      rocblas_int                   ldinvA);

// trtri_batched
rocblas_status rocblas_strtri_batched_fortran(rocblas_handle     handle,
                                              rocblas_fill       uplo,
                                              rocblas_diagonal   diag,
                                              rocblas_int        n,
                                              const float* const A[],
                                              rocblas_int        lda,
                                              float* const       invA[],
                                              rocblas_int        ldinvA,
                                              rocblas_int        batch_count);

rocblas_status rocblas_dtrtri_batched_fortran(rocblas_handle      handle,
                                              rocblas_fill        uplo,
                                              rocblas_diagonal    diag,
                                              rocblas_int         n,
                                              const double* const A[],
                                              rocblas_int         lda,
                                              double* const       invA[],
                                              rocblas_int         ldinvA,
                                              rocblas_int         batch_count);

rocblas_status rocblas_ctrtri_batched_fortran(rocblas_handle                     handle,
                                              rocblas_fill                       uplo,
                                              rocblas_diagonal                   diag,
                                              rocblas_int                        n,
                                              const rocblas_float_complex* const A[],
                                              rocblas_int                        lda,
                                              rocblas_float_complex* const       invA[],
                                              rocblas_int                        ldinvA,
                                              rocblas_int                        batch_count);

rocblas_status rocblas_ztrtri_batched_fortran(rocblas_handle                      handle,
                                              rocblas_fill                        uplo,
                                              rocblas_diagonal                    diag,
                                              rocblas_int                         n,
                                              const rocblas_double_complex* const A[],
                                              rocblas_int                         lda,
                                              rocblas_double_complex* const       invA[],
                                              rocblas_int                         ldinvA,
                                              rocblas_int                         batch_count);

// trtri_strided_batched
rocblas_status rocblas_strtri_strided_batched_fortran(rocblas_handle   handle,
                                                      rocblas_fill     uplo,
                                                      rocblas_diagonal diag,
                                                      rocblas_int      n,
                                                      const float*     A,
                                                      rocblas_int      lda,
                                                      rocblas_stride   stride_a,
                                                      float*           invA,
                                                      rocblas_int      ldinvA,
                                                      rocblas_stride   stride_invA,
                                                      rocblas_int      batch_count);

rocblas_status rocblas_dtrtri_strided_batched_fortran(rocblas_handle   handle,
                                                      rocblas_fill     uplo,
                                                      rocblas_diagonal diag,
                                                      rocblas_int      n,
                                                      const double*    A,
                                                      rocblas_int      lda,
                                                      rocblas_stride   stride_a,
                                                      double*          invA,
                                                      rocblas_int      ldinvA,
                                                      rocblas_stride   stride_invA,
                                                      rocblas_int      batch_count);

rocblas_status rocblas_ctrtri_strided_batched_fortran(rocblas_handle               handle,
                                                      rocblas_fill                 uplo,
                                                      rocblas_diagonal             diag,
                                                      rocblas_int                  n,
                                                      const rocblas_float_complex* A,
                                                      rocblas_int                  lda,
                                                      rocblas_stride               stride_a,
                                                      rocblas_float_complex*       invA,
                                                      rocblas_int                  ldinvA,
                                                      rocblas_stride               stride_invA,
                                                      rocblas_int                  batch_count);

rocblas_status rocblas_ztrtri_strided_batched_fortran(rocblas_handle                handle,
                                                      rocblas_fill                  uplo,
                                                      rocblas_diagonal              diag,
                                                      rocblas_int                   n,
                                                      const rocblas_double_complex* A,
                                                      rocblas_int                   lda,
                                                      rocblas_stride                stride_a,
                                                      rocblas_double_complex*       invA,
                                                      rocblas_int                   ldinvA,
                                                      rocblas_stride                stride_invA,
                                                      rocblas_int                   batch_count);

// trsm
rocblas_status rocblas_strsm_fortran(rocblas_handle    handle,
                                     rocblas_side      side,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     float*            B,
                                     rocblas_int       ldb);

rocblas_status rocblas_dtrsm_fortran(rocblas_handle    handle,
                                     rocblas_side      side,
                                     rocblas_fill      uplo,
                                     rocblas_operation transA,
                                     rocblas_diagonal  diag,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     double*           B,
                                     rocblas_int       ldb);

rocblas_status rocblas_ctrsm_fortran(rocblas_handle               handle,
                                     rocblas_side                 side,
                                     rocblas_fill                 uplo,
                                     rocblas_operation            transA,
                                     rocblas_diagonal             diag,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     rocblas_float_complex*       B,
                                     rocblas_int                  ldb);

rocblas_status rocblas_ztrsm_fortran(rocblas_handle                handle,
                                     rocblas_side                  side,
                                     rocblas_fill                  uplo,
                                     rocblas_operation             transA,
                                     rocblas_diagonal              diag,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     rocblas_double_complex*       B,
                                     rocblas_int                   ldb);

// trsm_batched
rocblas_status rocblas_strsm_batched_fortran(rocblas_handle     handle,
                                             rocblas_side       side,
                                             rocblas_fill       uplo,
                                             rocblas_operation  transA,
                                             rocblas_diagonal   diag,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             float* const       B[],
                                             rocblas_int        ldb,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dtrsm_batched_fortran(rocblas_handle      handle,
                                             rocblas_side        side,
                                             rocblas_fill        uplo,
                                             rocblas_operation   transA,
                                             rocblas_diagonal    diag,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             double* const       B[],
                                             rocblas_int         ldb,
                                             rocblas_int         batch_count);

rocblas_status rocblas_ctrsm_batched_fortran(rocblas_handle                     handle,
                                             rocblas_side                       side,
                                             rocblas_fill                       uplo,
                                             rocblas_operation                  transA,
                                             rocblas_diagonal                   diag,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             rocblas_float_complex* const       B[],
                                             rocblas_int                        ldb,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_ztrsm_batched_fortran(rocblas_handle                      handle,
                                             rocblas_side                        side,
                                             rocblas_fill                        uplo,
                                             rocblas_operation                   transA,
                                             rocblas_diagonal                    diag,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             rocblas_double_complex* const       B[],
                                             rocblas_int                         ldb,
                                             rocblas_int                         batch_count);

// trsm_strided_batched
rocblas_status rocblas_strsm_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_side      side,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const float*      alpha,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     float*            B,
                                                     rocblas_int       ldb,
                                                     rocblas_stride    stride_b,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dtrsm_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_side      side,
                                                     rocblas_fill      uplo,
                                                     rocblas_operation transA,
                                                     rocblas_diagonal  diag,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const double*     alpha,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     double*           B,
                                                     rocblas_int       ldb,
                                                     rocblas_stride    stride_b,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_ctrsm_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_side                 side,
                                                     rocblas_fill                 uplo,
                                                     rocblas_operation            transA,
                                                     rocblas_diagonal             diag,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_a,
                                                     rocblas_float_complex*       B,
                                                     rocblas_int                  ldb,
                                                     rocblas_stride               stride_b,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_ztrsm_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_side                  side,
                                                     rocblas_fill                  uplo,
                                                     rocblas_operation             transA,
                                                     rocblas_diagonal              diag,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_a,
                                                     rocblas_double_complex*       B,
                                                     rocblas_int                   ldb,
                                                     rocblas_stride                stride_b,
                                                     rocblas_int                   batch_count);

// gemm
rocblas_status rocblas_sgemm_fortran(rocblas_handle    handle,
                                     rocblas_operation transA,
                                     rocblas_operation transB,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     rocblas_int       k,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     const float*      B,
                                     rocblas_int       ldb,
                                     const float*      beta,
                                     float*            C,
                                     rocblas_int       ldc);

rocblas_status rocblas_dgemm_fortran(rocblas_handle    handle,
                                     rocblas_operation transA,
                                     rocblas_operation transB,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     rocblas_int       k,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     const double*     B,
                                     rocblas_int       ldb,
                                     const double*     beta,
                                     double*           C,
                                     rocblas_int       ldc);

rocblas_status rocblas_hgemm_fortran(rocblas_handle      handle,
                                     rocblas_operation   transA,
                                     rocblas_operation   transB,
                                     rocblas_int         m,
                                     rocblas_int         n,
                                     rocblas_int         k,
                                     const rocblas_half* alpha,
                                     const rocblas_half* A,
                                     rocblas_int         lda,
                                     const rocblas_half* B,
                                     rocblas_int         ldb,
                                     const rocblas_half* beta,
                                     rocblas_half*       C,
                                     rocblas_int         ldc);

rocblas_status rocblas_cgemm_fortran(rocblas_handle               handle,
                                     rocblas_operation            transA,
                                     rocblas_operation            transB,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     rocblas_int                  k,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* B,
                                     rocblas_int                  ldb,
                                     const rocblas_float_complex* beta,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zgemm_fortran(rocblas_handle                handle,
                                     rocblas_operation             transA,
                                     rocblas_operation             transB,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     rocblas_int                   k,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* B,
                                     rocblas_int                   ldb,
                                     const rocblas_double_complex* beta,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// gemm_batched
rocblas_status rocblas_sgemm_batched_fortran(rocblas_handle     handle,
                                             rocblas_operation  transA,
                                             rocblas_operation  transB,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             rocblas_int        k,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const B[],
                                             rocblas_int        ldb,
                                             const float*       beta,
                                             float* const       C[],
                                             rocblas_int        ldc,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dgemm_batched_fortran(rocblas_handle      handle,
                                             rocblas_operation   transA,
                                             rocblas_operation   transB,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             rocblas_int         k,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const B[],
                                             rocblas_int         ldb,
                                             const double*       beta,
                                             double* const       C[],
                                             rocblas_int         ldc,
                                             rocblas_int         batch_count);

rocblas_status rocblas_hgemm_batched_fortran(rocblas_handle            handle,
                                             rocblas_operation         transA,
                                             rocblas_operation         transB,
                                             rocblas_int               m,
                                             rocblas_int               n,
                                             rocblas_int               k,
                                             const rocblas_half*       alpha,
                                             const rocblas_half* const A[],
                                             rocblas_int               lda,
                                             const rocblas_half* const B[],
                                             rocblas_int               ldb,
                                             const rocblas_half*       beta,
                                             rocblas_half* const       C[],
                                             rocblas_int               ldc,
                                             rocblas_int               batch_count);

rocblas_status rocblas_cgemm_batched_fortran(rocblas_handle                     handle,
                                             rocblas_operation                  transA,
                                             rocblas_operation                  transB,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             rocblas_int                        k,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const B[],
                                             rocblas_int                        ldb,
                                             const rocblas_float_complex*       beta,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zgemm_batched_fortran(rocblas_handle                      handle,
                                             rocblas_operation                   transa,
                                             rocblas_operation                   transb,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             rocblas_int                         k,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const B[],
                                             rocblas_int                         ldb,
                                             const rocblas_double_complex*       beta,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// gemm_strided_batched
rocblas_status rocblas_sgemm_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation transA,
                                                     rocblas_operation transB,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     rocblas_int       k,
                                                     const float*      alpha,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     const float*      B,
                                                     rocblas_int       ldb,
                                                     rocblas_stride    stride_b,
                                                     const float*      beta,
                                                     float*            C,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_c,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dgemm_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation transA,
                                                     rocblas_operation transB,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     rocblas_int       k,
                                                     const double*     alpha,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     const double*     B,
                                                     rocblas_int       ldb,
                                                     rocblas_stride    stride_b,
                                                     const double*     beta,
                                                     double*           C,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_c,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_hgemm_strided_batched_fortran(rocblas_handle      handle,
                                                     rocblas_operation   transA,
                                                     rocblas_operation   transB,
                                                     rocblas_int         m,
                                                     rocblas_int         n,
                                                     rocblas_int         k,
                                                     const rocblas_half* alpha,
                                                     const rocblas_half* A,
                                                     rocblas_int         lda,
                                                     rocblas_stride      stride_a,
                                                     const rocblas_half* B,
                                                     rocblas_int         ldb,
                                                     rocblas_stride      stride_b,
                                                     const rocblas_half* beta,
                                                     rocblas_half*       C,
                                                     rocblas_int         ldc,
                                                     rocblas_stride      stride_c,
                                                     rocblas_int         batch_count);

rocblas_status rocblas_cgemm_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_operation            transa,
                                                     rocblas_operation            transb,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     rocblas_int                  k,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_a,
                                                     const rocblas_float_complex* B,
                                                     rocblas_int                  ldb,
                                                     rocblas_stride               stride_b,
                                                     const rocblas_float_complex* beta,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_c,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zgemm_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_operation             transa,
                                                     rocblas_operation             transb,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     rocblas_int                   k,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_a,
                                                     const rocblas_double_complex* B,
                                                     rocblas_int                   ldb,
                                                     rocblas_stride                stride_b,
                                                     const rocblas_double_complex* beta,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_c,
                                                     rocblas_int                   batch_count);

// dgmm
rocblas_status rocblas_sdgmm_fortran(rocblas_handle handle,
                                     rocblas_side   side,
                                     rocblas_int    m,
                                     rocblas_int    n,
                                     const float*   A,
                                     rocblas_int    lda,
                                     const float*   x,
                                     rocblas_int    incx,
                                     float*         C,
                                     rocblas_int    ldc);

rocblas_status rocblas_ddgmm_fortran(rocblas_handle handle,
                                     rocblas_side   side,
                                     rocblas_int    m,
                                     rocblas_int    n,
                                     const double*  A,
                                     rocblas_int    lda,
                                     const double*  x,
                                     rocblas_int    incx,
                                     double*        C,
                                     rocblas_int    ldc);

rocblas_status rocblas_cdgmm_fortran(rocblas_handle               handle,
                                     rocblas_side                 side,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* x,
                                     rocblas_int                  incx,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zdgmm_fortran(rocblas_handle                handle,
                                     rocblas_side                  side,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* x,
                                     rocblas_int                   incx,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// dgmm_batched
rocblas_status rocblas_sdgmm_batched_fortran(rocblas_handle     handle,
                                             rocblas_side       side,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float* const x[],
                                             rocblas_int        incx,
                                             float* const       C[],
                                             rocblas_int        ldc,
                                             rocblas_int        batch_count);

rocblas_status rocblas_ddgmm_batched_fortran(rocblas_handle      handle,
                                             rocblas_side        side,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double* const x[],
                                             rocblas_int         incx,
                                             double* const       C[],
                                             rocblas_int         ldc,
                                             rocblas_int         batch_count);

rocblas_status rocblas_cdgmm_batched_fortran(rocblas_handle                     handle,
                                             rocblas_side                       side,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex* const x[],
                                             rocblas_int                        incx,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zdgmm_batched_fortran(rocblas_handle                      handle,
                                             rocblas_side                        side,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex* const x[],
                                             rocblas_int                         incx,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// dgmm_strided_batched
rocblas_status rocblas_sdgmm_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_side   side,
                                                     rocblas_int    m,
                                                     rocblas_int    n,
                                                     const float*   A,
                                                     rocblas_int    lda,
                                                     rocblas_stride stride_a,
                                                     const float*   x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stride_x,
                                                     float*         C,
                                                     rocblas_int    ldc,
                                                     rocblas_stride stride_c,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_ddgmm_strided_batched_fortran(rocblas_handle handle,
                                                     rocblas_side   side,
                                                     rocblas_int    m,
                                                     rocblas_int    n,
                                                     const double*  A,
                                                     rocblas_int    lda,
                                                     rocblas_stride stride_a,
                                                     const double*  x,
                                                     rocblas_int    incx,
                                                     rocblas_stride stride_x,
                                                     double*        C,
                                                     rocblas_int    ldc,
                                                     rocblas_stride stride_c,
                                                     rocblas_int    batch_count);

rocblas_status rocblas_cdgmm_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_side                 side,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_a,
                                                     const rocblas_float_complex* x,
                                                     rocblas_int                  incx,
                                                     rocblas_stride               stride_x,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_c,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zdgmm_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_side                  side,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_a,
                                                     const rocblas_double_complex* x,
                                                     rocblas_int                   incx,
                                                     rocblas_stride                stride_x,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_c,
                                                     rocblas_int                   batch_count);

// geam
rocblas_status rocblas_sgeam_fortran(rocblas_handle    handle,
                                     rocblas_operation transA,
                                     rocblas_operation transB,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const float*      alpha,
                                     const float*      A,
                                     rocblas_int       lda,
                                     const float*      beta,
                                     const float*      B,
                                     rocblas_int       ldb,
                                     float*            C,
                                     rocblas_int       ldc);

rocblas_status rocblas_dgeam_fortran(rocblas_handle    handle,
                                     rocblas_operation transA,
                                     rocblas_operation transB,
                                     rocblas_int       m,
                                     rocblas_int       n,
                                     const double*     alpha,
                                     const double*     A,
                                     rocblas_int       lda,
                                     const double*     beta,
                                     const double*     B,
                                     rocblas_int       ldb,
                                     double*           C,
                                     rocblas_int       ldc);

rocblas_status rocblas_cgeam_fortran(rocblas_handle               handle,
                                     rocblas_operation            transA,
                                     rocblas_operation            transB,
                                     rocblas_int                  m,
                                     rocblas_int                  n,
                                     const rocblas_float_complex* alpha,
                                     const rocblas_float_complex* A,
                                     rocblas_int                  lda,
                                     const rocblas_float_complex* beta,
                                     const rocblas_float_complex* B,
                                     rocblas_int                  ldb,
                                     rocblas_float_complex*       C,
                                     rocblas_int                  ldc);

rocblas_status rocblas_zgeam_fortran(rocblas_handle                handle,
                                     rocblas_operation             transA,
                                     rocblas_operation             transB,
                                     rocblas_int                   m,
                                     rocblas_int                   n,
                                     const rocblas_double_complex* alpha,
                                     const rocblas_double_complex* A,
                                     rocblas_int                   lda,
                                     const rocblas_double_complex* beta,
                                     const rocblas_double_complex* B,
                                     rocblas_int                   ldb,
                                     rocblas_double_complex*       C,
                                     rocblas_int                   ldc);

// geam_batched
rocblas_status rocblas_sgeam_batched_fortran(rocblas_handle     handle,
                                             rocblas_operation  transA,
                                             rocblas_operation  transB,
                                             rocblas_int        m,
                                             rocblas_int        n,
                                             const float*       alpha,
                                             const float* const A[],
                                             rocblas_int        lda,
                                             const float*       beta,
                                             const float* const B[],
                                             rocblas_int        ldb,
                                             float* const       C[],
                                             rocblas_int        ldc,
                                             rocblas_int        batch_count);

rocblas_status rocblas_dgeam_batched_fortran(rocblas_handle      handle,
                                             rocblas_operation   transA,
                                             rocblas_operation   transB,
                                             rocblas_int         m,
                                             rocblas_int         n,
                                             const double*       alpha,
                                             const double* const A[],
                                             rocblas_int         lda,
                                             const double*       beta,
                                             const double* const B[],
                                             rocblas_int         ldb,
                                             double* const       C[],
                                             rocblas_int         ldc,
                                             rocblas_int         batch_count);

rocblas_status rocblas_cgeam_batched_fortran(rocblas_handle                     handle,
                                             rocblas_operation                  transA,
                                             rocblas_operation                  transB,
                                             rocblas_int                        m,
                                             rocblas_int                        n,
                                             const rocblas_float_complex*       alpha,
                                             const rocblas_float_complex* const A[],
                                             rocblas_int                        lda,
                                             const rocblas_float_complex*       beta,
                                             const rocblas_float_complex* const B[],
                                             rocblas_int                        ldb,
                                             rocblas_float_complex* const       C[],
                                             rocblas_int                        ldc,
                                             rocblas_int                        batch_count);

rocblas_status rocblas_zgeam_batched_fortran(rocblas_handle                      handle,
                                             rocblas_operation                   transA,
                                             rocblas_operation                   transB,
                                             rocblas_int                         m,
                                             rocblas_int                         n,
                                             const rocblas_double_complex*       alpha,
                                             const rocblas_double_complex* const A[],
                                             rocblas_int                         lda,
                                             const rocblas_double_complex*       beta,
                                             const rocblas_double_complex* const B[],
                                             rocblas_int                         ldb,
                                             rocblas_double_complex* const       C[],
                                             rocblas_int                         ldc,
                                             rocblas_int                         batch_count);

// geam_strided_batched
rocblas_status rocblas_sgeam_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation transA,
                                                     rocblas_operation transB,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const float*      alpha,
                                                     const float*      A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     const float*      beta,
                                                     const float*      B,
                                                     rocblas_int       ldb,
                                                     rocblas_stride    stride_b,
                                                     float*            C,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_c,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_dgeam_strided_batched_fortran(rocblas_handle    handle,
                                                     rocblas_operation transA,
                                                     rocblas_operation transB,
                                                     rocblas_int       m,
                                                     rocblas_int       n,
                                                     const double*     alpha,
                                                     const double*     A,
                                                     rocblas_int       lda,
                                                     rocblas_stride    stride_a,
                                                     const double*     beta,
                                                     const double*     B,
                                                     rocblas_int       ldb,
                                                     rocblas_stride    stride_b,
                                                     double*           C,
                                                     rocblas_int       ldc,
                                                     rocblas_stride    stride_c,
                                                     rocblas_int       batch_count);

rocblas_status rocblas_cgeam_strided_batched_fortran(rocblas_handle               handle,
                                                     rocblas_operation            transA,
                                                     rocblas_operation            transB,
                                                     rocblas_int                  m,
                                                     rocblas_int                  n,
                                                     const rocblas_float_complex* alpha,
                                                     const rocblas_float_complex* A,
                                                     rocblas_int                  lda,
                                                     rocblas_stride               stride_a,
                                                     const rocblas_float_complex* beta,
                                                     const rocblas_float_complex* B,
                                                     rocblas_int                  ldb,
                                                     rocblas_stride               stride_b,
                                                     rocblas_float_complex*       C,
                                                     rocblas_int                  ldc,
                                                     rocblas_stride               stride_c,
                                                     rocblas_int                  batch_count);

rocblas_status rocblas_zgeam_strided_batched_fortran(rocblas_handle                handle,
                                                     rocblas_operation             transA,
                                                     rocblas_operation             transB,
                                                     rocblas_int                   m,
                                                     rocblas_int                   n,
                                                     const rocblas_double_complex* alpha,
                                                     const rocblas_double_complex* A,
                                                     rocblas_int                   lda,
                                                     rocblas_stride                stride_a,
                                                     const rocblas_double_complex* beta,
                                                     const rocblas_double_complex* B,
                                                     rocblas_int                   ldb,
                                                     rocblas_stride                stride_b,
                                                     rocblas_double_complex*       C,
                                                     rocblas_int                   ldc,
                                                     rocblas_stride                stride_c,
                                                     rocblas_int                   batch_count);

/* ==========
 *    Ext.
 * ========== */

// axpy_ex
rocblas_status rocblas_axpy_ex_fortran(rocblas_handle    handle,
                                       const rocblas_int N,
                                       const void*       alpha,
                                       rocblas_datatype  alpha_type,
                                       const void*       x,
                                       rocblas_datatype  x_type,
                                       const rocblas_int incx,
                                       void*             y,
                                       rocblas_datatype  y_type,
                                       const rocblas_int incy,
                                       rocblas_datatype  execution_type);

rocblas_status rocblas_axpy_batched_ex_fortran(rocblas_handle    handle,
                                               const rocblas_int N,
                                               const void*       alpha,
                                               rocblas_datatype  alpha_type,
                                               const void*       x,
                                               rocblas_datatype  x_type,
                                               const rocblas_int incx,
                                               void*             y,
                                               rocblas_datatype  y_type,
                                               const rocblas_int incy,
                                               const rocblas_int batch_count,
                                               rocblas_datatype  execution_type);

rocblas_status rocblas_axpy_strided_batched_ex_fortran(rocblas_handle       handle,
                                                       const rocblas_int    N,
                                                       const void*          alpha,
                                                       rocblas_datatype     alpha_type,
                                                       const void*          x,
                                                       rocblas_datatype     x_type,
                                                       const rocblas_int    incx,
                                                       const rocblas_stride stridex,
                                                       void*                y,
                                                       rocblas_datatype     y_type,
                                                       const rocblas_int    incy,
                                                       const rocblas_stride stridey,
                                                       const rocblas_int    batch_count,
                                                       rocblas_datatype     execution_type);

// dot_ex
rocblas_status rocblas_dot_ex_fortran(rocblas_handle   handle,
                                      rocblas_int      n,
                                      const void*      x,
                                      rocblas_datatype x_type,
                                      rocblas_int      incx,
                                      const void*      y,
                                      rocblas_datatype y_type,
                                      rocblas_int      incy,
                                      void*            result,
                                      rocblas_datatype result_type,
                                      rocblas_datatype execution_type);

rocblas_status rocblas_dotc_ex_fortran(rocblas_handle   handle,
                                       rocblas_int      n,
                                       const void*      x,
                                       rocblas_datatype x_type,
                                       rocblas_int      incx,
                                       const void*      y,
                                       rocblas_datatype y_type,
                                       rocblas_int      incy,
                                       void*            result,
                                       rocblas_datatype result_type,
                                       rocblas_datatype execution_type);

rocblas_status rocblas_dot_batched_ex_fortran(rocblas_handle   handle,
                                              rocblas_int      n,
                                              const void*      x,
                                              rocblas_datatype x_type,
                                              rocblas_int      incx,
                                              const void*      y,
                                              rocblas_datatype y_type,
                                              rocblas_int      incy,
                                              rocblas_int      batch_count,
                                              void*            result,
                                              rocblas_datatype result_type,
                                              rocblas_datatype execution_type);

rocblas_status rocblas_dotc_batched_ex_fortran(rocblas_handle   handle,
                                               rocblas_int      n,
                                               const void*      x,
                                               rocblas_datatype x_type,
                                               rocblas_int      incx,
                                               const void*      y,
                                               rocblas_datatype y_type,
                                               rocblas_int      incy,
                                               rocblas_int      batch_count,
                                               void*            result,
                                               rocblas_datatype result_type,
                                               rocblas_datatype execution_type);

rocblas_status rocblas_dot_strided_batched_ex_fortran(rocblas_handle   handle,
                                                      rocblas_int      n,
                                                      const void*      x,
                                                      rocblas_datatype x_type,
                                                      rocblas_int      incx,
                                                      rocblas_stride   stride_x,
                                                      const void*      y,
                                                      rocblas_datatype y_type,
                                                      rocblas_int      incy,
                                                      rocblas_stride   stride_y,
                                                      rocblas_int      batch_count,
                                                      void*            result,
                                                      rocblas_datatype result_type,
                                                      rocblas_datatype execution_type);

rocblas_status rocblas_dotc_strided_batched_ex_fortran(rocblas_handle   handle,
                                                       rocblas_int      n,
                                                       const void*      x,
                                                       rocblas_datatype x_type,
                                                       rocblas_int      incx,
                                                       rocblas_stride   stride_x,
                                                       const void*      y,
                                                       rocblas_datatype y_type,
                                                       rocblas_int      incy,
                                                       rocblas_stride   stride_y,
                                                       rocblas_int      batch_count,
                                                       void*            result,
                                                       rocblas_datatype result_type,
                                                       rocblas_datatype execution_type);

// nrm2_ex
rocblas_status rocblas_nrm2_ex_fortran(rocblas_handle   handle,
                                       rocblas_int      n,
                                       const void*      x,
                                       rocblas_datatype x_type,
                                       rocblas_int      incx,
                                       void*            result,
                                       rocblas_datatype result_type,
                                       rocblas_datatype execution_type);

rocblas_status rocblas_nrm2_batched_ex_fortran(rocblas_handle   handle,
                                               rocblas_int      n,
                                               const void*      x,
                                               rocblas_datatype x_type,
                                               rocblas_int      incx,
                                               rocblas_int      batch_count,
                                               void*            result,
                                               rocblas_datatype result_type,
                                               rocblas_datatype execution_type);

rocblas_status rocblas_nrm2_strided_batched_ex_fortran(rocblas_handle   handle,
                                                       rocblas_int      n,
                                                       const void*      x,
                                                       rocblas_datatype x_type,
                                                       rocblas_int      incx,
                                                       rocblas_stride   stride_x,
                                                       rocblas_int      batch_count,
                                                       void*            result,
                                                       rocblas_datatype result_type,
                                                       rocblas_datatype execution_type);

// rot_ex
rocblas_status rocblas_rot_ex_fortran(rocblas_handle   handle,
                                      rocblas_int      n,
                                      void*            x,
                                      rocblas_datatype x_type,
                                      rocblas_int      incx,
                                      void*            y,
                                      rocblas_datatype y_type,
                                      rocblas_int      incy,
                                      const void*      c,
                                      const void*      s,
                                      rocblas_datatype cs_type,
                                      rocblas_datatype execution_type);

rocblas_status rocblas_rot_batched_ex_fortran(rocblas_handle   handle,
                                              rocblas_int      n,
                                              void*            x,
                                              rocblas_datatype x_type,
                                              rocblas_int      incx,
                                              void*            y,
                                              rocblas_datatype y_type,
                                              rocblas_int      incy,
                                              const void*      c,
                                              const void*      s,
                                              rocblas_datatype cs_type,
                                              rocblas_int      batch_count,
                                              rocblas_datatype execution_type);

rocblas_status rocblas_rot_strided_batched_ex_fortran(rocblas_handle   handle,
                                                      rocblas_int      n,
                                                      void*            x,
                                                      rocblas_datatype x_type,
                                                      rocblas_int      incx,
                                                      rocblas_stride   stride_x,
                                                      void*            y,
                                                      rocblas_datatype y_type,
                                                      rocblas_int      incy,
                                                      rocblas_stride   stride_y,
                                                      const void*      c,
                                                      const void*      s,
                                                      rocblas_datatype cs_type,
                                                      rocblas_int      batch_count,
                                                      rocblas_datatype execution_type);

// scal_ex
rocblas_status rocblas_scal_ex_fortran(rocblas_handle    handle,
                                       const rocblas_int N,
                                       const void*       alpha,
                                       rocblas_datatype  alpha_type,
                                       void*             x,
                                       rocblas_datatype  x_type,
                                       const rocblas_int incx,
                                       rocblas_datatype  execution_type);

rocblas_status rocblas_scal_batched_ex_fortran(rocblas_handle    handle,
                                               const rocblas_int N,
                                               const void*       alpha,
                                               rocblas_datatype  alpha_type,
                                               void*             x,
                                               rocblas_datatype  x_type,
                                               const rocblas_int incx,
                                               const rocblas_int batch_count,
                                               rocblas_datatype  execution_type);

rocblas_status rocblas_scal_strided_batched_ex_fortran(rocblas_handle       handle,
                                                       const rocblas_int    N,
                                                       const void*          alpha,
                                                       rocblas_datatype     alpha_type,
                                                       void*                x,
                                                       rocblas_datatype     x_type,
                                                       const rocblas_int    incx,
                                                       const rocblas_stride stridex,
                                                       const rocblas_int    batch_count,
                                                       rocblas_datatype     execution_type);

// gemm_ex
rocblas_status rocblas_gemm_ex_fortran(rocblas_handle    handle,
                                       rocblas_operation transA,
                                       rocblas_operation transB,
                                       rocblas_int       m,
                                       rocblas_int       n,
                                       rocblas_int       k,
                                       const void*       alpha,
                                       const void*       a,
                                       rocblas_datatype  a_type,
                                       rocblas_int       lda,
                                       const void*       b,
                                       rocblas_datatype  b_type,
                                       rocblas_int       ldb,
                                       const void*       beta,
                                       const void*       c,
                                       rocblas_datatype  c_type,
                                       rocblas_int       ldc,
                                       void*             d,
                                       rocblas_datatype  d_type,
                                       rocblas_int       ldd,
                                       rocblas_datatype  compute_type,
                                       rocblas_gemm_algo algo,
                                       int32_t           solution_index,
                                       uint32_t          flags);

// gemm_ex3
rocblas_status rocblas_gemm_ex3_fortran(rocblas_handle      handle,
                                        rocblas_operation   transA,
                                        rocblas_operation   transB,
                                        rocblas_int         m,
                                        rocblas_int         n,
                                        rocblas_int         k,
                                        const void*         alpha,
                                        const void*         a,
                                        rocblas_datatype    a_type,
                                        rocblas_int         lda,
                                        const void*         b,
                                        rocblas_datatype    b_type,
                                        rocblas_int         ldb,
                                        const void*         beta,
                                        const void*         c,
                                        rocblas_datatype    c_type,
                                        rocblas_int         ldc,
                                        void*               d,
                                        rocblas_datatype    d_type,
                                        rocblas_int         ldd,
                                        rocblas_computetype compute_type,
                                        rocblas_gemm_algo   algo,
                                        int32_t             solution_index,
                                        uint32_t            flags);

rocblas_status rocblas_gemm_batched_ex_fortran(rocblas_handle    handle,
                                               rocblas_operation transA,
                                               rocblas_operation transB,
                                               rocblas_int       m,
                                               rocblas_int       n,
                                               rocblas_int       k,
                                               const void*       alpha,
                                               const void*       a,
                                               rocblas_datatype  a_type,
                                               rocblas_int       lda,
                                               const void*       b,
                                               rocblas_datatype  b_type,
                                               rocblas_int       ldb,
                                               const void*       beta,
                                               const void*       c,
                                               rocblas_datatype  c_type,
                                               rocblas_int       ldc,
                                               void*             d,
                                               rocblas_datatype  d_type,
                                               rocblas_int       ldd,
                                               rocblas_int       batch_count,
                                               rocblas_datatype  compute_type,
                                               rocblas_gemm_algo algo,
                                               int32_t           solution_index,
                                               uint32_t          flags);

rocblas_status rocblas_gemm_strided_batched_ex_fortran(rocblas_handle    handle,
                                                       rocblas_operation transA,
                                                       rocblas_operation transB,
                                                       rocblas_int       m,
                                                       rocblas_int       n,
                                                       rocblas_int       k,
                                                       const void*       alpha,
                                                       const void*       a,
                                                       rocblas_datatype  a_type,
                                                       rocblas_int       lda,
                                                       rocblas_stride    stride_a,
                                                       const void*       b,
                                                       rocblas_datatype  b_type,
                                                       rocblas_int       ldb,
                                                       rocblas_stride    stride_b,
                                                       const void*       beta,
                                                       const void*       c,
                                                       rocblas_datatype  c_type,
                                                       rocblas_int       ldc,
                                                       rocblas_stride    stride_c,
                                                       void*             d,
                                                       rocblas_datatype  d_type,
                                                       rocblas_int       ldd,
                                                       rocblas_stride    stride_d,
                                                       rocblas_int       batch_count,
                                                       rocblas_datatype  compute_type,
                                                       rocblas_gemm_algo algo,
                                                       int32_t           solution_index,
                                                       uint32_t          flags);

rocblas_status rocblas_gemm_ext2_fortran(rocblas_handle    handle,
                                         rocblas_int       m,
                                         rocblas_int       n,
                                         rocblas_int       k,
                                         const void*       alpha,
                                         const void*       a,
                                         rocblas_datatype  a_type,
                                         rocblas_stride    row_stride_a,
                                         rocblas_stride    col_stride_a,
                                         const void*       b,
                                         rocblas_datatype  b_type,
                                         rocblas_stride    row_stride_b,
                                         rocblas_stride    col_stride_b,
                                         const void*       beta,
                                         const void*       c,
                                         rocblas_datatype  c_type,
                                         rocblas_stride    row_stride_c,
                                         rocblas_stride    col_stride_c,
                                         void*             d,
                                         rocblas_datatype  d_type,
                                         rocblas_stride    row_stride_d,
                                         rocblas_stride    col_stride_d,
                                         rocblas_datatype  compute_type,
                                         rocblas_gemm_algo algo,
                                         int32_t           solution_index,
                                         uint32_t          flags);

// trsm_ex
rocblas_status rocblas_trsm_ex_fortran(rocblas_handle    handle,
                                       rocblas_side      side,
                                       rocblas_fill      uplo,
                                       rocblas_operation transA,
                                       rocblas_diagonal  diag,
                                       rocblas_int       m,
                                       rocblas_int       n,
                                       const void*       alpha,
                                       const void*       A,
                                       rocblas_int       lda,
                                       void*             B,
                                       rocblas_int       ldb,
                                       const void*       invA,
                                       rocblas_int       invA_size,
                                       rocblas_datatype  compute_type);

rocblas_status rocblas_trsm_batched_ex_fortran(rocblas_handle    handle,
                                               rocblas_side      side,
                                               rocblas_fill      uplo,
                                               rocblas_operation transA,
                                               rocblas_diagonal  diag,
                                               rocblas_int       m,
                                               rocblas_int       n,
                                               const void*       alpha,
                                               const void*       A,
                                               rocblas_int       lda,
                                               void*             B,
                                               rocblas_int       ldb,
                                               rocblas_int       batch_count,
                                               const void*       invA,
                                               rocblas_int       invA_size,
                                               rocblas_datatype  compute_type);

rocblas_status rocblas_trsm_strided_batched_ex_fortran(rocblas_handle    handle,
                                                       rocblas_side      side,
                                                       rocblas_fill      uplo,
                                                       rocblas_operation transA,
                                                       rocblas_diagonal  diag,
                                                       rocblas_int       m,
                                                       rocblas_int       n,
                                                       const void*       alpha,
                                                       const void*       A,
                                                       rocblas_int       lda,
                                                       rocblas_stride    stride_A,
                                                       void*             B,
                                                       rocblas_int       ldb,
                                                       rocblas_stride    stride_B,
                                                       rocblas_int       batch_count,
                                                       const void*       invA,
                                                       rocblas_int       invA_size,
                                                       rocblas_stride    stride_invA,
                                                       rocblas_datatype  compute_type);

// geam_ex
rocblas_status rocblas_geam_ex_fortran(rocblas_handle            handle,
                                       rocblas_operation         transA,
                                       rocblas_operation         transB,
                                       rocblas_int               m,
                                       rocblas_int               n,
                                       rocblas_int               k,
                                       const void*               alpha,
                                       const void*               a,
                                       rocblas_datatype          a_type,
                                       rocblas_int               lda,
                                       const void*               b,
                                       rocblas_datatype          b_type,
                                       rocblas_int               ldb,
                                       const void*               beta,
                                       const void*               c,
                                       rocblas_datatype          c_type,
                                       rocblas_int               ldc,
                                       void*                     d,
                                       rocblas_datatype          d_type,
                                       rocblas_int               ldd,
                                       rocblas_datatype          compute_type,
                                       rocblas_geam_ex_operation geam_ex_op);
}
