// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         Multi-threaded compression
// ****************************************************************************

// Copyright 2008, 2009, 2010 Guy Voncken
//
// This file is part of guymager.
//
// guymager is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// guymager is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with guymager. If not, see <http://www.gnu.org/licenses/>.

#include <QtCore>

#include "libewf.h"

#include "common.h"
#include "device.h"
#include "threadcompress.h"
#include "threadwrite.h"
#include "config.h"
#include "aaff.h"

class t_ThreadCompressLocal
{
   public:
      t_pDevice pDevice;
      int        ThreadNr;
};

t_ThreadCompress::t_ThreadCompress (void)
{
   CHK_EXIT (ERROR_THREADCOMPRESS_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised


t_ThreadCompress::t_ThreadCompress (t_pDevice pDevice, int ThreadNr)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADCOMPRESS_CONSTRUCTOR_NOT_SUPPORTED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADCOMPRESS_ZLIB_FAILED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADCOMPRESS_LIBEWF_FAILED))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADCOMPRESS_COMPRESSION_BUFFER_TOO_SMALL))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_THREADCOMPRESS_INVALID_FORMAT))

      Initialised = true;
   }

   pOwn = new t_ThreadCompressLocal;
   pOwn->pDevice  = pDevice;
   pOwn->ThreadNr = ThreadNr;

   CHK_QT_EXIT (connect (this, SIGNAL(finished()), this, SLOT(SlotFinished())))
}

t_ThreadCompress::~t_ThreadCompress (void)
{
   delete pOwn;
}


void t_ThreadCompress::run (void)
{
   t_pDevice      pDevice;
   t_pFifoStd     pFifoIn;
   t_pFifoStd     pFifoOut;
   t_pFifoBlock   pFifoBlockIn;
   t_pFifoBlock   pFifoBlockOut;
   bool            Finished  = false;
   quint64         Blocks    = 0;
   const int       SubFifoNr = pOwn->ThreadNr;
   size_t          CompressedMaxSize = 0;
   size_t          CompressedSize    = 0;
   LIBEWF_HANDLE *pEwfHandle;
   ssize_t         rc;

   LOG_INFO ("Acquisition of %s: Compression thread #%d started", QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice), pOwn->ThreadNr)

   pDevice  = pOwn->pDevice;

   if (pDevice->Acquisition.Format == t_File::EWF)
      CompressedMaxSize = (size_t) (pDevice->FifoBlockSize * 1.001) + 12; // see zlib documentation

   CHK_EXIT (pDevice->pThreadWrite->GetpFileHandle ((void **)&pEwfHandle))
   CHK_EXIT (pDevice->pFifoCompressIn ->GetSubFifo (SubFifoNr, pFifoIn ))
   CHK_EXIT (pDevice->pFifoCompressOut->GetSubFifo (SubFifoNr, pFifoOut))

   while (!Finished && !pDevice->AbortRequest)
   {
      CHK_EXIT (pFifoIn->Get (pFifoBlockIn))
      if (pFifoBlockIn)
      {
         Blocks++;
         switch (pDevice->Acquisition.Format)
         {
            case t_File::EWF:
               CHK_EXIT (t_Fifo::Create (pDevice->pFifoMemory, pFifoBlockOut, pDevice->FifoAllocBlockSize))

               CompressedSize          = CompressedMaxSize;   // Must be initialised with the max buffer size (we use this one instead of MULTITHREADED_COMPRESSION_FIFO_BLOCK_SIZE in order to check if ti works as tols in the zlib docu)
               pFifoBlockOut->Nr       = pFifoBlockIn->Nr;
               pFifoBlockOut->DataSize = pFifoBlockIn->DataSize;

               rc = libewf_raw_write_prepare_buffer (pEwfHandle, pFifoBlockIn ->Buffer, pFifoBlockIn->DataSize,
                                                                 pFifoBlockOut->Buffer, &CompressedSize,
                                                                &pFifoBlockOut->EwfCompressionUsed,
                                                                &pFifoBlockOut->EwfChunkCRC,
                                                                &pFifoBlockOut->EwfWriteCRC);
               if (pFifoBlockOut->EwfCompressionUsed)
               {
                  pFifoBlockOut->EwfDataSize = CompressedSize;   // Data to be forwarded is contained in
                  CHK_EXIT (t_Fifo::Destroy (pDevice->pFifoMemory, pFifoBlockIn))     // pFifoBlockOut, pFifoBlockIn no longer needed.
               }
               else
               {
                  pFifoBlockIn->EwfDataSize        = pFifoBlockIn ->DataSize;
                  pFifoBlockIn->EwfCompressionUsed = pFifoBlockOut->EwfCompressionUsed;
                  pFifoBlockIn->EwfChunkCRC        = pFifoBlockOut->EwfChunkCRC;
                  pFifoBlockIn->EwfWriteCRC        = pFifoBlockOut->EwfWriteCRC;
                  CHK_EXIT (t_Fifo::Destroy (pDevice->pFifoMemory, pFifoBlockOut))    // No compression, we'll forward the
                  pFifoBlockOut = pFifoBlockIn;                                       // original block we received
               }

               if (rc != (ssize_t) pFifoBlockOut->EwfDataSize)
               {
                  LOG_INFO ("libewf_raw_write_prepare_buffer returned %d. DataSize=%u, CompressedMaxSize=%zd, CompressedSize=%zd, EwfCompressionUsed=%d", (int)rc,
                            pFifoBlockOut->DataSize, CompressedMaxSize, CompressedSize, pFifoBlockOut->EwfCompressionUsed)
                  CHK_EXIT (ERROR_THREADCOMPRESS_LIBEWF_FAILED)
               }
               pFifoBlockOut->EwfPreprocessed = true;
               break;

            case t_File::AFF:
//               LOG_ERROR ("AaffPreprocess")
               CHK_EXIT (t_Fifo::Create (pDevice->pFifoMemory, pFifoBlockOut, pDevice->FifoAllocBlockSize))
               pFifoBlockOut->Nr       = pFifoBlockIn->Nr;
               pFifoBlockOut->DataSize = pFifoBlockIn->DataSize;

               CHK_EXIT (AaffPreprocess (&pFifoBlockOut->pAaffPreprocess, pFifoBlockIn->Buffer, pFifoBlockIn->DataSize, pFifoBlockOut->Buffer, pFifoBlockOut->BufferSize))

               if (pFifoBlockOut->pAaffPreprocess->Compressed)
               {
                  CHK_EXIT (t_Fifo::Destroy (pDevice->pFifoMemory, pFifoBlockIn))
               }
               else
               {
                  pFifoBlockIn->pAaffPreprocess = pFifoBlockOut->pAaffPreprocess;
                  CHK_EXIT (t_Fifo::Destroy (pDevice->pFifoMemory, pFifoBlockOut))
                  pFifoBlockOut = pFifoBlockIn;
               }
               break;

            default:
               CHK_EXIT (ERROR_THREADCOMPRESS_INVALID_FORMAT)
         }
         CHK_EXIT (pFifoOut->Insert (pFifoBlockOut))
      }
      else
      {
         LOG_INFO ("Dummy block")
         Finished = true;
      }
   }
   CHK_EXIT (pDevice->pThreadWrite->ReleaseFileHandle ())

   LOG_INFO ("Compression thread #%d exits now (device %s, %Ld blocks processed)", pOwn->ThreadNr, QSTR_TO_PSZ (pOwn->pDevice->LinuxDevice), Blocks)
}


void t_ThreadCompress::SlotFinished (void)
{
   emit SignalEnded (pOwn->pDevice, pOwn->ThreadNr);
}

