// MOLDYN.CPP

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "libghemicalconfig2.h"
#include "moldyn.h"

#include "atom.h"
#include "model.h"

#include "eng1_sf.h"	// the langevin stuff needs this...

#include <sstream>
using namespace std;

#define T_BUFF_SIZE	500

/*################################################################################################*/

moldyn::moldyn(engine * p1, f64 p2)
{
	eng = p1;
	
	tstep1 = p2;						// [1.0e-15 s]
	tstep2 = tstep1 * tstep1;				// [1.0e-30 s^2]
	
	vel = new f64[eng->GetAtomCount() * 3];			// [1.0e+3 m/s]
	acc = new f64[eng->GetAtomCount() * 3];			// [1.0e+12 m/s^2]
	
	mass = new f64[eng->GetAtomCount()];			// [kg/mol]
	
	locked = new char[eng->GetAtomCount()];
	
	step_counter = 0;
	
	sum_of_masses = 0.0;					// [kg/mol]
	
	atom ** glob_atmtab = eng->GetSetup()->GetAtoms();
	
	num_locked = 0; i32s counter = 0;
	while (counter < eng->GetAtomCount())
	{
		bool lflag = false;
		if (glob_atmtab[counter]->flags & ATOMFLAG_USER_LOCKED)
		{
			lflag = true;
			num_locked++;
		}
		
		mass[counter] = glob_atmtab[counter]->mass;
		mass[counter] *= 1.6605402e-27 * 6.0221367e+23;
		
		sum_of_masses += mass[counter];		// kg/mol ; all atoms.
		
		locked[counter] = lflag;
		
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			vel[counter * 3 + n1] = 0.0;
			acc[counter * 3 + n1] = 0.0;
		}
		
		counter++;
	}
	
	// the rest are just default values; can be modified...
	// the rest are just default values; can be modified...
	// the rest are just default values; can be modified...
	
	target_temperature = 300.0;	// [K]
	temperature_rtime = 100.0;	// [fs] ; should be > 100x timestep.
	
	target_pressure = 1.000;	// [bar]
	pressure_rtime = 1000.0;	// [fs] ; should be > 100x timestep.
	isoth_compr = 4.57e-5;		// [1/bar]
}

moldyn::~moldyn(void)
{
	delete[] vel;
	delete[] acc;
	
	delete[] mass;
	
	delete[] locked;
}

void moldyn::TakeMDStep(bool enable_temperature_control, bool enable_pressure_control)
{
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 tmpA = acc[n1 * 3 + n2];
			
			f64 tmp1 = tstep1 * vel[n1 * 3 + n2] * 1.0e-3;
			f64 tmp2 = tstep2 * tmpA * 0.5e-9;
			
			if (locked[n1])
			{
				tmpA = 0.0;	// make sure that locked atoms remain locked!
				tmp1 = 0.0;	// make sure that locked atoms remain locked!
				tmp2 = 0.0;	// make sure that locked atoms remain locked!
				
				// the engine class really cannot compute and return zero forces
				// for the locked atoms.
				
				// then how to ensure translational invariance and stuff like that?
			}
			
			eng->crd[n1 * 3 + n2] += tmp1 + tmp2;
			
			vel[n1 * 3 + n2] += tstep1 * tmpA * 0.5e-6;
		}
	}
	
	eng->DoSHAKE();
	
	eng->Compute(1, enable_pressure_control);	// ask to calculate virial if pressure needed...
	epot = eng->energy;
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		if (locked[n1]) continue;
		
		acc[n1 * 3 + 0] = -eng->d1[n1 * 3 + 0] / mass[n1];
		acc[n1 * 3 + 1] = -eng->d1[n1 * 3 + 1] / mass[n1];
		acc[n1 * 3 + 2] = -eng->d1[n1 * 3 + 2] / mass[n1];
		
		vel[n1 * 3 + 0] += tstep1 * acc[n1 * 3 + 0] * 0.5e-6;
		vel[n1 * 3 + 1] += tstep1 * acc[n1 * 3 + 1] * 0.5e-6;
		vel[n1 * 3 + 2] += tstep1 * acc[n1 * 3 + 2] * 0.5e-6;
	}
	
	f64 ekinCOMP[3];
	ekin = KineticEnergy(ekinCOMP);
	
	f64 current_temperature = ConvEKinTemp(ekin);
	
	if (enable_temperature_control)
	{
		// do a Berendsen-type temperature control step.
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		const f64 tmp1 = (target_temperature / ConvEKinTemp(ekin)) - 1.0;
		const f64 tmp2 = tstep1 / temperature_rtime;
		const f64 delta = sqrt(1.0 + tmp2 * tmp1);
		
		ekin *= delta;
		ekinCOMP[0] *= delta;
		ekinCOMP[1] *= delta;
		ekinCOMP[2] *= delta;
		
		SetEKin(ekin);
		current_temperature = ConvEKinTemp(ekin);
	}
	
	if (enable_pressure_control)
	{
		// the unit of virial is this:
		// [nm] * [kJ / (mol * nm)] = [kJ/mol] ; the kilo must be handled.
		
		// volume must be m^3 / mol (multiplied by Avogadro constant).
		
		f64 pressure[3] =
		{
			target_pressure,
			target_pressure,
			target_pressure
		};
		
		f64 volume;
		
	// eng_pbc != NULL if we will use a system with periodic boundary conditions...
		engine_pbc * eng_pbc = dynamic_cast<engine_pbc *>(eng);
		if (eng_pbc != NULL)
		{
			f64 tmpVOL = eng_pbc->box_HALFdim[0] * eng_pbc->box_HALFdim[1] * eng_pbc->box_HALFdim[2];
			
			// exponent = 23 - 9 - 9 - 9 = -4
			// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
			// -9 comes from nanometers (in all tree dimensions).
			
			tmpVOL *= 8.0 * 6.0221367e-4;	// 2^3 = 8 for half-dimensions.
			volume = tmpVOL;		// store it as m^3 / mol...
			
			// +3 comes from the kilo prefix of Ekin and virial.
			// -5 comes from Pa -> bar conversion.
			
			for (i32s n1 = 0;n1 < 3;n1++)
			{
				pressure[n1] = 1.0e-2 * (2.0 * ekinCOMP[n1] + eng->virial[n1]) / tmpVOL;
			}
		}
		
		// the pressure components are now in bar units.
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		const f64 tot_P = (pressure[0] + pressure[1] + pressure[2]) / 3.0;
		
		// do a Berendsen-type pressure control step.
		// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		const f64 tmp1 = (target_pressure - tot_P);
		const f64 tmp2 = isoth_compr * tstep1 / pressure_rtime;
		const f64 delta = pow(1.0 - tmp2 * tmp1, 0.3333333333333333);
		
		eng->ScaleCRD(delta, delta, delta);
		
		if (eng_pbc != NULL)
		{
			eng_pbc->box_HALFdim[0] *= delta;
			eng_pbc->box_HALFdim[1] *= delta;
			eng_pbc->box_HALFdim[2] *= delta;
			
			model * mdl = eng->GetSetup()->GetModel();
			mdl->saved_periodic_box_HALFdim[0] = eng_pbc->box_HALFdim[0];
			mdl->saved_periodic_box_HALFdim[1] = eng_pbc->box_HALFdim[1];
			mdl->saved_periodic_box_HALFdim[2] = eng_pbc->box_HALFdim[2];
		}
		
		saved_pressure = tot_P;
		saved_density = 0.001 * sum_of_masses / volume;		// kg/dm^3
	}
	
	step_counter++;
}

f64 moldyn::KineticEnergy(f64 * comp)
{
	if (comp != NULL)
	{
		comp[0] = 0.0;
		comp[1] = 0.0;
		comp[2] = 0.0;
	}
	
	f64 energy = 0.0;
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		const f64 tmpX = 500.0 * mass[n1];
		
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			const f64 tmp1 = vel[n1 * 3 + n2];
			const f64 tmp2 = tmpX * tmp1 * tmp1;
			
			energy += tmp2;
			
			if (comp != NULL)
			{
				comp[n2] += tmp2;
			}
		}
	}
	
	return energy;
}

// <Ekin> = 3/2 * k * T		// <Ekin> = <0.5 * m * v^2) = average kinetic energy of a molecule.
// EKin = 3N/2 * k * T		// Ekin = total kinetic energy, N = number of atoms (3N = degrees of freedom).
// EKin = 3N/2 * R * T		// if Ekin is expressed "per mole", change the constant k to per-mole-constant R.

f64 moldyn::ConvTempEKin(f64 p1)
{
	return (3.0 / 2.0) * p1 * ((eng->GetAtomCount() - num_locked) * 8.314510) / 1000.0;
}

f64 moldyn::ConvEKinTemp(f64 p1)
{
	return (2.0 / 3.0) * p1 * 1000.0 / ((eng->GetAtomCount() - num_locked) * 8.314510);
}

void moldyn::SetEKin(f64 p1)
{
	f64 tmp1 = p1 / KineticEnergy();
	f64 tmp2 = (tmp1 < 0.0 ? 0.0 : sqrt(tmp1));
	
	i32u tmp3 = eng->GetAtomCount() * 3;
	for (i32u n1 = 0;n1 < tmp3;n1++) vel[n1] *= tmp2;
}

/*################################################################################################*/

moldyn_langevin::moldyn_langevin(engine * p1, f64 p2) : moldyn(p1, p2)
{
	if (num_locked != 0)
	{
		cout << "locked atoms not supported in moldyn_langevin!" << endl;
		exit(EXIT_FAILURE);
	}
	
	eng1_sf * engsf = dynamic_cast<eng1_sf *>(eng);
	
	// at the moment this code is limited to SF implicit solvation motion damping...
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
	
	if (engsf == NULL) { cout << "engsf cast failed!" << endl; exit(EXIT_FAILURE); }
	if (!engsf->use_implicit_solvent) { cout << "engsf must use implicit solvation!" << endl; exit(EXIT_FAILURE); }
	
	langevin_r_forces = new f64[eng->GetAtomCount() * 3];
	
	langevin_weight = new f64[eng->GetAtomCount()];
	langevin_frict = new f64[eng->GetAtomCount()];
	
	i32s counter = 0;
	while (counter < eng->GetAtomCount())
	{
		langevin_weight[counter] = NOT_DEFINED;		// this is -1.0 -> disabled!!!
		langevin_frict[counter] = 0.0;
		
		counter++;
	}
	
	// the rest are just default values; can be modified...
	// the rest are just default values; can be modified...
	// the rest are just default values; can be modified...
	
// Pastor RW, Brooks BR, Szabo A : "An analysis of the accuracy of Langevin
// and molecular dynamics algorithm", Mol Phys, 65, 1409-1419 (1988)

// the above article says frict should be 50 ps^-1 for H2O???
// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// not sure what it means but seems that the constant here should be much more than that...
// setting langevin_frict_fc = 4.0e-6 1/s will yield v_loss/stp = 1.0% ; roughly at range then.

//	langevin_frict_fc = 8.0e-6;	// see model::DoMolDyn() ; v_loss/stp = 2.0% ; Bfact ??? ; 20050124
	langevin_frict_fc = 16.0e-6;	// see model::DoMolDyn() ; v_loss/stp = 4.0% ; Bfact OK ; 20050124
	
	langevin_random_fc = 0.00;	// see model::DoMolDyn() ; 3450.0
	langevin_coupling = 0.00;	// see model::DoMolDyn() ; 0.0005
	
	for (i32s n1 = 0;n1 < engsf->GetSetup()->GetSFAtomCount() - engsf->num_solvent;n1++)
	{
		f64 weight = 1.0;	// make this depend on SA??? update dynamically below then.
		
		langevin_weight[n1] = weight;
		langevin_frict[n1] = weight * langevin_frict_fc;
	}
}

moldyn_langevin::~moldyn_langevin(void)
{
	delete[] langevin_r_forces;
	
	delete[] langevin_weight;
	delete[] langevin_frict;
}

void moldyn_langevin::TakeMDStep(bool enable_temperature_control)
{
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 tmp1 = tstep1 * vel[n1 * 3 + n2] * 1.0e-3;
			f64 tmp2 = tstep2 * acc[n1 * 3 + n2] * 0.5e-9;
			eng->crd[n1 * 3 + n2] += tmp1 + tmp2;
			
			vel[n1 * 3 + n2] += tstep1 * acc[n1 * 3 + n2] * 0.5e-6;
		}
	}
	
	eng->DoSHAKE();
	
	eng->Compute(1);
	epot = eng->energy;
	
// maintaining the correct temperature in langevin dynamics seems to be a bit problematic.
// constant energy simulations are very difficult, but it's quite easy to calculate the energy loss.

// the friction term will gain strength when T is increased, while the random term only depends on
// the force constant -> for each pair of friction/random settings, there is an equilibrium temperature
// that will be maintained without any heating.

// strategy: find friction/random settings with equilibrium T a bit below simulation temperature,
// and use mild temperature control to maintain the simulation temperature???
	
	// how to make sure that the random numbers really stay random???
	// a lot of rand() calls are used here -> periodicity near/after RAND_MAX numbers!!!
	//////////////////////
	//srand(time(NULL));
	//////////////////////
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		if (langevin_weight[n1] >= 0.0)
		{
			// generate gaussian random numbers using a (modified???) Box-Muller transformation:
			// y = sqrt(-2*ln(x1))*cos(2*pi*x2)	// x1 and x2 = linear random numbers ]0..1].
			
			i32u rnum;	// return value 0 from rand() would lead to ln(0.0)...
			do { rnum = rand(); } while (rnum == 0); f64 r1 = (f64) rnum / (f64) RAND_MAX;
			do { rnum = rand(); } while (rnum == 0); f64 r2 = (f64) rnum / (f64) RAND_MAX;
			do { rnum = rand(); } while (rnum == 0); f64 r3 = (f64) rnum / (f64) RAND_MAX;
			
			f64 rA = sqrt(-2.0 * log(r1)) * cos(2.0 * M_PI * r2);
			f64 rB = sqrt(-2.0 * log(r2)) * cos(2.0 * M_PI * r3);
			f64 rC = sqrt(-2.0 * log(r3)) * cos(2.0 * M_PI * r1);
			
			langevin_r_forces[n1 * 3 + 0] = rA * langevin_weight[n1];
			langevin_r_forces[n1 * 3 + 1] = rB * langevin_weight[n1];
			langevin_r_forces[n1 * 3 + 2] = rC * langevin_weight[n1];
		}
	}
	
	f64 net_random_x = 0.0;
	f64 net_random_y = 0.0;
	f64 net_random_z = 0.0;
	
	i32s langevin_counter = 0;
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		if (langevin_weight[n1] >= 0.0)
		{
			net_random_x += langevin_r_forces[n1 * 3 + 0];
			net_random_y += langevin_r_forces[n1 * 3 + 1];
			net_random_z += langevin_r_forces[n1 * 3 + 2];
			
			langevin_counter++;
		}
	}
	
	if (langevin_counter > 0)
	{
		net_random_x /= (f64) langevin_counter;
		net_random_y /= (f64) langevin_counter;
		net_random_z /= (f64) langevin_counter;
	}
	
	for (i32s n1 = 0;n1 < eng->GetAtomCount();n1++)
	{
		acc[n1 * 3 + 0] = -eng->d1[n1 * 3 + 0] / mass[n1];
		acc[n1 * 3 + 1] = -eng->d1[n1 * 3 + 1] / mass[n1];
		acc[n1 * 3 + 2] = -eng->d1[n1 * 3 + 2] / mass[n1];
		
		if (langevin_weight[n1] >= 0.0)
		{
			// ma = -grad - bmv + f(t)	;; b = frictional constant
			// a = -grad/m - bv + f(t)/m	;; b = frictional constant
			
			// the frictional constant b must have a unit 1/s ; [b] = 1/s
			// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
			// is the effect independent of MD timestep??? should be...
			
			// friction...
			// friction...
			// friction...
			
			f64 frict_const = langevin_frict[n1] * 1.0e+9;
			acc[n1 * 3 + 0] -= frict_const * vel[n1 * 3 + 0];
			acc[n1 * 3 + 1] -= frict_const * vel[n1 * 3 + 1];
			acc[n1 * 3 + 2] -= frict_const * vel[n1 * 3 + 2];
			
//static int lcnt = 0; if (n1 == 15 && !(lcnt++ % 1000)) {
//cout << "ATOM " << n1 << " velocity losses = ";
//cout << (frict_const * vel[n1 * 3 + 0] * tstep1 * 0.5e-6) / vel[n1 * 3 + 0] << " ";
//cout << (frict_const * vel[n1 * 3 + 1] * tstep1 * 0.5e-6) / vel[n1 * 3 + 1] << " ";
//cout << (frict_const * vel[n1 * 3 + 2] * tstep1 * 0.5e-6) / vel[n1 * 3 + 2] << endl; }
			
			// random forces...
			// random forces... NOT WORKING CORRECTLY YET???
			// random forces...
			
			f64 random_fc = langevin_random_fc / (tstep1 * mass[n1]);
			acc[n1 * 3 + 0] += (langevin_r_forces[n1 * 3 + 0] - net_random_x) * random_fc;
			acc[n1 * 3 + 1] += (langevin_r_forces[n1 * 3 + 1] - net_random_y) * random_fc;
			acc[n1 * 3 + 2] += (langevin_r_forces[n1 * 3 + 2] - net_random_z) * random_fc;
		}
		
		vel[n1 * 3 + 0] += tstep1 * acc[n1 * 3 + 0] * 0.5e-6;
		vel[n1 * 3 + 1] += tstep1 * acc[n1 * 3 + 1] * 0.5e-6;
		vel[n1 * 3 + 2] += tstep1 * acc[n1 * 3 + 2] * 0.5e-6;
	}
	
	ekin = KineticEnergy();
	if (enable_temperature_control)
	{
		f64 delta = (target_temperature / ConvEKinTemp(ekin)) - 1.0;
		
		f64 tc = sqrt(1.0 + temperature_coupling * delta);
		SetEKin(ekin * tc);
		
		if (langevin_coupling > 0.0)
		{
			f64 lc = sqrt(1.0 + langevin_coupling * delta);
			langevin_random_fc *= lc;
			
			if (!(step_counter%100))
			{
				cout << "langevin_random_fc = " << langevin_random_fc << endl;
			}
		}
	}
	
	step_counter++;
}

/*################################################################################################*/

// eof
