/***************************************************************************
                           map.cpp  -  map class
                             -------------------
    begin                : Tue Oct 14 2003
    copyright            : (C) 2003 by Paul Robson
    email                : autismuk@autismuk.freeserve.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cob.hxx"

//	**************************************************************************************************************************
//
//							                    Reset the map to empty
//
//	**************************************************************************************************************************

void Map::ResetMap(int xs,int ys)
{
    surface.SetColour(0,0,0);surface.FillRect();// Clear the background
    xsize = xs;ysize = ys;                      // Set size
    mapremaining = 0;
    hiddenflag = 0;                             // Nothing hidden
    for (int x = 0;x < xs;x++)                  // Clear map
        for (int y = 0;y < ys;y++)
                            map[x][y] = 0;
}

//	**************************************************************************************************************************
//
//							                    Write a cell on the map
//
//	**************************************************************************************************************************

void Map::Write(int x,int y,int n)
{
    SDWASSERT(x >= 0 && x < xsize);             // Check values
    SDWASSERT(y >= 0 && y < ysize);
    if (Read(x,y) == n) return;                 // No change
    if (Read(x,y) == 0 && (n & 15) != 0)        // Created an uncoloured, increment count
                                    mapremaining++;
    if ((Read(x,y) ^ n) & M_COLOUR)             // Colour changed ?
    {
        if (n & M_COLOUR) mapremaining--;       // Gone to colour, decrement count
        else              mapremaining++;       // Else increment count
    }
    map[x][y] = n;                              // Update value

    int xc,yc;
    xc = SPRITESIZE * (n % 16);yc = 0;          // Cell to use
    if (n & M_COLOUR) yc = SPRITESIZE;          // Different colour if filled in
    if (n & M_HIDDEN) xc = yc = 0;              // If hidden, write as blank
    Rect r(xc,yc,xc+SPRITESIZE-1,yc+SPRITESIZE-1);
    graphics.Copy(surface,r,x*SPRITESIZE,y*SPRITESIZE);
}

//	**************************************************************************************************************************
//
//							                    Draw a vertical line
//
//	**************************************************************************************************************************

void Map::VLine(int x,int y1,int y2)
{
    if (y1 > y2) { int t = y1;y1 = y2;y2 = t; }
    for (int y = y1;y <= y2;y++)
    {
        int n = Read(x,y);
        if (y > y1) n = n | M_UP;
        if (y < y2) n = n | M_DOWN;
        Write(x,y,n);
    }
}

//	**************************************************************************************************************************
//
//							                    Draw a horizontal line
//
//	**************************************************************************************************************************

void Map::HLine(int x1,int x2,int y)
{
    if (x1 > x2) { int t = x1;x1 = x2;x2 = t; }
    for (int x = x1;x <= x2;x++)
    {
        int n = Read(x,y);
        if (x > x1) n = n | M_LEFT;
        if (x < x2) n = n | M_RIGHT;
        Write(x,y,n);
    }
}

//	**************************************************************************************************************************
//
//							                            Draw a box
//
//	**************************************************************************************************************************

void Map::Box(int x1,int y1,int x2,int y2)
{
    HLine(x1,x2,y1);HLine(x1,x2,y2);
    VLine(x1,y1,y2);VLine(x2,y1,y2);
}

//	**************************************************************************************************************************
//
//							                    Hide a square on the map
//
//	**************************************************************************************************************************

void Map::Hide(int x,int y)
{
    if (hiddenflag != 0) return;                // One hidden square at a time (!)
    xhidden = x;yhidden = y;                    // Update hidden values
    hiddenflag = 1;                             // Mark as hidden
    Write(x,y,Read(x,y) | M_HIDDEN);            // Hide it
    restore.ResetTimer(2500);                   // Hide for 2.5 seconds
}    

//	**************************************************************************************************************************
//
//							     Check to see if it is time to restore the hidden square
//
//	**************************************************************************************************************************

void Map::CheckHide(void)
{
    if (hiddenflag != 0)                        // If there is a hidden square
        if (restore.TimedOut())                 // And the timer has timed out.
        {
            hiddenflag = 0;                     // Unhide it.
            Write(xhidden,yhidden,Read(xhidden,yhidden) - M_HIDDEN);
        }
}

//	**************************************************************************************************************************
//
//							     Load a level layout (there are four in the original)
//
//	**************************************************************************************************************************

int Map::Load(int Level)
{
    int Result = 1;
    int r,g,b,Black = 1;
    ResetMap(32,24);                            // Reset the map
    switch(Level)
    {
        case 1:
            r = b = 0;g = 255;
            Box(1,1,9,9);Box(1,13,9,21);
            Box(22,1,30,9);Box(22,13,30,21);
            Box(9,9,22,13);
            break;
        case 2:
            r = 0;b = g = 255;
            HLine(1,13,1);HLine(18,30,1);
            VLine(1,1,21);VLine(30,1,21);
            HLine(1,11,21);HLine(30,20,21);
            VLine(11,18,21);VLine(20,18,21);
            VLine(13,1,18);VLine(18,1,18);
            Box(13,18,3,7);Box(18,18,28,7);
            HLine(3,18,7);
            break;
        case 3:
            r = 255;g = b = 0;Black = 0;
            VLine(1,1,21);HLine(1,30,21);
            HLine(1,27,1);VLine(27,1,19);
            HLine(4,27,19);VLine(4,19,3);
            HLine(4,24,3);VLine(24,3,17);
            HLine(7,24,17);VLine(7,17,5);
            HLine(7,21,5);VLine(21,5,15);
            HLine(21,10,15);VLine(10,15,7);
            HLine(10,18,7);VLine(18,7,13);
            HLine(18,13,13);VLine(13,13,10);
            HLine(13,30,10);VLine(30,10,21);
            break;
        case 4:
            r = g = 255;b = 0;
            HLine(1,29,11);
            for (int i = 3;i < 29;i = i + 3)
            {
                VLine(i,1,21);
                if (i % 2 != 0)
                {
                    HLine(i,i - ((i == 3) ? 2 : 3),21);
                    HLine(i,i + ((i == 27) ? 2 : 3),1);
                }
            }
            VLine(1,11,21);VLine(29,1,11);
            break;            
        default:
            Result = 0;break;
    }
    if (Result)
    {
        Rect rc(0,767-SPRITESIZE-4,1023,767);
        char Buffer[32];
        surface.SetColour(r/2,g/2,b/2);
        surface.FillRect(rc);
        surface.SetColour(255,255,255);
        rc.Top += 4;
        if (Black) surface.SetColour(0,0,0);
        rc.Left = 128;rc.Right = 1023-rc.Left;
        sprintf(Buffer,"- - - F R A M E %d - - -",Level);
        surface.String(rc,Buffer);
    }
    return Result;
}

