/*
 * cddb.cpp
 */
#include "config.h"
#include "misc_utils.h"

#include <stdbool.h>
#include <stdio.h>

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <stddef.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
#endif

#ifdef HAVE_STRING_H
# if !defined STDC_HEADERS && defined HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif

#include <glib.h>
#include <glib/gi18n.h>

#include "cddb.h"
#include "cddbp.h"
#include "id3v1genres.h"

#include "misc_utils.h"
#include "select_frame_handler.h"
#include "main_window_handler.h"

#include <string>
#include <sstream>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <memory>
#include <cmath>
#include <map>

using std::ostringstream;
using std::setw;
using std::setfill;
using std::string;
using std::map;

#define NAME "CATraxx"
#define MAX_CDDB_FILE_SIZE 15360

// #define DEBUG

enum errors
{
    OK, REMOTE_OK, LOCAL_OK, NOT_FOUND, NO_CONNECT, CONNECT_REFUSED, SERVER_ERROR, PARAM_ERROR
};
enum lookup_protocol { CDDBP, HTTP };
enum lookup_order { NONE, LOCAL, REMOTE, LOCAL_REMOTE, REMOTE_LOCAL };

int cddb_handle_data(const string& data, string& artist, string& dtitle, std::map<int,string> &titles,
                     int &totaltracks, string& year, string& dgenre);
int do_cddb(string& result, string& disc_category, int tracknum, int duration,
            std::vector<long int>& offset,
            const string& server, int port, int proto);


int
cddb_handle_data(const string& data, string& artist, string& dtitle, std::map<int,string> &titles,
                 int &totaltracks, string& year, string& dgenre)
{  
  const string tmp = data;
  int i, j, counter = 0, track, previoustrack = 100, ttcounter = 0;
  
  if(data.find("# xmcd") != 0)
    {
#ifdef DEBUG
      std::cout << "cddb_handle_data: ERROR in format " << std::endl;
#endif
      return -1;
    }
    
    dtitle = "";

    int itmp = 0;
    while(itmp+1 < tmp.length())  	/* check against end of string */
      {
      /* get the row */
      const int itmp_new1 = tmp.find("\r", itmp);
      const int itmp_new2 = tmp.find("\n", itmp);
      const int itmp_new = std::min(itmp_new1, itmp_new2);
      const string row = tmp.substr(itmp, itmp_new - itmp);
      itmp = itmp_new;

#ifdef DEBUG
      std::cout << "cddb_handle_data, row: " << row << std::endl;
#endif

      while(tmp[itmp] == '\r' || tmp[itmp] == '\n')
        ++itmp;
      
        /* eval the row of type xxxx = yyyy*/

        const string tempstr = row.substr(row.find('=')+1, row.length()-row.find('=')-1);
        
        if(row.find("DYEAR") == 0)       /* CD Year */
          {
            year = tempstr;
            //remove_non_ascii_chars(*year);
        }
        else if(row.find("DGENRE") == 0)          /* Disc Genre */
          {
            dgenre = tempstr;
            //remove_non_ascii_chars(*dgenre);
        }
        else if(row.find("TTITLE") == 0)          /* Track Title */
        {
            /* get the track number before going on */
            /* skip the TTITLE */
          string track_str = row.substr(6, row.find("=", 6));
	  track = std::stoi(trim(track_str, " "));
	  // printf ("title = %s\n", mark);
	  
            if(previoustrack != track)
            {
                ttcounter++;
            }
            /* put in the track name */
	    
	    titles[track] = tempstr;
            trim(titles[track]);
            previoustrack = track;

            //printf("before %s\n", titles[track]);            
            //remove_non_ascii_chars(titles[ track ]);
            //printf("after %s\n", titles[track]);

#ifdef DEBUG
            printf("Track %d: \'%s\'\n", track, titles[ track ].c_str());
#endif
        }
        else if((row.find("DTITLE") == 0) && dtitle.empty())              /* CD Title */
        {
	    /* skip to the data */
	    //const char * mark = row.c_str()+row.find('=')+1;
	    
            // tm:  hack around bogus CDDB entries
            if (tempstr.find(" / ") != string::npos)
            {
                artist = tempstr.substr(0, tempstr.find(" / "));
                dtitle = tempstr.substr(tempstr.find(" / ")+3, tempstr.length()-tempstr.find(" / ")-3);
            }
            else
            {
#ifdef DEBUG
                printf("malformed DTITLE, copying full DTITLE into both artist and dtitle\n");
#endif
                artist = tempstr;
                dtitle = tempstr;
            }

            trim(artist, " ");
            trim(dtitle, " ");

            //remove_non_ascii_chars(*artist);
            //remove_non_ascii_chars(*dtitle);
            std::replace( artist.begin(), artist.end(), '/', '_');
            std::replace( dtitle.begin(), dtitle.end(), '/', '-'); 
            //convert_slashes(*artist, '_'); // dc: _ Artist, - others
            //convert_slashes(*dtitle, '-');

#ifdef CONVERT_SPACES_IN_ID3_DATA
            if(config.cddb_config.convert_spaces == true)
            {
               std::replace( artist.begin(), artist.end(), ' ', '_');
               std::replace( dtitle.begin(), dtitle.end(), ' ', '_'); 
              //convert_spaces(*artist, '_');
              //convert_spaces(*dtitle, '_');
            }
#endif
#ifdef DEBUG
            printf("Artist: %s\n", artist.c_str());
            printf("Dtitle: %s\n", dtitle.c_str());
#endif
        }

        /* ignore any other results */
    }

    totaltracks = ttcounter;
    return 0;
}

unsigned long int cddb_disk_id(int length, int tracknum, std::vector<long int>& offset)
{
    int track;
    long int result = 0, tmp;

    for(track = 0; track < tracknum; track++)
    {
        tmp = (offset[ track ] + 150) / 75;

        do
        {
            result += tmp % 10;
            tmp /= 10;
        }
        while(tmp != 0);
    }

    return (result % 0xff) << 24 | (length + 2 - ((offset[ 0 ] + 150) / 75)) << 8 | tracknum;
}

int do_cddb(string& result, string& disc_category, int tracknum, int duration,
            std::vector<long int>& offset,
            const string& server, int port, int proto)
{
    FILE *sock = NULL;
    int status, matches, i;
    char **category = NULL, **title = NULL, **alt_id = NULL;
    string final_cat, final_id, cd_id;
    
    /* chop up the URL */
    string wwwserver(server);
    string uri = wwwserver.substr(wwwserver.find("/")+1);
    wwwserver = wwwserver.substr(0, wwwserver.find("/"));

    ostringstream cd_id_str;
    cd_id_str << std::setw(8) << std::setfill('0') << std::hex << cddb_disk_id(duration, tracknum, offset);
    cd_id = cd_id_str.str();

#ifdef DEBUG
    printf("do_cddb\n");
    printf("cddb_disk_id returned '%s'\n", cd_id.c_str());
    printf("wwwserver %s, uri %s \n", wwwserver.c_str(), uri.c_str());
#endif

    if(proto == CDDBP)
    {
        sock = socket_init(wwwserver, port);

        if(!sock)
        {
            return NO_CONNECT;
        }

        /* connect to the cddb server */
        status = cddbp_signon(sock);

        switch(status)
        {
            case 200 :
            case 201 :
                break;         	/* might want to differenciate when write() is supported on the server side */
            case 432 :
            case 433 :
            case 434 :
                fclose(sock);
                return CONNECT_REFUSED;
            default :
                fclose(sock);
                return SERVER_ERROR;
        }

        /* do the hello handshake */
        status = cddbp_handshake(sock, NAME, VERSION);

        switch(status)
        {
            case 200 :
            case 402 :
                break;
            case 431 :
                fclose(sock);
                return CONNECT_REFUSED;
            default :
                fclose(sock);
                return SERVER_ERROR;
        }

        status = cddbp_query(sock, cd_id.c_str(), tracknum, &(offset.front()),
                             duration, &matches,
                             &category, &title, &alt_id);
    }

#if 0
    /* get cddb information from a local http server (plain text file) */
    /* must change CDDB URL to be http://localhost/testentry.txt */
    strcpy(final_cat, "test");
    strcpy(final_id, "test");
#else
    else
    {
        if(config.cddb_config.proxy_server.empty())
            status = http_query(wwwserver.c_str(), port, uri.c_str(), cd_id.c_str(), tracknum,
                                &(offset.front()), duration, &matches,
                                &category, &title, &alt_id, NAME, VERSION);
        else
            status = http_query_proxy(wwwserver.c_str(), port,
                                      config.cddb_config.proxy_server.c_str(), config.cddb_config.proxy_port,
                                      uri.c_str(), cd_id.c_str(), tracknum, &(offset.front()), duration, &matches,
                                      &category, &title, &alt_id, NAME, VERSION);

    }

    switch(status)
    {
        case 200 :
        case 211 :
            break;
        case 999:
            return NO_CONNECT;
        case 202 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return NOT_FOUND;
        case 403 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return SERVER_ERROR;
        default :

            if(proto == CDDBP)
            {
                fclose(sock);
            }

            return SERVER_ERROR;
    }

    /* we don't care about multiple matches, just grab the first one */
    final_cat = category[ 0 ];
    disc_category = final_cat;
    final_id = alt_id[ 0 ];

    for(i = 0; i < matches; i++)
    {
        free(category[ i ]);
        free(title[ i ]);
        free(alt_id[ i ]);
    }

    free(category);
    free(title);
    free(alt_id);
#endif

    /* now finally grab the data for the disc id and category */
    if(proto == CDDBP)
    {
        status = cddbp_read(sock, final_cat.c_str(), final_id.c_str(), result);
    }
    else if(config.cddb_config.proxy_server.empty())
    {
      status = http_read(wwwserver.c_str(), port, uri.c_str(), final_cat.c_str(), final_id.c_str(), result, NAME, VERSION);
    }
    else
    {
        status = http_read_proxy(wwwserver.c_str(), port, config.cddb_config.proxy_server.c_str(), config.cddb_config.proxy_port, uri.c_str(), final_cat.c_str(), final_id.c_str(), result, NAME, VERSION);
    }

    switch(status)
    {
        case 210 :
            break;
        case 401 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return NOT_FOUND;
        case 403 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return SERVER_ERROR;
        default :

            if(proto == CDDBP)
            {
                fclose(sock);
            }

            return SERVER_ERROR;
    }

    if(proto == CDDBP)
    {
        cddbp_signoff(sock);
    }

    //free(final_id);
    return REMOTE_OK;
}

/*
	  check the local file system to see if the file
	  already exists by looping thru the catagories and the
	  predefined prefix
*/
static int read_local_file(string& result, int tracknum, int duration, std::vector<long int>& offset)
{
    int i;

    string cd_id;
    ostringstream cd_id_str;
    cd_id_str << std::setw(8) << std::setfill('0') << std::hex << cddb_disk_id(duration, tracknum, offset);
    cd_id = cd_id_str.str();

    /* first, check the directory pointed to by the config file */
    const string file_check = config.cddb_path + '/' + cd_id;

#ifdef DEBUG
    printf("read_local_file\n");
    printf("cddb_disk_id returned '%s'\n", cd_id.c_str());
    printf("path: %s\n", config.cddb_path.c_str());
    printf("file: %s\n", file_check.c_str());
#endif
    
    std::ifstream datafile(file_check.c_str());
    if (!datafile.is_open())
      {
        /*
         *  since the file didn't exist in the directory
         * 	check the category sub-directories
         */

      const TagLib::StringList genre_description = TagLib::ID3v1::genreList();
      std::string file_check_genre;
      
      for(i = 0; i < genre_description.size(); i++)
        {
	  file_check_genre  = config.cddb_path;              /*   start with the base path  */
	  file_check_genre += '/';                           /*   add the seperator  */
	  file_check_genre += genre_description[i].to8Bit(); /*   add the catagory  */
	  file_check_genre += '/';                           /*   add the seperator  */
	  file_check_genre += cd_id;                         /*   add the filename  */
	  
	  datafile.open(file_check_genre.c_str());
          if (!datafile.is_open())
            {
	      break;
            }
        }
    }

    if(!datafile.is_open())
    {
#ifdef DEBUG
        printf("could not open cache file '%s'\n", file_check.c_str());
#endif
        return (NOT_FOUND);
    }
    
    datafile.seekg(0, std::ios::end);   
    result.reserve(datafile.tellg());
    datafile.seekg(0, std::ios::beg);

    result.assign((std::istreambuf_iterator<char>(datafile)),
                  std::istreambuf_iterator<char>());
 
    datafile.close();

#ifdef DEBUG
      printf("retrieved from cache: '%s'\n", result.c_str());
#endif

    return (REMOTE_OK);    
}

int do_cddb_proxy(string& result, string& disc_category, int tracknum, int duration,
                  std::vector<long int>& offset, const string& server, int port, int proto)
{
    FILE *sock = NULL;
    int status, matches, i;
    char **category = NULL, **title = NULL, **alt_id = NULL;
    string final_cat, final_id, cd_id;

    /* chop up the URL */
    string wwwserver(server);
    string uri = wwwserver.substr(0, wwwserver.find("/"));
    wwwserver = uri;

    final_cat = "";
    final_id = "";
    cd_id = "";

    ostringstream cd_id_str;
    cd_id_str << std::setw(8) << std::setfill('0') << std::hex << cddb_disk_id(duration, tracknum, offset);
    cd_id = cd_id_str.str();

    if(proto == CDDBP)
    {
        sock = socket_init(wwwserver, port);
        
        if(!sock)
        {
            return NO_CONNECT;
        }

        /* connect to the cddb server */
        status = cddbp_signon(sock);

        switch(status)
        {
            case 200 :
            case 201 :
                break;         	/* might want to differentiate when write() is supported on the server side */
            case 432 :
            case 433 :
            case 434 :
                fclose(sock);
                return CONNECT_REFUSED;
            default :
                fclose(sock);
                return SERVER_ERROR;
        }

        /* do the hello handshake */
        status = cddbp_handshake(sock, NAME, VERSION);

        switch(status)
        {
            case 200 :
            case 402 :
                break;
            case 431 :
                fclose(sock);
                return CONNECT_REFUSED;
            default :
                fclose(sock);
                return SERVER_ERROR;
        }

        status = cddbp_query(sock, cd_id.c_str(), tracknum, &(offset.front()), duration, &matches,
                             &category, &title, &alt_id);
    }

#ifdef TESTLOCALHTTP
    /* get cddb information from a local http server (plain text file) */
    /* must change CDDB URL to be http://localhost/testentry.txt */
    strcpy(final_cat, "test");
    strcpy(final_id, "test");
#else
    else
    {
        status = http_query(wwwserver.c_str(), port, uri.c_str(), cd_id.c_str(), tracknum,
                            &(offset.front()), duration, &matches,
                            &category, &title, &alt_id, NAME, VERSION);
    }

    switch(status)
    {
        case 200 :
        case 211 :
            break;
        case 999:
            return NO_CONNECT;
        case 202 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return NOT_FOUND;
        case 403 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return SERVER_ERROR;
        default :

            if(proto == CDDBP)
            {
                fclose(sock);
            }

            return SERVER_ERROR;
    }

    /* we don't care about multiple matches, just grab the first one */
    final_cat = category[ 0 ];
    disc_category = final_cat;
    final_id = alt_id[ 0 ];

    for(i = 0; i < matches; i++)
    {
        free(category[ i ]);
        free(title[ i ]);
        free(alt_id[ i ]);
    }

    free(category);
    free(title);
    free(alt_id);
#endif

    /* now finally grab the data for the disc id and category */
    if(proto == CDDBP)
    {
        status = cddbp_read(sock, final_cat.c_str(), final_id.c_str(), result);
    }
    else
        status = http_read(wwwserver.c_str(), port, uri.c_str(), final_cat.c_str(), final_id.c_str(), result,
                           NAME, VERSION);

    switch(status)
    {
        case 210 :
            break;
        case 401 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return NOT_FOUND;
        case 403 :

            if(proto == CDDBP)
            {
                cddbp_signoff(sock);
            }

            return SERVER_ERROR;
        default :

            if(proto == CDDBP)
            {
                fclose(sock);
            }

            return SERVER_ERROR;
    }

    if(proto == CDDBP)
    {
        cddbp_signoff(sock);
    }

    //free(final_id);
    return REMOTE_OK;
}

int cddb_main(_main_data *main_data)
{
    int err, i, totaltracks;

    string result;
    string artist, dtitle, category, year, dgenre;
    map<int, string> titles;

    int tracknum = main_data->num_tracks();
    int duration = main_data->total_length;
    std::vector<long int> offset(tracknum, 0);

    main_window_handler(MW_UPDATE_STATUSBAR, _("Contacting CDDB server..."), NULL);

    while(gtk_events_pending())
    {
        gtk_main_iteration();
    }

    /* grab the track offsets from the main_data structure */
    for(i = 0; i < tracknum; i++)
    {
        offset[ i ] = main_data->track[ i ].begin;
    }

    /* check for a local file first */
    err = read_local_file(result, tracknum, duration, offset);

#ifdef DEBUG
    std::cout << " www server: " << config.cddb_config.server << std::endl;
#endif
    
    /* connect to the cddb server and grab the results */
    if(err != REMOTE_OK)
    {
      result = "";

#ifdef DEBUG
      std::cout << " contacting server: p=" << config.cddb_config.port << std::endl;
#endif
      
      err = do_cddb(result, category, tracknum, duration, offset,
                    config.cddb_config.server, config.cddb_config.port, config.cddb_config.use_http);
    }
    
    main_window_handler(MW_UPDATE_STATUSBAR, _("Grabbing Completed..."), NULL);

    while(gtk_events_pending())
    {
        gtk_main_iteration();
    }

    switch(err)
    {
        case REMOTE_OK :
          {
            /* successful lookup, now parse the returned data */
            cddb_handle_data(result, artist, dtitle, titles, totaltracks, year, dgenre);
	    
            main_data->disc_artist =  artist;
            main_data->disc_title = dtitle;
            main_data->disc_category = dgenre;
            main_data->disc_year = year;

            /* copy the data into the main_data structure */
            int icount = 0;
            for (std::map<int,string>::const_iterator i=titles.begin(); i!=titles.end(); ++i) {
              put_track_title(i->second, main_data, icount++);
            }

#ifdef DEBUG
            main_data->Dump();
#endif
            break;
          }
        case NO_CONNECT :
            err_handler(CDDB_NO_CONNECT_ERR, NULL);
            break;
        case CONNECT_REFUSED :
            err_handler(CDDB_CONNECT_REFUSED_ERR, NULL);
            break;
        case SERVER_ERROR :
            err_handler(CDDB_SERVER_ERR, NULL);
            break;
        case NOT_FOUND :
            err_handler(CDDB_NOT_FOUND_ERR, NULL);
            break;
    }

    // auto-select all tracks
    // FIXME - this seems to get the state out of sync with the buttons.
    select_frame_handler(SF_SELECT_ALL, 0, main_data);
    // this is needed to update the display with the new data
    select_frame_handler(SF_SYNC_SELECT_FRAME, 0, main_data);

#ifdef DEBUG
    main_data->Dump();
#endif

    main_window_handler(MW_CLEAR_STATUSBAR, NULL, NULL);

    while(gtk_events_pending())
    {
        gtk_main_iteration();
    }

    return 0;
}
