#!/bin/bash

set -o verbose
set -o errexit

# This script downloads and imports timelib
# timelib does not use any autotools/cmake/config system to it is a simple import.

# This script is designed to run on Linux or Mac OS X
# Parsers make use of re2c, which needs to be installed and be version 0.15.3
# *only*. re2c 0.16 introduces an issues with clang which causes any date
# parser to hang.
#

VERSION=2022.04
NAME=timelib
TARBALL=$VERSION.tar.gz
TARBALL_DIR=$NAME-$VERSION
if grep -q Microsoft /proc/version; then
    TEMP_DIR=$(wslpath -u $(powershell.exe -Command "Get-ChildItem Env:TEMP | Get-Content | Write-Host"))
else
    TEMP_DIR="/tmp"
fi
TEMP_DIR=$(mktemp -d $TEMP_DIR/$NAME.XXXXXX)
DEST_DIR=`git rev-parse --show-toplevel`/src/third_party/$NAME

# Check prerequisites: re2c, wget
if ! [ -x "$(command -v re2c)" ]; then
    echo 'Error: re2c is not installed.' >&2
    exit 1
fi

RE2C_VERSION=`re2c --version`
if ! [ "re2c 0.15.3" == "$RE2C_VERSION" ]; then
    echo 'Error: re2c MUST be version 0.15.3.' >&2
    exit 1
fi

if ! [ -x "$(command -v wget)" ]; then
    echo 'Error: wget is not installed.' >&2
    exit 1
fi

# Derick may not upload a tarball for a release. In that case, better to create a tarball manually.
# Can create $TARBALL manually using the following commands. For example,
# $ cd ~
# $ VERSION=2022.04
# $ TARBALL=$VERSION.tar.gz
# $ git clone git@github.com:derickr/timelib.git
# $ cd timelib
# Derick made the change to v2022 branch for 2022.04 version
# $ git checkout v2022
# .git sub-directory is unnecessary in the tarball
# $ rm -rf .git
# $ cd ..
# tarball directory should be in the directory timelib-$VERSION
# $ mv timelib timelib-$VERSION
# $ tar czvf $TARBALL timelib-$VERSION
# $ mv $TARBALL ~/mongo/src/third_party/scripts
if [ ! -f $TARBALL ]; then
    echo "Get tarball"
    wget https://github.com/derickr/timelib/archive/$TARBALL
fi

echo $TARBALL
tar -zxvf $TARBALL

rm -rf $TEMP_DIR
mv $TARBALL_DIR $TEMP_DIR
mkdir $DEST_DIR || true

cp -r $TEMP_DIR/* $DEST_DIR

cd $DEST_DIR

# Prune files
rm -rf $DEST_DIR/tests
rm $DEST_DIR/zones/old-tzcode-32-bit-output.tar.gz || true

# Create parsers
echo "Creating parsers"
make parse_date.c parse_iso_intervals.c

# Create SConscript file
cat <<EOF > SConscript
# This is a generated file, please do not modify. It is generated by
# timelib_get_sources.sh

Import('env')

env = env.Clone()

try:
    env.AppendUnique(CCFLAGS=[
        '-DHAVE_GETTIMEOFDAY',
        '-DHAVE_STRING_H',
    ])
    if env.TargetOSIs('windows'):
        env.AppendUnique(CCFLAGS=[
            '-DHAVE_IO_H',
            '-DHAVE_WINSOCK2_H',
        ])

        # C4996: '...': was declared deprecated
        env.Append(CCFLAGS=['/wd4996'])
    elif env.TargetOSIs('solaris'):
        env.AppendUnique(CCFLAGS=[
            '-DHAVE_DIRENT_H',
            '-DHAVE_STRINGS_H',
            '-DHAVE_UNISTD_H',
            '-D_POSIX_C_SOURCE=200112L',
        ])
    elif env.TargetOSIs('darwin'):
        env.AppendUnique(CCFLAGS=[
            '-DHAVE_DIRENT_H',
            '-DHAVE_SYS_TIME_H',
            '-DHAVE_UNISTD_H',
        ])
    else:
        env.AppendUnique(CCFLAGS=[
            '-DHAVE_DIRENT_H',
            '-DHAVE_SYS_TIME_H',
            '-DHAVE_UNISTD_H',
            '-D_GNU_SOURCE',
        ])
except ValueError:
    pass

env.Library(
    target='timelib',
    source=[
        'astro.c',
        'dow.c',
        'interval.c',
        'parse_date.c',
        'parse_iso_intervals.c',
        'parse_tz.c',
        'parse_posix.c',
        'parse_zoneinfo.c',
        'timelib.c',
        'tm2unixtime.c',
        'unixtime2tm.c',
    ],
    LIBDEPS_TAGS=[
        'init-no-global-side-effects',
    ],
)
EOF

echo "Done"
