/*
 *
 * Copyright 2021 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package xds

import (
	"net"

	"google.golang.org/grpc"
	iserver "google.golang.org/grpc/xds/internal/server"
)

// ServingModeCallback returns a grpc.ServerOption which allows users to
// register a callback to get notified about serving mode changes.
func ServingModeCallback(cb ServingModeCallbackFunc) grpc.ServerOption {
	return &smcOption{cb: cb}
}

type serverOption interface {
	applyServerOption(*serverOptions)
}

// smcOption is a server option containing a callback to be invoked when the
// serving mode changes.
type smcOption struct {
	// Embedding the empty server option makes it safe to pass it to
	// grpc.NewServer().
	grpc.EmptyServerOption
	cb ServingModeCallbackFunc
}

func (s *smcOption) applyServerOption(o *serverOptions) {
	o.modeCallback = s.cb
}

type serverOptions struct {
	modeCallback ServingModeCallbackFunc
}

// ServingMode indicates the current mode of operation of the server.
type ServingMode = iserver.ServingMode

const (
	// ServingModeServing indicates the the server contains all required xDS
	// configuration is serving RPCs.
	ServingModeServing = iserver.ServingModeServing
	// ServingModeNotServing indicates that the server is not accepting new
	// connections. Existing connections will be closed gracefully, allowing
	// in-progress RPCs to complete. A server enters this mode when it does not
	// contain the required xDS configuration to serve RPCs.
	ServingModeNotServing = iserver.ServingModeNotServing
)

// ServingModeCallbackFunc is the callback that users can register to get
// notified about the server's serving mode changes. The callback is invoked
// with the address of the listener and its new mode.
//
// Users must not perform any blocking operations in this callback.
type ServingModeCallbackFunc func(addr net.Addr, args ServingModeChangeArgs)

// ServingModeChangeArgs wraps the arguments passed to the serving mode callback
// function.
type ServingModeChangeArgs struct {
	// Mode is the new serving mode of the server listener.
	Mode ServingMode
	// Err is set to a non-nil error if the server has transitioned into
	// not-serving mode.
	Err error
}
