/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/
#include "../curl_setup.h"

#include "warnless.h"

#if defined(__INTEL_COMPILER) && defined(__unix__)

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#endif /* __INTEL_COMPILER && __unix__ */

#define CURL_MASK_UCHAR   ((unsigned char)~0)

#define CURL_MASK_USHORT  ((unsigned short)~0)

#define CURL_MASK_UINT    ((unsigned int)~0)
#define CURL_MASK_SINT    (CURL_MASK_UINT >> 1)

#define CURL_MASK_ULONG   ((unsigned long)~0)

#define CURL_MASK_USIZE_T ((size_t)~0)
#define CURL_MASK_SSIZE_T (CURL_MASK_USIZE_T >> 1)

/*
** unsigned long to unsigned char
*/

unsigned char curlx_ultouc(unsigned long ulnum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(ulnum <= (unsigned long)CURL_MASK_UCHAR);
  return (unsigned char)(ulnum & (unsigned long)CURL_MASK_UCHAR);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** unsigned size_t to signed int
*/

int curlx_uztosi(size_t uznum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(uznum <= (size_t)CURL_MASK_SINT);
  return (int)(uznum & (size_t)CURL_MASK_SINT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** unsigned size_t to unsigned long
*/

unsigned long curlx_uztoul(size_t uznum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

#if ULONG_MAX < SIZE_MAX
  DEBUGASSERT(uznum <= (size_t)CURL_MASK_ULONG);
#endif
  return (unsigned long)(uznum & (size_t)CURL_MASK_ULONG);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** unsigned size_t to unsigned int
*/

unsigned int curlx_uztoui(size_t uznum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

#if UINT_MAX < SIZE_MAX
  DEBUGASSERT(uznum <= (size_t)CURL_MASK_UINT);
#endif
  return (unsigned int)(uznum & (size_t)CURL_MASK_UINT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** signed long to signed int
*/

int curlx_sltosi(long slnum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(slnum >= 0);
#if INT_MAX < LONG_MAX
  DEBUGASSERT((unsigned long)slnum <= (unsigned long)CURL_MASK_SINT);
#endif
  return (int)(slnum & (long)CURL_MASK_SINT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** signed long to unsigned int
*/

unsigned int curlx_sltoui(long slnum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(slnum >= 0);
#if UINT_MAX < LONG_MAX
  DEBUGASSERT((unsigned long)slnum <= (unsigned long)CURL_MASK_UINT);
#endif
  return (unsigned int)(slnum & (long)CURL_MASK_UINT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** signed long to unsigned short
*/

unsigned short curlx_sltous(long slnum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(slnum >= 0);
  DEBUGASSERT((unsigned long)slnum <= (unsigned long)CURL_MASK_USHORT);
  return (unsigned short)(slnum & (long)CURL_MASK_USHORT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** unsigned size_t to signed ssize_t
*/

ssize_t curlx_uztosz(size_t uznum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(uznum <= (size_t)CURL_MASK_SSIZE_T);
  return (ssize_t)(uznum & (size_t)CURL_MASK_SSIZE_T);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** signed curl_off_t to unsigned size_t
*/

size_t curlx_sotouz(curl_off_t sonum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(sonum >= 0);
  return (size_t)(sonum & (curl_off_t)CURL_MASK_USIZE_T);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** signed ssize_t to signed int
*/

int curlx_sztosi(ssize_t sznum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(sznum >= 0);
#if INT_MAX < SSIZE_MAX
  DEBUGASSERT((size_t)sznum <= (size_t)CURL_MASK_SINT);
#endif
  return (int)(sznum & (ssize_t)CURL_MASK_SINT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** unsigned int to unsigned short
*/

unsigned short curlx_uitous(unsigned int uinum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(uinum <= (unsigned int)CURL_MASK_USHORT);
  return (unsigned short)(uinum & (unsigned int)CURL_MASK_USHORT);

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

/*
** signed int to unsigned size_t
*/

size_t curlx_sitouz(int sinum)
{
#ifdef __INTEL_COMPILER
#pragma warning(push)
#pragma warning(disable:810) /* conversion may lose significant bits */
#endif

  DEBUGASSERT(sinum >= 0);
  return (size_t)sinum;

#ifdef __INTEL_COMPILER
#pragma warning(pop)
#endif
}

size_t curlx_uitouz(unsigned int uinum)
{
  return (size_t)uinum;
}

size_t curlx_sotouz_range(curl_off_t sonum, size_t uzmin, size_t uzmax)
{
  if(sonum < 0)
    return uzmin;
#if SIZEOF_CURL_OFF_T > SIZEOF_SIZE_T
  if(sonum > SIZE_MAX)
    return uzmax;
#endif
  return CURLMIN(CURLMAX((size_t)sonum, uzmin), uzmax);
}

bool curlx_sztouz(ssize_t sznum, size_t *puznum)
{
  if(sznum < 0) {
    *puznum = 0;
    return FALSE;
  }
  *puznum = (size_t)sznum;
  return TRUE;
}

bool curlx_sotouz_fits(curl_off_t sonum, size_t *puznum)
{
  if(sonum < 0) {
    *puznum = 0;
    return FALSE;
  }
#if SIZEOF_CURL_OFF_T > SIZEOF_SIZE_T
  if(sonum > SIZE_MAX) {
    *puznum = 0;
    return FALSE;
  }
#endif
  *puznum = (size_t)sonum;
  return TRUE;
}

bool curlx_sltouz(long slnum, size_t *puznum)
{
  if(slnum < 0) {
    *puznum = 0;
    return FALSE;
  }
  /* We error in curl_setup.h if SIZEOF_LONG > SIZEOF_SIZE_T */
  *puznum = (size_t)slnum;
  return TRUE;
}

curl_off_t curlx_uztoso(size_t uznum)
{
#if SIZEOF_SIZE_T >= SIZEOF_CURL_OFF_T
  if(uznum > (size_t)CURL_OFF_T_MAX)
    return CURL_OFF_T_MAX;
#endif
  return (curl_off_t)uznum;
}
