#!/bin/sh
#
# Copyright (c) 2006 Carl D. Worth
#

test_description='Test of git add, including the -- option.'

. ./test-lib.sh

test_expect_success \
    'Test of git add' \
    'touch foo && git add foo'

test_expect_success \
    'Post-check that foo is in the index' \
    'git ls-files foo | grep foo'

test_expect_success \
    'Test that "git add -- -q" works' \
    'touch -- -q && git add -- -q'

test_expect_success \
	'git add: Test that executable bit is not used if core.filemode=0' \
	'git config core.filemode 0 &&
	 echo foo >xfoo1 &&
	 chmod 755 xfoo1 &&
	 git add xfoo1 &&
	 case "`git ls-files --stage xfoo1`" in
	 100644" "*xfoo1) echo ok;;
	 *) echo fail; git ls-files --stage xfoo1; (exit 1);;
	 esac'

test_expect_success 'git add: filemode=0 should not get confused by symlink' '
	rm -f xfoo1 &&
	ln -s foo xfoo1 &&
	git add xfoo1 &&
	case "`git ls-files --stage xfoo1`" in
	120000" "*xfoo1) echo ok;;
	*) echo fail; git ls-files --stage xfoo1; (exit 1);;
	esac
'

test_expect_success \
	'git update-index --add: Test that executable bit is not used...' \
	'git config core.filemode 0 &&
	 echo foo >xfoo2 &&
	 chmod 755 xfoo2 &&
	 git update-index --add xfoo2 &&
	 case "`git ls-files --stage xfoo2`" in
	 100644" "*xfoo2) echo ok;;
	 *) echo fail; git ls-files --stage xfoo2; (exit 1);;
	 esac'

test_expect_success 'git add: filemode=0 should not get confused by symlink' '
	rm -f xfoo2 &&
	ln -s foo xfoo2 &&
	git update-index --add xfoo2 &&
	case "`git ls-files --stage xfoo2`" in
	120000" "*xfoo2) echo ok;;
	*) echo fail; git ls-files --stage xfoo2; (exit 1);;
	esac
'

test_expect_success \
	'git update-index --add: Test that executable bit is not used...' \
	'git config core.filemode 0 &&
	 ln -s xfoo2 xfoo3 &&
	 git update-index --add xfoo3 &&
	 case "`git ls-files --stage xfoo3`" in
	 120000" "*xfoo3) echo ok;;
	 *) echo fail; git ls-files --stage xfoo3; (exit 1);;
	 esac'

test_expect_success '.gitignore test setup' '
	echo "*.ig" >.gitignore &&
	mkdir c.if d.ig &&
	>a.ig && >b.if &&
	>c.if/c.if && >c.if/c.ig &&
	>d.ig/d.if && >d.ig/d.ig
'

test_expect_success '.gitignore is honored' '
	git add . &&
	! git ls-files | grep "\\.ig"
'

test_expect_success 'error out when attempting to add ignored ones without -f' '
	! git add a.?? &&
	! git ls-files | grep "\\.ig"
'

test_expect_success 'error out when attempting to add ignored ones without -f' '
	! git add d.?? &&
	! git ls-files | grep "\\.ig"
'

test_expect_success 'add ignored ones with -f' '
	git add -f a.?? &&
	git ls-files --error-unmatch a.ig
'

test_expect_success 'add ignored ones with -f' '
	git add -f d.??/* &&
	git ls-files --error-unmatch d.ig/d.if d.ig/d.ig
'

test_expect_success 'add ignored ones with -f' '
	rm -f .git/index &&
	git add -f d.?? &&
	git ls-files --error-unmatch d.ig/d.if d.ig/d.ig
'

test_expect_success '.gitignore with subdirectory' '

	rm -f .git/index &&
	mkdir -p sub/dir &&
	echo "!dir/a.*" >sub/.gitignore &&
	>sub/a.ig &&
	>sub/dir/a.ig &&
	git add sub/dir &&
	git ls-files --error-unmatch sub/dir/a.ig &&
	rm -f .git/index &&
	(
		cd sub/dir &&
		git add .
	) &&
	git ls-files --error-unmatch sub/dir/a.ig
'

mkdir 1 1/2 1/3
touch 1/2/a 1/3/b 1/2/c
test_expect_success 'check correct prefix detection' '
	rm -f .git/index &&
	git add 1/2/a 1/3/b 1/2/c
'

test_expect_success 'git add with filemode=0, symlinks=0, and unmerged entries' '
	for s in 1 2 3
	do
		echo $s > stage$s
		echo "100755 $(git hash-object -w stage$s) $s	file"
		echo "120000 $(printf $s | git hash-object -w -t blob --stdin) $s	symlink"
	done | git update-index --index-info &&
	git config core.filemode 0 &&
	git config core.symlinks 0 &&
	echo new > file &&
	echo new > symlink &&
	git add file symlink &&
	git ls-files --stage | grep "^100755 .* 0	file$" &&
	git ls-files --stage | grep "^120000 .* 0	symlink$"
'

test_expect_success 'git add with filemode=0, symlinks=0 prefers stage 2 over stage 1' '
	git rm --cached -f file symlink &&
	(
		echo "100644 $(git hash-object -w stage1) 1	file"
		echo "100755 $(git hash-object -w stage2) 2	file"
		echo "100644 $(printf 1 | git hash-object -w -t blob --stdin) 1	symlink"
		echo "120000 $(printf 2 | git hash-object -w -t blob --stdin) 2	symlink"
	) | git update-index --index-info &&
	git config core.filemode 0 &&
	git config core.symlinks 0 &&
	echo new > file &&
	echo new > symlink &&
	git add file symlink &&
	git ls-files --stage | grep "^100755 .* 0	file$" &&
	git ls-files --stage | grep "^120000 .* 0	symlink$"
'

test_expect_success 'git add --refresh' '
	>foo && git add foo && git commit -a -m "commit all" &&
	test -z "`git diff-index HEAD -- foo`" &&
	git read-tree HEAD &&
	case "`git diff-index HEAD -- foo`" in
	:100644" "*"M	foo") echo ok;;
	*) echo fail; (exit 1);;
	esac &&
	git add --refresh -- foo &&
	test -z "`git diff-index HEAD -- foo`"
'

test_done
