/*
 * Copyright 2007, Intel Corporation
 *
 * This file is part of kerneloops.org
 *
 * This program file is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file named COPYING; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 *
 * Authors:
 * 	Arjan van de Ven <arjan@linux.intel.com>
 */

#define _BSD_SOURCE
#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <asm/unistd.h>

#include "kerneloops.h"


/*
 * we keep track of 16 checksums of the last submitted oopses; this allows us to
 * make sure we don't submit the same oops twice (this allows us to not have to do
 * really expensive things during non-destructive dmesg-scanning)
 *
 * This also limits the number of oopses we'll submit per session;
 * it's important that this is bounded to avoid feedback loops
 * for the scenario where submitting an oopses causes a warning/oops
 */
#define MAX_CHECKSUMS 16
static unsigned int checksums[MAX_CHECKSUMS];
static int submitted;

struct oops;

struct oops {
	struct oops *next;
	char *text;
	unsigned int checksum;
};

/* we queue up oopses, and then submit in a batch.
 * This is useful to be able to cancel all submissions, in case
 * we later find our marker indicating we submitted everything so far already
 * previously.
 */
static struct oops *queued_oopses;
static int newoops;

/* For communicating details to the applet, we write the
 * details in a file, and provide the filename to the applet
 */
static char *detail_filename;


static unsigned int checksum(char *ptr)
{
	unsigned int temp = 0;
	unsigned char *c;
	c = (unsigned char *) ptr;
	while (c && *c) {
		temp = (temp) + *c;
		c++;
	}
	return temp;
}

/* Pipe the oops into another program for submission
 * to other reporting data sinks.
 * Silently return if the pipe executable/script is not
 * there.  This allows for safe removal of the other package
 * without major panic attacks and log file chatter.
 */
static void pipe_oops(char *oops) {
	FILE *pipe_file = NULL;
	int err;
	size_t wrote;

	if (submit_pipe && access(submit_pipe, X_OK) == 0)
		;
	else
		return;
	
	openlog("kerneloops", 0, LOG_KERN);
	pipe_file = popen(submit_pipe, "w");
	if (pipe_file) {
		wrote = fwrite(oops, 1, strlen(oops), pipe_file);
		if (wrote == strlen(oops)) {
			syslog(LOG_INFO,
			       "Wrote %d bytes of oops log to %s", wrote, submit_pipe);
		} else {
			syslog(LOG_ERR,
			       "Write to %s failed: %s\n", submit_pipe,
			       ferror(pipe_file) ? strerror(errno) : "short write");
		}
		err = pclose(pipe_file);
		if (err != 0) {
			syslog(LOG_ERR,
			       "Close of pipe to %s failed: %s\n", submit_pipe,
			       err < 0 ? strerror(errno) : "non-zero exit from child");
		}
	} else {
		syslog(LOG_WARNING,
		       "popen to %s failed: %s\n",
		       submit_pipe, strerror(errno));
	}
	closelog();
}

void queue_oops(char *oops)
{
	int i;
	unsigned int sum;
	struct oops *new;

	if (submitted >= MAX_CHECKSUMS-1)
		return;
	/* first, check if we haven't already submitted the oops */
	sum = checksum(oops);
	for (i = 0; i < submitted; i++) {
		if (checksums[i] == sum) {
			printf("Match with oops %i (%x)\n", i, sum);
			return;
		}
	}
	checksums[submitted++] = sum;

	new = malloc(sizeof(struct oops));
	memset(new, 0, sizeof(struct oops));
	new->next = queued_oopses;
	new->checksum = sum;
	new->text = strdup(oops);
	queued_oopses = new;
	newoops = 1;
}


static void write_detail_file(void)
{
	int temp_fileno;
	FILE *tmpf;
	struct oops *oops;
	int count = 0;

	detail_filename = strdup("/tmp/kerneloops.XXXXXX");
	temp_fileno = mkstemp(detail_filename);
	if (temp_fileno < 0) {
		free(detail_filename);
		detail_filename = NULL;
		return;
	}
	/* regular user must be able to read this detail file to be
	 * useful; there is nothing worth doing if fchmod fails.
	 */
	fchmod(temp_fileno, 0644);
	tmpf = fdopen(temp_fileno, "w");
	oops = queued_oopses;
	while (oops) {
		count++; /* Users are not programmers, start at 1 */
		fprintf(tmpf, "Kernel failure message %d:\n", count);
		fprintf(tmpf, "%s", oops->text);
		fprintf(tmpf, "\n\n");
		oops = oops->next;
	}
	fclose(tmpf);
	close(temp_fileno);
}

static void unlink_detail_file(void)
{
	if (detail_filename) {
		unlink(detail_filename);
		free(detail_filename);
		detail_filename = NULL;
	}
}

static void pipe_oopses(void)
{
	struct oops *oops;
	struct oops *queue;

	if (access(submit_pipe, X_OK) != 0) {
		printf("Cant access %s, not submitting oopses\n", submit_pipe);
		return;
	}

	queue = queued_oopses;
	queued_oopses = NULL;
	barrier();
	oops = queue;
	while (oops) {
		FILE *pipe_file;
		int len;
		char pipe[4096];

		if (strstr(oops->text, "WARNING:"))
			return;

                if (strstr(oops->text, "NETDEV WATCHDOG:"))
                        return;

		len = snprintf(pipe, 4096, "%s %d", submit_pipe, oops->checksum);
		if (len > 4096) {
			printf("Pipe name too long, aborting\n");
			return;
		}

		pipe_file = popen(pipe, "w");
		if (pipe_file) {
			size_t written = 0;
			int err;
			char *buf = oops->text;
			size_t to_write = strlen(oops->text);

			do {
				written = fwrite(buf, 1, to_write, pipe_file);
				if (written == to_write)
					break;
				buf += written;
				to_write -= written;
			} while (errno == EINTR);
			if (written < to_write) {
				perror("aborting, writing to pipe failed");
			}

			err = pclose(pipe_file);
			if (err != 0)
				perror("closing pipe failed");
		} else {
			perror("aborting, popen failed");
			return;
		}

		oops = oops->next;
	}
}

static void print_queue(void)
{
	struct oops *oops;
	struct oops *queue;
	int count = 0;

	queue = queued_oopses;
	queued_oopses = NULL;
	barrier();
	oops = queue;
	while (oops) {
		struct oops *next;

		printf("Submit text is:\n---[start of oops]---\n%s\n---[end of oops]---\n", oops->text);
		next = oops->next;
		free(oops->text);
		free(oops);
		oops = next;
		count++;
	}

}

static void write_logfile(int count, char *result_url)
{
	openlog("kerneloops", 0, LOG_KERN);
	syslog(LOG_WARNING, "Submitted %i kernel oopses to www.kerneloops.org", count);
	if (result_url && result_url[0])
		syslog(LOG_WARNING, "kerneloops.org: oops is posted as %s", result_url);
	closelog();
}

static char result_url[4096];

int submit_oops(struct oops *oops)
{
	char *submit_prog = "kerneloops-submit";
	int in_pipe[2];
	int out_pipe[2];
	int pid;

	if (pipe(in_pipe)) {
		perror("pipe");
		return 1;
	}

	if (pipe(out_pipe)){
		perror("pipe");
		return 1;
	}

	pid = fork();
	if (pid < 0) {
		perror("fork");
	} else if (pid == 0) {
		//child
		close(in_pipe[1]);
		close(out_pipe[0]);
		dup2(in_pipe[0], 0);
		dup2(out_pipe[1], 1);
		if (execvp(submit_prog, NULL)) {
			perror("execl");
			exit(255);
		}
		exit(0);
	} else {
		// parent
		size_t written = 0;
		char *buf = oops->text;
		size_t to_write = strlen(oops->text);
		int status;
		ssize_t count;
		char url[4096];
		close(in_pipe[0]);
		close(out_pipe[1]);
		do {
			written = write(in_pipe[1], buf, to_write);
			if (written == to_write)
				break;
			buf += written;
			to_write -= written;
		} while (errno == EINTR);
		if (written < to_write) {
			perror("aborting, writing to pipe failed");
			return 1;
		}
		close(in_pipe[1]);
		waitpid(pid, &status, 0);
		if (!WIFEXITED(status) || WEXITSTATUS(status) != 0) {
			perror("child failed");
			close(out_pipe[0]);
			return 1;
		}
		count = read(out_pipe[0], url, 4095);
		if (count < 0) {
			perror("read from child");
			return 1;
		}
		close(out_pipe[0]);
		printf("url is %s\n", url);
		strncpy(result_url, url, 4095);
	}
	return 0;
}

void submit_queue(void)
{
	struct oops *oops;
	struct oops *queue;
	int count = 0;

	unlink_detail_file();
	memset(result_url, 0, 4096);

	if (testmode) {
		print_queue();
		return;
	}

	queue = queued_oopses;
	queued_oopses = NULL;
	barrier();
	oops = queue;
	while (oops) {
		int err;
		struct oops *next;
		err = submit_oops(oops);
		next = oops->next;
		free(oops->text);
		free(oops);
		oops = next;
		if (!err)
			count++;
	}

	if (count && !testmode)
		write_logfile(count, result_url);

	if (count)
		dbus_say_thanks(result_url);
	/*
	 * If we've reached the maximum count, we'll exit the program,
	 * the program won't do any useful work anymore going forward.
	 */
	if (submitted >= MAX_CHECKSUMS-1)
		exit(EXIT_SUCCESS);
}

void clear_queue(void)
{
	struct oops *oops, *next;
	struct oops *queue;

	queue = queued_oopses;
	queued_oopses = NULL;
	barrier();
	oops = queue;
	while (oops) {
		next = oops->next;
		free(oops->text);
		free(oops);
		oops = next;
	}
	unlink_detail_file();
	write_logfile(0, NULL);
}

void ask_permission(void)
{
	if (!newoops)
		return;
	if (!submit_pipe) {
		if (!pinged)
			return;
		pinged = 0;
		newoops = 0;
		if (queued_oopses) {
			write_detail_file();
			dbus_ask_permission(detail_filename);
		}
	} else {
		if (queued_oopses) {
			pipe_oopses();
		}
	}
}
