/*
 * Copyright 2010 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authors:
 *    Cody Russell <crussell@canonical.com>
 */

#include "gripgesturemanager.h"

#include <gdk/gdkx.h>
#include <geis/geis.h>

typedef struct _GripGestureRegistration GripGestureRegistration;
typedef struct _GripGestureBinding      GripGestureBinding;
typedef struct _GripRegistrationRequest GripRegistrationRequest;

struct _GripGestureManagerPrivate
{
  GHashTable *hash;
  GList      *requests;
};

struct _GripGestureBinding
{
  GripGestureType      type;
  GtkWidget           *widget;
  gint                 touches;
  GripGestureCallback  callback;
  gpointer             data;
  GDestroyNotify       destroy;
};

struct _GripGestureRegistration
{
  GtkWindow         *window;
  GList             *bindings;
  GPtrArray         *gesture_list;
  GeisInstance       instance;
  GIOChannel        *iochannel;
};

struct _GripRegistrationRequest
{
  GripGestureManager *manager;
  GtkWidget          *widget;
  GripGestureType     gesture_type;
  gint                touch_points;
  GripGestureCallback callback;
  gpointer            user_data;
  GDestroyNotify      destroy;
};

static const gchar *geis_gesture_types[] = {
  GEIS_GESTURE_TYPE_DRAG1,
  GEIS_GESTURE_TYPE_DRAG2,
  GEIS_GESTURE_TYPE_DRAG3,
  GEIS_GESTURE_TYPE_DRAG4,
  GEIS_GESTURE_TYPE_DRAG5,
  GEIS_GESTURE_TYPE_PINCH1,
  GEIS_GESTURE_TYPE_PINCH2,
  GEIS_GESTURE_TYPE_PINCH3,
  GEIS_GESTURE_TYPE_PINCH4,
  GEIS_GESTURE_TYPE_PINCH5,
  GEIS_GESTURE_TYPE_ROTATE1,
  GEIS_GESTURE_TYPE_ROTATE2,
  GEIS_GESTURE_TYPE_ROTATE3,
  GEIS_GESTURE_TYPE_ROTATE4,
  GEIS_GESTURE_TYPE_ROTATE5,
  GEIS_GESTURE_TYPE_TAP1,
  GEIS_GESTURE_TYPE_TAP2,
  GEIS_GESTURE_TYPE_TAP3,
  GEIS_GESTURE_TYPE_TAP4,
  GEIS_GESTURE_TYPE_TAP5,
};

static void gesture_added (void              *cookie,
                           GeisGestureType    gesture_type,
                           GeisGestureId      gesture_id,
                           GeisSize           attr_count,
                           GeisGestureAttr   *attrs);

static void gesture_removed (void              *cookie,
                             GeisGestureType    gesture_type,
                             GeisGestureId      gesture_id,
                             GeisSize           attr_count,
                             GeisGestureAttr   *attrs);

static void gesture_start (void              *cookie,
                           GeisGestureType    gesture_type,
                           GeisGestureId      gesture_id,
                           GeisSize           attr_count,
                           GeisGestureAttr   *attrs);

static void gesture_update (void              *cookie,
                            GeisGestureType    gesture_type,
                            GeisGestureId      gesture_id,
                            GeisSize           attr_count,
                            GeisGestureAttr   *attrs);

static void gesture_finish (void              *cookie,
                            GeisGestureType    gesture_type,
                            GeisGestureId      gesture_id,
                            GeisSize           attr_count,
                            GeisGestureAttr   *attrs);

static void window_mapped_cb (GtkWidget       *widget,
                              GdkEvent        *event,
                              gpointer         user_data);

static GripGestureManager *manager_singleton = NULL;
static GeisGestureFuncs gesture_funcs = {
  gesture_added,
  gesture_removed,
  gesture_start,
  gesture_update,
  gesture_finish
};

G_DEFINE_TYPE (GripGestureManager, grip_gesture_manager, G_TYPE_OBJECT)

#define GRIP_GESTURE_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GRIP_TYPE_GESTURE_MANAGER, GripGestureManagerPrivate))

static void
grip_gesture_manager_dispose (GObject *object)
{
}

static void
free_registration (gpointer key,
		   gpointer value,
		   gpointer user_data)
{
  GripGestureRegistration *reg = (GripGestureRegistration *)value;
  GList *tmp = NULL;

  geis_unsubscribe (reg->instance,
		    (GeisGestureType*)reg->gesture_list->pdata);

  /* We don't need to free the values in the GPtrArray. */
  g_ptr_array_free (reg->gesture_list, TRUE);

  for (tmp = reg->bindings; tmp != NULL; tmp = tmp->next)
    {
      g_free (tmp->data);
    }

  g_list_free (reg->bindings);
}

static void
grip_gesture_manager_finalize (GObject *object)
{
  GripGestureManagerPrivate *priv = GRIP_GESTURE_MANAGER_GET_PRIVATE (object);

  if (priv->hash != NULL)
    {
      g_hash_table_foreach (priv->hash,
			    free_registration,
			    NULL);

      g_hash_table_unref (priv->hash);
      priv->hash = NULL;
    }
}

static GObject *
grip_gesture_manager_constructor (GType                  type,
                                  guint                  n_params,
                                  GObjectConstructParam *params)
{
  GObject *object;

  if (manager_singleton != NULL)
    {
      object = g_object_ref (manager_singleton);
    }
  else
    {
      object = G_OBJECT_CLASS (grip_gesture_manager_parent_class)->constructor (type,
                                                                                n_params,
                                                                                params);

      manager_singleton = GRIP_GESTURE_MANAGER (object);
      g_object_add_weak_pointer (object, (gpointer) &manager_singleton);
    }

  return object;
}

static void
grip_gesture_manager_class_init (GripGestureManagerClass *class)
{
  GObjectClass     *gobject_class;

  gobject_class = G_OBJECT_CLASS (class);

  grip_gesture_manager_parent_class = g_type_class_peek_parent (class);

  g_type_class_add_private (gobject_class, sizeof (GripGestureManagerPrivate));

  gobject_class->constructor = grip_gesture_manager_constructor;
  gobject_class->dispose     = grip_gesture_manager_dispose;
  gobject_class->finalize    = grip_gesture_manager_finalize;
}

static gint
pinch_gesture_handle_properties (GripEventGesturePinch *event,
                                 GeisSize               attr_count,
                                 GeisGestureAttr       *attrs)
{
  gint i = 0;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          touches = attrs[i].integer_val;
        }
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_RADIUS_DELTA) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->radius_delta = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_RADIAL_VELOCITY) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->radial_velocity = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_RADIUS) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->radius = attrs[i].float_val;
        }
    }

  return touches;
}

static gint
drag_gesture_handle_properties (GripEventGestureDrag *event,
                                GeisSize              attr_count,
                                GeisGestureAttr      *attrs)
{
  gint i;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          touches = attrs[i].integer_val;
        }
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_DELTA_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->delta_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_DELTA_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->delta_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_VELOCITY_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->velocity_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_VELOCITY_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->velocity_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_y = attrs[i].float_val;
        }
    }

  return touches;
}

static gint
rotate_gesture_handle_properties (GripEventGestureRotate *event,
                                  GeisSize                attr_count,
                                  GeisGestureAttr        *attrs)
{
  gint i;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          touches = attrs[i].integer_val;
        }
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_ANGLE_DELTA) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->angle_delta = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_ANGULAR_VELOCITY) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->angular_velocity = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_ANGLE) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->angle = attrs[i].float_val;
        }
    }

  return touches;
}

static gint
tap_gesture_handle_properties (GripEventGestureTap *event,
			       GeisSize             attr_count,
			       GeisGestureAttr     *attrs)
{
  gint i;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
	  attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
	{
	  touches = attrs[i].integer_val;
	}
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TAP_TIME) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->tap_time = attrs[i].float_val;
        }
    }

  return touches;
}

static void
gesture_added (void              *cookie,
               GeisGestureType    gesture_type,
               GeisGestureId      gesture_id,
               GeisSize           attr_count,
               GeisGestureAttr   *attrs)
{
}

static void
gesture_removed (void              *cookie,
                 GeisGestureType    gesture_type,
                 GeisGestureId      gesture_id,
                 GeisSize           attr_count,
                 GeisGestureAttr   *attrs)
{
}

static gboolean
matches_widget (GtkWidget *widget,
                GdkWindow *window,
                gint       x,
                gint       y)
{
  GtkAllocation alloc;
  gint ax, ay;

  gtk_widget_get_allocation (widget, &alloc);
  gdk_window_get_root_coords (window, alloc.x, alloc.y, &ax, &ay);

  return (x >= ax && x < ax + alloc.width && y >= ay && y < ay + alloc.height);
}

static void
gesture_start (void              *cookie,
               GeisGestureType    type,
               GeisGestureId      id,
               GeisSize           attr_count,
               GeisGestureAttr   *attrs)
{
  GripGestureRegistration *reg = (GripGestureRegistration *)cookie;
  GList *l = NULL;

  for (l = reg->bindings; l != NULL; l = l->next)
    {
      GripGestureBinding *binding = (GripGestureBinding *)l->data;

      if (binding->type == type)
        {
	  GripGestureEvent *event = grip_gesture_event_new (type);

          if (type == GRIP_GESTURE_DRAG)
            {
              GripEventGestureDrag *drag = (GripEventGestureDrag *)event;

              drag->type    = type;
              drag->id      = id;
              drag->fingers = drag_gesture_handle_properties (drag,
							      attr_count,
							      attrs);

              if (drag->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)drag->focus_x,
                                      (gint)drag->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_START,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_PINCH)
            {
              GripEventGesturePinch *pinch = (GripEventGesturePinch *)event;

              pinch->type    = type;
              pinch->id      = id;
              pinch->fingers = pinch_gesture_handle_properties (pinch,
								attr_count,
								attrs);

              if (pinch->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)pinch->focus_x,
                                      (gint)pinch->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_START,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_ROTATE)
            {
              GripEventGestureRotate *rotate = (GripEventGestureRotate *)event;

              rotate->type    = type;
              rotate->id      = id;
              rotate->fingers = rotate_gesture_handle_properties (rotate,
								  attr_count,
								  attrs);

              if (rotate->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)rotate->focus_x,
                                      (gint)rotate->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_START,
                                         event,
                                         binding->data);
                    }
                }
            }
	  else if (type == GRIP_GESTURE_TAP)
	    {
	      GripEventGestureTap *tap = (GripEventGestureTap *)event;

	      tap->type    = type;
	      tap->id      = id;
	      tap->fingers = tap_gesture_handle_properties (tap,
							    attr_count,
							    attrs);

	      if (tap->fingers == binding->touches)
		{
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)tap->focus_x,
                                      (gint)tap->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_START,
                                         event,
                                         binding->data);
                    }
                }
	    }

	  grip_gesture_event_free (event);
        }
    }
}

static void
gesture_update (void              *cookie,
                GeisGestureType    type,
                GeisGestureId      id,
                GeisSize           attr_count,
                GeisGestureAttr   *attrs)
{
  GripGestureRegistration *reg = (GripGestureRegistration *)cookie;
  GList *l = NULL;

  for (l = reg->bindings; l != NULL; l = l->next)
    {
      GripGestureBinding *binding = (GripGestureBinding *)l->data;

      if (binding->type == type)
        {
	  GripGestureEvent *event = grip_gesture_event_new (type);

          if (type == GRIP_GESTURE_DRAG)
            {
              GripEventGestureDrag *drag = (GripEventGestureDrag *)event;

              drag->type    = type;
              drag->id      = id;
              drag->fingers = drag_gesture_handle_properties (drag,
							      attr_count,
							      attrs);

              if (drag->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)drag->focus_x,
                                      (gint)drag->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_UPDATE,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_PINCH)
            {
              GripEventGesturePinch *pinch = (GripEventGesturePinch *)event;

              pinch->type    = type;
              pinch->id      = id;
              pinch->fingers = pinch_gesture_handle_properties (pinch,
								attr_count,
								attrs);

              if (pinch->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)pinch->focus_x,
                                      (gint)pinch->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_UPDATE,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_ROTATE)
            {
              GripEventGestureRotate *rotate = (GripEventGestureRotate *)event;

              rotate->type    = type;
              rotate->id      = id;
              rotate->fingers = rotate_gesture_handle_properties (rotate,
								  attr_count,
								  attrs);

              if (rotate->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)rotate->focus_x,
                                      (gint)rotate->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_UPDATE,
                                         event,
                                         binding->data);
                    }
                }
            }
	  else if (type == GRIP_GESTURE_TAP)
	    {
	      GripEventGestureTap *tap = (GripEventGestureTap *)event;

	      tap->type    = type;
	      tap->id      = id;
	      tap->fingers = tap_gesture_handle_properties (tap,
							    attr_count,
							    attrs);

	      if (tap->fingers == binding->touches)
		{
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)tap->focus_x,
                                      (gint)tap->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_UPDATE,
                                         event,
                                         binding->data);
                    }
                }
	    }

	  grip_gesture_event_free (event);
        }
    }
}

static void
gesture_finish (void              *cookie,
                GeisGestureType    type,
                GeisGestureId      id,
                GeisSize           attr_count,
                GeisGestureAttr   *attrs)
{
  GripGestureRegistration *reg = (GripGestureRegistration *)cookie;
  GList *l = NULL;

  for (l = reg->bindings; l != NULL; l = l->next)
    {
      GripGestureBinding *binding = (GripGestureBinding *)l->data;

      if (binding->type == type)
        {
	  GripGestureEvent *event = grip_gesture_event_new (type);

          if (type == GRIP_GESTURE_DRAG)
            {
              GripEventGestureDrag *drag = (GripEventGestureDrag *)event;

              drag->type    = type;
              drag->id      = id;
              drag->fingers = drag_gesture_handle_properties (drag,
							      attr_count,
							      attrs);

              if (drag->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)drag->focus_x,
                                      (gint)drag->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_END,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_PINCH)
            {
              GripEventGesturePinch *pinch = (GripEventGesturePinch *)event;

              pinch->type    = type;
              pinch->id      = id;
              pinch->fingers = pinch_gesture_handle_properties (pinch,
								attr_count,
								attrs);

              if (pinch->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)pinch->focus_x,
                                      (gint)pinch->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_END,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_ROTATE)
            {
              GripEventGestureRotate *rotate = (GripEventGestureRotate *)event;

              rotate->type    = type;
              rotate->id      = id;
              rotate->fingers = rotate_gesture_handle_properties (rotate,
								  attr_count,
								  attrs);

              if (rotate->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)rotate->focus_x,
                                      (gint)rotate->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_END,
                                         event,
                                         binding->data);
                    }
                }
            }
	  else if (type == GRIP_GESTURE_TAP)
	    {
	      GripEventGestureTap *tap = (GripEventGestureTap *)event;

	      tap->type    = type;
	      tap->id      = id;
	      tap->fingers = tap_gesture_handle_properties (tap,
							    attr_count,
							    attrs);

	      if (tap->fingers == binding->touches)
		{
                  if (matches_widget (binding->widget,
                                      GTK_WIDGET (reg->window)->window,
                                      (gint)tap->focus_x,
                                      (gint)tap->focus_y))
                    {
                      binding->callback (binding->widget,
                                         GRIP_TIME_END,
                                         event,
                                         binding->data);
                    }
                }
	    }

	  grip_gesture_event_free (event);
        }
    }
}

static void
grip_gesture_manager_init (GripGestureManager *item)
{
  GripGestureManagerPrivate *priv;

  priv = item->priv = GRIP_GESTURE_MANAGER_GET_PRIVATE (item);

  priv->hash = g_hash_table_new (g_direct_hash, g_direct_equal);
}

static gboolean
io_callback (GIOChannel   *source,
             GIOCondition  condition,
             gpointer      data)
{
  GripGestureRegistration *reg = (GripGestureRegistration *)data;

  geis_event_dispatch (reg->instance);

  return TRUE;
}

static void
window_destroyed_cb (GtkObject *object,
                     gpointer   user_data)
{
  GripGestureManager *manager = (GripGestureManager *)user_data;
  GripGestureManagerPrivate *priv = manager->priv;
  GripGestureRegistration *reg = g_hash_table_lookup (priv->hash, object);
  GList *list;

  for (list = reg->bindings; list != NULL; list = list->next)
    {
      GripGestureBinding *binding = (GripGestureBinding *)list->data;

      if (binding->destroy)
        {
          GDestroyNotify d = binding->destroy;

          d (binding->data);
        }

      g_free (binding);
    }

  g_list_free (reg->bindings);

  g_io_channel_shutdown (reg->iochannel, TRUE, NULL);

  geis_finish (reg->instance);

  g_hash_table_remove (priv->hash, object);
  g_free (reg);
}

static const gchar *
grip_type_to_geis_type (GripGestureType gesture_type,
			gint            touch_points)
{
  /* grail taps begin at 15, so let's convert that into something we
   * can index in geis_gesture_types. */
  int t = gesture_type == 15 ? 3 : gesture_type;

  return geis_gesture_types[(t * 5 + touch_points) - 1];
}


/* Public API */
GripGestureManager *
grip_gesture_manager_get (void)
{
  return g_object_new (GRIP_TYPE_GESTURE_MANAGER, NULL);
}

static void
register_internal (GripGestureManager *manager,
                   GtkWidget          *widget,
                   GripGestureType     gesture_type,
                   gint                touch_points,
                   GripGestureCallback callback,
                   gpointer            user_data,
                   GDestroyNotify      destroy)
{
  GripGestureManagerPrivate *priv;
  GripGestureRegistration *reg;
  GripGestureBinding *binding;
  GtkWidget *toplevel;

  g_return_if_fail (GRIP_IS_GESTURE_MANAGER (manager));
  g_return_if_fail (GTK_IS_WIDGET (widget));

  toplevel = gtk_widget_get_toplevel (widget);

  g_return_if_fail (GTK_IS_WINDOW (toplevel));

  priv = manager->priv;

  if (!(reg = g_hash_table_lookup (priv->hash, toplevel)))
    {
      GeisInstance  instance;
      GIOChannel   *iochannel;
      gint fd = -1;
      GeisXcbWinInfo xcb_win_info = {
        .display_name = NULL,
        .screenp      = NULL,
        .window_id    = GDK_DRAWABLE_XID (toplevel->window)
      };
      GeisWinInfo win_info = {
        GEIS_XCB_FULL_WINDOW,
        &xcb_win_info
      };

      if (geis_init (&win_info, &instance) != GEIS_STATUS_SUCCESS)
        {
          g_warning ("Failed to initialize gesture manager.");
          return;
        }

      if (geis_configuration_supported (instance,
                                        GEIS_CONFIG_UNIX_FD) != GEIS_STATUS_SUCCESS)
        {
          g_warning ("Gesture manager does not support UNIX fd.");
          return;
        }

      if (geis_configuration_get_value (instance,
                                        GEIS_CONFIG_UNIX_FD,
                                        &fd) != GEIS_STATUS_SUCCESS)
        {
          g_error ("Gesture manager failed to obtain UNIX fd.");
          return;
        }

      reg = g_new0 (GripGestureRegistration, 1);

      reg->window   = GTK_WINDOW (toplevel);
      reg->instance = instance;

      g_signal_connect (toplevel,
                        "destroy",
                        G_CALLBACK (window_destroyed_cb),
                        manager);

      iochannel = g_io_channel_unix_new (fd);
      g_io_add_watch (iochannel,
                      G_IO_IN,
                      io_callback,
                      reg);

      reg->iochannel = iochannel;

      reg->gesture_list = g_ptr_array_new ();
    }
  else
    {
      geis_unsubscribe (reg->instance,
			(GeisGestureType*)reg->gesture_list->pdata);
    }

  if (reg->gesture_list->len)
    g_ptr_array_remove_index (reg->gesture_list,
			      reg->gesture_list->len - 1);

  g_ptr_array_add (reg->gesture_list,
		   (gchar *)grip_type_to_geis_type (gesture_type, touch_points));
  g_ptr_array_add (reg->gesture_list,
		   NULL);

  geis_subscribe (reg->instance,
		  GEIS_ALL_INPUT_DEVICES,
		  (const char**)reg->gesture_list->pdata,
		  &gesture_funcs,
		  reg);

  /* XXX - check for duplicates in reg->bindings first */
  binding = g_new0 (GripGestureBinding, 1);

  binding->type     = gesture_type;
  binding->widget   = widget;
  binding->touches  = touch_points;
  binding->callback = callback;
  binding->data     = user_data;
  binding->destroy  = destroy;

  reg->bindings = g_list_append (reg->bindings, binding);

  g_hash_table_insert (priv->hash,
                       toplevel,
                       reg);
}

static void
toplevel_notify_cb (GtkWidget    *widget,
                    GParamSpec   *pspec,
                    gpointer      user_data)
{
  if (pspec->name == g_intern_static_string ("parent"))
    {
      GtkWidget *toplevel = gtk_widget_get_toplevel (widget);
      GripRegistrationRequest *req = (GripRegistrationRequest *)user_data;

      if (GTK_IS_WINDOW (toplevel))
        {
          g_signal_handlers_disconnect_by_func (widget,
                                                G_CALLBACK (toplevel_notify_cb),
                                                user_data);

          if (gtk_widget_get_mapped (GTK_WIDGET (toplevel)))
            {
              register_internal (req->manager,
                                 req->widget,
                                 req->gesture_type,
                                 req->touch_points,
                                 req->callback,
                                 req->user_data,
                                 req->destroy);

              g_free (req);
            }
          else
            {
              GripGestureManagerPrivate *priv = req->manager->priv;

              priv->requests = g_list_append (priv->requests, req);

              g_signal_connect (toplevel,
                                "map-event",
                                G_CALLBACK (window_mapped_cb),
                                req->manager);
            }
        }
      else
        {
          g_signal_connect (G_OBJECT (toplevel),
                            "notify",
                            G_CALLBACK (toplevel_notify_cb),
                            user_data);
        }
    }
}

static void
register_widget (GripGestureManager *manager,
                 GtkWidget          *widget,
                 GripGestureType     gesture_type,
                 gint                touch_points,
                 GripGestureCallback callback,
                 gpointer            user_data,
                 GDestroyNotify      destroy)
{
  GtkWidget *toplevel = gtk_widget_get_toplevel (widget);

  if (GTK_IS_WINDOW (toplevel))
    {
      register_internal (manager,
                         widget,
                         gesture_type,
                         touch_points,
                         callback,
                         user_data,
                         destroy);
    }
  else
    {
      GripRegistrationRequest *req = g_new0 (GripRegistrationRequest, 1);

      req->manager      = manager;
      req->widget       = widget;
      req->gesture_type = gesture_type;
      req->touch_points = touch_points;
      req->callback     = callback;
      req->user_data    = user_data;
      req->destroy      = destroy;

      g_signal_connect (toplevel,
                        "notify",
                        G_CALLBACK (toplevel_notify_cb),
                        req);
    }
}

static void
window_mapped_cb (GtkWidget   *widget,
                  GdkEvent    *event,
                  gpointer     user_data)
{
  GripGestureManager *manager = (GripGestureManager *)user_data;
  GripGestureManagerPrivate *priv = manager->priv;
  GList *tmp;

  for (tmp = priv->requests; tmp != NULL; tmp = tmp->next)
    {
      GripRegistrationRequest *req = tmp->data;

      register_widget (req->manager,
                       req->widget,
                       req->gesture_type,
                       req->touch_points,
                       req->callback,
                       req->user_data,
                       req->destroy);

      g_free (req);
    }

  g_list_free (priv->requests);
  priv->requests = NULL;

  g_signal_handlers_disconnect_by_func (widget,
                                        window_mapped_cb,
                                        user_data);
}

/**
 * grip_gesture_manager_register_window:
 * @manager: A #GripGestureManager instance.
 * @window: A #GtkWindow to register the gesture event for.
 * @gesture_type: The type of gesture event to register.
 * @touch_points: Number of touch points for this gesture.
 * @callback: Called when a gesture starts, updates, or ends.
 * @user_data: User data
 * @destroy: Destroy callback for user data.
 *
 * Registers a toplevel window to receive gesture events.
 * The callback parameters provided will be called by the
 * #GripGestureManager whenever the user initiates a gesture
 * on the specified window.
 */
void
grip_gesture_manager_register_window (GripGestureManager  *manager,
                                      GtkWidget           *widget,
                                      GripGestureType      gesture_type,
                                      gint                 touch_points,
                                      GripGestureCallback  callback,
                                      gpointer             user_data,
                                      GDestroyNotify       destroy)
{
  GtkWidget *toplevel;

  g_return_if_fail (GRIP_IS_GESTURE_MANAGER (manager));
  g_return_if_fail (GTK_IS_WIDGET (widget));

  toplevel = gtk_widget_get_toplevel (widget);

  if (GTK_IS_WINDOW (toplevel))
    {
      if (gtk_widget_get_mapped (GTK_WIDGET (toplevel)))
        {
          register_internal (manager,
                             widget,
                             gesture_type,
                             touch_points,
                             callback,
                             user_data,
                             destroy);
        }
      else
        {
          GripRegistrationRequest *req = g_new0 (GripRegistrationRequest, 1);
          GripGestureManagerPrivate *priv = manager->priv;

          req->manager      = manager;
          req->widget       = widget;
          req->gesture_type = gesture_type;
          req->touch_points = touch_points;
          req->callback     = callback;
          req->user_data    = user_data;
          req->destroy      = destroy;

          priv->requests = g_list_append (priv->requests, req);

          g_signal_connect (toplevel,
                            "map-event",
                            G_CALLBACK (window_mapped_cb),
                            manager);
        }
    }
  else
    {
      GripRegistrationRequest *req = g_new0 (GripRegistrationRequest, 1);

      req->manager      = manager;
      req->widget       = widget;
      req->gesture_type = gesture_type;
      req->touch_points = touch_points;
      req->callback     = callback;
      req->user_data    = user_data;
      req->destroy      = destroy;

      g_signal_connect (toplevel,
                        "notify",
                        G_CALLBACK (toplevel_notify_cb),
                        req);
    }
}

GType
grip_gesture_event_get_type (void)
{
  static GType type = 0;

  if (type == 0)
    {
      type = g_boxed_type_register_static (g_intern_static_string ("GripGestureEvent"),
					   (GBoxedCopyFunc)grip_gesture_event_copy,
					   (GBoxedFreeFunc)grip_gesture_event_free);
    }

  return type;
}

GripGestureEvent *
grip_gesture_event_new (GripGestureType type)
{
  GripGestureEvent *event = g_slice_new0 (GripGestureEvent);

  event->any.type = type;

  return event;
}

GripGestureEvent *
grip_gesture_event_copy (const GripGestureEvent *event)
{
  GripGestureEvent *new_event;

  g_return_val_if_fail (event != NULL, NULL);

  new_event = grip_gesture_event_new (event->type);
  *new_event = *event;

  return new_event;
}

void
grip_gesture_event_free (GripGestureEvent *event)
{
  g_return_if_fail (event != NULL);

  g_slice_free (GripGestureEvent, event);
}
