/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

private class ScopeProxyRemote : GLib.Object, ScopeProxy
{
  public string dbus_name { get; construct; }
  public string dbus_path { get; construct; }

  public OptionsFilter sources { get; set; }
  public bool search_in_global { get; set; }

  public Dee.SerializableModel results_model {
    get { return _results_model; }
    set {
      _results_model = value as Dee.SharedModel;
    }
  }

  public Dee.SerializableModel global_results_model {
    get { return _global_results_model; }
    set {
      _global_results_model = value as Dee.SharedModel;
    }
  }

  public Dee.SerializableModel filters_model {
    get { return _filters_model; }
    set {
      _filters_model = value as Dee.SharedModel;
    }
  }

  public ViewType view_type {
    get { return _view_type; }
    // make sure we do dbus calls only on changes
    set { if (_view_type != value) set_view_type (value); }
  }

  private ViewType _view_type;
  private DBusConnection _bus;
  private uint _watcher;

  private ScopeService _service;

  private Dee.SharedModel? _results_model;
  private Dee.SharedModel? _global_results_model;
  private Dee.SharedModel? _filters_model;
  private uint _reconnection_id = 0;
  private bool synchronized = false;
  private int64 _last_scope_crash = 0;
  private uint _scope_crashes = 0;

  public ScopeProxyRemote (string dbus_name_, string dbus_path_)
  {
    Object (dbus_name:dbus_name_, dbus_path:dbus_path_);
  }

  construct
  {
    sources = new CheckOptionFilter ("sources", "Sources", null, true);
    try {
      _bus = Bus.get_sync (BusType.SESSION);
      _watcher = Bus.watch_name_on_connection (_bus,
                                               dbus_name,
                                               BusNameWatcherFlags.NONE,
                                               on_scope_appeared,
                                               on_scope_vanished);

      /* If it doesn't exist when we load up, start it */
      start_reconnection_timeout ();
    } catch (Error e) {
      critical (@"Unable to connect to session bus: $(e.message)");
    }
  }

  private async void connect_to_scope ()
  {
    try {
      _service = yield _bus.get_proxy (dbus_name, dbus_path);
      _service.changed.connect (on_changed);
      yield _service.info_request ();

    } catch (Error e) {
      warning (@"Unable to connect to Scope ($dbus_path @ $dbus_name): $(e.message)");
    }
  }
 
  private void on_scope_appeared (DBusConnection conn,
                                  string         name,
                                  string         name_owner)
  {
    Trace.log_object (this, "Scope appeared: %s", dbus_name);
    if (_reconnection_id != 0)
      Source.remove (_reconnection_id);
    connect_to_scope.begin ();
  }

  private void on_scope_vanished (DBusConnection conn,
                                  string         name)
  {
    Trace.log_object (this, "Scope vanished: %s", dbus_name);

    synchronized = false;
    search_in_global = false;
    sources = new CheckOptionFilter ("sources", "Sources", null, true);
    if (_results_model != null)
      _results_model.clear ();

    if (_global_results_model != null)
      _global_results_model.clear ();

    /* No need to clear the filters model, it's read-only for the scope and
     * it would just cause warnings from filters synchronizer */
    _filters_model = null;

    if (_service != null)
    {
      /* Here comes the protected-restarting logic - the scope will be
       * restarted unless it crashed more than 10 times during the past
       * 15 minutes */
      _scope_crashes++;
      var cur_time = get_monotonic_time ();
      var time_since_last_crash = cur_time - _last_scope_crash;

      if (time_since_last_crash >= 15*60000000) // 15 minutes
      {
        _last_scope_crash = cur_time;
        // reset crash counter, it's not that bad
        _scope_crashes = 1;
      }
      else if (_scope_crashes >= 10)
      {
        // more than 10 crashes in the past 15 minutes
        warning ("Scope %s is crashing too often, disabling it", dbus_name);
        return;
      }

      start_reconnection_timeout ();
    }
    else
    {
      start_reconnection_timeout ();
    }
  }

  private void start_reconnection_timeout ()
  {
    if (_reconnection_id != 0)
      Source.remove (_reconnection_id);

    _reconnection_id = Timeout.add_seconds (2, () =>
    {
      if (_service == null)
        connect_to_scope ();
      else if ((_service as DBusProxy).g_name_owner == null)
        _service.info_request (); // ping the service to autostart it

      _reconnection_id = 0;
      return false;
    });
  }

  /*
   * Implementation of the ScopeService interface
   */
  public async ActivationReplyRaw activate (string uri, uint action_type)
  {
    if (synchronized)
    {
      try {
        var raw = yield _service.activate (uri, action_type);
        return raw;
      } catch (Error e) {
        warning (@"Unable to search scope ($dbus_path): $(e.message)");
      }
    }
    return ActivationReplyRaw ();
  }

  public async HashTable<string, Variant> search (
      string search_string, HashTable<string, Variant> hints)
  {
    if (synchronized)
    {
      try {
        var ht = yield _service.search (search_string, hints);
        return ht;
      } catch (ScopeError.SEARCH_CANCELLED scope_error) {
        // that's fine
      } catch (Error e) {
        warning (@"Unable to search scope ($dbus_path - '$search_string'): $(e.message)");
      }
    }

    return new HashTable<string, Variant> (str_hash, str_equal);
  }

  public async HashTable<string, Variant> global_search (
      string search_string, HashTable<string, Variant> hints)
  {
    if (synchronized)
    {
      try {
        var result = yield _service.global_search (search_string, hints);
        return result;
      } catch (ScopeError.SEARCH_CANCELLED scope_error) {
        // that's fine
      } catch (Error e) {
        warning (@"Unable to global_search scope ($dbus_path - '$search_string'): $(e.message)");
      }
    }

    return new HashTable<string, Variant> (str_hash, str_equal);
  }

  public async PreviewReplyRaw preview (string uri)
  {
    if (synchronized)
    {
      try {
        var raw = yield _service.preview (uri);
        return raw;
      } catch (Error e) {
        warning (@"Unable to search scope ($dbus_path): $(e.message)");
      }
    }
    return PreviewReplyRaw();
  }

  public async void set_view_type (ViewType view_type)
  {
    _view_type = view_type;
    if (synchronized)
    {
      try {
        // FIXME: no need to set HOME_VIEW if !search_in_global
        yield _service.set_view_type (view_type);
      } catch (Error e) {
        warning (@"Unable to set_active ($dbus_path): $(e.message)");
      }
    }
  }

  public async void set_active_sources (string[] sources)
  {
    if (synchronized)
    {
      try {
        yield _service.set_active_sources (sources);
      } catch (Error e) {
        warning (@"Unable to set_sources ($dbus_path): $(e.message)");
      }
    }
  }
  
  /* This is where we figure out the diff between states */
  public void on_changed (ScopeInfo scope_info)
  {
    if (dbus_path != scope_info.dbus_path)
    {
      warning (@"Unable to handle Scope changed signal: dbus_path mismatch. Expected $dbus_path got $(scope_info.dbus_path)");
      return;
    }
    Trace.log_object (this, "Processing changed signal for %s", dbus_path);
    search_in_global = scope_info.search_in_global;

    if (results_model == null ||
        _results_model.get_swarm_name () != scope_info.results_model_name)
    {
      results_model = new Dee.SharedModel (scope_info.results_model_name);
      results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");
    }

    if (global_results_model == null ||
        _global_results_model.get_swarm_name () != scope_info.global_results_model_name)
    {
      global_results_model = new Dee.SharedModel (scope_info.global_results_model_name);
      global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");
    }

    if (filters_model == null ||
        _filters_model.get_swarm_name () != scope_info.filters_model_name)
    {
      filters_model = new Dee.SharedModel (scope_info.filters_model_name);
      filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");
    }

    /* extract the sources */
    sources.update (scope_info.sources);
    this.notify_property ("sources");

    if (!synchronized)
    {
      synchronized = true;
      set_view_type (_view_type);
      // FIXME: make sure active_sources are set, dispatch last (missed) search
    }
  }
}

} /* Namespace */
