/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;

namespace Unity {

public class Lens : GLib.Object
{
  public bool active { get; private set; default = false; }
  public bool visible { get; set; default = true; }
  public bool searching { get; private set; default = false; }
  public bool exported { get; private set; default = false; }
  public bool search_in_global { get; set; default = true; }
  
  // default category name to be used by Home Lens when no search query is provided
  public string home_lens_default_name { get; set; default = ""; }
  public string id { get; construct; }
  public string dbus_path { get; construct; }
  public string search_hint { get; set; }
  public List<Filter> filters {
    get {
      return _filters;
    }
    set {
      _filters = new List<Filter> ();
      foreach (Filter f in value)
        _filters.prepend (f);
      _filters.reverse ();
      _pimpl.load_filters (_filters);
    }
  }
  
  public List<Category> categories {
    get {
      return _categories;
    }
    set {
      if (_categories == null)
      {
        _categories = new List<Category> ();
        foreach (Category cat in value)
          _categories.prepend (cat);
        _categories.reverse ();
        _pimpl.load_categories (_categories);
      }
      else
        warning ("Categories can only be set once");
    }
  }
  
  public MergeStrategy merge_strategy {
    get {
      return _pimpl.results_sync.merge_strategy;
    }
    set {
      _pimpl.results_sync.merge_strategy = value;
    }
  }
  
  public MergeStrategy global_merge_strategy {
    get {
      return _pimpl.global_results_sync.merge_strategy;
    }
    set {
      _pimpl.global_results_sync.merge_strategy = value;
    }
  }

  public string sources_display_name { get; construct set; }

  internal const string SOURCES_FILTER_ID = "unity-sources";

  private LensImpl _pimpl;
  private List<Category> _categories = null;
  private List<Filter> _filters = null;

  public Lens (string dbus_path_, string id_)
  {
    Object (dbus_path:dbus_path_, id:id_, sources_display_name:"");
  }

  construct
  {
    _pimpl = new LensImpl (this);
  }

  public void export () throws IOError
  {
    if (!exported)
    {
      _pimpl.export ();
      exported = true;
    }
  }

  public void add_local_scope (Scope scope)
  {
    _pimpl.add_local_scope (scope);
  }

  internal void set_active_internal (bool is_active)
  {
    active = is_active;
  }

  internal unowned OptionsFilter get_sources_internal ()
  {
    return _pimpl.get_sources ();
  }

  internal unowned Dee.Model get_model_internal (int index)
  {
    return _pimpl.get_model (index);
  }
}

} /* namespace */
