/* -*- Mode: vala; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Michal Hruby <michal.hruby@canonical.com>
 *
 */
using Unity;
using Unity.Protocol;

namespace Unity.Test
{
  public class ScopeSuite
  {
    public ScopeSuite ()
    {
      GLib.Test.add_data_func ("/Unit/Scope/Signal/NoSearchChanged",
                               test_no_search_changed);
      GLib.Test.add_data_func ("/Unit/Scope/Signal/SearchChanged",
                               test_search_changed);
      GLib.Test.add_data_func ("/Unit/Scope/Signal/SearchKey",
                               test_search_key);
      GLib.Test.add_data_func ("/Unit/Scope/Signal/SearchKeyDetails",
                               test_search_key_details);
      GLib.Test.add_data_func ("/Unit/Scope/Signals/MultipleQueueCalls",
                               test_multiple_queue_calls);
      GLib.Test.add_data_func ("/Unit/Scope/Signals/Cancellable",
                               test_cancellable);
    }

    static void test_no_search_changed ()
    {
      MainLoop ml = new MainLoop ();
      var scope = new Unity.Scope ("com/canonical/Scope/Test");
      assert (scope != null);

      bool got_signal = false;
      scope.search_changed.connect (() =>
      {
        got_signal = true;
        ml.quit ();
      });

      // queue_search_changed shouldn't emit the search_changed signal if
      // appropriate view isn't active
      scope.queue_search_changed (SearchType.DEFAULT);

      uint tid = Timeout.add (500, () => { ml.quit (); return false; });

      ml.run ();
      assert (got_signal == false);
      Source.remove (tid);
    }

    static void test_search_changed ()
    {
      MainLoop ml = new MainLoop ();
      var scope = new Unity.Scope ("com/canonical/Scope/Test");
      assert (scope != null);
      scope.set_view_type_internal (ViewType.LENS_VIEW);

      bool got_signal = false;
      scope.search_changed.connect (() =>
      {
        got_signal = true;
        ml.quit ();
      });

      scope.queue_search_changed (SearchType.DEFAULT);

      // we don't want to wait indefinitely
      uint tid = Timeout.add (500, () => { assert_not_reached (); });

      ml.run ();
      assert (got_signal == true);
      Source.remove (tid);
    }

    static void test_search_key ()
    {
      MainLoop ml = new MainLoop ();
      var scope = new Unity.Scope ("com/canonical/Scope/Test");
      assert (scope != null);
      scope.set_view_type_internal (ViewType.LENS_VIEW);

      bool got_changed_signal = false;
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        got_changed_signal = true;
        ml.quit ();
      });

      bool got_gen_signal = false;
      scope.generate_search_key.connect ((lens_search) =>
      {
        got_gen_signal = true;
        return "foo";
      });

      scope.queue_search_changed (SearchType.DEFAULT);

      // we don't want to wait indefinitely
      uint tid = Timeout.add (500, () => { assert_not_reached (); });

      ml.run ();
      assert (got_changed_signal == true);
      assert (got_gen_signal == true);

      Source.remove (tid);
    }
    
    static void test_search_key_details ()
    {
      MainLoop ml = new MainLoop ();
      var scope = new Unity.Scope ("com/canonical/Scope/Test");
      assert (scope != null);
      scope.set_view_type_internal (ViewType.LENS_VIEW);

      int num_changed_signals = 0;
      int num_gen_signals = 0;
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        num_changed_signals++;

        if (num_changed_signals == 1)
          scope.set_view_type_internal (ViewType.HOME_VIEW);
        else if (num_changed_signals >= 2) ml.quit ();
      });

      scope.generate_search_key["default"].connect ((lens_search) =>
      {
        num_gen_signals++;
        return "foo";
      });

      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);

      // we don't want to wait indefinitely
      uint tid = Timeout.add (500, () => { assert_not_reached (); });

      ml.run ();
      assert (num_changed_signals == 2);
      assert (num_gen_signals == 1);

      Source.remove (tid);
    }
    
    static void test_multiple_queue_calls ()
    {
      MainLoop ml = new MainLoop ();
      var scope = new Unity.Scope ("com/canonical/Scope/Test");
      assert (scope != null);
      scope.set_view_type_internal (ViewType.LENS_VIEW);

      int num_changed = 0;
      scope.search_changed.connect (() =>
      {
        num_changed++;
      });

      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);
      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);
      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);
      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.DEFAULT);
      scope.queue_search_changed (SearchType.GLOBAL);
      scope.queue_search_changed (SearchType.GLOBAL);

      // we don't want to wait indefinitely
      Timeout.add (500, () => { ml.quit (); return false; });

      ml.run ();
      // we expect just one signal (for SearchType.DEFAULT, LENS_VIEW is active)
      assert (num_changed == 1);
    }

    static void test_cancellable ()
    {
      MainLoop ml = new MainLoop ();
      var scope = new Unity.Scope ("com/canonical/Scope/Test");
      assert (scope != null);
      scope.set_view_type_internal (ViewType.LENS_VIEW);

      Cancellable? canc = null;
      scope.search_changed.connect ((lens_search, search_type, cancellable) =>
      {
        if (canc == null)
        {
          canc = cancellable;
          scope.queue_search_changed (SearchType.DEFAULT);
        }
        else
        {
          assert (cancellable.is_cancelled () == false);
          ml.quit ();
        }
      });

      scope.queue_search_changed (SearchType.DEFAULT);

      // we don't want to wait indefinitely
      uint tid = Timeout.add (500, () => { assert_not_reached (); });

      ml.run ();
      assert (canc != null);
      assert (canc.is_cancelled () == true);

      Source.remove (tid);
    }
  }
}
