

import utils
import libxml2
import urlparse
from blueprintbase import BlueprintInfo
from buglistbase import LPBugList, LPBugPage
from lphelper import blueprint_sort, user, unicode_for_libxml2

#deactivate error messages from the validation [libxml2.htmlParseDoc]
def noerr(ctx, str):
    pass

libxml2.registerErrorHandler(noerr, None)

class BPInfo(BlueprintInfo):
    def __init__(self, priority, spec, title, url, design, delivery, assignee, project, mentorship):
        url = utils.valid_lp_url(url,utils.BLUEPRINT)
        assert url, "Invalid launchpad url %s" %url
        BlueprintInfo.__init__(self, priority, spec, title, url, design, delivery, assignee, project, mentorship)
        

class Project(object):
    def __init__(self, url, project):
        self.url = url
        self.project = project
    def __str__(self):
        return str(self.project)
        

class BlueprintPage(LPBugPage):
    """
    grab content of a single BluePrint-table    
    """
    @staticmethod
    def find_parse_function(connection, url, all_tasks):
        lp_content = connection.get(url)
        xmldoc = libxml2.htmlParseDoc(unicode_for_libxml2(lp_content.text), "UTF-8")
        u = urlparse.urlsplit(url)
        if "+milestone" in u[2]:
            result = BlueprintPage.parse_html_milestone_specs(xmldoc, all_tasks, url)
        else:
            result = BlueprintPage.parse_html_blueprintpage(xmldoc, all_tasks, url)
        return result
        
    @staticmethod
    def parse_html_blueprintpage(xmldoc, all_tasks, url):
        def _parse():
            if not xmldoc.xpathEval('//table[@id="speclisting"]'):
                xmldoc.freeDoc()
                return
                
            blueprinttable = xmldoc.xpathEval('//table[@id="speclisting"]//tbody//tr')
            for row in blueprinttable:
                m = row.xpathEval('td[1]//span[not(@class="sortkey")]')
                assert m
                priority = m[0].prop("class")
                
                m = row.xpathEval('td[2]//a')
                assert m
                url = m[0].prop("href")
                title = m[0].prop("title")
                spec = m[0].content
                
                mentorship = bool(row.xpathEval('td[2]//img[@alt="mentoring"]'))
                #add INFORMATIONAL
                
                m = row.xpathEval('td[3]//span[not(@class="sortkey")]')
                assert m
                status = m[0].prop("class")
                
                m = row.xpathEval('td[4]//span[not(@class="sortkey")]')
                assert m
                delivery = m[0].prop("class")
                
                m = row.xpathEval('td[5]//a')
                if m:
                    assignee = user.parse_html_user(m[0])
                else:
                    assignee = user(None)
                
                m = row.xpathEval('td[6]//a')
                # on personal blueprint pages this column does not exist
                if m:
                    project = Project(m[0].prop("href"), m[0].content)
                else:
                    project = None
                
                yield BPInfo(priority, spec, title, url, status, delivery, assignee, project, mentorship)

        next = xmldoc.xpathEval('//a[@rel="next"]//@href')
        if next:
            return _parse(), utils.sanitize_html(next[0].content)
        return _parse(), False
        
    
    @staticmethod
    def parse_html_milestone_specs(xmldoc, all_tasks, url):
        def _parse():
            if not xmldoc.xpathEval('//table[@id="milestone_specs"]'):
                xmldoc.freeDoc()
                return
                
            blueprinttable = xmldoc.xpathEval('//table[@id="milestone_specs"]//tbody//tr')
            for row in blueprinttable:
                m = row.xpathEval('td[1]//a')
                assert m
                url = m[0].prop("href")
                title = m[0].prop("title")
                spec = m[0].content
                
                m = row.xpathEval('td[2]//span[not(@class="sortkey")]')
                assert m
                priority = m[0].prop("class")
                
                # is the mentorship-icon used?
                mentorship = bool(row.xpathEval('td[2]//img[@alt="mentoring"]'))
                #add INFORMATIONAL
                
                m = row.xpathEval('td[3]/a')
                if m:
                    assignee = user.parse_html_user(m[0])
                else:
                    assignee = user(None)
                
                m = row.xpathEval('td[4]//span[not(@class="sortkey")]')
                assert m
                delivery = m[0].prop("class")
                
                yield BPInfo(priority, spec, title, url, None, delivery, assignee, None, mentorship)

        next = xmldoc.xpathEval('//a[@rel="next"]//@href')
        assert not next, "milestone_specs are supposed to be single-paged, does it changed? url='%s'" %url
        return _parse(), False
        

class BlueprintList(LPBugList):
    """
    returns a SET of LPBugInfo objects
    searches baseurl and its following pages
    """
    def __init__(self, baseurl, connection=None, filter=None, length=True,
                    all_tasks=False, start_bugs=[], helper_bugpage=None):
        LPBugList.__init__(self, baseurl, connection, filter, length,
                                all_tasks, start_bugs, BlueprintPage)
    
    def sort(self, optsort):
        """ returns a LIST of bugs sorted by optsort """
        return sorted(self, cmp=lambda x,y: blueprint_sort(x,y,optsort.strip("-")),
                        reverse=optsort.startswith("-"))
        
        


    
