/* RemoveElements.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.undoable;

import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

import org.grinvin.graphs.Edge;
import org.grinvin.graphs.Element;
import org.grinvin.graphs.Vertex;

/**
 * Undoable change for removing elements from a graph.
 * <p>
 * <a name="i18n"><b>Internationalization:</b></a> By default, the caption
 * key prefix for this command is <code>RemoveElements</code>.
 * </p>
 */
public class RemoveElements extends InternationalizedUndoableChange {
    
    /**
     * Graph from which the elements were removed.
     */
    private UndoableGraph graph;
    
    /**
     * Collection of elements removed from the graph.
     */
    private List<Element> elements;
    
    //
    private static final String RESOURCE_KEY = "RemoveElements";
    
    /**
     * Create an undoable change object for removing elements from a graph.
     * @param graph Graph from which the elements were removed
     * @param elements Elements removed from this graph.
     */
    public RemoveElements(UndoableGraph graph, List<Element> elements) {
        super(RESOURCE_KEY);
        this.graph = graph;
        this.elements = elements;
    }
    
    /**
     * Create an undoable object for removing a single element from a graph.
     */
    public RemoveElements(UndoableGraph graph, Element element) {
        super(RESOURCE_KEY);
        this.graph = graph;
        this.elements = new ArrayList<Element> ();
        this.elements.add(element);
    }
    
    /**
     * Undoes the effects of this change by restoring the elements
     * in reverse chronological order.
     */
    public void undo() {
        for (ListIterator<Element> iterator = elements.listIterator(elements.size());
        iterator.hasPrevious(); ) {
            Element el = iterator.previous();
            if (el instanceof Vertex)
                graph.restore((Vertex)el);
            else if (el instanceof Edge)
                graph.restore((Edge)el);
        }
    }
    
    /**
     * Redoes this change by removing the elements
     * in chronological order.
     */
    public void redo() {
        for (Element el : elements) {
            if (el instanceof Vertex)
                graph.remove((Vertex)el); // TODO: this is too slow
            else if (el instanceof Edge)
                graph.remove((Edge)el);
        }
    }
}
