/*
 * Copyright (c) 2006 Bea Lam. All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//
//  BBOBEXClientSession.h
//  BTUtil
//
//  A Bluetooth OBEX client session. 
//
//  This can be used to create a client session and perform CONNECT, DISCONNECT,
//  PUT, GET, SETPATH and ABORT operations.
//
//  See the included BTUtil "FileClient" example.
//

#import <Cocoa/Cocoa.h>
#import <Foundation/Foundation.h>

#import <IOBluetooth/Bluetooth.h>
#import <IOBluetooth/OBEX.h>

@class IOBluetoothOBEXSession, BBOBEXFileReader;

@interface BBOBEXClientSession : NSObject {	
	IOBluetoothOBEXSession*	mOBEXSession;	
	id mDelegate;
	NSString *mLastErrorDesc;	
	BOOL mUserWantsAbort;
	BOOL mClosed;
	
	OBEXOpCode mCurrentOp;
	OBEXMaxPacketLength	mMaxPacketLength;	// max packet length for the session
	NSData *mWhoHeaderValue;
	long mConnectionID;
	CFDataRef mConnectionIDDataRef;
	
	CFMutableDataRef mConnectHeadersDataRef;
	CFMutableDataRef mPutHeadersDataRef;
	CFMutableDataRef mGetHeadersDataRef;
	CFMutableDataRef mSetPathHeadersDataRef;
	
	BBOBEXFileReader *mCurrentPutBodyHandler;
	NSData*	mTempPutDataBuffer;
	BOOL mClosePutFileWhenDone;
	
	NSFileHandle *mCurrentGetFile;
}

/*
 * Sets whether debug messages should be displayed (default is NO).
 */
+ (void)setDebug:(BOOL)doDebug;

/*
 * Create a BBOBEXClientSession.
 *
 * Arguments:
 *	- delegate: the delegate which will be notified when OBEX events have
 *	  occured. The methods which the delegate may implement are described in 
 *    the BTOBEXClientDelegate informal protocol (in the bottom of this header 
 *	  file). 
 */
- (id)initWithDelegate:(id)delegate;

/*
 * Connect the OBEX session to an OBEX server. 
 * Returns kOBEXSuccess if the CONNECT request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 *
 * Arguments:
 *	- device: the device to connect to
 *	- channelID: the RFCOMM channel to connect to
 *	- target: the intended target on the server, if necessary. This can be nil.
 *
 * The delegate will be notified through the obexClientConnectComplete:error:
 * method when the connection process is complete.
 */
- (OBEXError)connectToDevice:(IOBluetoothDevice *)device
			   withChannelID:(BluetoothRFCOMMChannelID)channelID
					toTarget:(NSData *)target;

/*
 * Connect the OBEX session to an OBEX server. 
 * This is the same as connectToDevice:withChannelID:toTarget: except that the 
 * remote device can be specified by a device address string instead of an 
 * IOBluetoothDevice object.
 */
- (OBEXError)connectToDeviceWithAddress:(NSString *)address 
						  withChannelID:(BluetoothRFCOMMChannelID)channelID
							   toTarget:(NSData *)target;

/*
 * Disconnect the OBEX Session. You should call closeTransportConnection: after
 * this to close the underlying RFCOMM transport connection.
 * Returns kOBEXSuccess if the DISCONNECT request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 *
 * The delegate will be notified through the obexClientDisconnectComplete:error:
 * method when the disconnection process is complete.
 */
- (OBEXError)disconnect;

/*
 * Send a file to the server. 
 * Returns kOBEXSuccess if the PUT request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 *
 * Arguments:
 *	- localFilePath: the path to the local file to be sent. This cannot be nil.
 *
 * The delegate will be notified through the obexClientPutComplete:error:
 * method when the PUT process is complete. It will also be notified each time
 * data is sent for the PUT (as the operation may require multiple packets)
 * through the obexClientPutProgress:byAmount: method.
 */
- (OBEXError)putFileAtPath:(NSString *)localFilePath;

/*
 * Sends the contents of an NSFileHandle to the server. 
 * Returns kOBEXSuccess if the PUT request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 *
 * Arguments:
 *	- file: the file from which data will be read and sent to the server. This
 *	  can be nil, if you don't want to send the body header.
 *	- name: the name to be sent to the server to describe this data. This 
 *	  cannot be nil.
 *	- type: the content-type of the data. This can be nil, if you don't want to
 *    send a type header.
 *	- totalLength: the length of the file data. This can be -1 if the length is
 *	  unknown. In this case, the length header will not be sent.
 *
 * The delegate will be notified through the obexClientPutComplete:error:
 * method when the PUT process is complete. It will also be notified each time
 * data is sent for the PUT (as the operation may require multiple packets)
 * through the obexClientPutProgress:byAmount: method.
 */
- (OBEXError)putFile:(NSFileHandle *)file
			withName:(NSString *)name
				type:(NSString *)type
			  length:(int)totalLength;

/*
 * Requests a file from the server.
 * Returns kOBEXSuccess if the GET request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 *
 * Arguments:
 *	- name: the name of the requested file
 *	- type: the type of the requested file
 *	- file: the file handle to which the requested file data will be written.
 *	  This cannot be nil.
 * Either the name or type values may be nil, but not both. If the value is nil,
 * the corresponding header will not be sent.
 * 
 * The delegate will be notified through the obexClientGetComplete:error:
 * method when the GET process is complete. It will also be notified each time
 * data is received for the GET (as the operation may require multiple packets)
 * through the obexClientGetProgress:byAmount: method.
 */
- (OBEXError)getWithName:(NSString *)name
					type:(NSString *)type
			 writeToFile:(NSFileHandle *)file;

/*
 * Sends a SETPATH request to the server. 
 * Returns kOBEXSuccess if the SETPATH request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 *
 * Arguments:
 *	- pathName: the name of the requested path. If this is nil, the name header
 *    will not be sent.
 *	- flags: any flags to be sent as part of the request. This can be 0 if no
 *	  flags are required.
 *
 * The delegate will be notified through the obexClientSetPathComplete:error:
 * method when the SETPATH process is complete.
 */
- (OBEXError)setPath:(NSString *)pathName 
		   withFlags:(OBEXFlags)flags;

/*
 * Aborts the current operation. (This is only possible during PUT or GET
 * operations.)
 * Returns kOBEXSuccess if the ABORT request was successfully sent to the 
 * server, or one of the other OBEX error codes if there was an error.
 * Returns kOBEXSessionBadRequestError if there is no PUT or GET in progress.
 *
 * The delegate will be notified through the obexClientAbortComplete:error:
 * method when the ABORT process is complete.
 */
- (OBEXError)abort;


- (void)setDelegate:(id)delegate;
- (id)delegate;

- (BOOL)isBusy;

/*
 * Sets the maximum packet length for an OBEX session. This could change after
 * the session is connected if the server negotiates a lower max packet length.
 * This should be set before the OBEX session is connected.
 * The max packet length defaults to 0x2000 bytes.
 */
- (void)setMaxPacketLength:(OBEXMaxPacketLength)length;

/*
 * Sets the value for the Who header that is sent when a connection is
 * requested.
 */
- (void)setWhoHeaderValue:(NSData *)whoValue;

/*
 * Returns the current Who header value.
 */
- (NSData *)whoHeaderValue;


/*
 * Sets the Connection ID to use when sending requests to the server. If a 
 * Connection ID is received in a connection response and no Connection ID has
 * been set, this method will automatically be called to set the session to use
 * the new Connection ID.
 */
- (void)setConnectionID:(long)connectionID;

/*
 * Returns the Connection ID for the session, or -1 if there is no Connection ID.
 */
- (long)connectionID;

/*
 * Returns whether the client has a connection to an OBEX server.
 */
- (BOOL)hasOpenOBEXConnection;

/*
 * Returns whether the client has an open transport connection (i.e. an open 
 * RFCOMM connection) to a remote device.
 */
- (BOOL)hasOpenTransportConnection;

/*
 * Closes this client session and its underlying (RFCOMM) transport connection. 
 * This client session cannot be used after this.
 * Returns an OBEX error code if there was an error while closing the transport
 * connection. If the session is already closed, this method does nothing and
 * returns kOBEXSessionNotConnectedError.
 */
- (OBEXError)close;

/*
 * Returns the associated RFCOMM channel for this OBEX session, or nil if the 
 * client has not started an OBEX session.
 */
- (IOBluetoothRFCOMMChannel *)getRFCOMMChannel;

/*
 * Returns a string description of the last error that occured. If a method
 * returns an OBEXError value other than kOBEXSuccess, you can call this method
 * to get a description of that error.
 */
- (NSString *)lastErrorDescription;

@end


/*
 * Informal protocol to describe the BBOBEXClientSession delegate methods that can
 * be implemented.
 * All methods are optional. However it seems useful for the delegate to 
 * implement at least the "complete" methods for any corresponding request that 
 * will be sent. For example, if the client putFileAtPath: method is called,
 * the delegate will not know when the file has been sent unless it has 
 * implemented the obexClientPutComplete:error: method.
 */
@protocol BBOBEXClientSessionDelegate

/*
 * Called when a CONNECT operation is complete.
 *
 * Arguments:
 *	- session: the client which made the initial CONNECT request
 *	- error: an error code. This is kOBEXSuccess if the operation was successful, 
 *    or an OBEXError value from <IOBluetooth/OBEX.h> if an error occured.
 *	- who: the value of the 'Who' header sent by the server, or nil if no Who 
 *	  header was received.
 *	- connectionID: the value of the ConnectionID header sent by the server, 
 *	  or -1 if no ConnectionID header was received. (If a ConnectionID is present
 *	  and this session has no current ConnectionID, it will automatically be set
 *	  to use this new ConnectionID.)
 */
- (void)obexClientSessionConnectComplete:(BBOBEXClientSession *)session 
								   error:(OBEXError)error
								 fromWho:(NSData *)who
							connectionID:(long)connectionID;

/*
 * Called when the disconnection process is complete.
 *
 * Arguments:
 *	- session: the client which made the initial DISCONNECT request
 *	- error: an error code. This is kOBEXSuccess if the operation was successful, 
 *    or an OBEXError value from <IOBluetooth/OBEX.h> if an error occured.
 */
- (void)obexClientSessionDisconnectComplete:(BBOBEXClientSession *)session 
									  error:(OBEXError)error;

/*
 * Called each time data is sent for a PUT operation. This is useful if the 
 * data is sent over multiple packets and the delegate needs regular updates
 * about the progress of the sending of the file.
 *
 * Arguments:
 *	- session: the client which made the initial PUT request
 *	- bytesSent: the number of bytes of the file that was sent in this packet. 
 */
- (void)obexClientSessionPutProgress:(BBOBEXClientSession *)session 
							byAmount:(int)bytesSent;

/*
 * Called when a PUT process is complete.
 *
 * Arguments:
 *	- session: the client which made the initial PUT request
 *	- error: an error code. This is kOBEXSuccess if the operation was successful, 
 *    or an OBEXError value from <IOBluetooth/OBEX.h> if an error occured.
 */
- (void)obexClientSessionPutComplete:(BBOBEXClientSession *)session 
							   error:(OBEXError)error;

/*
 * Called each time data is received for a GET operation. This is useful if the 
 * data is received over multiple packets and the delegate needs regular updates
 * about the progress of the receiving of the file.
 *
 * Arguments:
 *	- session: the client which made the initial PUT request
 *	- bytesSent: the number of bytes of the file that was received in this packet. 
 */
- (void)obexClientSessionGetProgress:(BBOBEXClientSession *)session 
							byAmount:(int)bytesSent;

/*
 * Called when a GET process is complete.
 *
 * Arguments:
 *	- session: the client which made the initial GET request
 *	- error: an error code. This is kOBEXSuccess if the operation was successful, 
 *    or an OBEXError value from <IOBluetooth/OBEX.h> if an error occured.
 */
- (void)obexClientSessionGetComplete:(BBOBEXClientSession *)session 
							   error:(OBEXError)error;

/*
 * Called when a SETPATH process is complete.
 *
 * Arguments:
 *	- session: the client which made the initial SETPATH request
 *	- error: an error code. This is kOBEXSuccess if the operation was successful, 
 *    or an OBEXError value from <IOBluetooth/OBEX.h> if an error occured.
 */
- (void)obexClientSessionSetPathComplete:(BBOBEXClientSession *)session 
								   error:(OBEXError)error;

/*
 * Called when an ABORT process is complete.
 *
 * Arguments:
 *	- session: the client which made the initial ABORT request
 *	- error: an error code. This is kOBEXSuccess if the operation was successful, 
 *    or an OBEXError value from <IOBluetooth/OBEX.h> if an error occured.
 */
- (void)obexClientSessionAbortComplete:(BBOBEXClientSession *)session 
								 error:(OBEXError)error;


@end