%  Copyright (C) 2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module DateMatcher ( parseDateMatcher ) where

import System.Time
import IsoDate ( parseDate )
import Monad ( liftM )
import Char ( toLower )

sameDate :: CalendarTime -> CalendarTime -> Bool
sameDate a b = ctDay a == ctDay b &&
               ctMonth a == ctMonth b &&
               ctYear a == ctYear b

dateRange :: CalendarTime -> CalendarTime -> CalendarTime -> Bool
dateRange a b c = toClockTime a <= toClockTime c &&
                  toClockTime b >= toClockTime c

parseDateMatcher :: String -> IO (CalendarTime -> Bool)
parseDateMatcher d
    | map toLower d == "today"
        = sameDate `liftM` now
    | map toLower d == "yesterday"
        = sameDate `liftM` ago (TimeDiff 0 0 (-1) 0 0 0 0)
    | map toLower d == "day before yesterday"
        = sameDate `liftM` ago (TimeDiff 0 0 (-2) 0 0 0 0)
    | map toLower d == "last week"
        = do weekago <- ago (TimeDiff 0 0 (-7) 0 0 0 0)
             today <- now
             return $ dateRange weekago today
    | map toLower d == "last month"
        = do monthago <- ago (TimeDiff 0 (-1) 0 0 0 0 0)
             today <- now
             return $ dateRange monthago today
parseDateMatcher d =
    case parseDate d of
    Right ct -> return $ sameDate ct
    _ -> error "Can't support fancy dates."

ago :: TimeDiff -> IO CalendarTime
ago td = (toUTCTime . addToClockTime td . toClockTime) `liftM` now

now :: IO CalendarTime
now = toUTCTime `liftM` getClockTime
\end{code}

