#!/usr/bin/perl
#
# $Id: ftpstats.pl,v 1.64 2001/11/25 14:30:37 chris Exp $
#
# read stdin and put contents into MySQL
#
# looks for config file, ftpstats.ini, in the current directory!
#
# ----------------------------------------------------------------------------
# "THE BEER-WARE LICENSE" (Revision 43):
# <chris@shagged.org> & <freaky@aagh.net> wrote this file.  As long as you
# retain this notice you can do whatever you want with this stuff. If we 
# meet some day, and you think this stuff is worth it, you can buy us a
# beer in return. Chris Elsworth & Thomas Hurst
# ----------------------------------------------------------------------------


use DBI;
use Socket;
use Time::Local;
use Getopt::Long;

use diagnostics;

# Default config values and month hash {{{
%conf = (	'logfile' => 'xferlog',
		'db_name' => 'ftpstats',
		'db_host' => '127.0.0.1',
		'db_user' => 'ftpstats',
		'db_pass' => 'ftpstats'); 

%months = ('Jan' => 1, 'Feb' => 2, 'Mar' => 3, 'Apr' => 4, 'May' => 5, 
	'Jun' => 6, 'Jul' => 7, 'Aug' => 8, 'Sep' => 9, 'Oct' => 10,
	'Nov' => 11, 'Dec' => 12);

my $opt_config = "ftpstats.ini";
my $opt_interval = 500;
my $min_ver = "3.23.40"; # }}}

GetOptions(	"a|all" => \$opt_all,
		"c|config=s" => \$opt_config,
		"i|interval=i" => \$opt_interval,
		"h|help" => \$opt_help);

if ($opt_help) {
	print "usage: ftpstats.pl [-a] [-i N] [-h] <file|stream

-a, --all		Parse all transfers [default no (update)]
-c, --config FILE	Where to find config file [default ./ftpstats.ini]
-i, --interval N	Insert into MySQL every N lines [default 500]

-h, --help		Help

";
	exit 1;
}

my ($result, $query);
%conf = &parse_conf(); # read in config

# open MySQL connection {{{
$db_conn = DBI->connect( "DBI:mysql:$conf{'db_name'}:$conf{'db_host'}",
	$conf{'db_user'}, $conf{'db_pass'})
	or die ("Can't connect to MySQL server: $DBI::errstr\n"); # }}}

# Get version of server and check its not too old {{{
$query = "SHOW VARIABLES LIKE 'version'";
exit 1 if (($result = &doQuery($query)) == -1);
my $s_ver = ($result->fetchrow_array)[1];
if ($s_ver lt $min_ver) {
	print STDERR "Your MySQL version (v$s_ver) is too old. ";
	print STDERR "You need v$min_ver.\n";
	exit;
} # }}}

# get a list of tables so we know what to create later
$existing_tables = join ' ', $db_conn->tables;

# get the latest xfer in the table (but not if we have $opt_all) {{{
if ((!$opt_all) && ($existing_tables =~ m/xfers\s/)) {
	$query = "SELECT UNIX_TIMESTAMP(MAX(xferTime)) AS foo FROM xfers";
	$result = &doQuery($query);
	if (($result = &doQuery($query)) == -1) {
		&repair("xfers");
		exit 1 if (($result = &doQuery($query)) == -1);
	}
	$latest_xfer = $result->fetchrow_array;
}
$latest_xfer ||= 0; # set to zero if it doesn't exist }}}

$size_of_query = $count = 0;

# some standard variables. It'd be nice to get these into a global config file
$query = $std_query = "INSERT INTO xfers.static (\n" .
	"Username, xferTime, xferDur, xferFile, xferSize, xferDirn, xferIP) VALUES \n";

$std_table = "
  xferID   MEDIUMINT UNSIGNED AUTO_INCREMENT PRIMARY KEY NOT NULL,
  Username CHAR(16) NOT NULL,
  xferTime DATETIME NOT NULL,
  xferDur  MEDIUMINT UNSIGNED NOT NULL,
  xferFile CHAR(255) NOT NULL,
  xferSize INT UNSIGNED NOT NULL,
  xferDirn ENUM('Up','Dn') NOT NULL,
  xferCmpl ENUM('COMP','INCM','ABOR','PERM','NENT','MISC'),
  xferIP   INT UNSIGNED NOT NULL
";

# use a sub called logfiletype_parse to parse this logfile
$parser = $conf{'logfile'} . "_parse";

while (<STDIN>)
{
	# Ignore FreeBSD newsyslog lines.. change if required
	next if /newsyslog/;

	# Ignore blank lines
	next if /^\n$/;

	# Quote everything so MySQL won't complain later {{{
	s/\\/\\\\/g;		# replace \ with \\
	s/'/\\'/g;		# replace ' with \' }}}

	%elements = &$parser($_);

	if (%elements && $elements{'tstamp'} > $latest_xfer) {

		# make sure "host" is always an IP
		if ($elements{"host"} !~ m/^(\d{1,3}\.){3}\d{1,3}$/) {
			$addr = inet_aton($elements{"host"});
			if ($addr) {
				$elements{"host"} = inet_ntoa($addr);
			} else {
				# unresolvable, use 0.0.0.0
				$elements{"host"} = "0.0.0.0";
			}
		}

		# now convert it to a 32 bit integer
		@ip_bits = split /\./, $elements{"host"};
		$elements{"host"} = ($ip_bits[0] << 24) + ($ip_bits[1] << 16) + ($ip_bits[2] << 8) + $ip_bits[3];

		# convert unix stamp into MySQL friendly date
		@ds = localtime($elements{'tstamp'});
		$ds[5] += 1900; $ds[4]++;
		$elements{'date'} = "$ds[5]-$ds[4]-$ds[3] $ds[2]:$ds[1]:$ds[0]";

		# work out which table this xfer should go into
		$old_table_id = $table_id;
		if ($conf{'table_rotate'} eq "daily") {
			$table_id = "xfers_d" . $ds[7] . "_" . $ds[5];
		} elsif ($conf{'table_rotate'} eq "weekly") {
			$table_id = "xfers_w" . int($ds[7]/7) . "_" . $ds[5];
		} elsif ($conf{'table_rotate'} eq "monthly") {
			$table_id = "xfers_m" . $ds[4] . "_" . $ds[5];
		} else { # yearly is the default, safeguard
			$table_id = "xfers_y" . $ds[5];
		}

		if ($count == 0) {
			# first xfer, won't have an $old_table_id
			$old_table_id = $table_id;
			$query =~ s/xfers\.static/$table_id/;
		}
		# is this xfer in a new timescale than the old one?
		if ($table_id ne $old_table_id) {
			# new query - execute the old one first.
			if (!($existing_tables =~ m/$old_table_id/)) {
				# create the table, doesn't exist
				&create_subtable($old_table_id);
				$existing_tables .= " " . $old_table_id;
			}
			$query =~ s/xfers\.static/$old_table_id/;
		
			chop $query; chop $query;
			$result = &doQuery($query);
			$size_of_query = 0;

			$query = $std_query;
			$query =~ s/xfers\.static/$table_id/;
		}

		$query .= "	( '$elements{\"user\"}', '$elements{\"date\"}'," .
		" $elements{\"durn\"}, '$elements{\"file\"}', $elements{\"size\"}," .
		" '$elements{\"dirn\"}', '$elements{\"host\"}'),\n"; 

		$size_of_query++;

		$count++;

		if ($size_of_query >= $opt_interval) {
			if (!($existing_tables =~ m/$table_id/)) {
				# create the table, doesn't exist
				&create_subtable($table_id);
				$existing_tables .= " " . $table_id;
			}
			$query =~ s/xfers\.static/$table_id/;
			chop $query; chop $query;
			$result = &doQuery($query);
			$query = $std_query;
			print $query . "\n";
			$size_of_query = 0;
		}
	}
}

# if there's anything left in the query queue, enter it
if ($size_of_query > 0) {
	if (!($existing_tables =~ m/$table_id/)) {
		# create the table, doesn't exist
		&create_subtable($table_id);
		$existing_tables .= " " . $table_id;
	}
	$query =~ s/xfers\.static/$table_id/;
	chop $query; chop $query;
	if (($result = &doQuery($query)) == -1) {
		&repair($table_id);
		exit 1 if (($result = &doQuery($query)) == -1);
	}
}

# convert spaces to commas, but make sure there isn't one at the beginning
$existing_tables =~ s/ /\,/g;
$existing_tables =~ s/^\,//g;

# This could be improved to ask MySQL directly if xfers exists
# if it does, nuke it before re-creating
if ($existing_tables =~ m/xfers\,/) { 
	$existing_tables =~ s/xfers\,//;
	$query = "DROP TABLE xfers";
	$result = &doQuery($query);
}

$query = "CREATE TABLE xfers ($std_table) TYPE=MERGE UNION=($existing_tables)";
$result = &doQuery($query);

print "Added $count transfers\n";

exit 0;


sub doQuery # Do a MySQL query {{{
{
	my ($query) = @_;

	my $result = $db_conn->prepare($query);
	if (!$result) {
		warn("Preparation error:\n$query\n$DBI::errstr");
	} else {
		if (!($r = $result->execute)) {
			warn("\nMySQL error, trying again in 5 secs\n\n");
			sleep 5;
			if (!($r = $result->execute)) {
				warn("\nMySQL error on:\n$query\n" .
				"Returned:\n$DBI::errstr\n\n");
				return -1;
			}
		}
	}

	return $result;
} # }}}

sub parse_conf # Parse .ini file {{{
{
	local *FILE;
	my $conf;
	open(FILE, $opt_config) or die "Can't open config";
		while(<FILE>) {
			chomp($_);
			if ($_ =~ m/^(.+)\s+=\s+(.+)$/) {
				$conf{"$1"} = $2;
			}
		}
	close(FILE);
	return(%conf);

} # }}}

sub repair # Try to repair a table {{{
{
	my ($table) = @_;

	my $result = $db_conn->prepare("REPAIR TABLE $table");
	my $r      = $result->execute;

	return;
}
sub create_subtable # create a subtable for the MERGE {{{
{
	my ($table_id) = @_;

	my $query = "CREATE TABLE $table_id ( $std_table ) ";
	$query .= "TYPE=MyISAM PACK_KEYS=1 CHECKSUM=1 DELAY_KEY_WRITE=1";
	$result = &doQuery($query);

	my @columns = ('xferID', 'Username', 'xferTime', 'xferDur',
	'xferFile', 'xferSize', 'xferDirn', 'xferIP');

	# make indexes for this new table
	foreach $column (@columns) {
		my $query = "CREATE INDEX $column ON $table_id($column)";
		$result = &doQuery($query);
	}

}

# }}}

# Log formats

sub pureftpd_parse # parse a PureFTPd logline {{{
{
	my ($logline) = @_;

	($elements{'tstamp'}, $elements{"uniq_id"}, $elements{'user'},
	$elements{"host"},
	$dirn, $elements{'size'}, $elements{'durn'}, @file)
		= split /\s+/, $logline;

	$elements{'file'} = join ' ', @file;

	$elements{'dirn'} = 'Up' if ($dirn eq 'U');
	$elements{'dirn'} = 'Dn' if ($dirn eq 'D');

	return %elements;

} # }}}

sub xferlog_parse # parse an xferlog logline {{{
{
	my ($logline) = @_;

	(undef, $mnth, $date, $time, $year,
	$elements{"durn"}, $elements{"host"}, $elements{"size"}, @rest)
		= split /\s+/, $logline;

	# 0 byte files 1s to upload - avoid bugs in working stuff out
	$elements{"durn"} = 1 if ($elements{"size"} == 0);

	($hours, $min, $sec) = split /\:/, $time;

	$elements{'tstamp'} = timelocal($sec, $min,
		$hours, $date, $months{$mnth}-1, $year);

	pop @rest if (/\s[ic]$/); # trailing i or c?

	(undef, undef, undef, $elements{"user"}, $ax_mode, $dirn, undef, $type)
		= (pop @rest, pop @rest, pop @rest, pop @rest,
		   pop @rest, pop @rest, pop @rest, pop @rest);

	$elements{"file"} = join '', @rest;

	# $type: a = ascii  b = binary


	# $dirn: i = incoming (upld)  o = outgoing (dnld)  d = delete
	if ($dirn eq "i") {
		$elements{"dirn"} = "Up";
	} elsif ($dirn eq "o") {
		$elements{"dirn"} = "Dn";
	} else {
		return;
	}

	# $ax_mode: a = anonymous  g = passworded guest  r = local user
	$elements{"user"} = "ftp" if ($ax_mode eq "a");

	return %elements;
} # }}}

sub ncftpd_parse # parse ncftpd logline {{{
{
	my ($logline) = @_;

	$logline =~ s/^(\d+)-(\d+)-(\d+) (\d+):(\d+):(\d+).*\| //;
	$year = $1; $month = $2; $date = $3;
	$hours = $4; $min = $5; $sec = $6;

	$elements{'tstamp'} = timelocal($sec, $min,
		$hours, $date, $month-1, $year);


	($dirn, $elements{"file"}, $elements{"size"}, $elements{"durn"},
		undef, $elements{"user"}, undef, $elements{"host"}, undef,
		$elements{"completion"}, $type, undef, undef)	
			= split /\,/, $logline;

	# $type: A = ascii  I = binary

	if ($dirn eq "S") {
		$elements{"dirn"} = "Up";
	} elsif ($dirn eq "R") {
		$elements{"dirn"} = "Dn";
	} else {
		return;
	}

	$elements{"user"} = "ftp" if ($elements{"user"} eq "anonymous");

	return %elements;

} # }}}
