/************************************* */
/* Rule Set Based Access Control       */
/* Author and (c) 1999-2001:           */
/*   Amon Ott <ao@rsbac.org>           */
/* Network access control data structs */
/* Last modified: 30/Oct/2001          */
/************************************* */

#ifndef __RSBAC_NETWORK_TYPES_H
#define __RSBAC_NETWORK_TYPES_H

#define RSBAC_NET_ANY 0
#define RSBAC_NET_UNKNOWN 0

#define RSBAC_NET_TEMP_VERSION 1
#define RSBAC_NET_TEMP_KEY 0x815affe
#define RSBAC_NET_TEMP_NAME "nettemp"

typedef __u32 rsbac_net_temp_id_t;

#define RSBAC_NET_MAX_ADDRESS_LEN 128
#define RSBAC_NET_TEMP_NAMELEN 16

struct rsbac_net_temp_data_t
  {
    /* must be first for alignment */
    char              address[RSBAC_NET_MAX_ADDRESS_LEN];
    __u8              address_family;
    __u8              valid_len; /* Bytes for AF_UNIX, Bits for all others */
    __u8              type;
    __u8              protocol;
    rsbac_netdev_id_t netdev;
    __u16             min_port; /* for those address families that support them */
    __u16             max_port;
    char              name[RSBAC_NET_TEMP_NAMELEN];
  };

/* default template */
#define RSBAC_NET_TEMP_DEFAULT_DATA \
  { \
    "",      /* address (not used, because len=0) */ \
    AF_MAX,  /* impossible family */ \
    0,       /* valid_len (match any) */ \
    RSBAC_NET_ANY, /* type (not used) */ \
    RSBAC_NET_ANY, /* protocol (not used) */ \
    "",      /* netdev (not used) */ \
    0,       /* min_port (not used) */ \
    0,       /* max_port (not used) */ \
    "DEFAULT" /* name */ \
  }

/* template to match all UNIX objects */
#define RSBAC_NET_TEMP_UNIX_ID 100010
#define RSBAC_NET_TEMP_UNIX_DATA \
  { \
    "",      /* address (not used, because len=0) */ \
    AF_UNIX, /* family */ \
    0,       /* valid_len (match any) */ \
    0,       /* type (not used) */ \
    0,       /* protocol (not used) */ \
    "",      /* netdev (not used) */ \
    0,       /* min_port (not used) */ \
    0,       /* max_port (not used) */ \
    "AF_UNIX" /* name */ \
  }

/* template to match all INET (IPv4) objects in the localnet */
#define RSBAC_NET_TEMP_LNET_ID 100101
#define RSBAC_NET_TEMP_LNET_ADDRESS "127.0.0.0"
#define RSBAC_NET_TEMP_LNET_DATA \
  { \
    "",            /* address filled in init code - do not enter dot-notation here! */ \
    AF_INET,       /* family */ \
    8,             /* valid_len */ \
    RSBAC_NET_ANY, /* type */ \
    RSBAC_NET_ANY, /* protocol */ \
    "",            /* netdev (any) */ \
    0,             /* min_port */ \
    65535,         /* max_port */ \
    "localnet"     /* name */ \
  }

/* template to match all INET (IPv4) objects in the internal LAN */
#define RSBAC_NET_TEMP_LAN_ID 100102
#define RSBAC_NET_TEMP_LAN_ADDRESS "192.168.0.0"
#define RSBAC_NET_TEMP_LAN_DATA \
  { \
    "",            /* address filled in init code - do not enter dot-notation here! */ \
    AF_INET,       /* family */ \
    16,            /* valid_len */ \
    RSBAC_NET_ANY, /* type */ \
    RSBAC_NET_ANY, /* protocol */ \
    "",            /* netdev (any) */ \
    0,             /* min_port */ \
    65535,         /* max_port */ \
    "Internal LAN" /* name */ \
  }

/* template to match all INET (IPv4) objects with 0.0.0.0 address */
/* (bind to all addresses / let system select address) */
#define RSBAC_NET_TEMP_AUTO_ID 100105
#define RSBAC_NET_TEMP_AUTO_ADDRESS "0.0.0.0"
#define RSBAC_NET_TEMP_AUTO_DATA \
  { \
    "",            /* address filled in init code - do not enter dot-notation here! */ \
    AF_INET,       /* family */ \
    32,            /* valid_len */ \
    RSBAC_NET_ANY, /* type */ \
    RSBAC_NET_ANY, /* protocol */ \
    "",            /* netdev (any) */ \
    0,             /* min_port */ \
    65535,         /* max_port */ \
    "Auto-IPv4"    /* name */ \
  }

/* template to match all INET (IPv4) objects */
#define RSBAC_NET_TEMP_INET_ID 100110
#define RSBAC_NET_TEMP_INET_DATA \
  { \
    "",            /* address (not used, because len=0) */ \
    AF_INET,       /* family */ \
    0,             /* valid_len (match any) */ \
    RSBAC_NET_ANY, /* type */ \
    RSBAC_NET_ANY, /* protocol */ \
    "",            /* netdev (any) */ \
    0,             /* min_port */ \
    65535,         /* max_port */ \
    "AF_INET" /* name */ \
  }

/* template to match all objects */
#define RSBAC_NET_TEMP_ALL_ID ((rsbac_net_temp_id_t) -1)
#define RSBAC_NET_TEMP_ALL_DATA \
  { \
    "",            /* address (not used, no family) */ \
    RSBAC_NET_ANY, /* family */ \
    0,             /* valid_len (match any) */ \
    RSBAC_NET_ANY, /* type */ \
    RSBAC_NET_ANY, /* protocol */ \
    "",            /* netdev (any) */ \
    0,             /* min_port */ \
    65535,         /* max_port */ \
    "ALL" /* name */ \
  }

struct rsbac_net_description_t
  {
    __u8              address_family;
    void            * address;
    __u8              address_len;
    __u8              type;
    __u8              protocol;
    rsbac_netdev_id_t netdev;
    __u16             port;
  };

enum rsbac_net_temp_syscall_t
  {
    NTS_new_template,
    NTS_copy_template,
    NTS_delete_template,
    NTS_check_id,
    NTS_get_address,
    NTS_get_address_family,
    NTS_get_valid_len,
    NTS_get_type,
    NTS_get_protocol,
    NTS_get_netdev,
    NTS_get_min_port,
    NTS_get_max_port,
    NTS_get_name,
    NTS_set_address,
    NTS_set_address_family,
    NTS_set_valid_len,
    NTS_set_type,
    NTS_set_protocol,
    NTS_set_netdev,
    NTS_set_min_port,
    NTS_set_max_port,
    NTS_set_name,
    NTS_none
  };

union rsbac_net_temp_syscall_data_t
  {
    rsbac_net_temp_id_t id;
    char                address[RSBAC_NET_MAX_ADDRESS_LEN];
    __u8                address_family;
    __u8                valid_len; /* Bytes for AF_UNIX, Bits for all others */
    __u8                type;
    __u8                protocol;
    rsbac_netdev_id_t   netdev;
    __u16               min_port; /* for those address families that support them */
    __u16               max_port;
    char                name[RSBAC_NET_TEMP_NAMELEN];
  };

/*
 *      Display an IP address in readable format.
 */

#ifndef NIPQUAD
#define NIPQUAD(addr) \
	((unsigned char *)&addr)[0], \
	((unsigned char *)&addr)[1], \
	((unsigned char *)&addr)[2], \
	((unsigned char *)&addr)[3]

#define HIPQUAD(addr) \
	((unsigned char *)&addr)[3], \
	((unsigned char *)&addr)[2], \
	((unsigned char *)&addr)[1], \
	((unsigned char *)&addr)[0]
#endif

#endif
