/*********************************** */
/* Rule Set Based Access Control     */
/* Author and (c)1999-2003:          */
/*   Amon Ott <ao@rsbac.org>         */
/* API: Data types for attributes    */
/*      and standard module calls    */
/* Last modified: 25/Jul/2003        */
/*********************************** */

#ifndef __RSBAC_TYPES_H
#define __RSBAC_TYPES_H

/* trigger module dependency for EXPORT_SYMBOL */
#ifdef CONFIG_MODULES
#endif

#define RSBAC_VERSION "v1.2.2"
#define RSBAC_VERSION_MAJOR 1
#define RSBAC_VERSION_MID 2
#define RSBAC_VERSION_MINOR 2
#define RSBAC_VERSION_NR \
 ((RSBAC_VERSION_MAJOR << 16) | (RSBAC_VERSION_MID << 8) | RSBAC_VERSION_MINOR)
#define RSBAC_VERSION_MAKE_NR(x,y,z) \
 ((x << 16) & (y << 8) & z)

#include <linux/types.h>

#ifdef __KERNEL__
#include <linux/fs.h>
#include <linux/socket.h>
#include <linux/pipe_fs_i.h>
#include <linux/kdev_t.h>

/* version checks */
#ifndef LINUX_VERSION_CODE
#include <linux/version.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
#error "RSBAC: unsupported kernel version"
#endif

#endif

/* General */

#ifndef NULL
#define NULL ((void *) 0)
#endif

#define rsbac_min(a,b) (((a)<(b))?(a):(b))
#define rsbac_max(a,b) (((a)>(b))?(a):(b))

#define RSBAC_OLD_NO_USER 65533
#define RSBAC_OLD_ALL_USERS 65532
#define RSBAC_NO_USER ((rsbac_uid_t) -3)
#define RSBAC_ALL_USERS ((rsbac_uid_t) -4)

#ifndef __cplusplus
#if defined(FALSE) || defined(TRUE) || defined(boolean)
#ifndef boolean
typedef int boolean;
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif
#else
typedef enum {FALSE, TRUE} boolean;
#endif
#else
typedef bool boolean;
#endif

typedef __u8 rsbac_boolean_int_t;

#define RSBAC_IFNAMSIZ 16
typedef u_char rsbac_netdev_id_t[RSBAC_IFNAMSIZ + 1];

#define RSBAC_SEC_DEL_CHUNK_SIZE 65536

/* Adjust these, if you have to, but if you do, adjust them all! */
/* Note: no / allowed, file must be exactly in second level! */
#define RSBAC_AUTH_LOGIN_PATH "/bin/login"
#define RSBAC_AUTH_LOGIN_PATH_DIR "bin"
#define RSBAC_AUTH_LOGIN_PATH_FILE "login"

/* These data structures work parallel to the Linux data structures, */
/* so all data for RSBAC decisions is maintained seperately.         */
/* Any change to RSBAC data will NOT modify any other linux data,    */
/* e.g. userlists, process lists or inodes.                          */

typedef __u32 rsbac_version_t;
typedef __u32 rsbac_uid_t;                   /* Same as user in Linux kernel */
typedef __u32 rsbac_gid_t;                   /* Same as group in Linux kernel */
typedef __u16 rsbac_old_uid_t;               /* Same as user in Linux kernel */
typedef __u16 rsbac_old_gid_t;               /* Same as group in Linux kernel */
typedef __u32 rsbac_time_t;                  /* Same as time_t in Linux kernel */
typedef __u32 rsbac_cap_vector_t;            /* Same as kernel_cap_t in Linux kernel */

/* Special generic lists time-to-live (ttl) value to keep old setting */
#define RSBAC_LIST_TTL_KEEP ((rsbac_time_t) -1)

typedef __u8 rsbac_enum_t; /* internally used for all enums */

#define RSBAC_SYSADM_UID   0
#define RSBAC_BIN_UID      1
#ifdef CONFIG_RSBAC_SECOFF_UID
#define RSBAC_SECOFF_UID   CONFIG_RSBAC_SECOFF_UID
#else
#define RSBAC_SECOFF_UID 400
#endif
#define RSBAC_DATAPROT_UID (RSBAC_SECOFF_UID+1)
#define RSBAC_TPMAN_UID    (RSBAC_SECOFF_UID+2)
#define RSBAC_AUDITOR_UID  (RSBAC_SECOFF_UID+4)

typedef __u32 rsbac_pseudo_t;               /* For Pseudonymic Logging */
typedef __u32 rsbac_pid_t;                   /* Same as pid in Linux  */

enum    rsbac_cap_process_hiding_t {PH_off, PH_from_other_users, PH_full,
                              PH_none};
typedef rsbac_enum_t rsbac_cap_process_hiding_int_t;


typedef __u8 rsbac_security_level_t;
#define SL_max            252
#define SL_min            0
// #define SL_rsbac_internal 253
#define SL_inherit        254
#define SL_none           255
enum    rsbac_old_security_level_t {SL_unclassified, SL_confidential, SL_secret,
                                    SL_top_secret, SL_old_rsbac_internal,
                                    SL_old_inherit, SL_old_none};
                                             /* MAC security levels   */
typedef __u64 rsbac_mac_category_vector_t;   /* MAC category sets */
#define RSBAC_MAC_GENERAL_CATEGORY 0
#define RSBAC_MAC_DEF_CAT_VECTOR ((rsbac_mac_category_vector_t) 1)
  /* 1 << GENERAL_CAT */
#define RSBAC_MAC_MAX_CAT_VECTOR ((rsbac_mac_category_vector_t) -1)
  /* all bits set */
#define RSBAC_MAC_MIN_CAT_VECTOR ((rsbac_mac_category_vector_t) 0)
  /* no bits set */
#define RSBAC_MAC_INHERIT_CAT_VECTOR ((rsbac_mac_category_vector_t) 0)
  /* for fd: no bits set */
#define RSBAC_MAC_NR_CATS 64
#define RSBAC_MAC_MAX_CAT 63

#define RSBAC_MAC_CAT_VECTOR(x) ((rsbac_mac_category_vector_t) 1 << (x))

typedef u_int rsbac_cwi_relation_id_t;

/* For MAC, FC, SIM, FF, AUTH */
enum    rsbac_system_role_t {SR_user, SR_security_officer, SR_administrator,
                             SR_auditor, SR_none};
typedef rsbac_enum_t rsbac_system_role_int_t;

/* For FC  */
enum    rsbac_object_category_t {OC_general, OC_security, OC_system,
                                 OC_inherit, OC_none};
#define RSBAC_FC_OC_DEF OC_inherit
#define RSBAC_FC_OC_ROOT_DEF OC_general
/* internal type */
typedef rsbac_enum_t rsbac_fc_oc_t;

/* For SIM */
enum    rsbac_data_type_t {DT_none, DT_SI, DT_inherit};
#define RSBAC_SIM_DT_DEF DT_inherit
#define RSBAC_SIM_DT_ROOT_DEF DT_none
/* internal type */
typedef rsbac_enum_t rsbac_sim_dt_t;

/* For all models */
enum    rsbac_scd_type_t {ST_time_strucs, ST_clock, ST_host_id,
                          ST_net_id, ST_ioports, ST_rlimit,
                          ST_swap, ST_syslog, ST_rsbac, ST_rsbaclog,
                          ST_other, ST_kmem, ST_network, ST_firewall,
                          ST_none};

enum    rsbac_dev_type_t {D_block, D_char, D_none};


enum    rsbac_ipc_type_t {I_sem, I_msg, I_shm, I_none};
union   rsbac_ipc_id_t
  {
    u_long id_nr;
  };

typedef __u32 rsbac_inode_nr_t;

enum    rsbac_linux_dac_disable_t {LDD_false, LDD_true, LDD_inherit, LDD_none};
typedef rsbac_enum_t rsbac_linux_dac_disable_int_t;

#ifdef __KERNEL__
/* We need unique identifiers for each file/dir. inode means inode in */
/* the file system.                                                   */
struct rsbac_fs_file_t
    {
      kdev_t               device;
      rsbac_inode_nr_t     inode;
      struct dentry      * dentry_p;  /* used for inheritance recursion */
    };

/* We need unique ids for dev objects */
struct rsbac_dev_t
    {
      enum  rsbac_dev_type_t     type;
            kdev_t               id;
    };
#endif /* __KERNEL */

/* And we need unique ids for ipc objects */
struct rsbac_ipc_t
    {
      enum  rsbac_ipc_type_t     type;
      union rsbac_ipc_id_t       id;
    };

/* log levels: nothing, denied requests only, all, refer to request log level */
enum    rsbac_log_level_t {LL_none, LL_denied, LL_full, LL_request, LL_invalid};
typedef __u64 rsbac_log_array_t;

/* request bitvectors */
typedef __u64 rsbac_request_vector_t;
#define RSBAC_REQUEST_VECTOR(x) ((rsbac_request_vector_t) 1 << (x))

/* The max length of each filename is kept in a macro */
#define RSBAC_MAXNAMELEN     256

/* MAC */

typedef __u8 rsbac_mac_user_flags_t;
typedef __u16 rsbac_mac_process_flags_t;
typedef __u8 rsbac_mac_file_flags_t;

#define MAC_override		1
#define MAC_auto		2
#define MAC_trusted     	4
#define MAC_write_up		8
#define MAC_read_up		16
#define MAC_write_down		32
#define MAC_allow_auto		64
#define MAC_prop_trusted	128
#define MAC_program_auto	256

#define RSBAC_MAC_U_FLAGS (MAC_override | MAC_trusted | MAC_write_up | MAC_read_up | MAC_write_down | MAC_allow_auto)
#define RSBAC_MAC_P_FLAGS (MAC_override | MAC_auto | MAC_trusted | MAC_write_up | MAC_read_up | MAC_write_down | MAC_prop_trusted | MAC_program_auto)
#define RSBAC_MAC_F_FLAGS (MAC_auto | MAC_trusted | MAC_write_up | MAC_read_up | MAC_write_down)

#define RSBAC_MAC_DEF_U_FLAGS 0
#define RSBAC_MAC_DEF_SYSADM_U_FLAGS MAC_allow_auto
#define RSBAC_MAC_DEF_SECOFF_U_FLAGS MAC_override

#define RSBAC_MAC_DEF_P_FLAGS 0
#define RSBAC_MAC_DEF_INIT_P_FLAGS MAC_auto

typedef rsbac_enum_t rsbac_mac_auto_int_t;
enum    rsbac_mac_auto_t {MA_no, MA_yes, MA_inherit};

/* PM */

#include <rsbac/pm_types.h>

/* MS */
typedef __u32 rsbac_ms_scanned_t;
#define MS_unscanned 0
#define MS_rejected 1
#define MS_active_rejected 2
#define MS_scan_error 3
#define MS_scanning 4
#define DEFAULT_MS_FD_SCANNED MS_unscanned

#define MS_need_scan_no 0
#define MS_need_scan_exec 1
#define MS_need_scan_full 2
#define MS_need_scan_inherit 3
#define DEFAULT_MS_FD_NEED_SCAN MS_need_scan_inherit
#define DEFAULT_MS_ROOT_DIR_NEED_SCAN MS_need_scan_no
typedef rsbac_enum_t rsbac_ms_need_scan_t;

enum    rsbac_ms_trusted_t {MT_not_trusted, MT_read, MT_full, MT_none};
typedef rsbac_enum_t rsbac_ms_trusted_int_t;

enum    rsbac_ms_sock_trusted_t {MS_not_trusted, MS_active, MS_full, MS_none};
typedef rsbac_enum_t rsbac_ms_sock_trusted_int_t;

struct ms_segment_t
  {
    int len;
    char * ubuf;
    enum {MP_TCP,MP_UDP} prot;
  };

#ifdef __KERNEL__
typedef int rsbac_ms_do_scan_t(struct dentry * dentry_p);
extern rsbac_ms_do_scan_t * rsbac_ms_do_scan;
extern int rsbac_ms_scan_level;
#endif

/* FF */

typedef __u16 rsbac_ff_flags_t;
#define FF_read_only       1
#define FF_execute_only    2
#define FF_search_only     4
#define FF_write_only      8
#define FF_secure_delete  16
#define FF_no_execute     32
#define FF_no_delete_or_rename 64
#define FF_append_only   256
#define FF_no_mount      512

#define FF_add_inherited 128

#define RSBAC_FF_DEF FF_add_inherited
#define RSBAC_FF_ROOT_DEF 0

/***** RC *****/

#include <rsbac/rc_types.h>

/**** AUTH ****/
/* special cap value, replaced by process owner at execute time */
#define RSBAC_AUTH_MAX_MAXNUM 1000000
#define RSBAC_AUTH_OLD_OWNER_F_CAP (rsbac_old_uid_t) -3
#define RSBAC_AUTH_OWNER_F_CAP (rsbac_uid_t) -3
#define RSBAC_AUTH_MAX_RANGE_UID (rsbac_uid_t) -10
typedef struct rsbac_fs_file_t rsbac_auth_file_t;
struct rsbac_auth_cap_range_t
  {
    rsbac_uid_t first;
    rsbac_uid_t last;
  };
enum    rsbac_auth_cap_type_t {ACT_real, ACT_eff, ACT_fs, ACT_none};
typedef rsbac_enum_t rsbac_auth_cap_type_int_t;

/**** ACL ****/
/* include at end of types.h */

/**** CAP ****/
#include <linux/capability.h>
#define CAP_NONE 29
#define RSBAC_CAP_MAX CAP_NONE

/**** JAIL ****/

#define RSBAC_JAIL_VERSION 1

typedef __u32 rsbac_jail_id_t;
#define RSBAC_JAIL_DEF_ID 0
typedef __u32 rsbac_jail_ip_t;

typedef __u32 rsbac_jail_flags_t;
#define JAIL_allow_external_ipc 1
#define JAIL_allow_all_net_family 2
#define JAIL_allow_rlimit 4
#define JAIL_allow_inet_raw 8
#define JAIL_auto_adjust_inet_any 16
#define JAIL_allow_inet_localhost 32

#define RSBAC_JAIL_LOCALHOST ((1 << 24) | 127)

/**** RES ****/

typedef __u32 rsbac_res_limit_t;
#define RSBAC_RES_UNSET 0

#define RSBAC_RES_MAX 10 /* RLIMIT_LOCKS in 2.4.x kernels */
#define RSBAC_RES_NONE 11

typedef rsbac_res_limit_t rsbac_res_array_t[RSBAC_RES_MAX + 1];

/**** REG ****/
typedef __s32 rsbac_reg_handle_t;


/****************************************************************************/
/* ADF types                                                                */
/****************************************************************************/

#include <rsbac/network_types.h>

#ifdef __KERNEL__
    typedef struct socket * rsbac_net_obj_id_t;
#else
    typedef void * rsbac_net_obj_id_t;
#endif

struct rsbac_net_obj_desc_t
  {
    rsbac_net_obj_id_t sock_p;
    void * local_addr;
    u_int  local_len;
    void * remote_addr;
    u_int  remote_len;
  };

#define RSBAC_ADF_REQUEST_ARRAY_VERSION 2

enum  rsbac_adf_request_t {
                        R_ADD_TO_KERNEL,
                        R_ALTER,
                        R_APPEND_OPEN,
                        R_CHANGE_GROUP,
                        R_CHANGE_OWNER,
                        R_CHDIR,
                        R_CLONE,
                        R_CLOSE,
                        R_CREATE,
                        R_DELETE,
                        R_EXECUTE,
                        R_GET_PERMISSIONS_DATA,
                        R_GET_STATUS_DATA,
                        R_LINK_HARD,
                        R_MODIFY_ACCESS_DATA,
                        R_MODIFY_ATTRIBUTE,
                        R_MODIFY_PERMISSIONS_DATA,
                        R_MODIFY_SYSTEM_DATA,
                        R_MOUNT,
                        R_READ,
                        R_READ_ATTRIBUTE,
                        R_READ_WRITE_OPEN,
                        R_READ_OPEN,
                        R_REMOVE_FROM_KERNEL,
                        R_RENAME,
                        R_SEARCH,
                        R_SEND_SIGNAL,
                        R_SHUTDOWN,
                        R_SWITCH_LOG,
                        R_SWITCH_MODULE,
                        R_TERMINATE,
                        R_TRACE,
                        R_TRUNCATE,
                        R_UMOUNT,
                        R_WRITE,
                        R_WRITE_OPEN,
                        R_MAP_EXEC,
                        R_BIND,
                        R_LISTEN,
                        R_ACCEPT,
                        R_CONNECT,
                        R_SEND,
                        R_RECEIVE,
                        R_NET_SHUTDOWN,
                        R_CHANGE_DAC_EFF_OWNER,
                        R_CHANGE_DAC_FS_OWNER,
                        R_NONE
                      };

typedef rsbac_enum_t rsbac_adf_request_int_t;

#include <rsbac/request_groups.h>

/* This type is returned from the rsbac_adf_request() function. Since a */
/* decision of undefined means an error, it is never returned.          */

enum  rsbac_adf_req_ret_t {NOT_GRANTED,GRANTED,DO_NOT_CARE,UNDEFINED};

/****************************************************************************/
/* ACI types                                                                */
/****************************************************************************/

/* For switching adf-modules */
enum  rsbac_switch_target_t {GEN,MAC,FC,SIM,PM,MS,FF,RC,AUTH,REG,ACL,CAP,JAIL,RES,SOFTMODE,DAC_DISABLE,SW_NONE};
#define RSBAC_MAX_MOD (SOFTMODE - 1)
typedef rsbac_enum_t rsbac_switch_target_int_t;

/****************************************************************************/
/* For objects, users and processes all manipulation is encapsulated by the */
/* function calls rsbac_set_attr, rsbac_get_attr and rsbac_remove_target.   */

/* For those, we declare some extra types to specify target and attribute.  */

enum   rsbac_target_t {T_FILE, T_DIR, T_FIFO, T_SYMLINK, T_DEV, T_IPC, T_SCD, T_USER, T_PROCESS,
                       T_NETDEV, T_NETTEMP, T_NETOBJ, T_NETTEMP_NT,
                       T_FD,
                       T_NONE};

union  rsbac_target_id_t
       {
#ifdef __KERNEL__
          struct rsbac_fs_file_t    file;
          struct rsbac_fs_file_t    dir;
          struct rsbac_fs_file_t    fifo;
          struct rsbac_fs_file_t    symlink;
          struct rsbac_dev_t        dev;
#endif
          struct rsbac_ipc_t        ipc;
          enum rsbac_scd_type_t     scd;
          rsbac_uid_t               user;
          rsbac_pid_t               process;
          rsbac_netdev_id_t         netdev;
          rsbac_net_temp_id_t       nettemp;
          struct rsbac_net_obj_desc_t netobj;
          int                       dummy;
       };

#ifdef __KERNEL__
typedef rsbac_enum_t rsbac_log_entry_t[T_NONE+1];

/* used with CREATE on DIR with REG turned on only */
struct rsbac_create_data_t
  {
    enum   rsbac_target_t   target;
    struct dentry         * dentry_p;
           int              mode;
           kdev_t           device; /* for mknod etc. */
  };
#endif

enum   rsbac_attribute_t
  {
    A_pseudo,
    A_security_level,
    A_initial_security_level,
    A_local_sec_level,
    A_remote_sec_level,
    A_min_security_level,
    A_mac_categories,
    A_mac_initial_categories,
    A_local_mac_categories,
    A_remote_mac_categories,
    A_mac_min_categories,
    A_mac_user_flags,
    A_mac_process_flags,
    A_mac_file_flags,
    A_object_category,
    A_local_object_category,
    A_remote_object_category,
    A_data_type,
    A_local_data_type,
    A_remote_data_type,
    A_system_role,
    A_mac_role,
    A_fc_role,
    A_sim_role,
    A_ms_role,
    A_ff_role,
    A_auth_role,
    A_cap_role,
    A_jail_role,
    A_current_sec_level,
    A_mac_curr_categories,
    A_min_write_open,
    A_min_write_categories,
    A_max_read_open,
    A_max_read_categories,
    A_mac_auto,
    A_mac_trusted_for_user,
    A_mac_check,
    A_mac_prop_trusted,
    A_pm_role,
    A_pm_process_type,
    A_pm_current_task,
    A_pm_object_class,
    A_local_pm_object_class,
    A_remote_pm_object_class,
    A_pm_ipc_purpose,
    A_local_pm_ipc_purpose,
    A_remote_pm_ipc_purpose,
    A_pm_object_type,
    A_local_pm_object_type,
    A_remote_pm_object_type,
    A_pm_program_type,
    A_pm_tp,
    A_pm_task_set,
    A_ms_scanned,
    A_ms_trusted,
    A_ms_backbuf,
    A_ms_buflen,
    A_ms_str_nr,
    A_ms_str_offset,
    A_ms_sock_trusted_tcp,
    A_ms_sock_trusted_udp,
    A_ms_need_scan,
    A_ff_flags,
    A_rc_type,
    A_local_rc_type,
    A_remote_rc_type,
    A_rc_type_fd,
    A_rc_type_nt,
    A_rc_force_role,
    A_rc_initial_role,
    A_rc_role,
    A_rc_def_role,
    A_auth_may_setuid,
    A_auth_may_set_cap,
    A_min_caps,
    A_max_caps,
    A_jail_id,
    A_jail_ip,
    A_jail_flags,
    A_res_role,
    A_res_min,
    A_res_max,
    A_log_array_low,
    A_local_log_array_low,
    A_remote_log_array_low,
    A_log_array_high,
    A_local_log_array_high,
    A_remote_log_array_high,
    A_log_program_based,
    A_log_user_based,
    A_symlink_add_uid,
    A_symlink_add_mac_level,
    A_symlink_add_rc_role,
    A_linux_dac_disable,
    A_cap_process_hiding,
#ifdef __KERNEL__
    /* adf-request helpers */
    A_owner,
    A_group,
    A_signal,
    A_mode,
    A_nlink,
    A_switch_target,
    A_mod_name,
    A_request,
    A_ms_segment,
    A_trace_request,
    A_auth_add_f_cap,
    A_auth_remove_f_cap,
    A_auth_get_caplist,
    A_prot_bits,
    A_internal,
    /* used with CREATE on DIR */
    A_create_data,
    A_new_object,
    A_rlimit,
    A_new_dir_dentry_p,
#endif
    A_none};

#include <rsbac/ms_strings.h>

union  rsbac_attribute_value_t
  {
         rsbac_uid_t                 owner;           /* process owner */
         rsbac_pseudo_t              pseudo;
         rsbac_security_level_t      security_level;
         rsbac_mac_category_vector_t mac_categories;
         rsbac_fc_oc_t               object_category;
         rsbac_sim_dt_t              data_type;
         rsbac_system_role_int_t     system_role;
         rsbac_security_level_t      current_sec_level;
         rsbac_security_level_t      min_write_open;
         rsbac_security_level_t      max_read_open;
         rsbac_mac_user_flags_t      mac_user_flags;
         rsbac_mac_process_flags_t   mac_process_flags;
         rsbac_mac_file_flags_t      mac_file_flags;
         rsbac_uid_t                 mac_trusted_for_user;
         rsbac_mac_auto_int_t        mac_auto;
         boolean                     mac_check;
         boolean                     mac_prop_trusted;
         rsbac_pm_role_int_t         pm_role;
         rsbac_pm_process_type_int_t pm_process_type;
         rsbac_pm_task_id_t          pm_current_task;
         rsbac_pm_object_class_id_t  pm_object_class;
         rsbac_pm_purpose_id_t       pm_ipc_purpose;
         rsbac_pm_object_type_int_t  pm_object_type;
         rsbac_pm_program_type_int_t pm_program_type;
         rsbac_pm_tp_id_t            pm_tp;
         rsbac_pm_task_set_id_t      pm_task_set;
         rsbac_ms_scanned_t          ms_scanned;
         rsbac_ms_trusted_int_t      ms_trusted;
         char                      * ms_backbuf;
         int                         ms_buflen;
         int                         ms_str_nr[RSBAC_MS_NR_MALWARE];
         int                         ms_str_offset[RSBAC_MS_NR_MALWARE];
         rsbac_ms_sock_trusted_int_t ms_sock_trusted_tcp;
         rsbac_ms_sock_trusted_int_t ms_sock_trusted_udp;
         rsbac_ms_need_scan_t        ms_need_scan;
         rsbac_ff_flags_t            ff_flags;
         rsbac_rc_type_id_t          rc_type;
         rsbac_rc_type_id_t          rc_type_fd;
         rsbac_rc_role_id_t          rc_force_role;
         rsbac_rc_role_id_t          rc_initial_role;
         rsbac_rc_role_id_t          rc_role;
         rsbac_rc_role_id_t          rc_def_role;
         boolean                     auth_may_setuid;
         boolean                     auth_may_set_cap;
         rsbac_pid_t                 auth_p_capset;
         rsbac_inode_nr_t            auth_f_capset;
         rsbac_cap_vector_t          min_caps;
         rsbac_cap_vector_t          max_caps;
         rsbac_jail_id_t             jail_id;
         rsbac_jail_ip_t             jail_ip;
         rsbac_jail_flags_t          jail_flags;
         rsbac_res_array_t           res_array;
         rsbac_log_array_t           log_array_low;
         rsbac_log_array_t           log_array_high;
         rsbac_request_vector_t      log_program_based;
         rsbac_request_vector_t      log_user_based;
         boolean                     symlink_add_uid;
         boolean                     symlink_add_mac_level;
         boolean                     symlink_add_rc_role;
         rsbac_linux_dac_disable_int_t linux_dac_disable;
//         rsbac_net_temp_id_t         net_temp;
         rsbac_cap_process_hiding_int_t    cap_process_hiding;
#ifdef __KERNEL__
         rsbac_gid_t                 group;        /* process/fd group */
    struct sockaddr                * sockaddr_p; /* socket-ipc address */
         long                        signal;        /* signal for kill */
         int                         mode;    /* mode for create/mount */
         int                         nlink;       /* for DELETE/unlink */
    enum rsbac_switch_target_t       switch_target; /* for SWITCH_MODULE */
         char                      * mod_name;    /* for ADD_TO_KERNEL */
    enum rsbac_adf_request_t         request;        /* for SWITCH_LOG */
         struct ms_segment_t         ms_segment;
         long                        trace_request; /* request for sys_trace */
    struct rsbac_auth_cap_range_t    auth_cap_range;
    	 int                         prot_bits;/* prot bits for mmap()/mprotect() */
         boolean                     internal;
    /* used with CREATE on DIR */
    struct rsbac_create_data_t       create_data;
    /* newly created object in OPEN requests? */
         boolean                     new_object;
         u_int                       rlimit;
         struct dentry             * new_dir_dentry_p;
#endif
         u_char                      u_char_dummy;
         u_short                     u_short_dummy;
         int                         dummy;
         u_int                       u_dummy;
         long                        long_dummy;
         u_long                      u_long_dummy;
       };


/**** ACL ****/

#include <rsbac/acl_types.h>

#endif
