/*******************************************************************************

  Copyright(c) 2002 - 2005 Promise Technology, Inc. All rights reserved.
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Promise Technology, Inc.
  <support@promise.com.tw>	[TAIWAN]
  <support@promise.com>		[U.S.A]
  <support-china@promise.com>	[CHINA]

*******************************************************************************/

/* Copyright (c) 2002 - 2005 Promise Technology, Inc.  All rights reserved.
 *
 * pdc618_mod.c - ULTRA 618/SATA378/SATA150 TX2plus/TX4 Linux driver
 *
 * Usage: insmod pdc-ultra.o [parameter=value]
 * ------
 * parameter:
 *        atapi		1/0, enable atapi deivce, default is 0
 *        debug		0-2, debug level, default is 0
 *
 * History
 * -------
 * 1. 07/09/2002 v1.00.0.1 - Initial release
 * 2. 08/12/2002 v1.00.0.2 - support ATAPI device
 * 3. 08/15/2002 v1.00.0.3 - support sg when using pio mode
 * 4. 08/27/2002 v1.00.0.4 - use CAM v1.00.0.12 and add a module parameter
 * 			     `atapi' (default is disable)
 * 5. 09/04/2002 v1.00.0.5 - use CAM v1.00.0.13. use scsi read_6 and write_6 
 *                           directly instead of converting to 10.
 * 6. 10/14/2002 v1.00.0.6 - use CAM v1.00.0.14. set the packet count of ata 
 *                           module to 1 in order to avoid overrun.
 * 7. 11/20/2002 v1.00.0.7 - use CAM v1.00.0.19. collect expired cam timer.
 * 			     add ATA_STATUS_RETURN in ioctl:0x3803 to update
 * 			     ATA registers when cmd is completed
 * 8. 12/11/2002 v1.00.0.8 - use CAM v1.00.0.20. support SATA 378.
 * 9. 05/22/2003 v1.00.0.9 - use CAM v1.00.0.41. Fix nForce and WD issues.
 * 9. 06/09/2003 v1.00.0.10 - register char device and support some ioctl
 *                            functions (flash memory).
 * 
 * (AK) Some minor changes, September 2003 
 * Copyright 2003 SuSE Labs
 * - Check pci_map_sg return value
 * - Free request in related error path
 * - Add MODULE_DEVICE_TABLE
 * - Fix scanning for multiple cards. 
 * - Fix gapping security holes in ioctl handler
 */
#include <stdarg.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/byteorder.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>
#include <linux/types.h>
#include <linux/hdreg.h>
#include <linux/wrapper.h>
#include <linux/interrupt.h>
#include <linux/reboot.h>
#include <linux/fs.h>
#include <scsi/sg.h>

#include "sd.h"
#include "scsi.h"
#include "hosts.h"

#include <cam_con.h>
#include <cam_def.h>
#include <cam_exp.h>

#include "pdc618_mod.h"

/*
 * function prototype
 */
static void hexdump(u8 *x, int len);
static inline int pdc618_ata_rw(Scsi_Cmnd *SCpnt);
void pdc618_scan_device(unsigned long drv);
static int pdc618_halt(struct notifier_block *nb, unsigned long event, void *buf);
static int pdc618dev_open(struct inode *inodep, struct file *filep);
static int pdc618dev_release(struct inode *inodep, struct file *filep);
static int pdc618dev_ioctl(struct inode *inodep, struct file *filep, unsigned int req, unsigned long arg);

#define minmax(a,b)	((a)<(b) ? (a):(b))

/* linux kernel version related */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#define KERNEL_VERSION_24x
#else
static inline int pci_enable_device(struct pci_dev *dev) {}
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,3)
static inline int pci_set_dma_mask(struct pci_dev *dev, u64 mask) {}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,47)
#define SCSI_DYNAMIC_DMA_MAPPING
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,18)
typedef u32 dma_addr_t;		/* 2.2.18 defines dma_addr_t */
#endif

/*
 * global data structure
 */
/* /proc directory entry for kernel v2.2 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,27)
static struct proc_dir_entry proc_scsi_pdc618 = {
    PROC_SCSI_NOT_PRESENT, 9, PDCNAME,
    S_IFDIR | S_IRUGO | S_IXUGO, 2
};
#endif

/*
 * The file operations structure for the ioctl interface of the driver
 */
static struct file_operations pdc618_fops = {
	open: 		pdc618dev_open,
	release:	pdc618dev_release,
	ioctl:		pdc618dev_ioctl,
};

/* notifier block to get a notifier on system shutdown/halt/poweroff */
static struct notifier_block pdc618_notifier = {
	pdc618_halt, NULL, 0
};

pdc618_adapter_t	pdc618_adapter[MAXADAPTER];
pdc618_channel_t	pdc618_channel[MAXCHANNEL];
pdc618_drive_t		pdc618_drive[MAXDRIVE];
pdc618_timer_t		pdc618_timer[MAXTIMER];
pdc618_req_t		*pdc618_req[QLENGTH+1], *PoolH, *PoolF;

/* user configurable values */
static int atapi = 0;			/* en/disable atapi support */
static int debug = 0;			/* The debug level */
static int nForce_present = 0;		/* Fix nForce chipset */

static DECLARE_WAIT_QUEUE_HEAD(wait);
static int pdc618_unlock;
static int pdc618_num_adapters;		/* total adapters */
static int pdc618_major;		/* char dev major number */

/*
 *  Deal with DMA mapping/unmapping.
 */
#ifndef SCSI_DYNAMIC_DMA_MAPPING

#define __unmap_scsi_data(pdev, cmd)	  do {; } while (0)
#define __map_scsi_single_data(pdev, cmd) (virt_to_bus((cmd)->request_buffer))
#define __map_scsi_sg_data(pdev, cmd)	  ((cmd)->use_sg)
#if 0
#define __sync_scsi_data(pdev, cmd)	  do {; } while (0)
#endif

#define scsi_sg_dma_address(sc)		  virt_to_bus((sc)->address)
#define scsi_sg_dma_len(sc)		  ((sc)->length)

#else

/* To keep track of the dma mapping (sg/single) that has been set */
#define __data_mapped(cmd)	(cmd)->SCp.phase
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,13)
#define __data_mapping(cmd)	(cmd)->SCp.have_data_in
#else
#define __data_mapping(cmd)	(cmd)->SCp.dma_handle
#endif

static void __unmap_scsi_data(struct pci_dev *pdev, Scsi_Cmnd *SCpnt)
{
	int dma_dir = scsi_to_pci_dma_dir(SCpnt->sc_data_direction);

	switch(__data_mapped(SCpnt)) {
	case 2:
		pci_unmap_sg(pdev, SCpnt->buffer, SCpnt->use_sg, dma_dir);
		break;
	case 1:
		pci_unmap_single(pdev, __data_mapping(SCpnt), SCpnt->request_bufflen, dma_dir);
		break;
	}
	__data_mapped(SCpnt) = 0;
}

static dma_addr_t __map_scsi_single_data(struct pci_dev *pdev, Scsi_Cmnd *SCpnt)
{
	dma_addr_t mapping;
	int dma_dir = scsi_to_pci_dma_dir(SCpnt->sc_data_direction);

	if (SCpnt->request_bufflen == 0)
		return 0;

	mapping = pci_map_single(pdev, SCpnt->request_buffer, SCpnt->request_bufflen, dma_dir);
	__data_mapped(SCpnt) = 1;
	__data_mapping(SCpnt) = mapping;

	return mapping;
}

static int __map_scsi_sg_data(struct pci_dev *pdev, Scsi_Cmnd *SCpnt)
{
	int use_sg;
	int dma_dir = scsi_to_pci_dma_dir(SCpnt->sc_data_direction);

	if (SCpnt->use_sg == 0)
		return 0;

	use_sg = pci_map_sg(pdev, SCpnt->buffer, SCpnt->use_sg, dma_dir);
	__data_mapped(SCpnt) = 2;
	__data_mapping(SCpnt) = use_sg;

	return use_sg;
}

#if 0
static void __sync_scsi_data(struct pci_dev *pdev, Scsi_Cmnd *SCpnt)
{
	int dma_dir = scsi_to_pci_dma_dir(SCpnt->sc_data_direction);

	switch(__data_mapped(SCpnt)) {
	case 2:
		pci_dma_sync_sg(pdev, SCpnt->buffer, SCpnt->use_sg, dma_dir);
		break;
	case 1:
		pci_dma_sync_single(pdev, __data_mapping(SCpnt), SCpnt->request_bufflen, dma_dir);
		break;
	}
}
#endif

#define scsi_sg_dma_address(sc)		sg_dma_address((sc))
#define scsi_sg_dma_len(sc)		sg_dma_len((sc))

#endif	/* SCSI_DYNAMIC_DMA_MAPPING */

#define unmap_scsi_data(np, cmd)	__unmap_scsi_data((np)->pci_dev, (cmd))
#define map_scsi_single_data(np, cmd)	__map_scsi_single_data((np)->pci_dev, (cmd))
#define map_scsi_sg_data(np, cmd)	__map_scsi_sg_data((np)->pci_dev, (cmd))
#if 0
#define sync_scsi_data(np, cmd)		__sync_scsi_data((np)->pci_dev, (cmd))
#endif

/*
 * dump hex data
 */
static void hexdump(u8 *x, int len)
{
	int i;

	printk("[ ");
	for (i = 0; i < len; i++)
		printk("%x ", x[i]);
	printk("]\n");
}

/*
 * initial global data structure
 */
int pdc618_init_gds(void)
{
	unsigned char i,j;

	/* initialize pdc618_adapter[] */
	for (i = 0; i < MAXADAPTER; i++) {
		pdc618_adapter_t *pada = &pdc618_adapter[i];
		memset(pada,0,sizeof(pdc618_adapter_t));
		pada->id = i;
		for (j = 0; j < ChPAda; j++) {
			pada->pchannel[j] = &pdc618_channel[i*ChPAda+j];
		}

		spin_lock_init(&pada->pdc618_lock);
		pada->cam_ada = kmalloc(sizeof(ADAPTER_CONFIG_INFO), GFP_ATOMIC);
		memset(pada->cam_ada, 0, sizeof(ADAPTER_CONFIG_INFO));
		pada->cam_info = kmalloc(sizeof(CAM_INFO), GFP_ATOMIC);
		memset(pada->cam_info, 0, sizeof(CAM_INFO));
	}
	/* initialize pdc618_channel[] */
	for (i = 0; i < MAXCHANNEL; i++) {
		pdc618_channel_t *pchn = &pdc618_channel[i];
		memset(pchn,0,sizeof(pdc618_channel_t));
		pchn->id = i;
		pchn->padapter = &pdc618_adapter[i/ChPAda];
		for (j = 0 ; j < DrvPCh; j++) {
			pchn->pdrive[j] = &pdc618_drive[i*DrvPCh+j];
		}
		pchn->cam_chn = kmalloc(sizeof(CHNL_CONFIG), GFP_ATOMIC);
		memset(pchn->cam_chn,0 ,sizeof(CHNL_CONFIG));
	}
	/* initialize pdc618_drive[] */
	for (i = 0; i < MAXDRIVE; i++) {
		pdc618_drive_t *pdrv = &pdc618_drive[i];
		memset(pdrv,0,sizeof(pdc618_drive_t));
		pdc618_drive[i].pchannel = &pdc618_channel[i/DrvPCh];
		pdrv->cam_dev = kmalloc(sizeof(DEV_CONFIG), GFP_ATOMIC);
		memset(pdrv->cam_dev, 0, sizeof(DEV_CONFIG));
	}

	/* initialize pdc618_timer[] */
	for ( i = 0; i < MAXTIMER; i++) {
		pdc618_timer[i].status = camFREE;
	}

	/* initialize submit ata request queue */
	for ( i = 0 ; i < QLENGTH +1; i++ ) {
		pdc618_req[i] = kmalloc(sizeof(pdc618_req_t), GFP_ATOMIC|__GFP_DMA);
		if (!pdc618_req[i]) {
		       	printk(KERN_ERR "%s:[error] can't allocate enough memory for queue\n",PDCNAME);
			break;
		}
		memset(pdc618_req[i],0, sizeof(pdc618_req_t));
		if ( i == 0 )
			pdc618_req[i]->next = NULL;
		else
			pdc618_req[i]->next = pdc618_req[i-1];
	}
	PoolH = pdc618_req[QLENGTH];
	PoolF = pdc618_req[0];

	return(0);
}

/*
 * interrupt handler
 */
void pdc618_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long flags	= 0;
	unsigned long irqhandle = 0;
	pdc618_adapter_t *pada	= (pdc618_adapter_t *) dev_id;
	unsigned char ua;
	spinlock_t *host_lock;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,5,3)
	host_lock = &io_request_lock;
#else
	host_lock = pada->host->host_lock;
#endif

	/* check adapter data */
	if ( (!pada) ) {
		return;
	}


	/* set adapter bit according to irq */
	for (ua = 0; ua < MAXADAPTER; ua++) {
		if(pdc618_adapter[ua].present) {
			if (pdc618_adapter[ua].irq == irq)
				irqhandle |= 1<<ua;

		}
	}

	/* not our irq */
	if (!irqhandle) {
		return;
	}

	/* debug */
	if (debug & LOGATA) {
		printk(KERN_DEBUG "%s:[D] ISR handle=%#lx\n", 
				PDCNAME, irqhandle);
	}

	spin_lock_irqsave(host_lock, flags);
	CAM_ISR(irqhandle);
	spin_unlock_irqrestore(host_lock, flags);
	return;
}

/*
 * initialize ioport and irq
 */
int pdc618_init_ioport(unsigned char ada)
{
	unsigned char	uc;
	pdc618_adapter_t *pada = &pdc618_adapter[ada];
	struct pci_dev	*pdc618_pci_dev = pada->pci_dev;
	unsigned short	command;

	/* read base address and get ioport range */
	for (uc = 0; uc < 6; uc++) {
	       	unsigned int	base = 0 , range = 0;
		pci_read_config_dword(pdc618_pci_dev, PCI_BASE_ADDRESS_0+4*uc, &base);
		pci_write_config_dword(pdc618_pci_dev,PCI_BASE_ADDRESS_0+4*uc,0xffffffff);
		pci_read_config_dword(pdc618_pci_dev, PCI_BASE_ADDRESS_0+4*uc, &range);
		range = ~range + 1;
		pada->range[uc] = range + ((uc < 3) ? 1 : 0);

		pci_write_config_dword(pdc618_pci_dev,PCI_BASE_ADDRESS_0+4*uc,base);
		pada->base[uc] = base & PCI_BASE_ADDRESS_IO_MASK;
		/* printk("base[%d]=%#x\n",uc,pada->base[uc]); */
	}

	/* enable bus master */
	pci_read_config_word(pdc618_pci_dev,PCI_COMMAND,&command);
	command |= (PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY);
	pci_write_config_word(pdc618_pci_dev,PCI_COMMAND,command);
	pci_read_config_dword(pdc618_pci_dev,0x30,&pada->rombase);

	/* register ioport */
	if (!check_region(pada->base[0],pada->range[0])) {
		request_region(pada->base[0],pada->range[0],PDCNAME"(ATA)");
	}
	if (!check_region(pada->base[1],pada->range[1]))
		request_region(pada->base[1],pada->range[1],PDCNAME"(XOR)");
	if (!check_region(pada->base[2],pada->range[2]))
		request_region(pada->base[2],pada->range[2],PDCNAME"(HOST)");

	if (request_irq(pdc618_pci_dev->irq, pdc618_interrupt, SA_INTERRUPT| SA_SHIRQ, PDCNAME,pada)) {
	       	printk(KERN_ERR "%s:[error] adapter%d request irq failed\n",PDCNAME,ada);
		return(2);
	}
	return(0);
}

/*
 * detect , initialize and register PDC618 adapters.
 * return non-zero number on detection.
 */
int pdc618_findcards(void) 
{
	unsigned char	class;
	unsigned char	adapter;
#ifdef	KERNEL_VERSION_24x
	struct pci_dev	*pdc618_pci_dev = NULL ;
#else
	struct pci_dev	*pdc618_pci_dev = pci_devices ;
#endif
	struct pci_dev	*p_dev = NULL ;
	unsigned short	nVidiaVID = 0x10de ;
	unsigned int	nVidiaCLASS = 0x060000 ;

	class = adapter = 0;
	while (PDC618_DEVID[class]) {
		/* search pci bus according to vendorID, deviceID */
		while( (pdc618_pci_dev = pci_find_device(PCI_VENDOR_ID_PROMISE, PDC618_DEVID[class], pdc618_pci_dev)) ) {
			if (pci_enable_device(pdc618_pci_dev))
				continue;

#ifdef SCSI_DYNAMIC_DMA_MAPPING
			/* check dma capable */
			if (pci_set_dma_mask(pdc618_pci_dev, 0xffffffff)) {
			       	printk(KERN_ERR "%s:[error] no suitable DMA available\n",PDCNAME);
				continue;
			}
#endif

			/* class code is 0x0180 or PDC378*/
			if(pdc618_pci_dev->class>>8 == PCI_CLASS_STORAGE_OTHER || PDC618_DEVID[class] == DID378) {
				pdc618_adapter_t  *pada;

				/* pdc378 class:0x0104 */
				/* offset 0x0C  SATA:0x91 FT:0x90 */
				if (PDC618_DEVID[class] == DID378) {
					unsigned char code = 0;
					pci_read_config_byte(pdc618_pci_dev, 0x0c, &code);
					/* SATA is 0x91 */
					if (code != 0x91)
						continue;
				}

				/* over max cards */
				if (adapter >= MAXADAPTER ) {
				       	printk(KERN_WARNING "%s:[warning] max %d adapter(s) support\n",PDCNAME,MAXADAPTER);
					return(adapter);
				}
				pada = &pdc618_adapter[adapter];
				pada->present = 1;
				pada->pci_dev = pdc618_pci_dev;
				pada->deviceid = pdc618_pci_dev->device;
				pada->irq = pdc618_pci_dev->irq;
				adapter++;
			}
		}
		class++;
		pdc618_pci_dev = NULL;
	}
	/* >> find nForce chipset for fix */
	if ( adapter ) {
		while ( (p_dev = pci_find_device(nVidiaVID, PCI_ANY_ID, p_dev) ) ) {
			if ( p_dev->class == nVidiaCLASS )
				nForce_present = 1;
		}
	}
	/* Add by Hank 2003.05.22 << */
	return(adapter);
}

/*
 * start cam request packet
 */
pdc618_req_t *pdc618_request_start(void)
{
       	pdc618_req_t *request, *nrequest;

	request = PoolH;
	nrequest = request->next;
	memset(request, 0, sizeof(pdc618_req_t));
	PoolH = nrequest;

	return request;
}
/*
 * finish cam request packet
 */
void pdc618_request_finish(pdc618_req_t *request)
{
	PoolF->next = request;
	PoolF = request;
	PoolF->next = NULL;
	return;
}
/*
 * step down udma mode 
 */
void pdc618_downmode(unsigned char drv)
{
	pdc618_drive_t *pdrv;
	DEV_CONFIG *pcam_dev; 
	pdrv = &pdc618_drive[drv];
	pcam_dev = pdrv->cam_dev;

	if (pcam_dev->UDMAMode != 0xFF) {
		if (pcam_dev->UDMAMode > 2) {
			unsigned char oldmode = pcam_dev->UDMAMode;
		       	pcam_dev->UDMAMode = (pcam_dev->UDMAMode-1 >= 2) ? (pcam_dev->UDMAMode-1): 2;
		       	if (CAM_ConfigATADevice(pcam_dev)) {
			       	CAM_ResetATAChannel(pcam_dev->bCh);
			       	printk(KERN_INFO "%s:[info] disk%d downmode from %d to %d\n",PDCNAME,drv+1,oldmode,pcam_dev->UDMAMode);
			       	return;
		       	}
		}
	}
	return;
}

/*
 * Function:	pdc618_decide_disposition()
 *
 * Purpose:	decode ATAStatus
 *
 * Parameters:	PATA_TASK	- pointer to ATA_TASK
 *
 * Return:	SCSI error code
 *
 * Notes:	n/a
 */
static inline int pdc618_decide_disposition( PATA_TASK ptask )
{
	pdc618_req_t *req = (pdc618_req_t *) ptask->pTaskExtension;
       	unsigned char status;
	int errcode = 0;

	status = ptask->ATAStatus;
       	if (status & ATA_ERROR) {
	       	unsigned char id = 0;
	       	unsigned char ch = 0;
	       	unsigned long lba = 0;
		pdc618_drive_t *pdrv;
	       	id = ptask->bCh * DrvPCh + ptask->bID;
	       	ch = ptask->bCh;
		/* have to use backup LBAl */
	       	lba = req->cam_task2.Cmd.ATACmd.LBAl;
		pdrv = &pdc618_drive[id];
		switch (status & ~ATA_ERROR) {
	        	case ATA_TIMEOUT: {
		       	    printk(KERN_WARNING "%s:[warning] disk%d ATA timeoutat LBA %#lx \n",PDCNAME,id+1,lba);
			    if (pdrv->cam_dev->DevFlag & DEV_EXIST) {
				    errcode = DID_BUS_BUSY << 16;
			    }
			    else {
				    errcode = DID_TIME_OUT << 16;
			    }
			    break;
		           }
			case ATAERR_OVERRUN: 
		       	    printk(KERN_ERR "%s:[error] disk%d ATA overrun\n",PDCNAME,id+1);
			    errcode = DID_ERROR << 16;
			    break;
			case ATAERR_UNDERRUN:
		       	    printk(KERN_ERR "%s:[error] disk%d ATA underrun\n",PDCNAME,id+1);
			    errcode = DID_ERROR << 16;
			    break;
			case ATAERR_PCI:
		       	    printk(KERN_ERR "%s:[error] PCI error\n",PDCNAME);
			    errcode = DID_ERROR << 16;
			    break;
			case ATAERR_PARITY:
		       	    printk(KERN_ERR "%s:[error] PCI parity check error\n",PDCNAME);
			    errcode = DID_ERROR << 16;
			    break;
			case ATAERR_DRV: {
			    /* ICRC bit on */
		       	    if (ptask->Cmd.ATACmd.bError & 0x80) {
				    printk(KERN_ERR "%s:[error] disk%d crc error at LBA %#lx\n",PDCNAME,id+1,lba);
				    pdc618_downmode(id);
			    }
			    else {
				    printk(KERN_ERR "%s:[error] disk%d error at LBA %#lx cmd=%#x status=%#x error=%#x\n",PDCNAME,id+1,lba,ptask->Cmd.ATACmd.bCmd,ptask->Cmd.ATACmd.bStatus,ptask->Cmd.ATACmd.bError);
			    }
			    errcode = DID_ERROR << 16;
			    break;
			   }
			default: {
			    errcode = DID_ERROR << 16;
			   }

		}
       	}
	else 
		errcode = DID_OK << 16;

	return(errcode);
}

/*
 * submit ata callback function for ioctl
 */
void pdc618_cbk_ioctl ( PATA_TASK ptask )
{
	pdc618_req_t *req = (pdc618_req_t *) ptask->pTaskExtension;
	unsigned char *argbuf = req->cam_task.DataBuffer;
	unsigned char *args = argbuf-8;

	unsigned char status;

	status = ptask->ATAStatus;

	if (status & ATA_ERROR) {
	       	pdc618_decide_disposition(ptask);
       	}

	/* get register status */
	args[0] = ptask->Cmd.ATACmd.bFeature;
	args[1] = ptask->Cmd.ATACmd.bCount;
	args[2] = ptask->Cmd.ATACmd.LBAl & 0xFF;
	args[3] = ptask->Cmd.ATACmd.LBAl>>8 & 0xFF;
	args[4] = ptask->Cmd.ATACmd.LBAl>>16 & 0xFF;
	args[5] = ptask->Cmd.ATACmd.bCmd;
	args[6] = ptask->DataTransferLength >> 9;

	/* copy data to user */
	if (args[6]) {
	       	kfree(argbuf-8);
	}

	pdc618_request_finish(req);

	pdc618_unlock = 1;
	wake_up_interruptible(&wait);

	return;
}

/*
 * scan device
 */
void pdc618_scan_device(unsigned long drv)
{
	pdc618_drive_t *pdrv;
	pdc618_channel_t *pchn;
	DEV_CONFIG *pcam_dev; 
	CHNL_CONFIG *pcam_chn;
	unsigned char bch = (char)drv / DrvPCh;
	unsigned char bid = (char)drv % DrvPCh;

	pchn = &pdc618_channel[bch];
	pcam_chn = pchn->cam_chn;
	pdrv = &pdc618_drive[drv];
	pcam_dev = pdrv->cam_dev;

	if (CAM_EnumATAChannel(pcam_chn) != camSUCCESS)
		return;

	if (CAM_InitATAChannel(bch) != camSUCCESS)
		return;

	/* find device for each channel */
	memset(pcam_dev, 0, sizeof(DEV_CONFIG));
	pcam_dev->bCh = bch;
	pcam_dev->bID = bid;
	CAM_EnumATADevice(pcam_dev);
	if (pcam_dev->DevFlag & DEV_EXIST) {

		/* check atapi is enable or disable */
		if (pcam_dev->DevFlag & DEV_ATAPI_DEVICE)
			if (!atapi)
				return;

		printk(KERN_INFO "%s:[info] Drive%2ld: %s ",PDCNAME, drv+1, pcam_dev->Model);
		if (pcam_dev->MAXLBAL) {
			int sz;

			/* scsi will add 1 in sd.c */
			pcam_dev->MAXLBAL -= 1;
			sz = pcam_dev->MAXLBAL * 2;
			printk("%10lds %6dMB  ", pcam_dev->MAXLBAL, (sz/2 - sz/1250 + 974)/1950);
		}
		else
			printk("%22s"," ");
		if (pcam_dev->UDMAMode != 0xFF)
			printk("UDMA%d\n", pcam_dev->UDMAMode);
		else if (pcam_dev->MDMAMode != 0xFF)
			printk("MDMA%d\n", pcam_dev->MDMAMode);
		else
			printk("PIO%d\n", pcam_dev->PIOMode);
		if (pcam_dev->pEnclosure) {
			CAM_CtrlLED(bch, LED_Green);
		}
	}
	return;
}
/*
 * initialize cam
 */
int pdc618_init_cam(unsigned char ada)
{
	unsigned char i,uc,ud;
	unsigned char order = 0;
	unsigned long cam_size;
	unsigned long *cam_mem;
	pdc618_adapter_t *pada = &pdc618_adapter[ada];
	ADAPTER_CONFIG_INFO *pcam_ada = pada->cam_ada;
	CAM_INFO *pcam_info = pada->cam_info;
       	MODULE_PACKET_COUNT pcnt;

	CAM_GetInfo(pcam_info);
	pcam_ada->Adapter_ID = pada->id;

	/* assign baseaddress */
	for (i = 0; i < 6; i++) {
#ifdef _MMIO_
		if (pcam_info->VirMemBase & bit(i) ) {
			pcam_ada->BaseAddress[i] = (unsigned long)ioremap(pada->base[i], pada->range[i]);
		}
		else 
#endif
			pcam_ada->BaseAddress[i] = (unsigned long)pada->base[i];
	}
	/* cam need physically contiguous memory */
	cam_size = (pcam_info->MinMemSize>>PAGE_SHIFT)+1; /* in pages */
	while ( (1<<order) < cam_size ) order++;
	cam_mem = (unsigned long *) __get_dma_pages(GFP_ATOMIC,order);
	if ( !cam_mem) {
	       	printk(KERN_ERR "%s:[error] can't allocate enough memory for cam\n",PDCNAME);
		return(1);
	}
	memset(cam_mem, 0, (1<<order)<<PAGE_SHIFT);
	pcam_ada->MemPhyAddress = virt_to_bus(cam_mem);
	pcam_ada->MemVirAddr = cam_mem;
	pcam_ada->MemSize = (unsigned long) (1 << order) << PAGE_SHIFT;

	/* init cam */
	if (CAM_Init(pcam_ada) != camSUCCESS)
	       	printk(KERN_ERR "%s:[error] cam init failed\n",PDCNAME);

	/* Fix nForce chipset */
	if ( nForce_present )
		CAM_DisableBMRLongBurst( ada );

	/* scan device */
	for (ud = 0; ud < DrvPCh*ChPAda; ud++) {
		unsigned long drv = ada*DrvPCh*ChPAda + ud;
		pdc618_scan_device(drv);
	}

       	/* get cam packet count */
       	CAM_GetPacketCount(ada, &pcnt);

	/* set cam packet count to 1 to avoid overrun */
       	for (uc = 0; uc < ChPAda; uc++) {
	       	pcnt.ATAModule[uc] = 1;
       	}

       	/* set cam packet count */
       	CAM_SetPacketCount(ada, &pcnt);

	return(0);
}

/*
 * free resource: adapter dependent
 */
void pdc618_cleanup(int ada)
{
	unsigned char i,j;
	pdc618_adapter_t *pada = &pdc618_adapter[ada];
#ifdef _MMIO_
	CAM_INFO *pcam_info = pada->cam_info;
#endif
	if (!pada->present)
	       	return;

	/* free irq */
       	free_irq(pada->pci_dev->irq,pada);

	/* unregister ioport */
       	if (pada->base[0]) {
	       	release_region(pada->base[0],pada->range[0]);
	       	release_region(pada->base[1],pada->range[1]);
	       	release_region(pada->base[2],pada->range[2]);
       	}

	/* iounmap */
       	for (i = 0; i < 6; i++) {
#ifdef _MMIO_
	       	if (pcam_info->VirMemBase & bit(i) ) {
		       	iounmap((void *)pada->cam_ada->BaseAddress[i]);
	       	}
#endif
       	}

	/* release cam memeory */
       	if (pada->cam_ada->MemSize) {
	       	unsigned char order = 0;
	       	while ( (1<<order) < (pada->cam_ada->MemSize >> 12) ) order++;
	       	free_pages((unsigned long)pada->cam_ada->MemVirAddr,order);
       	}

	/* free memory */
	if (pada->cam_ada) {
		kfree(pada->cam_ada);
	}
	if (pada->cam_info) {
	       	kfree(pada->cam_info);
	}
	for (i = 0; i < ChPAda; i++) {
		if (pada->pchannel[i]->cam_chn) {
			kfree(pada->pchannel[i]->cam_chn);
		}
		for (j = 0; j < DrvPCh; j++) {
			if (pada->pchannel[i]->pdrive[j]->cam_dev) {
			       	kfree(pada->pchannel[i]->pdrive[j]->cam_dev);
			}
		}
	}
	return;
}
/*
 * free resource: adapter independent
 */
void pdc618_cleanup2(void)
{
	unsigned char i;

	/* remove timer */
	for ( i = 0; i < MAXTIMER; i++) {
		del_timer(&pdc618_timer[i].timer);
	}

	/* free queue memory */
	for ( i = 0 ; i < QLENGTH +1; i++ ) {
		if (pdc618_req[i]) {
		       	kfree(pdc618_req[i]);
		}
	}

	return;
}

/*
 * system shutdown routine
 */
static int pdc618_halt(struct notifier_block *nb, unsigned long event, void *buf)
{
	switch (event) {
		case SYS_RESTART:
		case SYS_HALT:
		case SYS_POWER_OFF:
			break;
		default:
			return NOTIFY_DONE;
	}

	unregister_reboot_notifier(&pdc618_notifier);

	return NOTIFY_DONE;
}
/* 
 * detect , initialize and register PDC618 adapters.
 * return non-zero number on detection.
 */
int pdc618_detect(Scsi_Host_Template *pdc618_host) 
{
	unsigned char ua;
	unsigned char adapters;
	struct Scsi_Host *shpnt;
	pdc618_adapter_t	*pada;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,27)
       	pdc618_host->proc_dir  = &proc_scsi_pdc618;
#else
       	pdc618_host->proc_name = PDCNAME;
#endif

	printk("PROMISE %s Series Linux Driver v%s\n", PRODUCT, VERSION);

	/* initialize global data structure */
	if(pdc618_init_gds()) {
	       	printk(KERN_ERR "%s:[error] initialize global data structure failed\n",PDCNAME);
		return(0);
	}

	/* find pdc618 */
	if ( !(adapters = pdc618_findcards()) ) {
		printk(KERN_WARNING "%s:[warning] No valid cards found\n", PDCNAME);
		return(0);
	}

	/* register ioport */
	for(ua=0; ua < adapters; ua++) {
		if(!pdc618_adapter[ua].present)
			continue;

	       	if (pdc618_init_ioport(ua)) {
		       	printk(KERN_ERR "%s:[error] adapter%d: init ioport failed\n",PDCNAME,ua);
		       	return(0);
	       	}
	}

	/* initialize cam */
	for (ua = 0; ua < adapters; ua++) {
		if(!pdc618_adapter[ua].present)
			continue;

	       	if (pdc618_init_cam(ua)) {
		       	printk(KERN_ERR "%s:[error] adapter%d: init cam failed\n",PDCNAME,ua);
		       	return(0);
	       	}
	}

	/* register scsi host */
	for (ua = 0; ua < adapters; ua++) {
	       	shpnt = scsi_register(pdc618_host,sizeof(pdc618_adapter_t));
	       	if (shpnt == NULL) {
		       	printk(KERN_ERR "%s:[error] adapter%d: register scsi host failed\n",PDCNAME,ua);
		       	pdc618_cleanup(ua);
		       	return(0);
	       	}
	       	shpnt->max_channel=0;
	       	shpnt->max_id=DrvPCh*ChPAda;
	       	shpnt->max_lun=1;
	       	if (!shpnt->hostdata) {
		       	printk(KERN_ERR "%s:[error] adapter%d: register scsi host failed\n",PDCNAME,ua);
		       	pdc618_cleanup(ua);
		       	return(0);
	       	}
	       	pada = (pdc618_adapter_t *) shpnt->hostdata;
	       	memcpy(pada, &pdc618_adapter[ua], sizeof(pdc618_adapter_t));
		pada->host = shpnt;
       	}

	pdc618_num_adapters = adapters ;

	/* register some device */
	if (adapters) {
		/* register reboot notifier */
		if (register_reboot_notifier(&pdc618_notifier)) {
			printk(KERN_WARNING "%s:[warning] shutdown routine register failed\n",PDCNAME); 
		}

		/* 
		 * register a character device for application to aceess 
		 * driver by ioctls.
		 */
		if ((pdc618_major = register_chrdev(0, "pdc618", 
						&pdc618_fops)) < 0 ) {
			printk(KERN_ERR "%s:[error] register character device failed\n",PDCNAME);
		}

	}
	return(adapters);
}

/*
 * unregister scsi host when unregister scsi module
 */
int pdc618_release(struct Scsi_Host *shpnt)
{
	pdc618_adapter_t *pada =(pdc618_adapter_t *)shpnt->hostdata;
	static int release_count;

	/* close cam */
       	if(pada->present) {
	       	ADAPTER_CONFIG_INFO *pcam_ada = pada->cam_ada;
	       	if (CAM_Close(pcam_ada) != camSUCCESS )
		       	printk(KERN_ERR "%s:[error] adapter%x cam close fail\n",PDCNAME,pada->id);
       	}

	/* free memory, irq, ioport */
	pdc618_cleanup(pada->id);

	if (pdc618_num_adapters == ++release_count) {
		/* free queue, timer */
		pdc618_cleanup2();

		/* unregister chr device */
		unregister_chrdev(pdc618_major, "pdc618");
	}

	/* unregister scsi host */
	scsi_unregister(shpnt);

	/* fake event to notify shutdown */
	pdc618_halt(0, SYS_DOWN, 0);

	return(0);
}

/*
 * Ioctl Interface
 */
int pdc618_ioctl(Scsi_Device *device, int cmd, void *arg)
{
	unsigned char drv;
	unsigned char *args;
	pdc618_adapter_t *pada;
	pdc618_drive_t *pdrv;

	pada =(pdc618_adapter_t *)device->host->hostdata;

	args = (unsigned char *) arg;

	drv = device->id;
	pdrv = &pdc618_drive[drv];

       	if( drv >= DrvPCh*ChPAda ) {
	       	strcpy(args,"ioctl(): Drive not exist.");
	       	return(-1);
       	}
       	if( !(pdrv->cam_dev->DevFlag & DEV_EXIST) ) {
	       	strcpy(args,"ioctl(): Drive not exist.");
	       	return(-1);
       	}

	switch (cmd) {
	        /* RAW ATA CMD ** use carefully ** */
		case 0x3803: {
		     unsigned char *argbuf = args;
		     int argsize = 8;
		     pdc618_req_t *queue;

		     if ( !arg ) {
			     return(-1);
		     }

		     /* support ATA device only */
		     if( !(pdrv->cam_dev->DevFlag & DEV_ATA_DEVICE) ) {
			     strcpy(args,"ioctl(): ATA device only");
			     return(-1);
		     }

		     /* there is data to trasnfer */
		     if (args[6]) {
			     argsize += (512 * args[6]);
			     argbuf = kmalloc(argsize, GFP_ATOMIC);
			     if (argbuf == NULL)
				     return(-1);
			     memset(argbuf, 0, argsize);
			     memcpy(argbuf, args, 8);
		     }

		     /* prepare cam task */
		     queue = pdc618_request_start();
		     queue->cam_task.bCh=drv / DrvPCh;
		     queue->cam_task.bID=drv % DrvPCh;
		     queue->cam_task.ATACmdFlag=ATA_DEVICE+ATA_STATUS_RETURN;
		     if (args[6]) 
			     queue->cam_task.ATACmdFlag+=camDATA_XFER+PIO_XFER+camDATA_IN;
		     queue->cam_task.Cmd.ATACmd.bFeature = args[0];
		     queue->cam_task.Cmd.ATACmd.bCount=args[1];
		     queue->cam_task.Cmd.ATACmd.LBAl = args[2] | args[3] << 8| args[4] << 16;
		     queue->cam_task.Cmd.ATACmd.LBAh = 0;
		     queue->cam_task.Cmd.ATACmd.bCmd = args[5];
		     queue->cam_task.SGCount = 0;
		     queue->cam_task.DataTransferLength = args[6] << 9 ;
		     queue->cam_task.DataBuffer=argbuf+8;
		     queue->cam_task.callback = pdc618_cbk_ioctl;
		     queue->buffer_size = argsize;

		     /* attach queue and backup cam_task */
		     queue->cam_task.pTaskExtension = queue;
		     memcpy(&queue->cam_task2, &queue->cam_task, 
				     sizeof(ATA_TASK));

		     CAM_SubmitATA(&queue->cam_task);

		     pdc618_unlock = 0;
		     wait_event_interruptible(wait, pdc618_unlock);

		     if (args[6]) {
			     memcpy((void *)args, argbuf, argsize);
		     }

		     return(0);
		     }
		case SG_SET_TRANSFORM: {
		     return(0);
		     }
		default: {
		     if (debug & LOGATA) {
		       	 printk(KERN_WARNING "%s:[warning] unspported ioctl command(%#x)\n",PDCNAME,cmd); 
		     }
		     return(-1);
		     }
	}
}

/*
 * copy proc info, called by pdc618_copy_info()
 */
static void pdc618_copy_mem_info(pdc618_info_t *info, char *data, int len)
{
	if (info->position + len > info->length)
		len = info->length - info->position;

	if (info->position + len < info->offset) {
		info->position += len;
		return;
	}
	if (info->position < info->offset) {
		data += (info->offset - info->position);
		len  -= (info->offset - info->position);
	}
	if (len > 0) {
		memcpy(info->buffer + info->position, data, len);
		info->position += len;
	}
}

/*
 * copy proc info, called by pdc618_proc_info()
 */
static int pdc618_copy_info(pdc618_info_t *info, char *fmt, ...)
{
	va_list args;
	char buf[81];
	int len;

	va_start(args, fmt);
	len = vsprintf(buf, fmt, args);
	va_end(args);
	pdc618_copy_mem_info(info, buf, len);
	return len;
}

/*
 * proc info
 */
int pdc618_proc_info(char *buffer, char **start, off_t offset, int length, int hostno, int func)
{
	pdc618_info_t info;
	unsigned char i;
	unsigned char swap = 0;
	unsigned char disk = 0;

	info.buffer = buffer;
	info.length = length;
	info.offset = offset;
	info.position = 0;

	if (start)
		*start = buffer;

	pdc618_copy_info(&info, "PROMISE %s Series Linux Driver v%s\n", PRODUCT, VERSION);

	/* get IC status */
	for (i = 0; i < MAXADAPTER; i++) {
		unsigned short id;
		if ( pdc618_adapter[i].present != 1 )
			continue;
		id = pdc618_adapter[i].deviceid;
		if ( id ) {
			pdc618_copy_info(&info, "Adapter%d - ", i+1);
			switch (id) {
				case DID318:
					pdc618_copy_info(&info, "SATA150 TX4  ");
					break;
				case DID375:
					pdc618_copy_info(&info, "SATA150 TX2plus  ");
					break;
				case DID378:
					pdc618_copy_info(&info, "SATA 378  ");
					break;
				case DID618:
					pdc618_copy_info(&info, "Ultra 618  ");
					break;
				default:
					pdc618_copy_info(&info, "SATA150 Series  ");
					break;
			}
			pdc618_copy_info(&info, "[ ");
			pdc618_copy_info(&info, "IRQ%2d ", pdc618_adapter[i].irq);
			pdc618_copy_info(&info, "]\n");
		}
	}

	/* get hard disk status */
	for (i = 0; i < MAXDRIVE; i++) {
		pdc618_drive_t *pdrv = &pdc618_drive[i];
		DEV_CONFIG *pcam_dev = pdrv->cam_dev;

		/* check drive exist or not */
		if (!(pcam_dev->DevFlag & DEV_EXIST))
			continue;

		/* check atapi is enable or disable */
	       	if (pcam_dev->DevFlag & DEV_ATAPI_DEVICE)
		       	if (!atapi)
			       	continue;

		if ( ++disk == 1 )
		       	pdc618_copy_info(&info, "Drive    - \n");

		pdc618_copy_info(&info, "  %2d : ", i+1);
		pdc618_copy_info(&info, "%s ", pcam_dev->Model);
		pdc618_copy_info(&info, "Channel%d/", pcam_dev->bCh%ChPAda+1);
		pdc618_copy_info(&info, "%s ", (i%2)?"Slave ":"Master");
	       	if (pcam_dev->MAXLBAL) {
			int sz = pcam_dev->MAXLBAL * 2;
		       	pdc618_copy_info(&info, "%6ldMB  ", (sz/2 - sz/1250 + 974)/1950);
		}
		else {
		       	pdc618_copy_info(&info, "%10s", " ");
		}
		if (pcam_dev->UDMAMode != 0xFF)
		       	pdc618_copy_info(&info, "UDMA%d\n", pcam_dev->UDMAMode);
		else if (pcam_dev->MDMAMode != 0xFF)
		       	pdc618_copy_info(&info, "MDMA%d\n", pcam_dev->MDMAMode);
		else
		       	pdc618_copy_info(&info, "PIO%d\n", pcam_dev->PIOMode);
	}

	/* get enclosure status */
	for (i = 0; i < MAXCHANNEL; i++) {
	       	ENCLOSURE_TYPE type;
	       	ENCLOSURE_STATUS box;
	       	unsigned char boxid;
	        
		if ( !pdc618_channel[i].padapter->present )
			continue;

		if ( (boxid=CAM_EnumEnclosure(i, &type)) == 0xFF )
			continue;

		if (CAM_Get_Enclosure_Status(boxid, &box) != camSUCCESS)
		       	continue;

		if ( ++swap == 1 )
		       	pdc618_copy_info(&info, "Swapbox  - \n");

		pdc618_copy_info(&info,"  %2d : ", boxid+1);
	       	switch (type.BOXType) {
		       	case 0x01:
			       	pdc618_copy_info(&info,"SuperSwapBox  - ");
			       	break;
		       	case 0x00:
			       	pdc618_copy_info(&info,"SwapBox  - ");
			       	break;
	       	}
	       	pdc618_copy_info(&info,"RPM(%4ld) ",box.FANStatus);
	       	pdc618_copy_info(&info,"TEMP(%d.",box.TEMPStatus/2);
	       	pdc618_copy_info(&info,"%d%cC) ",(box.TEMPStatus&1)*5,0xB0);
	       	pdc618_copy_info(&info,"V5(%ld.",box.VOL5V/1000);
	       	pdc618_copy_info(&info,"%03ldV) ",box.VOL5V%1000);
	       	pdc618_copy_info(&info,"V12(%ld.",box.VOL12V/1000);
	       	pdc618_copy_info(&info,"%03ldV) ",box.VOL12V%1000);
	       	pdc618_copy_info(&info,"\n");
       	}

	if (info.position > info.offset)
		return (info.position - info.offset);
	else
		return 0;
}

/*
 * transform SCSI CDB to ATAPI one
 */
static inline void pdc618_scsi2atapi(pdc618_req_t *req)
{
	Scsi_Cmnd *SCpnt = req->cmd;
	unsigned char *cmd = req->cam_task.Cmd.Cdb;
	unsigned char *scsi_buf = SCpnt->request_buffer;
	unsigned char *atapi_buf;

	switch (cmd[0]) {
		case READ_6:
		case WRITE_6:
			cmd[8] = cmd[4];
			cmd[5] = cmd[3];
			cmd[4] = cmd[2];
			cmd[3] = cmd[1] & 0x1F;
			cmd[2] = 0;
			cmd[1] &= 0xE0;
			cmd[0] += (READ_10 - READ_6);
			break;
		case MODE_SELECT:
		case MODE_SENSE:
			if ((atapi_buf = kmalloc(req->buffer_size + 4, GFP_ATOMIC)) == NULL )
				return;
			memset(atapi_buf, 0, req->buffer_size + 4);
			cmd[9] = cmd[5];
			cmd[8] = cmd[4] + 4;
			cmd[0] |= 0x40;
			if ( cmd[4] + 4 > 255)
				cmd[7] = cmd[4] + 4 - 255;
			cmd[4] = 0;
			if (cmd[0] == MODE_SELECT_10) {
				atapi_buf[1] = scsi_buf[0];
				atapi_buf[2] = scsi_buf[1];
				atapi_buf[3] = scsi_buf[2];
				atapi_buf[7] = scsi_buf[3];
				memcpy(atapi_buf + 8, scsi_buf + 4, req->buffer_size - 4);
			}
			req->buffer = atapi_buf;
			req->buffer_size += 4;
			break;
		case 0x5b:	/* CLOSE TRACK/RZONE/SESSION/BORDER */
			cmd[1] |= 0x1;		/* set Immed bit */
			break;
		case 0xa1:	/* BLANK */
			cmd[1] |= 0x10;		/* set Immed bit */
			break;
		case 0xa6:	/* LOAD/UNLOAD MEDIUM */
			cmd[1] |= 0x1;		/* set Immed bit */
			break;
		case START_STOP: /* 0x1b */
			cmd[1] |= 0x1;		/* set Immed bit */
			break;
		case SYNCHRONIZE_CACHE:	/* 0x35 */
			cmd[1] |= 0x2;		/* set Immed bit */
			break;
		default:
			break;
	}
}
/*
 * transform ATAPI CDB to SCSI one
 */
static inline void pdc618_atapi2scsi(pdc618_req_t *req)
{
	Scsi_Cmnd *SCpnt = req->cmd;
	unsigned char *cmd = SCpnt->cmnd;
	unsigned char *scsi_buf = SCpnt->request_buffer;
	unsigned char *atapi_buf = req->buffer;

	switch (cmd[0]) {
		case INQUIRY:
			scsi_buf[2] |= 2;
			scsi_buf[3] = (scsi_buf[3] & 0xF0) | 2;
			break;
		case MODE_SENSE:
			if (req->cam_task.Cmd.Cdb[0] != MODE_SENSE_10) {
			       	if (atapi_buf && atapi_buf != scsi_buf)
				       	kfree(atapi_buf);
			       	break;
			}
			scsi_buf[0] = atapi_buf[1];
			scsi_buf[1] = atapi_buf[2];
			scsi_buf[2] = atapi_buf[3];
			scsi_buf[3] = atapi_buf[7];
			memcpy(scsi_buf + 4, atapi_buf + 8, req->buffer_size - 8);
			if (atapi_buf && atapi_buf != scsi_buf)
				kfree(atapi_buf);
			break;
		default:
			break;
	}
}
/*
 * Function:	pdc618_cbk_ata_rw()
 *
 * Purpose:	process cam task depend on ATAStatus
 *
 * Parameters:	PATA_TASK	- pointer to ATA_TASK
 *
 * Return:	nothing
 *
 * Notes:	n/a
 */
void pdc618_cbk_ata_rw ( PATA_TASK ptask )
{
	pdc618_req_t *req = (pdc618_req_t *) ptask->pTaskExtension;
	Scsi_Cmnd *SCpnt = req->cmd;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;
	pdc618_drive_t *pdrv;
	int result = DID_OK << 16;
	int id;

	id = ptask->bCh * DrvPCh + ptask->bID;
	pdrv = &pdc618_drive[id];

	/* debug */
	if (debug & LOGATA) {
		printk(KERN_DEBUG "%s:[D] task done  cmd=%#x lba=%#lx cnt=%#x status=%#x e=%#x s=%#x\n", 
				PDCNAME,
				req->cam_task2.Cmd.ATACmd.bCmd,
				req->cam_task2.Cmd.ATACmd.LBAl,
				req->cam_task2.Cmd.ATACmd.bCount,
				req->cam_task.ATAStatus,
				req->cam_task.Cmd.ATACmd.bError,
				req->cam_task.Cmd.ATACmd.bStatus);
	}

	unmap_scsi_data(pada, SCpnt);

	if (ptask->ATACmdFlag & camDATA_XFER) {
	       	result = pdc618_decide_disposition(ptask);
	}

	/* if error then retry 5 times */
	if ((result >> 16) != DID_OK) {
		if (++pdrv->retries < 5) {
			pdc618_request_finish(req);
			pdc618_ata_rw(SCpnt);
			return;
		}
		/* retry too many times, abort it */
		else {
			if (CAM_ResetATAChannel(ptask->bCh) != camSUCCESS) {
				printk(KERN_ERR "%s:[error] reset channel %d failed\n",PDCNAME, ptask->bCh);
			}

			result = DID_ABORT<<16;
		}
	}

	pdrv->retries = 0;
	SCpnt->result = result;
	SCpnt->scsi_done(SCpnt);
	pdc618_request_finish(req);

	return;
}
/*
 * SubmitATA callback for pdc618_atapi_dma
 */
void pdc618_cbk_atapi_dma( PATA_TASK ptask )
{
	pdc618_req_t *req = (pdc618_req_t *) ptask->pTaskExtension;
	Scsi_Cmnd *SCpnt = req->cmd;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;

	/* unmap DMA mapping */
	if (SCpnt->use_sg)
	       	unmap_scsi_data(pada, SCpnt);
	else {
	       	int dma_dir; 
		if (__data_mapped(SCpnt) == 1) {
		       	dma_dir = scsi_to_pci_dma_dir(SCpnt->sc_data_direction);
		       	pci_unmap_single(pada->pci_dev, __data_mapping(SCpnt), req->buffer_size, dma_dir);
		       	__data_mapped(SCpnt) = 0;
		}
	}

	pdc618_atapi2scsi(req);
	if (debug & LOGATAPI) {
	       	printk("    status=%2x   ",ptask->ATAStatus);
	       	if (SCpnt->request_bufflen) {
		       	printk("rst=");
		       	hexdump(SCpnt->request_buffer, minmax(16, SCpnt->request_bufflen));
	       	}
	       	else
		       	printk("\n");
       	}

	/* check ATAStatus */
	if (ptask->ATAStatus & ATA_ERROR) {
	       	if (ptask->ATAStatus & ATA_TIMEOUT)
		       	SCpnt->result = (DID_BUS_BUSY << 16);
	       	else if (ptask->ATAStatus & ATAERR_DRV) {
		       	SCpnt->result = (CHECK_CONDITION << 1) | (DID_OK << 16);
			if (debug & LOGATAPI) {
			       	printk("              sense=");
			       	hexdump(SCpnt->sense_buffer, 14);
			}
	       	}
	}
	else
	       	SCpnt->result = (DID_OK << 16);

	SCpnt->scsi_done(SCpnt);
	pdc618_request_finish(req);

	return;
}
/*
 * SubmitATA callback for pdc618_atapi_pio
 */
void pdc618_cbk_atapi_pio( PATA_TASK ptask )
{
	pdc618_req_t *req = (pdc618_req_t *) ptask->pTaskExtension;
	Scsi_Cmnd *SCpnt = req->cmd;

	pdc618_atapi2scsi(req);

	if (debug & LOGATAPI) {
	printk("    status=%2x   ",ptask->ATAStatus);
	printk("rst=");
       	hexdump(req->buffer, minmax(16, SCpnt->request_bufflen));
	}

       	if (SCpnt->use_sg) {
	       	if (SCpnt->sc_data_direction != SCSI_DATA_WRITE) {
		       	unsigned char *s, *t;
		       	struct scatterlist *sglist;
		       	int i, sgcnt, addr=0;

			sglist = (struct scatterlist *)SCpnt->request_buffer;
		       	sgcnt = SCpnt->use_sg;
		       	for (i = 0; i < sgcnt; i++) {
			       	unsigned int len = scsi_sg_dma_len(sglist+i);
			       	s = &req->buffer[addr];
			       	t = bus_to_virt(scsi_sg_dma_address(sglist+i));
			       	memcpy(t, s, len);
			       	addr += len;
		       	}
		}
		kfree(req->buffer);
	}

	if (ptask->ATAStatus & ATA_ERROR) {
	       	if (ptask->ATAStatus & ATA_TIMEOUT)
		       	SCpnt->result = (DID_BUS_BUSY << 16);
	       	else if (ptask->ATAStatus & ATAERR_DRV) {
		       	SCpnt->result = (CHECK_CONDITION << 1) | (DID_OK << 16);
			if (debug & LOGATAPI) {
			       	printk("              sense=");
			       	hexdump(SCpnt->sense_buffer, 14);
			}
	       	}
	}
	else
	       	SCpnt->result = (DID_OK << 16);

	SCpnt->scsi_done(SCpnt);
	pdc618_request_finish(req);
}
/*
 * Function:	pdc618_ata_rw()
 *
 * Purpose:	process scsi R/W commands and submit it to CAM
 *
 * Parameters:	SCpnt	- scsi command structure
 *
 * Return:	0 is success, 1 is fail
 *
 * Notes:	n/a
 */
static inline int pdc618_ata_rw(Scsi_Cmnd *SCpnt)
{
	unsigned char id;
	pdc618_req_t *queue;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;
	unsigned char *cmd = (unsigned char *) SCpnt->cmnd;

	queue = pdc618_request_start();

	id = pada->id*ChPAda*DrvPCh + SCpnt->target;

	/* attach SCpnt for DMA mapping */
	queue->cmd = SCpnt;
	queue->cam_task.bCh = id / DrvPCh;
	queue->cam_task.bID = id % DrvPCh;

       	queue->cam_task.ATACmdFlag = ATA_DEVICE+camDATA_XFER+PSEUDO_CMD;

	/* assign bCount and LBA depend on scsi cmd */
	switch (cmd[0]) {
		case READ_6:
		       	queue->cam_task.ATACmdFlag += camDATA_IN;
		       	queue->cam_task.Cmd.ATACmd.bCmd = IDE_COMMAND_READ_DMA;
		       	queue->cam_task.Cmd.ATACmd.bCount = cmd[4];
		       	queue->cam_task.Cmd.ATACmd.LBAl = ( ((cmd[1]&0x1F) << 16) | (cmd[2] << 8) | cmd[3]);
			break;
		case READ_10:
		       	queue->cam_task.ATACmdFlag += camDATA_IN;
		       	queue->cam_task.Cmd.ATACmd.bCmd = IDE_COMMAND_READ_DMA;
		       	queue->cam_task.Cmd.ATACmd.bCount = ( (cmd[7] << 8) | cmd[8]);
		       	queue->cam_task.Cmd.ATACmd.LBAl = ( (cmd[2] << 24) | (cmd[3] << 16) | (cmd[4] << 8) | cmd[5]);
			break;
		case WRITE_6:
		       	queue->cam_task.Cmd.ATACmd.bCmd = IDE_COMMAND_WRITE_DMA;
		       	queue->cam_task.Cmd.ATACmd.bCount = cmd[4];
		       	queue->cam_task.Cmd.ATACmd.LBAl = ( ((cmd[1]&0x1F) << 16) | (cmd[2] << 8) | cmd[3]);
			break;
		case WRITE_10:
		       	queue->cam_task.Cmd.ATACmd.bCmd = IDE_COMMAND_WRITE_DMA;
		       	queue->cam_task.Cmd.ATACmd.bCount = ( (cmd[7] << 8) | cmd[8]);
		       	queue->cam_task.Cmd.ATACmd.LBAl = ( (cmd[2] << 24) | (cmd[3] << 16) | (cmd[4] << 8) | cmd[5]);
			break;
	}

	queue->cam_task.Cmd.ATACmd.LBAh = 0;
	queue->cam_task.callback = pdc618_cbk_ata_rw;

	/* there is one sg table, request_buffer point to sg list */
	if (SCpnt->use_sg) {
		struct scatterlist *sglist;
		int i, sgcnt;

		sglist = (struct scatterlist *)SCpnt->request_buffer;
		sgcnt = map_scsi_sg_data(pada, SCpnt);
		if (sgcnt == 0) {
			pdc618_request_finish(queue); 
			return 1;
		}
		for (i = 0; i < sgcnt; i++) {
			unsigned int len = scsi_sg_dma_len(sglist+i);
			queue->cam_sg[i].StartAddr = cpu_to_le32(scsi_sg_dma_address(sglist+i));
			queue->cam_sg[i].wCount = cpu_to_le32(len);
		}
		queue->cam_sg[sgcnt-1].CtrlFlag = EOT;
		queue->cam_task.SGCount = sgcnt;
		queue->cam_task.pSG = (unsigned long)((PCAMSG)&queue->cam_sg[0]);
	}
	/* request_buffer is a virtual addr point to real sg address , 
	 * convert it into our sg table
	 */
	else {
		dma_addr_t address;
		PCAMSG psg;
		address = map_scsi_single_data(pada, SCpnt);
		psg = &queue->cam_sg[0];
		psg->StartAddr = cpu_to_le32(address);
		psg->wCount = cpu_to_le32(SCpnt->request_bufflen);
		psg->Reserved = 0;
		psg->CtrlFlag = EOT;
		queue->cam_task.SGCount = 1;
		queue->cam_task.pSG = (unsigned long)((PCAMSG)psg);
	}

	/* attach queue structure */
	queue->cam_task.pTaskExtension = queue;

	/* backup cam_task */
	memcpy(&queue->cam_task2, &queue->cam_task, sizeof(ATA_TASK));

	/* debug */
	if (debug & LOGATA) {
		printk(KERN_DEBUG "%s:[D] task start cmd=%#x lba=%#lx cnt=%#x\n", 
				PDCNAME,
				queue->cam_task.Cmd.ATACmd.bCmd,
				queue->cam_task.Cmd.ATACmd.LBAl,
				queue->cam_task.Cmd.ATACmd.bCount
				);
	}

	if (CAM_SubmitATA(&queue->cam_task) != camACCEPTED) {
		pdc618_request_finish(queue); 
	       	printk(KERN_WARNING "%s:[warning] submit cam busy\n",PDCNAME);
		return(1);
	}
	return(0);
}

/*
 * ATAPI CDB DMA mode
 */
static inline int pdc618_atapi_dma(Scsi_Cmnd *SCpnt)
{
	unsigned char *buff = (unsigned char *) SCpnt->request_buffer;
	unsigned char id;
	int bufflen = SCpnt->request_bufflen;
       	PCAMSG psg;
	pdc618_req_t *queue;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;

	id = pada->id*ChPAda*DrvPCh + SCpnt->target;

	queue = pdc618_request_start();
       	queue->cmd = SCpnt;
       	queue->buffer=buff;
       	queue->buffer_size=bufflen;
       	queue->cam_task.bCh=id / DrvPCh;
       	queue->cam_task.bID=id % DrvPCh;
       	queue->cam_task.ATACmdFlag=0;
       	memcpy(queue->cam_task.Cmd.Cdb, SCpnt->cmnd, SCpnt->cmd_len);
       	pdc618_scsi2atapi(queue);
       	if (SCpnt->use_sg) {
	       	struct scatterlist *sglist;
	       	int i, sgcnt;

		sglist = (struct scatterlist *)SCpnt->request_buffer;
	       	sgcnt = map_scsi_sg_data(pada, SCpnt);
		for (i = 0; i < sgcnt; i++) {
		       	unsigned int len = scsi_sg_dma_len(sglist+i);
			queue->cam_sg[i].StartAddr = cpu_to_le32(scsi_sg_dma_address(sglist+i));
			queue->cam_sg[i].wCount = cpu_to_le32(len);
	       	}
	       	queue->cam_sg[sgcnt-1].CtrlFlag = EOT;
	       	queue->cam_task.SGCount = sgcnt;
	       	queue->cam_task.pSG = (unsigned long)((PCAMSG)&queue->cam_sg[0]);
       	}
       	else { /* use_sg = 0 */
	       	if (queue->buffer_size) {
		       	dma_addr_t address;
		       	int dma_dir = scsi_to_pci_dma_dir(SCpnt->sc_data_direction);
		       	address = pci_map_single(pada->pci_dev, queue->buffer, queue->buffer_size, dma_dir);
		       	__data_mapped(SCpnt) = 1;
		       	__data_mapping(SCpnt) = address;

			psg = &queue->cam_sg[0];
		       	psg->StartAddr = cpu_to_le32(address);
		       	psg->wCount = cpu_to_le32(queue->buffer_size);
		       	psg->Reserved = 0;
		       	psg->CtrlFlag = EOT;
		       	queue->cam_task.SGCount = 1;
		       	queue->cam_task.pSG = (unsigned long)((PCAMSG)psg);
	       	}
       	}
       	if (queue->cam_task.SGCount) {
	       	queue->cam_task.ATACmdFlag+=camDATA_XFER;
	       	if (SCpnt->sc_data_direction != SCSI_DATA_WRITE)
		       	queue->cam_task.ATACmdFlag+=camDATA_IN;
       	}
       	queue->cam_task.callback = pdc618_cbk_atapi_dma;
       	queue->cam_task.SenseInfoLength = 14;
       	queue->cam_task.SenseInfoBuffer = SCpnt->sense_buffer;

	/* attach queue structure */
	queue->cam_task.pTaskExtension = queue;

	/* backup cam_task */
	memcpy(&queue->cam_task2, &queue->cam_task, sizeof(ATA_TASK));

	if (CAM_SubmitATA(&queue->cam_task) != camACCEPTED) {
	       	printk(KERN_WARNING "%s:[warning] submit cam busy\n",PDCNAME);
		return(1);
	}

	if (debug & LOGATAPI) {
	       	printk("dma: cmd=%x flag=%x len=%x sg=%x dir=%x\n",SCpnt->cmnd[0],queue->cam_task.ATACmdFlag,bufflen,SCpnt->use_sg,SCpnt->sc_data_direction);
	       	printk("pdc-ultra: sg=%d cmd=",SCpnt->use_sg);
	       	hexdump(queue->cam_task.Cmd.Cdb, SCpnt->cmd_len);
	}
	return(0);
}

/*
 * ATAPI CDB PIO mode
 */
static inline int pdc618_atapi_pio(Scsi_Cmnd *SCpnt)
{
	unsigned char *buff = (unsigned char *) SCpnt->request_buffer;
	unsigned char id;
	int bufflen = SCpnt->request_bufflen;
	pdc618_req_t *queue;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;

	id = pada->id*ChPAda*DrvPCh + SCpnt->target;

       	queue = pdc618_request_start();
       	queue->cmd = SCpnt;
       	queue->buffer=buff;
       	queue->buffer_size=bufflen;
       	queue->cam_task.bCh=id / DrvPCh;
       	queue->cam_task.bID=id % DrvPCh;
       	queue->cam_task.ATACmdFlag=0;
       	memcpy(queue->cam_task.Cmd.Cdb, SCpnt->cmnd, SCpnt->cmd_len);
       	pdc618_scsi2atapi(queue);
       	if (SCpnt->use_sg) {
	       	if ((queue->buffer = kmalloc(bufflen, GFP_ATOMIC|__GFP_DMA)) == NULL ) {
		       	printk(KERN_ERR "%s:[error] can't allocate enough memory for sg\n",PDCNAME);
		       	pdc618_request_finish(queue);
			return(1);
	       	}
	       	memset(queue->buffer, 0, bufflen);

	       	if (SCpnt->sc_data_direction == SCSI_DATA_WRITE) {
		       	unsigned char *s, *t;
		       	struct scatterlist *sglist;
		       	int i, sgcnt, addr=0;

			sglist = (struct scatterlist *)SCpnt->request_buffer;
		       	sgcnt = SCpnt->use_sg;
		       	for (i = 0; i < sgcnt; i++) {
			       	unsigned int len = scsi_sg_dma_len(sglist+i);
			       	s = bus_to_virt(scsi_sg_dma_address(sglist+i));
			       	t = &queue->buffer[addr];
			       	memcpy(t, s, len);
			       	addr += len;
		       	}
	       	}
	}
       	if (queue->buffer_size) {
	       	queue->cam_task.ATACmdFlag+=PIO_XFER;
	       	queue->cam_task.ATACmdFlag+=camDATA_XFER;
	       	if (SCpnt->sc_data_direction != SCSI_DATA_WRITE)
		       	queue->cam_task.ATACmdFlag+=camDATA_IN;
       	}
       	queue->cam_task.SGCount = 0;
       	queue->cam_task.DataBuffer=queue->buffer;
       	queue->cam_task.DataTransferLength=queue->buffer_size;
       	queue->cam_task.SenseInfoLength = 14;
       	queue->cam_task.SenseInfoBuffer = SCpnt->sense_buffer;
       	queue->cam_task.callback=pdc618_cbk_atapi_pio;

	/* attach queue structure */
	queue->cam_task.pTaskExtension = queue;

	/* backup cam_task */
	memcpy(&queue->cam_task2, &queue->cam_task, sizeof(ATA_TASK));

	if (CAM_SubmitATA(&queue->cam_task) != camACCEPTED) {
	       	printk(KERN_WARNING "%s:[warning] submit cam busy\n",PDCNAME);
		return(1);
	}

	if (debug & LOGATAPI) {
	       	printk("pio: cmd=%x flag=%x len=%x sg=%x dir=%x\n",SCpnt->cmnd[0],queue->cam_task.ATACmdFlag,bufflen,SCpnt->use_sg,SCpnt->sc_data_direction);
	       	printk("pdc-ultra: sg=%d cmd=",SCpnt->use_sg);
	       	hexdump(SCpnt->cmnd, SCpnt->cmd_len);
	}
	return(0);
}
/*
 * Function:	pdc618_queuecommand()
 *
 * Purpose:	scsi queue function to handle scsi commands
 *
 * Parameters:	SCpnt	- scsi command structure
 * 		done	- scsi done callback 
 *
 * Return:	only when use_new_eh_code = 1, return value is meaningful
 *
 * Notes:	n/a
 */
int pdc618_queuecommand(Scsi_Cmnd *SCpnt, void (*done)(Scsi_Cmnd *))
{
	unsigned char *cmd = (unsigned char *) SCpnt->cmnd;
	unsigned char *buff = (unsigned char *) SCpnt->request_buffer;
	unsigned char drv = 0;
	unsigned long lock = 0;
	unsigned long capacity;
	int bufflen = SCpnt->request_bufflen;
	int error = 0;
	pdc618_adapter_t *pada;
	pdc618_drive_t *pdrv;
	DEV_CONFIG *pcam_dev;

	__data_mapped(SCpnt) = 0;
	__data_mapping(SCpnt) = 0;

	if ( debug ) {
		printk(KERN_DEBUG "%s:[D] Qcmd t=%d cmd=",
				PDCNAME, SCpnt->target);
		hexdump(SCpnt->cmnd, SCpnt->cmd_len);
	}

	pada = (pdc618_adapter_t *) SCpnt->host->hostdata;
	if (pada == NULL ) {
	       	printk(KERN_ERR "%s:[error] invalid device in queue cmd\n",PDCNAME);
		SCpnt->result = (DID_ERROR << 16);
		done(SCpnt);
		return(0);
	}

	spin_lock_irqsave(&pada->pdc618_lock, lock);
	drv = pada->id*ChPAda*DrvPCh + SCpnt->target;
	pdrv = &pdc618_drive[drv];
	pcam_dev = pdrv->cam_dev;
	
	/* drive not exist */
	if (!(pcam_dev->DevFlag & DEV_EXIST)) {
	       	SCpnt->result = DID_BAD_TARGET << 16 ;
		done(SCpnt);
		spin_unlock_irqrestore(&pada->pdc618_lock, lock);
		return(0);
	}

	/* check atapi is enable or disable */
       	if (pcam_dev->DevFlag & DEV_ATAPI_DEVICE) {
	       	if (!atapi) {
		       	SCpnt->result = DID_BAD_TARGET << 16 ;
		       	done(SCpnt);
		       	spin_unlock_irqrestore(&pada->pdc618_lock, lock);
		       	return(0);
		}
	}

	SCpnt->scsi_done = done;
	
	/* ATAPI device */
	if (pcam_dev->DevFlag & DEV_ATAPI_DEVICE) {
		unsigned char pio = 1;

		/* use dma to do data transfer */
		switch (cmd[0]) {
			case WRITE_12:
			case WRITE_10:
			case WRITE_6:
			case READ_12:
			case READ_10:
			case READ_6:
			case 0xad:
			case 0xbe:
				pio = 0; break;
		}

		/* -45150 (FFFF4FA2h) to -1 (FFFFFFFFh) shall use PIO mode */
		if (cmd[0] == WRITE_10) {
			unsigned long lba;
			lba = cmd[5] | cmd[4] << 8 | cmd[3] << 16| cmd[2] << 24;
			if (lba >= 0xFFFF4FA2 && lba <= 0xFFFFFFFF) {
				pio = 1;
			}
		}

		/* device support only pio mode */
	       	if (pcam_dev->UDMAMode == 0xFF && pcam_dev->MDMAMode == 0xFF)
			pio = 1;

		/* atapi dma mode */
		if (!pio) {
			error = pdc618_atapi_dma(SCpnt);
		}
		else { /* atapi pio mode */
		       	error = pdc618_atapi_pio(SCpnt);
		}

	}
	else { /* ATA device */
	switch (cmd[0]) {
		case INQUIRY:
			{
			memset(buff,0 , bufflen);
			buff[0] = TYPE_DISK;	/* device type */
			buff[1] = 0;		/* RMB and modifier (SCSI-1) */
			buff[2] = 2;		/* claim SCSI 2 */
			buff[4] = 31;		/* Additional len (n-4) */
			memcpy(&buff[16], pcam_dev->Model, 16);
			memcpy(&buff[32], "    ", 4);
			SCpnt->result = (DID_OK << 16);
			done(SCpnt);
			}
			break;
		case TEST_UNIT_READY:
			SCpnt->result = (DID_OK << 16);
			done(SCpnt);
			break;
		case READ_CAPACITY:
			{
			capacity = pcam_dev->MAXLBAL;
			memset(buff,0 , bufflen);
			/* Number of LBA's */
			buff[0] = (unsigned char) (capacity >> 24);
			buff[1] = (unsigned char) (capacity >> 16) & 0xff;
			buff[2] = (unsigned char) (capacity >> 8) & 0xff;
			buff[3] = (unsigned char) capacity & 0xff;
			/* Block size in bytes (512) */
			buff[4] = 0;
			buff[5] = 0;
			buff[6] = 0x02;
			buff[7] = 0;
			}
			SCpnt->result = (DID_OK << 16);
			SCpnt->scsi_done = done;
			done(SCpnt);
			break;
		case READ_6:
		case WRITE_6:
		case READ_10:
		case WRITE_10:
			error = pdc618_ata_rw(SCpnt);
			break;
		default:
			if (debug & LOGATA) {
				printk(KERN_WARNING "%s:[warning] unknown scsi command: %#x\n",PDCNAME, cmd[0]);
			}
			SCpnt->result = (DID_BAD_TARGET << 16);
			done(SCpnt);
	}
	}

	if (error) {
		/* return and retry again */
		/* SCpnt->result = (BUSY<<1|COMMAND_COMPLETE<<8|DID_OK<<16); */
		SCpnt->result = DID_ERROR << 16;
		done(SCpnt);
	}

	spin_unlock_irqrestore(&pada->pdc618_lock, lock);
	return(0);
}

/*
 * scsi abort function
 */
int pdc618_scsi_abort(Scsi_Cmnd *SCpnt)
{
       	printk(KERN_INFO "%s:[info] scsi abort success\n",PDCNAME);
	if (SCpnt->serial_number != SCpnt->serial_number_at_timeout) {
		return(SCSI_ABORT_NOT_RUNNING);
	}
	SCpnt->result = DID_ABORT << 16;
	SCpnt->scsi_done(SCpnt);
	return(SCSI_ABORT_SUCCESS);
}
/*
 * scsi eh abort function
 */
int pdc618_scsi_eh_abort(Scsi_Cmnd *SCpnt)
{
	return(FAILED);
}
/*
 * scsi reset function
 */
int pdc618_scsi_reset(Scsi_Cmnd *SCpnt, unsigned int flags)
{

	unsigned char chn,id;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;

	id = pada->id*ChPAda*DrvPCh + SCpnt->target;
	chn = id / DrvPCh;
       	printk(KERN_WARNING "%s:[warning] scsi reset channel%x ",PDCNAME,chn+1);
	SCpnt->result = DID_RESET << 16;
	if (CAM_ResetATAChannel(chn) == camSUCCESS) {
		printk("OK\n");
	       	return(SCSI_RESET_SUCCESS);
	}
	else {
		printk("Failed\n");
	       	return(SCSI_RESET_ERROR);
	}
}
/*
 * scsi eh device reset function
 */
int pdc618_scsi_eh_device_reset(Scsi_Cmnd *SCpnt)
{
	unsigned char chn,id;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;
	pdc618_drive_t *pdrv;
	PDEV_CONFIG cam_dev;

	id = pada->id*ChPAda*DrvPCh + SCpnt->target;
	chn = id / DrvPCh;
	pdrv = &pdc618_drive[id];
	cam_dev = pdrv->cam_dev;
       	printk(KERN_WARNING "%s:[warning] scsi eh reset disk%x ",PDCNAME,id+1);
	if (CAM_EnumATADevice(cam_dev) == camSUCCESS) {
		printk("OK\n");
	       	return(SUCCESS);
	}
	else {
		printk("Failed\n");
		return(FAILED);
	}
}
/*
 * scsi eh host reset function
 */
int pdc618_scsi_eh_host_reset(Scsi_Cmnd *SCpnt)
{

	unsigned char chn,id;
	pdc618_adapter_t *pada =(pdc618_adapter_t *)SCpnt->host->hostdata;

	id = pada->id*ChPAda*DrvPCh + SCpnt->target;
	chn = id / DrvPCh;
       	printk(KERN_WARNING "%s:[warning] scsi eh reset channel%x ",PDCNAME,chn+1);
	if (CAM_ResetATAChannel(chn) == camSUCCESS) {
		printk("OK\n");
	       	return(SUCCESS);
	}
	else {
		printk("Failed\n");
	       	return(FAILED);
	}
}
/*
 * returns unit geometry in cylinders/heads/sectors
 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,27)
int pdc618_bios_param(Disk *disk, struct block_device *dev, int geom[])
#else
int pdc618_bios_param(Disk *disk, kdev_t dev, int geom[])
#endif
{
	int heads, sectors, cylinders;

	heads = 64;
	sectors = 32;
	cylinders = disk->capacity / (heads * sectors);

	if (disk->capacity > 0x200000) 	/* > 1GB */ {
		heads = 255;
		sectors = 63;
		cylinders = disk->capacity / (heads * sectors);

		/* > 526GB (C/H/S = 65535/255/63) */
		if ( cylinders > 65535 ) {
			cylinders = 65535;
			sectors = disk->capacity / (cylinders*heads);
		}
	}

	geom[0] = heads;
	geom[1] = sectors;
	geom[2] = cylinders;

	return(0);
}
/*
 * char dev operation
 */
static int pdc618dev_open(struct inode *inodep, struct file *filep)
{
	MOD_INC_USE_COUNT;
	return(0);
}
static int pdc618dev_release(struct inode *inodep, struct file *filep)
{
	MOD_DEC_USE_COUNT;
	return(0);
}
static int pdc618dev_ioctl(struct inode *inodep, struct file *filep, 
		unsigned int req, unsigned long arg)
{
	int retval = -1;
	unsigned char *args;

	if ( !arg ) {
		return retval;
	}

	if (!capable(CAP_SYS_ADMIN))
		return -EPERM;

	args = (unsigned char *)arg;
	switch (req) {
		/* get total adapter count */
		case STIO_GET_ADAPTERCOUNT: {
		tag_adaptercount_t *buf = (tag_adaptercount_t *)args;

		retval = 0;
		if (get_user(pdc618_num_adapters, &buf->adaptercount))
			retval = -EFAULT;
		break;
		}

		/* get adapter type */
		case STIO_GET_ADAPTERTYPE: {
		tag_adaptertype_t *buf = (tag_adaptertype_t *)args;
		int ada;
		if (get_user(ada, &buf->adapter))
			return -EFAULT;

		if (pdc618_adapter[ada].present) {
			switch (pdc618_adapter[ada].deviceid) {
				case DID618:
					put_user(ULTRA618, &buf->adaptertype);
					break;
				case DID375:
					put_user(SATA375, &buf->adaptertype);
					break;
				case DID318:
					put_user(SATA375, &buf->adaptertype);
					break;
				case DID378:
					put_user(SATA378, &buf->adaptertype);
					break;
				default:
					put_user(SATA150, &buf->adaptertype);
					break;
			}
			retval = 0;
		}

		break;
		}

		/* get adapter info */
		case STIO_GET_ADAPTERINFO: {
		tag_adapterinfo_t *buf = (tag_adapterinfo_t *)args;
		int ada;
		get_user(ada, &buf->adapter);

		if (pdc618_adapter[ada].present) {
			buf->deviceid = pdc618_adapter[ada].deviceid;
			retval = 0;
		}

		break;
		}
		/* find flash mem type */
		case STIO_FIND_FLASHMEM: {
		tag_flash_t *buf = (tag_flash_t *)args;
		int ret;
		int ada;
		void *buffer;
		get_user(ada, &buf->adapter);
		get_user(buffer, &buf->buffer);

		ret = CAM_Find_FlashMemory(ada, buffer);

		if (ret ==camFAIL) {
			return -1;
		}

		retval = 0;
		break;
		}

		/* erase flash memory */
		case STIO_ERASE_FLASHMEM: {
		int ret;
		tag_flash_t *buf = (tag_flash_t *) args;
		int ada;
		
		get_user(ada, &buf->adapter);

		ret = CAM_Erase_FlashMemory(ada);
		if (ret == camSUCCESS) {
			retval = 0;
		}

		break;
		}

		/* read flash memory */
		case STIO_READ_FLASHMEM: {
		int ret;
		tag_flash_t *buf = (tag_flash_t *) args;
		int ada;
		unsigned long offset, buffer_size;
		void *buffer;
		get_user(ada, &buf->adapter);
		get_user(offset, &buf->offset);
		get_user(buffer_size, &buf->buffer_size);
		get_user(buffer, &buf->buffer);
		
		ret = CAM_Read_FlashMemory(ada, offset, buffer_size, buffer);
		if (ret == camSUCCESS) {
			retval = 0;
			if (debug & LOGATA) {
				printk(KERN_DEBUG "%s:[D] read %ldK offset %ld success ", PDCNAME, buf->buffer_size, buf->offset);
				hexdump(buf->buffer, 10);
			}
		}

		break;
		}

		/* write flash memory */
		case STIO_WRITE_FLASHMEM: {
		int ret;
		tag_flash_t *buf = (tag_flash_t *) args;
		int ada;
		unsigned long offset, buffer_size;
		void *buffer;
		get_user(ada, &buf->adapter);
		get_user(offset, &buf->offset);
		get_user(buffer_size, &buf->buffer_size);
		get_user(buffer, &buf->buffer);

		ret = CAM_Write_FlashMemory(ada, offset, buffer_size, buffer);
		if (ret == camSUCCESS) {
			retval = 0;
			if (debug & LOGATA) {
				printk(KERN_DEBUG "%s:[D] write %ldK offset %ld success ", PDCNAME, buf->buffer_size, buf->offset);
				hexdump(buf->buffer, 10);
			}
		}

		break;
		}
		default:
			break;
	}
	return retval;
}
/*
 * Implement CAM Supportive Routines
 */
void camOutPortByte(U32 port, U8 data)
{
	outb(data, port);
}
U8 camInPortByte(U32 port)
{
	return inb(port);
}
void camOutPortWord(U32 port, U16 data)
{
	outw(data, port);
}
U16 camInPortWord(U32 port)
{
	return inw(port);
}

void camOutPortDword(U32 port, U32 data)
{
	outl(data, port);
}
U32 camInPortDword(U32 port)
{
	return inl(port);
}
/*
 * MMIO function for CAM
 */
void camWriteRegByte(U32 addr, U8 data)
{
	writeb(data, addr);
}
U8 camReadRegByte(U32 addr)
{
	return readb(addr);
}
void camWriteRegWord(U32 addr, U16 data)
{
	writew(data, addr);
}
U16 camReadRegWord(U32 addr)
{
	return readw(addr);
}
void camWriteRegDword(U32 addr, U32 data)
{
	writel(data, addr);
}
U32 camReadRegDword(U32 addr)
{
	return readl(addr);
}

/*
 * Function:	camCheckDriveStatus()
 *
 * Purpose:	Report to driver if the drive is plugged in or unplugged
 *
 * Parameters:	bCh	- channel # (0 - N)
 * 		bID	- 0 is master, 1 is slave
 * 		Status	- NEW_DRIVE: a new drive is plugged in
 * 			  NO_DRIVE:  the drive is unplugged
 *
 * Return:	nothing
 *
 * Notes:	use tasklet to scan device.
 * 		FIXME: there seems no function to notify scsi when disk is 
 * 		       unplugged in kernel 2.4.x
 */
void camCheckDriveStatus(U8 bCh, U8 bID, U8 Status)
{
	int drv;
	pdc618_drive_t *pdrv;
	DEV_CONFIG *pcam_dev; 

	drv = bCh*DrvPCh+bID;
	pdrv = &pdc618_drive[drv];
	pcam_dev = pdrv->cam_dev;

	/* SATA disk unplug */
	if (Status == NO_DRIVE) {
		pcam_dev->DevFlag = DEV_NOTFOUND;
		printk(KERN_INFO "%s:[info] Drive%2d Offline: %s \n",PDCNAME, drv+1, pcam_dev->Model);
	}
	/* new SATA disk plugin */
	else if (Status == NEW_DRIVE) {
		DECLARE_TASKLET(tasklet, pdc618_scan_device, drv);
		tasklet_schedule(&tasklet);
	}
	return;
}

/*
 * Function:	camStallExecution()
 *
 * Purpose:	provid CAM a OS-related interface to stalls under the OS.
 *
 * Parameters:	Delay	- specifies the delay interval in 10(ms)
 *
 * Return:	nothing
 *
 * Notes:	n/a
 */
void camStallExecution(U16 Delay)
{
	mdelay((unsigned long)Delay*10);
	return;
}

/*
 * Function:	pdc618_pre_cam_timer()
 *
 * Purpose:	when ATA timeout, call pre timer before TimerService to do some
 * 		task
 *
 * Parameters:	pointer to pdc618_timer_t
 *
 * Return:	nothing
 *
 * Notes:	when ATA timeout, call our pre timer then call cam TimerService.
 * 		when timeout, cam won't del timer, thus status will always be 
 * 		BUSY.
 */
void pdc618_pre_cam_timer(unsigned long ptr)
{
	pdc618_timer_t *t = (pdc618_timer_t *)ptr;
	t->status = camFREE;
	del_timer(&t->timer);
	t->callback();
	return;
}

/*
 * Function:	camTimer()
 *
 * Purpose:	export OS timer for CAM
 *
 * Parameters:	Delay		- the timeout (ticks=1/18 second)
 * 		TimerService	- timer expire function
 *
 * Return:	timer index
 *
 * Notes:	don't return 0 as index
 */
U32 camTimer(U32 Delay, void *TimerService)
{
	unsigned char i;

	if (Delay) {
		for (i = 0; i < MAXTIMER; i++) {
			/* add timer */
			if ( pdc618_timer[i].status == camFREE ) {
				memset(&pdc618_timer[i], 0, 
						sizeof(pdc618_timer_t));
				init_timer(&pdc618_timer[i].timer);
				pdc618_timer[i].status = camBUSY;
				pdc618_timer[i].timer.expires = jiffies + (Delay*100/18);
				pdc618_timer[i].timer.function = pdc618_pre_cam_timer;
				pdc618_timer[i].timer.data = (unsigned long)&pdc618_timer[i];
				pdc618_timer[i].id = i;
				pdc618_timer[i].callback = TimerService;
				add_timer(&pdc618_timer[i].timer);
				/* don't return 0 as index */
				return(i+1);
			}
		}
	}
	else {	/* remove timer */
		unsigned long num;
		num = (unsigned long)TimerService - 1;
		if (num > MAXTIMER) {
			printk("camTimer: bad timer to remove %lx\n", num); 
			return 0;
		}
		if (pdc618_timer[num].status == camBUSY)
			del_timer(&pdc618_timer[num].timer);
		pdc618_timer[num].status = camFREE;
		return(0);
	}

	return(0);
}

/*
 * Module Stuff
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10)
MODULE_AUTHOR("Promise Technology Inc.");
MODULE_DESCRIPTION("Promise SATA150 Series Linux Driver");
MODULE_LICENSE("GPL");
#endif
MODULE_PARM(atapi, "i");
MODULE_PARM(debug, "i");
MODULE_PARM_DESC(atapi, "ATAPI enable bit (1/0, default 0)");
MODULE_PARM_DESC(debug, "The debug level (0-2)");

#ifdef MODULE
Scsi_Host_Template driver_template = pdc618;
#include "scsi_module.c"
#endif

#if 0
#ifdef MODULE
Scsi_Host_Template driver_template = pdc618;
#include <linux/module.h>
int init_module(void)
{
	driver_template.module = &__this_module;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,1)
	scsi_register_host(&driver_template);
#else
	scsi_register_module(MODULE_SCSI_HA, &driver_template);
#endif
	if (driver_template.present) {
		return 0;
	}	

	/* unregister scsi module */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,1)
	scsi_unregister_host(&driver_template);
#else
	scsi_unregister_module(MODULE_SCSI_HA, &driver_template);
#endif
	return -1;
}

void cleanup_module(void) 
{

	unsigned char ua;
	/* close cam and cleanup */
	for (ua = 0; ua < MAXADAPTER; ua++) {
		if(pdc618_adapter[ua].present) {
			pdc618_adapter_t *pada = &pdc618_adapter[ua];
			ADAPTER_CONFIG_INFO *pcam_ada = pada->cam_ada;
			if (CAM_Close(pcam_ada) != camSUCCESS)
			       	printk(KERN_ERR "%s:[error] can't close cam\n",PDCNAME);

			/* free memory, irq, ioport */
			pdc618_cleanup(ua);
		}
	}

	/* free queue, timer */
	pdc618_cleanup2();

	/* unregister scsi module */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,5,1)
	scsi_unregister_host(&driver_template);
#else
	scsi_unregister_module(MODULE_SCSI_HA, &driver_template);
#endif
	return;

}
#endif
#endif

static struct pci_device_id pdc_ultra_pci_tbl[] __devinitdata = { 
	{ PCI_VENDOR_ID_PROMISE, 0x3318, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 }, 
	{ PCI_VENDOR_ID_PROMISE, 0x3375, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 }, 
	{ PCI_VENDOR_ID_PROMISE, 0x3373, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 }, 
	{ PCI_VENDOR_ID_PROMISE, 0x6626, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 }, 
	{ 0, }
};
MODULE_DEVICE_TABLE(pci,pdc_ultra_pci_tbl);
