// SPDX-License-Identifier: GPL-2.0
/*
 * R8A77470 processor support - PFC hardware block.
 *
 * Copyright (C) 2018 Renesas Electronics Corp.
 */

#include <linux/kernel.h>

#include "core.h"

#define PINMUX_IPSR_GPSR(ipsr, fn)					\
	PINMUX_DATA(fn##_MARK, FN_##fn, FN_##ipsr)

#define PORT_GP_17(bank, fn, sfx)					\
	PORT_GP_1(bank, 0,  fn, sfx), PORT_GP_1(bank, 1,  fn, sfx),	\
	PORT_GP_1(bank, 2,  fn, sfx), PORT_GP_1(bank, 3,  fn, sfx),	\
	PORT_GP_1(bank, 4,  fn, sfx), PORT_GP_1(bank, 5,  fn, sfx),	\
	PORT_GP_1(bank, 6,  fn, sfx), PORT_GP_1(bank, 7,  fn, sfx),	\
	PORT_GP_1(bank, 8,  fn, sfx), PORT_GP_1(bank, 9,  fn, sfx),	\
	PORT_GP_1(bank, 10, fn, sfx), PORT_GP_1(bank, 11, fn, sfx),	\
	PORT_GP_1(bank, 12, fn, sfx), PORT_GP_1(bank, 13, fn, sfx),	\
	PORT_GP_1(bank, 14, fn, sfx), PORT_GP_1(bank, 15, fn, sfx),	\
	PORT_GP_1(bank, 16, fn, sfx)

#define PORT_GP_23(bank, fn, sfx)					\
	PORT_GP_17(bank, fn, sfx),					\
	PORT_GP_1(bank, 17, fn, sfx), PORT_GP_1(bank, 18, fn, sfx),	\
	PORT_GP_1(bank, 19, fn, sfx), PORT_GP_1(bank, 20, fn, sfx),	\
	PORT_GP_1(bank, 21, fn, sfx), PORT_GP_1(bank, 22, fn, sfx)

#define PORT_GP_26(bank, fn, sfx)					\
	PORT_GP_23(bank, fn, sfx),					\
	PORT_GP_1(bank, 23, fn, sfx), PORT_GP_1(bank, 24, fn, sfx),	\
	PORT_GP_1(bank, 25, fn, sfx)

#define CPU_ALL_PORT(fn, sfx)						\
	PORT_GP_23(0, fn, sfx),						\
	PORT_GP_23(1, fn, sfx),						\
	PORT_GP_32(2, fn, sfx),						\
	PORT_GP_17(3, fn, sfx),						\
	PORT_GP_1(3, 27, fn, sfx),					\
	PORT_GP_1(3, 28, fn, sfx),					\
	PORT_GP_1(3, 29, fn, sfx),					\
	PORT_GP_26(4, fn, sfx),						\
	PORT_GP_32(5, fn, sfx)

enum {
	PINMUX_RESERVED = 0,

	PINMUX_DATA_BEGIN,
	GP_ALL(DATA),
	PINMUX_DATA_END,

	PINMUX_FUNCTION_BEGIN,
	GP_ALL(FN),

	/* GPSR0 */
	FN_USB0_PWEN, FN_USB0_OVC, FN_USB1_PWEN, FN_USB1_OVC, FN_CLKOUT,
	FN_IP0_3_0, FN_IP0_7_4, FN_IP0_11_8, FN_IP0_15_12, FN_IP0_19_16,
	FN_IP0_23_20, FN_IP0_27_24, FN_IP0_31_28, FN_MMC0_CLK_SDHI1_CLK,
	FN_MMC0_CMD_SDHI1_CMD, FN_MMC0_D0_SDHI1_D0, FN_MMC0_D1_SDHI1_D1,
	FN_MMC0_D2_SDHI1_D2, FN_MMC0_D3_SDHI1_D3, FN_IP1_3_0,
	FN_IP1_7_4, FN_MMC0_D6, FN_MMC0_D7,

	/* GPSR1 */
	FN_IP1_11_8, FN_IP1_15_12, FN_IP1_19_16, FN_IP1_23_20, FN_IP1_27_24,
	FN_IP1_31_28, FN_IP2_3_0, FN_IP2_7_4, FN_IP2_11_8, FN_IP2_15_12,
	FN_IP2_19_16, FN_IP2_23_20, FN_IP2_27_24, FN_IP2_31_28, FN_IP3_3_0,
	FN_IP3_7_4, FN_IP3_11_8, FN_IP3_15_12, FN_IP3_19_16, FN_IP3_23_20,
	FN_IP3_27_24, FN_IP3_31_28, FN_IP4_3_0,

	/* GPSR2 */
	FN_IP4_7_4, FN_IP4_11_8, FN_IP4_15_12, FN_IP4_19_16, FN_IP4_23_20,
	FN_IP4_27_24, FN_IP4_31_28, FN_IP5_3_0, FN_IP5_7_4, FN_IP5_11_8,
	FN_IP5_15_12, FN_IP5_19_16, FN_IP5_23_20, FN_IP5_27_24, FN_IP5_31_28,
	FN_IP6_3_0, FN_IP6_7_4, FN_IP6_11_8, FN_IP6_15_12, FN_IP6_19_16,
	FN_IP6_23_20, FN_IP6_27_24, FN_IP6_31_28, FN_IP7_3_0, FN_IP7_7_4,
	FN_IP7_11_8, FN_IP7_15_12, FN_IP7_19_16, FN_IP7_23_20, FN_IP7_27_24,
	FN_IP7_31_28, FN_IP8_3_0,

	/* GPSR3 */
	FN_IP8_7_4, FN_IP8_11_8, FN_IP8_15_12, FN_IP8_19_16, FN_IP8_23_20,
	FN_IP8_27_24, FN_IP8_31_28, FN_IP9_3_0, FN_IP9_7_4, FN_IP9_11_8,
	FN_IP9_15_12, FN_IP9_19_16, FN_IP9_23_20, FN_IP9_27_24, FN_IP9_31_28,
	FN_IP10_3_0, FN_IP10_7_4, FN_IP10_11_8, FN_IP10_15_12, FN_IP10_19_16,

	/* GPSR4 */
	FN_IP10_23_20, FN_IP10_27_24, FN_IP10_31_28, FN_IP11_3_0, FN_IP11_7_4,
	FN_IP11_11_8, FN_IP11_15_12, FN_IP11_19_16, FN_IP11_23_20,
	FN_IP11_27_24, FN_IP11_31_28, FN_IP12_3_0, FN_IP12_7_4, FN_IP12_11_8,
	FN_IP12_15_12, FN_IP12_19_16, FN_IP12_23_20, FN_IP12_27_24,
	FN_IP12_31_28, FN_IP13_3_0, FN_IP13_7_4, FN_IP13_11_8, FN_IP13_15_12,
	FN_IP13_19_16, FN_IP13_23_20, FN_IP13_27_24,

	/* GPSR5 */
	FN_IP13_31_28, FN_IP14_3_0, FN_IP14_7_4, FN_IP14_11_8, FN_IP14_15_12,
	FN_IP14_19_16, FN_IP14_23_20, FN_IP14_27_24, FN_IP14_31_28,
	FN_IP15_3_0, FN_IP15_7_4, FN_IP15_11_8, FN_IP15_15_12, FN_IP15_19_16,
	FN_IP15_23_20, FN_IP15_27_24, FN_IP15_31_28, FN_IP16_3_0, FN_IP16_7_4,
	FN_IP16_11_8, FN_IP16_15_12, FN_IP16_19_16, FN_IP16_23_20,
	FN_IP16_27_24, FN_IP16_31_28, FN_IP17_3_0, FN_IP17_7_4, FN_IP17_11_8,
	FN_IP17_15_12, FN_IP17_19_16, FN_IP17_23_20, FN_IP17_27_24,

	/* IPSR0 */
	FN_SD0_CLK, FN_SSI_SCK1_C, FN_RX3_C,
	FN_SD0_CMD, FN_SSI_WS1_C, FN_TX3_C,
	FN_SD0_DAT0, FN_SSI_SDATA1_C, FN_RX4_E,
	FN_SD0_DAT1, FN_SSI_SCK0129_B, FN_TX4_E,
	FN_SD0_DAT2, FN_SSI_WS0129_B, FN_RX5_E,
	FN_SD0_DAT3, FN_SSI_SDATA0_B, FN_TX5_E,
	FN_SD0_CD, FN_CAN0_RX_A,
	FN_SD0_WP, FN_IRQ7, FN_CAN0_TX_A,

	/* IPSR1 */
	FN_MMC0_D4, FN_SD1_CD,
	FN_MMC0_D5, FN_SD1_WP,
	FN_D0, FN_SCL3_B, FN_RX5_B, FN_IRQ4, FN_MSIOF2_RXD_C, FN_SSI_SDATA5_B,
	FN_D1, FN_SDA3_B, FN_TX5_B, FN_MSIOF2_TXD_C, FN_SSI_WS5_B,
	FN_D2, FN_RX4_B, FN_SCL0_D, FN_PWM1_C, FN_MSIOF2_SCK_C, FN_SSI_SCK5_B,
	FN_D3, FN_TX4_B, FN_SDA0_D, FN_PWM0_A, FN_MSIOF2_SYNC_C,
	FN_D4, FN_IRQ3, FN_TCLK1_A, FN_PWM6_C,
	FN_D5, FN_HRX2, FN_SCL1_B, FN_PWM2_C, FN_TCLK2_B,

	/* IPSR2 */
	FN_D6, FN_HTX2, FN_SDA1_B, FN_PWM4_C,
	FN_D7, FN_HSCK2, FN_SCIF1_SCK_C, FN_IRQ6, FN_PWM5_C,
	FN_D8, FN_HCTS2_N, FN_RX1_C, FN_SCL1_D, FN_PWM3_C,
	FN_D9, FN_HRTS2_N, FN_TX1_C, FN_SDA1_D,
	FN_D10, FN_MSIOF2_RXD_A, FN_HRX0_B,
	FN_D11, FN_MSIOF2_TXD_A, FN_HTX0_B,
	FN_D12, FN_MSIOF2_SCK_A, FN_HSCK0, FN_CAN_CLK_C,
	FN_D13,	FN_MSIOF2_SYNC_A, FN_RX4_C,

	/* IPSR3 */
	FN_D14, FN_MSIOF2_SS1, FN_TX4_C, FN_CAN1_RX_B, FN_AVB_AVTP_CAPTURE_A,
	FN_D15, FN_MSIOF2_SS2, FN_PWM4_A, FN_CAN1_TX_B, FN_IRQ2, FN_AVB_AVTP_MATCH_A,
	FN_QSPI0_SPCLK, FN_WE0_N,
	FN_QSPI0_MOSI_QSPI0_IO0, FN_BS_N,
	FN_QSPI0_MISO_QSPI0_IO1, FN_RD_WR_N,
	FN_QSPI0_IO2, FN_CS0_N,
	FN_QSPI0_IO3, FN_RD_N,
	FN_QSPI0_SSL, FN_WE1_N,

	/* IPSR4 */
	FN_EX_WAIT0, FN_CAN_CLK_B, FN_SCIF_CLK_A,
	FN_DU0_DR0, FN_RX5_C, FN_SCL2_D, FN_A0,
	FN_DU0_DR1, FN_TX5_C, FN_SDA2_D, FN_A1,
	FN_DU0_DR2, FN_RX0_D, FN_SCL0_E, FN_A2,
	FN_DU0_DR3, FN_TX0_D, FN_SDA0_E, FN_PWM0_B, FN_A3,
	FN_DU0_DR4, FN_RX1_D, FN_A4,
	FN_DU0_DR5, FN_TX1_D, FN_PWM1_B, FN_A5,
	FN_DU0_DR6, FN_RX2_C, FN_A6,

	/* IPSR5 */
	FN_DU0_DR7, FN_TX2_C, FN_PWM2_B, FN_A7,
	FN_DU0_DG0, FN_RX3_B, FN_SCL3_D, FN_A8,
	FN_DU0_DG1, FN_TX3_B, FN_SDA3_D, FN_PWM3_B, FN_A9,
	FN_DU0_DG2, FN_RX4_D, FN_A10,
	FN_DU0_DG3, FN_TX4_D, FN_PWM4_B, FN_A11,
	FN_DU0_DG4, FN_HRX0_A, FN_A12,
	FN_DU0_DG5, FN_HTX0_A, FN_PWM5_B, FN_A13,
	FN_DU0_DG6, FN_HRX1_C, FN_A14,

	/* IPSR6 */
	FN_DU0_DG7, FN_HTX1_C, FN_PWM6_B, FN_A15,
	FN_DU0_DB0, FN_SCL4_D, FN_CAN0_RX_C, FN_A16,
	FN_DU0_DB1, FN_SDA4_D, FN_CAN0_TX_C, FN_A17,
	FN_DU0_DB2, FN_HCTS0_N, FN_A18,
	FN_DU0_DB3, FN_HRTS0_N, FN_A19,
	FN_DU0_DB4, FN_HCTS1_N_C, FN_A20,
	FN_DU0_DB5, FN_HRTS1_N_C, FN_A21,
	FN_DU0_DB6, FN_A22,

	/* IPSR7 */
	FN_DU0_DB7, FN_A23,
	FN_DU0_DOTCLKIN, FN_A24,
	FN_DU0_DOTCLKOUT0, FN_A25,
	FN_DU0_DOTCLKOUT1, FN_MSIOF2_RXD_B, FN_CS1_N_A26,
	FN_DU0_EXHSYNC_DU0_HSYNC, FN_MSIOF2_TXD_B, FN_DREQ0_N,
	FN_DU0_EXVSYNC_DU0_VSYNC, FN_MSIOF2_SYNC_B, FN_DACK0,
	FN_DU0_EXODDF_DU0_ODDF_DISP_CDE, FN_MSIOF2_SCK_B, FN_DRACK0,
	FN_DU0_DISP, FN_CAN1_RX_C,

	/* IPSR8 */
	FN_DU0_CDE, FN_CAN1_TX_C,
	FN_VI1_CLK, FN_AVB_RX_CLK, FN_ETH_REF_CLK,
	FN_VI1_DATA0, FN_AVB_RX_DV, FN_ETH_CRS_DV,
	FN_VI1_DATA1, FN_AVB_RXD0, FN_ETH_RXD0,
	FN_VI1_DATA2, FN_AVB_RXD1, FN_ETH_RXD1,
	FN_VI1_DATA3, FN_AVB_RXD2, FN_ETH_MDIO,
	FN_VI1_DATA4, FN_AVB_RXD3, FN_ETH_RX_ER,
	FN_VI1_DATA5, FN_AVB_RXD4, FN_ETH_LINK,

	/* IPSR9 */
	FN_VI1_DATA6, FN_AVB_RXD5, FN_ETH_TXD1,
	FN_VI1_DATA7, FN_AVB_RXD6, FN_ETH_TX_EN,
	FN_VI1_CLKENB, FN_SCL3_A, FN_AVB_RXD7, FN_ETH_MAGIC,
	FN_VI1_FIELD, FN_SDA3_A, FN_AVB_RX_ER, FN_ETH_TXD0,
	FN_VI1_HSYNC_N,	FN_RX0_B, FN_SCL0_C, FN_AVB_GTXREFCLK, FN_ETH_MDC,
	FN_VI1_VSYNC_N,	FN_TX0_B, FN_SDA0_C, FN_AUDIO_CLKOUT_B, FN_AVB_TX_CLK,
	FN_VI1_DATA8, FN_SCL2_B, FN_AVB_TX_EN,
	FN_VI1_DATA9, FN_SDA2_B, FN_AVB_TXD0,

	/* IPSR10 */
	FN_VI1_DATA10, FN_CAN0_RX_B, FN_AVB_TXD1,
	FN_VI1_DATA11, FN_CAN0_TX_B, FN_AVB_TXD2,
	FN_AVB_TXD3, FN_AUDIO_CLKA_B, FN_SSI_SCK1_D, FN_RX5_F, FN_MSIOF0_RXD_B,
	FN_AVB_TXD4, FN_AUDIO_CLKB_B, FN_SSI_WS1_D, FN_TX5_F, FN_MSIOF0_TXD_B,
	FN_AVB_TXD5, FN_SCIF_CLK_B, FN_AUDIO_CLKC_B, FN_SSI_SDATA1_D, FN_MSIOF0_SCK_B,
	FN_SCL0_A, FN_RX0_C, FN_PWM5_A,	FN_TCLK1_B, FN_AVB_TXD6, FN_CAN1_RX_D, FN_MSIOF0_SYNC_B,
	FN_SDA0_A, FN_TX0_C, FN_IRQ5, FN_CAN_CLK_A, FN_AVB_GTX_CLK, FN_CAN1_TX_D, FN_DVC_MUTE,
	FN_SCL1_A, FN_RX4_A, FN_PWM5_D, FN_DU1_DR0, FN_SSI_SCK6_B, FN_VI0_G0,

	/* IPSR11 */
	FN_SDA1_A, FN_TX4_A, FN_DU1_DR1, FN_SSI_WS6_B, FN_VI0_G1,
	FN_MSIOF0_RXD_A, FN_RX5_A, FN_SCL2_C, FN_DU1_DR2, FN_QSPI1_MOSI_QSPI1_IO0, FN_SSI_SDATA6_B, FN_VI0_G2,
	FN_MSIOF0_TXD_A, FN_TX5_A, FN_SDA2_C, FN_DU1_DR3, FN_QSPI1_MISO_QSPI1_IO1, FN_SSI_WS78_B, FN_VI0_G3,
	FN_MSIOF0_SCK_A, FN_IRQ0, FN_DU1_DR4, FN_QSPI1_SPCLK, FN_SSI_SCK78_B, FN_VI0_G4,
	FN_MSIOF0_SYNC_A, FN_PWM1_A, FN_DU1_DR5, FN_QSPI1_IO2, FN_SSI_SDATA7_B,
	FN_MSIOF0_SS1_A, FN_DU1_DR6, FN_QSPI1_IO3, FN_SSI_SDATA8_B,
	FN_MSIOF0_SS2_A, FN_DU1_DR7, FN_QSPI1_SSL,
	FN_HRX1_A, FN_SCL4_A, FN_PWM6_A, FN_DU1_DG0, FN_RX0_A,

	/* IPSR12 */
	FN_HTX1_A, FN_SDA4_A, FN_DU1_DG1, FN_TX0_A,
	FN_HCTS1_N_A, FN_PWM2_A, FN_DU1_DG2, FN_REMOCON_B,
	FN_HRTS1_N_A, FN_DU1_DG3, FN_SSI_WS1_B, FN_IRQ1,
	FN_SD2_CLK, FN_HSCK1, FN_DU1_DG4, FN_SSI_SCK1_B,
	FN_SD2_CMD, FN_SCIF1_SCK_A, FN_TCLK2_A, FN_DU1_DG5, FN_SSI_SCK2_B, FN_PWM3_A,
	FN_SD2_DAT0, FN_RX1_A, FN_SCL1_E, FN_DU1_DG6, FN_SSI_SDATA1_B,
	FN_SD2_DAT1, FN_TX1_A, FN_SDA1_E, FN_DU1_DG7, FN_SSI_WS2_B,
	FN_SD2_DAT2, FN_RX2_A, FN_DU1_DB0, FN_SSI_SDATA2_B,

	/* IPSR13 */
	FN_SD2_DAT3, FN_TX2_A, FN_DU1_DB1, FN_SSI_WS9_B,
	FN_SD2_CD, FN_SCIF2_SCK_A, FN_DU1_DB2, FN_SSI_SCK9_B,
	FN_SD2_WP, FN_SCIF3_SCK, FN_DU1_DB3, FN_SSI_SDATA9_B,
	FN_RX3_A, FN_SCL1_C, FN_MSIOF1_RXD_B, FN_DU1_DB4, FN_AUDIO_CLKA_C, FN_SSI_SDATA4_B,
	FN_TX3_A, FN_SDA1_C, FN_MSIOF1_TXD_B, FN_DU1_DB5, FN_AUDIO_CLKB_C, FN_SSI_WS4_B,
	FN_SCL2_A, FN_MSIOF1_SCK_B, FN_DU1_DB6, FN_AUDIO_CLKC_C, FN_SSI_SCK4_B,
	FN_SDA2_A, FN_MSIOF1_SYNC_B, FN_DU1_DB7, FN_AUDIO_CLKOUT_C,
	FN_SSI_SCK5_A, FN_DU1_DOTCLKOUT1,

	/* IPSR14 */
	FN_SSI_WS5_A, FN_SCL3_C, FN_DU1_DOTCLKIN,
	FN_SSI_SDATA5_A, FN_SDA3_C, FN_DU1_DOTCLKOUT0,
	FN_SSI_SCK6_A, FN_DU1_EXODDF_DU1_ODDF_DISP_CDE,
	FN_SSI_WS6_A, FN_SCL4_C, FN_DU1_EXHSYNC_DU1_HSYNC,
	FN_SSI_SDATA6_A, FN_SDA4_C, FN_DU1_EXVSYNC_DU1_VSYNC,
	FN_SSI_SCK78_A, FN_SDA4_E, FN_DU1_DISP,
	FN_SSI_WS78_A, FN_SCL4_E, FN_DU1_CDE,
	FN_SSI_SDATA7_A, FN_IRQ8, FN_AUDIO_CLKA_D, FN_CAN_CLK_D, FN_VI0_G5,

	/* IPSR15 */
	FN_SSI_SCK0129_A, FN_MSIOF1_RXD_A, FN_RX5_D, FN_VI0_G6,
	FN_SSI_WS0129_A, FN_MSIOF1_TXD_A, FN_TX5_D, FN_VI0_G7,
	FN_SSI_SDATA0_A, FN_MSIOF1_SYNC_A, FN_PWM0_C, FN_VI0_R0,
	FN_SSI_SCK34, FN_MSIOF1_SCK_A, FN_AVB_MDC, FN_DACK1, FN_VI0_R1,
	FN_SSI_WS34, FN_MSIOF1_SS1_A, FN_AVB_MDIO, FN_CAN1_RX_A, FN_DREQ1_N, FN_VI0_R2,
	FN_SSI_SDATA3, FN_MSIOF1_SS2_A, FN_AVB_LINK, FN_CAN1_TX_A, FN_DREQ2_N, FN_VI0_R3,
	FN_SSI_SCK4_A, FN_AVB_MAGIC, FN_VI0_R4,
	FN_SSI_WS4_A, FN_AVB_PHY_INT, FN_VI0_R5,

	/* IPSR16 */
	FN_SSI_SDATA4_A, FN_AVB_CRS, FN_VI0_R6,
	FN_SSI_SCK1_A, FN_SCIF1_SCK_B, FN_PWM1_D, FN_IRQ9, FN_REMOCON_A, FN_DACK2, FN_VI0_CLK, FN_AVB_COL,
	FN_SSI_SDATA8_A, FN_RX1_B, FN_CAN0_RX_D, FN_AVB_AVTP_CAPTURE_B,	FN_VI0_R7,
	FN_SSI_WS1_A, FN_TX1_B, FN_CAN0_TX_D, FN_AVB_AVTP_MATCH_B, FN_VI0_DATA0_VI0_B0,
	FN_SSI_SDATA1_A, FN_HRX1_B, FN_VI0_DATA1_VI0_B1,
	FN_SSI_SCK2_A, FN_HTX1_B, FN_AVB_TXD7, FN_VI0_DATA2_VI0_B2,
	FN_SSI_WS2_A, FN_HCTS1_N_B, FN_AVB_TX_ER, FN_VI0_DATA3_VI0_B3,
	FN_SSI_SDATA2_A, FN_HRTS1_N_B, FN_VI0_DATA4_VI0_B4,

	/* IPSR17 */
	FN_SSI_SCK9_A, FN_RX2_B, FN_SCL3_E, FN_EX_WAIT1, FN_VI0_DATA5_VI0_B5,
	FN_SSI_WS9_A, FN_TX2_B, FN_SDA3_E, FN_VI0_DATA6_VI0_B6,
	FN_SSI_SDATA9_A, FN_SCIF2_SCK_B, FN_PWM2_D, FN_VI0_DATA7_VI0_B7,
	FN_AUDIO_CLKA_A, FN_SCL0_B, FN_VI0_CLKENB,
	FN_AUDIO_CLKB_A, FN_SDA0_B, FN_VI0_FIELD,
	FN_AUDIO_CLKC_A, FN_SCL4_B, FN_VI0_HSYNC_N,
	FN_AUDIO_CLKOUT_A, FN_SDA4_B, FN_VI0_VSYNC_N,

	/* MOD_SEL0 */
	FN_SEL_ADGA_0, FN_SEL_ADGA_1, FN_SEL_ADGA_2, FN_SEL_ADGA_3,
	FN_SEL_CANCLK_0, FN_SEL_CANCLK_1, FN_SEL_CANCLK_2, FN_SEL_CANCLK_3,
	FN_SEL_CAN1_0, FN_SEL_CAN1_1, FN_SEL_CAN1_2, FN_SEL_CAN1_3,
	FN_SEL_CAN0_0, FN_SEL_CAN0_1, FN_SEL_CAN0_2, FN_SEL_CAN0_3,
	FN_SEL_I2C04_0, FN_SEL_I2C04_1,	FN_SEL_I2C04_2, FN_SEL_I2C04_3, FN_SEL_I2C04_4,
	FN_SEL_I2C03_0,	FN_SEL_I2C03_1, FN_SEL_I2C03_2, FN_SEL_I2C03_3, FN_SEL_I2C03_4,
	FN_SEL_I2C02_0, FN_SEL_I2C02_1, FN_SEL_I2C02_2, FN_SEL_I2C02_3,
	FN_SEL_I2C01_0, FN_SEL_I2C01_1, FN_SEL_I2C01_2, FN_SEL_I2C01_3, FN_SEL_I2C01_4,
	FN_SEL_I2C00_0, FN_SEL_I2C00_1, FN_SEL_I2C00_2, FN_SEL_I2C00_3, FN_SEL_I2C00_4,
	FN_SEL_AVB_0, FN_SEL_AVB_1,

	/* MOD_SEL1 */
	FN_SEL_SCIFCLK_0, FN_SEL_SCIFCLK_1,
	FN_SEL_SCIF5_0, FN_SEL_SCIF5_1,	FN_SEL_SCIF5_2, FN_SEL_SCIF5_3, FN_SEL_SCIF5_4, FN_SEL_SCIF5_5,
	FN_SEL_SCIF4_0, FN_SEL_SCIF4_1, FN_SEL_SCIF4_2, FN_SEL_SCIF4_3,	FN_SEL_SCIF4_4,
	FN_SEL_SCIF3_0, FN_SEL_SCIF3_1, FN_SEL_SCIF3_2,
	FN_SEL_SCIF2_0, FN_SEL_SCIF2_1, FN_SEL_SCIF2_2,
	FN_SEL_SCIF2_CLK_0, FN_SEL_SCIF2_CLK_1,
	FN_SEL_SCIF1_0, FN_SEL_SCIF1_1, FN_SEL_SCIF1_2,	FN_SEL_SCIF1_3,
	FN_SEL_SCIF0_0, FN_SEL_SCIF0_1, FN_SEL_SCIF0_2,	FN_SEL_SCIF0_3,
	FN_SEL_MSIOF2_0, FN_SEL_MSIOF2_1, FN_SEL_MSIOF2_2,
	FN_SEL_MSIOF1_0, FN_SEL_MSIOF1_1,
	FN_SEL_MSIOF0_0, FN_SEL_MSIOF0_1,
	FN_SEL_RCN_0, FN_SEL_RCN_1,
	FN_SEL_TMU2_0, FN_SEL_TMU2_1,
	FN_SEL_TMU1_0, FN_SEL_TMU1_1,
	FN_SEL_HSCIF1_0, FN_SEL_HSCIF1_1, FN_SEL_HSCIF1_2,
	FN_SEL_HSCIF0_0, FN_SEL_HSCIF0_1,

	/* MOD_SEL2 */
	FN_SEL_ADGB_0, FN_SEL_ADGB_1, FN_SEL_ADGB_2,
	FN_SEL_ADGC_0, FN_SEL_ADGC_1, FN_SEL_ADGC_2,
	FN_SEL_SSI9_0, FN_SEL_SSI9_1,
	FN_SEL_SSI8_0, FN_SEL_SSI8_1,
	FN_SEL_SSI7_0, FN_SEL_SSI7_1,
	FN_SEL_SSI6_0, FN_SEL_SSI6_1,
	FN_SEL_SSI5_0, FN_SEL_SSI5_1,
	FN_SEL_SSI4_0, FN_SEL_SSI4_1,
	FN_SEL_SSI2_0, FN_SEL_SSI2_1,
	FN_SEL_SSI1_0, FN_SEL_SSI1_1, FN_SEL_SSI1_2, FN_SEL_SSI1_3,
	FN_SEL_SSI0_0, FN_SEL_SSI0_1,
	PINMUX_FUNCTION_END,

	PINMUX_MARK_BEGIN,

	USB0_PWEN_MARK, USB0_OVC_MARK, USB1_PWEN_MARK, USB1_OVC_MARK,
	CLKOUT_MARK, MMC0_CLK_SDHI1_CLK_MARK, MMC0_CMD_SDHI1_CMD_MARK,
	MMC0_D0_SDHI1_D0_MARK, MMC0_D1_SDHI1_D1_MARK,
	MMC0_D2_SDHI1_D2_MARK, MMC0_D3_SDHI1_D3_MARK, MMC0_D6_MARK,
	MMC0_D7_MARK,

	/* IPSR0 */
	SD0_CLK_MARK, SSI_SCK1_C_MARK, RX3_C_MARK,
	SD0_CMD_MARK, SSI_WS1_C_MARK, TX3_C_MARK,
	SD0_DAT0_MARK, SSI_SDATA1_C_MARK, RX4_E_MARK,
	SD0_DAT1_MARK, SSI_SCK0129_B_MARK, TX4_E_MARK,
	SD0_DAT2_MARK, SSI_WS0129_B_MARK, RX5_E_MARK,
	SD0_DAT3_MARK, SSI_SDATA0_B_MARK, TX5_E_MARK,
	SD0_CD_MARK, CAN0_RX_A_MARK,
	SD0_WP_MARK, IRQ7_MARK,	CAN0_TX_A_MARK,

	/* IPSR1 */
	MMC0_D4_MARK, SD1_CD_MARK,
	MMC0_D5_MARK, SD1_WP_MARK,
	D0_MARK, SCL3_B_MARK, RX5_B_MARK, IRQ4_MARK, MSIOF2_RXD_C_MARK,	SSI_SDATA5_B_MARK,
	D1_MARK, SDA3_B_MARK, TX5_B_MARK, MSIOF2_TXD_C_MARK, SSI_WS5_B_MARK,
	D2_MARK, RX4_B_MARK, SCL0_D_MARK, PWM1_C_MARK, MSIOF2_SCK_C_MARK, SSI_SCK5_B_MARK,
	D3_MARK, TX4_B_MARK, SDA0_D_MARK, PWM0_A_MARK, MSIOF2_SYNC_C_MARK,
	D4_MARK, IRQ3_MARK, TCLK1_A_MARK, PWM6_C_MARK,
	D5_MARK, HRX2_MARK, SCL1_B_MARK, PWM2_C_MARK, TCLK2_B_MARK,

	/* IPSR2 */
	D6_MARK, HTX2_MARK, SDA1_B_MARK, PWM4_C_MARK,
	D7_MARK, HSCK2_MARK, SCIF1_SCK_C_MARK, IRQ6_MARK, PWM5_C_MARK,
	D8_MARK, HCTS2_N_MARK, RX1_C_MARK, SCL1_D_MARK,	PWM3_C_MARK,
	D9_MARK, HRTS2_N_MARK, TX1_C_MARK, SDA1_D_MARK,
	D10_MARK, MSIOF2_RXD_A_MARK, HRX0_B_MARK,
	D11_MARK, MSIOF2_TXD_A_MARK, HTX0_B_MARK,
	D12_MARK, MSIOF2_SCK_A_MARK, HSCK0_MARK, CAN_CLK_C_MARK,
	D13_MARK, MSIOF2_SYNC_A_MARK, RX4_C_MARK,

	/* IPSR3 */
	D14_MARK, MSIOF2_SS1_MARK, TX4_C_MARK, CAN1_RX_B_MARK, AVB_AVTP_CAPTURE_A_MARK,
	D15_MARK, MSIOF2_SS2_MARK, PWM4_A_MARK, CAN1_TX_B_MARK, IRQ2_MARK, AVB_AVTP_MATCH_A_MARK,
	QSPI0_SPCLK_MARK, WE0_N_MARK,
	QSPI0_MOSI_QSPI0_IO0_MARK, BS_N_MARK,
	QSPI0_MISO_QSPI0_IO1_MARK, RD_WR_N_MARK,
	QSPI0_IO2_MARK, CS0_N_MARK,
	QSPI0_IO3_MARK, RD_N_MARK,
	QSPI0_SSL_MARK, WE1_N_MARK,

	/* IPSR4 */
	EX_WAIT0_MARK, CAN_CLK_B_MARK, SCIF_CLK_A_MARK,
	DU0_DR0_MARK, RX5_C_MARK, SCL2_D_MARK, A0_MARK,
	DU0_DR1_MARK, TX5_C_MARK, SDA2_D_MARK, A1_MARK,
	DU0_DR2_MARK, RX0_D_MARK, SCL0_E_MARK, A2_MARK,
	DU0_DR3_MARK, TX0_D_MARK, SDA0_E_MARK, PWM0_B_MARK, A3_MARK,
	DU0_DR4_MARK, RX1_D_MARK, A4_MARK,
	DU0_DR5_MARK, TX1_D_MARK, PWM1_B_MARK, A5_MARK,
	DU0_DR6_MARK, RX2_C_MARK, A6_MARK,

	/* IPSR5 */
	DU0_DR7_MARK, TX2_C_MARK, PWM2_B_MARK, A7_MARK,
	DU0_DG0_MARK, RX3_B_MARK, SCL3_D_MARK, A8_MARK,
	DU0_DG1_MARK, TX3_B_MARK, SDA3_D_MARK, PWM3_B_MARK, A9_MARK,
	DU0_DG2_MARK, RX4_D_MARK, A10_MARK,
	DU0_DG3_MARK, TX4_D_MARK, PWM4_B_MARK, A11_MARK,
	DU0_DG4_MARK, HRX0_A_MARK, A12_MARK,
	DU0_DG5_MARK, HTX0_A_MARK, PWM5_B_MARK, A13_MARK,
	DU0_DG6_MARK, HRX1_C_MARK, A14_MARK,

	/* IPSR6 */
	DU0_DG7_MARK, HTX1_C_MARK, PWM6_B_MARK, A15_MARK,
	DU0_DB0_MARK, SCL4_D_MARK, CAN0_RX_C_MARK, A16_MARK,
	DU0_DB1_MARK, SDA4_D_MARK, CAN0_TX_C_MARK, A17_MARK,
	DU0_DB2_MARK, HCTS0_N_MARK, A18_MARK,
	DU0_DB3_MARK, HRTS0_N_MARK, A19_MARK,
	DU0_DB4_MARK, HCTS1_N_C_MARK, A20_MARK,
	DU0_DB5_MARK, HRTS1_N_C_MARK, A21_MARK,
	DU0_DB6_MARK, A22_MARK,

	/* IPSR7 */
	DU0_DB7_MARK, A23_MARK,
	DU0_DOTCLKIN_MARK, A24_MARK,
	DU0_DOTCLKOUT0_MARK, A25_MARK,
	DU0_DOTCLKOUT1_MARK, MSIOF2_RXD_B_MARK, CS1_N_A26_MARK,
	DU0_EXHSYNC_DU0_HSYNC_MARK, MSIOF2_TXD_B_MARK, DREQ0_N_MARK,
	DU0_EXVSYNC_DU0_VSYNC_MARK, MSIOF2_SYNC_B_MARK, DACK0_MARK,
	DU0_EXODDF_DU0_ODDF_DISP_CDE_MARK, MSIOF2_SCK_B_MARK, DRACK0_MARK,
	DU0_DISP_MARK, CAN1_RX_C_MARK,

	/* IPSR8 */
	DU0_CDE_MARK, CAN1_TX_C_MARK,
	VI1_CLK_MARK, AVB_RX_CLK_MARK, ETH_REF_CLK_MARK,
	VI1_DATA0_MARK, AVB_RX_DV_MARK, ETH_CRS_DV_MARK,
	VI1_DATA1_MARK, AVB_RXD0_MARK, ETH_RXD0_MARK,
	VI1_DATA2_MARK, AVB_RXD1_MARK, ETH_RXD1_MARK,
	VI1_DATA3_MARK, AVB_RXD2_MARK, ETH_MDIO_MARK,
	VI1_DATA4_MARK, AVB_RXD3_MARK, ETH_RX_ER_MARK,
	VI1_DATA5_MARK, AVB_RXD4_MARK, ETH_LINK_MARK,

	/* IPSR9 */
	VI1_DATA6_MARK, AVB_RXD5_MARK, ETH_TXD1_MARK,
	VI1_DATA7_MARK,	AVB_RXD6_MARK, ETH_TX_EN_MARK,
	VI1_CLKENB_MARK, SCL3_A_MARK, AVB_RXD7_MARK, ETH_MAGIC_MARK,
	VI1_FIELD_MARK, SDA3_A_MARK, AVB_RX_ER_MARK, ETH_TXD0_MARK,
	VI1_HSYNC_N_MARK, RX0_B_MARK, SCL0_C_MARK, AVB_GTXREFCLK_MARK, ETH_MDC_MARK,
	VI1_VSYNC_N_MARK, TX0_B_MARK, SDA0_C_MARK, AUDIO_CLKOUT_B_MARK, AVB_TX_CLK_MARK,
	VI1_DATA8_MARK, SCL2_B_MARK, AVB_TX_EN_MARK,
	VI1_DATA9_MARK, SDA2_B_MARK, AVB_TXD0_MARK,

	/* IPSR10 */
	VI1_DATA10_MARK, CAN0_RX_B_MARK, AVB_TXD1_MARK,
	VI1_DATA11_MARK, CAN0_TX_B_MARK, AVB_TXD2_MARK,
	AVB_TXD3_MARK, AUDIO_CLKA_B_MARK, SSI_SCK1_D_MARK, RX5_F_MARK, MSIOF0_RXD_B_MARK,
	AVB_TXD4_MARK, AUDIO_CLKB_B_MARK, SSI_WS1_D_MARK, TX5_F_MARK, MSIOF0_TXD_B_MARK,
	AVB_TXD5_MARK, SCIF_CLK_B_MARK, AUDIO_CLKC_B_MARK, SSI_SDATA1_D_MARK, MSIOF0_SCK_B_MARK,
	SCL0_A_MARK, RX0_C_MARK, PWM5_A_MARK, TCLK1_B_MARK, AVB_TXD6_MARK, CAN1_RX_D_MARK, MSIOF0_SYNC_B_MARK,
	SDA0_A_MARK, TX0_C_MARK, IRQ5_MARK, CAN_CLK_A_MARK, AVB_GTX_CLK_MARK, CAN1_TX_D_MARK, DVC_MUTE_MARK,
	SCL1_A_MARK, RX4_A_MARK, PWM5_D_MARK, DU1_DR0_MARK, SSI_SCK6_B_MARK, VI0_G0_MARK,

	/* IPSR11 */
	SDA1_A_MARK, TX4_A_MARK, DU1_DR1_MARK, SSI_WS6_B_MARK, VI0_G1_MARK,
	MSIOF0_RXD_A_MARK, RX5_A_MARK, SCL2_C_MARK, DU1_DR2_MARK, QSPI1_MOSI_QSPI1_IO0_MARK, SSI_SDATA6_B_MARK, VI0_G2_MARK,
	MSIOF0_TXD_A_MARK, TX5_A_MARK, SDA2_C_MARK, DU1_DR3_MARK, QSPI1_MISO_QSPI1_IO1_MARK, SSI_WS78_B_MARK, VI0_G3_MARK,
	MSIOF0_SCK_A_MARK, IRQ0_MARK, DU1_DR4_MARK, QSPI1_SPCLK_MARK, SSI_SCK78_B_MARK, VI0_G4_MARK,
	MSIOF0_SYNC_A_MARK, PWM1_A_MARK, DU1_DR5_MARK, QSPI1_IO2_MARK, SSI_SDATA7_B_MARK,
	MSIOF0_SS1_A_MARK, DU1_DR6_MARK, QSPI1_IO3_MARK, SSI_SDATA8_B_MARK,
	MSIOF0_SS2_A_MARK, DU1_DR7_MARK, QSPI1_SSL_MARK,
	HRX1_A_MARK, SCL4_A_MARK, PWM6_A_MARK, DU1_DG0_MARK, RX0_A_MARK,

	/* IPSR12 */
	HTX1_A_MARK, SDA4_A_MARK, DU1_DG1_MARK, TX0_A_MARK,
	HCTS1_N_A_MARK, PWM2_A_MARK, DU1_DG2_MARK, REMOCON_B_MARK,
	HRTS1_N_A_MARK, DU1_DG3_MARK, SSI_WS1_B_MARK, IRQ1_MARK,
	SD2_CLK_MARK, HSCK1_MARK, DU1_DG4_MARK, SSI_SCK1_B_MARK,
	SD2_CMD_MARK, SCIF1_SCK_A_MARK, TCLK2_A_MARK, DU1_DG5_MARK, SSI_SCK2_B_MARK, PWM3_A_MARK,
	SD2_DAT0_MARK, RX1_A_MARK, SCL1_E_MARK, DU1_DG6_MARK, SSI_SDATA1_B_MARK,
	SD2_DAT1_MARK, TX1_A_MARK, SDA1_E_MARK, DU1_DG7_MARK, SSI_WS2_B_MARK,
	SD2_DAT2_MARK, RX2_A_MARK, DU1_DB0_MARK, SSI_SDATA2_B_MARK,

	/* IPSR13 */
	SD2_DAT3_MARK, TX2_A_MARK, DU1_DB1_MARK, SSI_WS9_B_MARK,
	SD2_CD_MARK, SCIF2_SCK_A_MARK, DU1_DB2_MARK, SSI_SCK9_B_MARK,
	SD2_WP_MARK, SCIF3_SCK_MARK, DU1_DB3_MARK, SSI_SDATA9_B_MARK,
	RX3_A_MARK, SCL1_C_MARK, MSIOF1_RXD_B_MARK, DU1_DB4_MARK, AUDIO_CLKA_C_MARK, SSI_SDATA4_B_MARK,
	TX3_A_MARK, SDA1_C_MARK, MSIOF1_TXD_B_MARK, DU1_DB5_MARK, AUDIO_CLKB_C_MARK, SSI_WS4_B_MARK,
	SCL2_A_MARK, MSIOF1_SCK_B_MARK, DU1_DB6_MARK, AUDIO_CLKC_C_MARK, SSI_SCK4_B_MARK,
	SDA2_A_MARK, MSIOF1_SYNC_B_MARK, DU1_DB7_MARK, AUDIO_CLKOUT_C_MARK,
	SSI_SCK5_A_MARK, DU1_DOTCLKOUT1_MARK,

	/* IPSR14 */
	SSI_WS5_A_MARK, SCL3_C_MARK, DU1_DOTCLKIN_MARK,
	SSI_SDATA5_A_MARK, SDA3_C_MARK, DU1_DOTCLKOUT0_MARK,
	SSI_SCK6_A_MARK, DU1_EXODDF_DU1_ODDF_DISP_CDE_MARK,
	SSI_WS6_A_MARK, SCL4_C_MARK, DU1_EXHSYNC_DU1_HSYNC_MARK,
	SSI_SDATA6_A_MARK, SDA4_C_MARK, DU1_EXVSYNC_DU1_VSYNC_MARK,
	SSI_SCK78_A_MARK, SDA4_E_MARK, DU1_DISP_MARK,
	SSI_WS78_A_MARK, SCL4_E_MARK, DU1_CDE_MARK,
	SSI_SDATA7_A_MARK, IRQ8_MARK, AUDIO_CLKA_D_MARK, CAN_CLK_D_MARK, VI0_G5_MARK,

	/* IPSR15 */
	SSI_SCK0129_A_MARK, MSIOF1_RXD_A_MARK, RX5_D_MARK, VI0_G6_MARK,
	SSI_WS0129_A_MARK, MSIOF1_TXD_A_MARK, TX5_D_MARK, VI0_G7_MARK,
	SSI_SDATA0_A_MARK, MSIOF1_SYNC_A_MARK, PWM0_C_MARK, VI0_R0_MARK,
	SSI_SCK34_MARK, MSIOF1_SCK_A_MARK, AVB_MDC_MARK, DACK1_MARK, VI0_R1_MARK,
	SSI_WS34_MARK, MSIOF1_SS1_A_MARK, AVB_MDIO_MARK, CAN1_RX_A_MARK, DREQ1_N_MARK, VI0_R2_MARK,
	SSI_SDATA3_MARK, MSIOF1_SS2_A_MARK, AVB_LINK_MARK, CAN1_TX_A_MARK, DREQ2_N_MARK, VI0_R3_MARK,
	SSI_SCK4_A_MARK, AVB_MAGIC_MARK, VI0_R4_MARK,
	SSI_WS4_A_MARK, AVB_PHY_INT_MARK, VI0_R5_MARK,

	/* IPSR16 */
	SSI_SDATA4_A_MARK, AVB_CRS_MARK, VI0_R6_MARK,
	SSI_SCK1_A_MARK, SCIF1_SCK_B_MARK, PWM1_D_MARK, IRQ9_MARK, REMOCON_A_MARK, DACK2_MARK, VI0_CLK_MARK, AVB_COL_MARK,
	SSI_SDATA8_A_MARK, RX1_B_MARK, CAN0_RX_D_MARK, AVB_AVTP_CAPTURE_B_MARK, VI0_R7_MARK,
	SSI_WS1_A_MARK,	TX1_B_MARK, CAN0_TX_D_MARK, AVB_AVTP_MATCH_B_MARK, VI0_DATA0_VI0_B0_MARK,
	SSI_SDATA1_A_MARK, HRX1_B_MARK, VI0_DATA1_VI0_B1_MARK,
	SSI_SCK2_A_MARK, HTX1_B_MARK, AVB_TXD7_MARK, VI0_DATA2_VI0_B2_MARK,
	SSI_WS2_A_MARK, HCTS1_N_B_MARK, AVB_TX_ER_MARK, VI0_DATA3_VI0_B3_MARK,
	SSI_SDATA2_A_MARK, HRTS1_N_B_MARK, VI0_DATA4_VI0_B4_MARK,

	/* IPSR17 */
	SSI_SCK9_A_MARK, RX2_B_MARK, SCL3_E_MARK, EX_WAIT1_MARK, VI0_DATA5_VI0_B5_MARK,
	SSI_WS9_A_MARK, TX2_B_MARK, SDA3_E_MARK, VI0_DATA6_VI0_B6_MARK,
	SSI_SDATA9_A_MARK, SCIF2_SCK_B_MARK, PWM2_D_MARK, VI0_DATA7_VI0_B7_MARK,
	AUDIO_CLKA_A_MARK, SCL0_B_MARK, VI0_CLKENB_MARK,
	AUDIO_CLKB_A_MARK, SDA0_B_MARK,	VI0_FIELD_MARK,
	AUDIO_CLKC_A_MARK, SCL4_B_MARK, VI0_HSYNC_N_MARK,
	AUDIO_CLKOUT_A_MARK, SDA4_B_MARK, VI0_VSYNC_N_MARK,

	PINMUX_MARK_END,
};

static const u16 pinmux_data[] = {
	PINMUX_DATA_GP_ALL(), /* PINMUX_DATA(GP_M_N_DATA, GP_M_N_FN...), */

	PINMUX_SINGLE(USB0_PWEN),
	PINMUX_SINGLE(USB0_OVC),
	PINMUX_SINGLE(USB1_PWEN),
	PINMUX_SINGLE(USB1_OVC),
	PINMUX_SINGLE(CLKOUT),
	PINMUX_SINGLE(MMC0_CLK_SDHI1_CLK),
	PINMUX_SINGLE(MMC0_CMD_SDHI1_CMD),
	PINMUX_SINGLE(MMC0_D0_SDHI1_D0),
	PINMUX_SINGLE(MMC0_D1_SDHI1_D1),
	PINMUX_SINGLE(MMC0_D2_SDHI1_D2),
	PINMUX_SINGLE(MMC0_D3_SDHI1_D3),
	PINMUX_SINGLE(MMC0_D6),
	PINMUX_SINGLE(MMC0_D7),

	/* IPSR0 */
	PINMUX_IPSR_GPSR(IP0_3_0, SD0_CLK),
	PINMUX_IPSR_MSEL(IP0_3_0, SSI_SCK1_C, SEL_SSI1_2),
	PINMUX_IPSR_MSEL(IP0_3_0, RX3_C, SEL_SCIF3_2),
	PINMUX_IPSR_GPSR(IP0_7_4, SD0_CMD),
	PINMUX_IPSR_MSEL(IP0_7_4, SSI_WS1_C, SEL_SSI1_2),
	PINMUX_IPSR_MSEL(IP0_7_4, TX3_C, SEL_SCIF3_2),
	PINMUX_IPSR_GPSR(IP0_11_8, SD0_DAT0),
	PINMUX_IPSR_MSEL(IP0_11_8, SSI_SDATA1_C, SEL_SSI1_2),
	PINMUX_IPSR_MSEL(IP0_11_8, RX4_E, SEL_SCIF4_4),
	PINMUX_IPSR_GPSR(IP0_15_12, SD0_DAT1),
	PINMUX_IPSR_MSEL(IP0_15_12, SSI_SCK0129_B, SEL_SSI0_1),
	PINMUX_IPSR_MSEL(IP0_15_12, TX4_E, SEL_SCIF4_4),
	PINMUX_IPSR_GPSR(IP0_19_16, SD0_DAT2),
	PINMUX_IPSR_MSEL(IP0_19_16, SSI_WS0129_B, SEL_SSI0_1),
	PINMUX_IPSR_MSEL(IP0_19_16, RX5_E, SEL_SCIF5_4),
	PINMUX_IPSR_GPSR(IP0_23_20, SD0_DAT3),
	PINMUX_IPSR_MSEL(IP0_23_20, SSI_SDATA0_B, SEL_SSI0_1),
	PINMUX_IPSR_MSEL(IP0_23_20, TX5_E, SEL_SCIF5_4),
	PINMUX_IPSR_GPSR(IP0_27_24, SD0_CD),
	PINMUX_IPSR_MSEL(IP0_27_24, CAN0_RX_A, SEL_CAN0_0),
	PINMUX_IPSR_GPSR(IP0_31_28, SD0_WP),
	PINMUX_IPSR_GPSR(IP0_31_28, IRQ7),
	PINMUX_IPSR_MSEL(IP0_31_28, CAN0_TX_A, SEL_CAN0_0),

	/* IPSR1 */
	PINMUX_IPSR_GPSR(IP1_3_0, MMC0_D4),
	PINMUX_IPSR_GPSR(IP1_3_0, SD1_CD),
	PINMUX_IPSR_GPSR(IP1_7_4, MMC0_D5),
	PINMUX_IPSR_GPSR(IP1_7_4, SD1_WP),
	PINMUX_IPSR_GPSR(IP1_11_8, D0),
	PINMUX_IPSR_MSEL(IP1_11_8, SCL3_B, SEL_I2C03_1),
	PINMUX_IPSR_MSEL(IP1_11_8, RX5_B, SEL_SCIF5_1),
	PINMUX_IPSR_GPSR(IP1_11_8, IRQ4),
	PINMUX_IPSR_MSEL(IP1_11_8, MSIOF2_RXD_C, SEL_MSIOF2_2),
	PINMUX_IPSR_MSEL(IP1_11_8, SSI_SDATA5_B, SEL_SSI5_1),
	PINMUX_IPSR_GPSR(IP1_15_12, D1),
	PINMUX_IPSR_MSEL(IP1_15_12, SDA3_B, SEL_I2C03_1),
	PINMUX_IPSR_MSEL(IP1_15_12, TX5_B, SEL_SCIF5_1),
	PINMUX_IPSR_MSEL(IP1_15_12, MSIOF2_TXD_C, SEL_MSIOF2_2),
	PINMUX_IPSR_MSEL(IP1_15_12, SSI_WS5_B, SEL_SSI5_1),
	PINMUX_IPSR_GPSR(IP1_19_16, D2),
	PINMUX_IPSR_MSEL(IP1_19_16, RX4_B, SEL_SCIF4_1),
	PINMUX_IPSR_MSEL(IP1_19_16, SCL0_D, SEL_I2C00_3),
	PINMUX_IPSR_GPSR(IP1_19_16, PWM1_C),
	PINMUX_IPSR_MSEL(IP1_19_16, MSIOF2_SCK_C, SEL_MSIOF2_2),
	PINMUX_IPSR_MSEL(IP1_19_16, SSI_SCK5_B, SEL_SSI5_1),
	PINMUX_IPSR_GPSR(IP1_23_20, D3),
	PINMUX_IPSR_MSEL(IP1_23_20, TX4_B, SEL_SCIF4_1),
	PINMUX_IPSR_MSEL(IP1_23_20, SDA0_D, SEL_I2C00_3),
	PINMUX_IPSR_GPSR(IP1_23_20, PWM0_A),
	PINMUX_IPSR_MSEL(IP1_23_20, MSIOF2_SYNC_C, SEL_MSIOF2_2),
	PINMUX_IPSR_GPSR(IP1_27_24, D4),
	PINMUX_IPSR_GPSR(IP1_27_24, IRQ3),
	PINMUX_IPSR_MSEL(IP1_27_24, TCLK1_A, SEL_TMU1_0),
	PINMUX_IPSR_GPSR(IP1_27_24, PWM6_C),
	PINMUX_IPSR_GPSR(IP1_31_28, D5),
	PINMUX_IPSR_GPSR(IP1_31_28, HRX2),
	PINMUX_IPSR_MSEL(IP1_31_28, SCL1_B, SEL_I2C01_1),
	PINMUX_IPSR_GPSR(IP1_31_28, PWM2_C),
	PINMUX_IPSR_MSEL(IP1_31_28, TCLK2_B, SEL_TMU2_1),

	/* IPSR2 */
	PINMUX_IPSR_GPSR(IP2_3_0, D6),
	PINMUX_IPSR_GPSR(IP2_3_0, HTX2),
	PINMUX_IPSR_MSEL(IP2_3_0, SDA1_B, SEL_I2C01_1),
	PINMUX_IPSR_GPSR(IP2_3_0, PWM4_C),
	PINMUX_IPSR_GPSR(IP2_7_4, D7),
	PINMUX_IPSR_GPSR(IP2_7_4, HSCK2),
	PINMUX_IPSR_MSEL(IP2_7_4, SCIF1_SCK_C, SEL_SCIF1_2),
	PINMUX_IPSR_GPSR(IP2_7_4, IRQ6),
	PINMUX_IPSR_GPSR(IP2_7_4, PWM5_C),
	PINMUX_IPSR_GPSR(IP2_11_8, D8),
	PINMUX_IPSR_GPSR(IP2_11_8, HCTS2_N),
	PINMUX_IPSR_MSEL(IP2_11_8, RX1_C, SEL_SCIF1_2),
	PINMUX_IPSR_MSEL(IP2_11_8, SCL1_D, SEL_I2C01_3),
	PINMUX_IPSR_GPSR(IP2_11_8, PWM3_C),
	PINMUX_IPSR_GPSR(IP2_15_12, D9),
	PINMUX_IPSR_GPSR(IP2_15_12, HRTS2_N),
	PINMUX_IPSR_MSEL(IP2_15_12, TX1_C, SEL_SCIF1_2),
	PINMUX_IPSR_MSEL(IP2_15_12, SDA1_D, SEL_I2C01_3),
	PINMUX_IPSR_GPSR(IP2_19_16, D10),
	PINMUX_IPSR_MSEL(IP2_19_16, MSIOF2_RXD_A, SEL_MSIOF2_0),
	PINMUX_IPSR_MSEL(IP2_19_16, HRX0_B, SEL_HSCIF0_1),
	PINMUX_IPSR_GPSR(IP2_23_20, D11),
	PINMUX_IPSR_MSEL(IP2_23_20, MSIOF2_TXD_A, SEL_MSIOF2_0),
	PINMUX_IPSR_MSEL(IP2_23_20, HTX0_B, SEL_HSCIF0_1),
	PINMUX_IPSR_GPSR(IP2_27_24, D12),
	PINMUX_IPSR_MSEL(IP2_27_24, MSIOF2_SCK_A, SEL_MSIOF2_0),
	PINMUX_IPSR_GPSR(IP2_27_24, HSCK0),
	PINMUX_IPSR_MSEL(IP2_27_24, CAN_CLK_C, SEL_CANCLK_2),
	PINMUX_IPSR_GPSR(IP2_31_28, D13),
	PINMUX_IPSR_MSEL(IP2_31_28, MSIOF2_SYNC_A, SEL_MSIOF2_0),
	PINMUX_IPSR_MSEL(IP2_31_28, RX4_C, SEL_SCIF4_2),

	/* IPSR3 */
	PINMUX_IPSR_GPSR(IP3_3_0, D14),
	PINMUX_IPSR_GPSR(IP3_3_0, MSIOF2_SS1),
	PINMUX_IPSR_MSEL(IP3_3_0, TX4_C, SEL_SCIF4_2),
	PINMUX_IPSR_MSEL(IP3_3_0, CAN1_RX_B, SEL_CAN1_1),
	PINMUX_IPSR_MSEL(IP3_3_0, AVB_AVTP_CAPTURE_A, SEL_AVB_0),
	PINMUX_IPSR_GPSR(IP3_7_4, D15),
	PINMUX_IPSR_GPSR(IP3_7_4, MSIOF2_SS2),
	PINMUX_IPSR_GPSR(IP3_7_4, PWM4_A),
	PINMUX_IPSR_MSEL(IP3_7_4, CAN1_TX_B, SEL_CAN1_1),
	PINMUX_IPSR_GPSR(IP3_7_4, IRQ2),
	PINMUX_IPSR_MSEL(IP3_7_4, AVB_AVTP_MATCH_A, SEL_AVB_0),
	PINMUX_IPSR_GPSR(IP3_11_8, QSPI0_SPCLK),
	PINMUX_IPSR_GPSR(IP3_11_8, WE0_N),
	PINMUX_IPSR_GPSR(IP3_15_12, QSPI0_MOSI_QSPI0_IO0),
	PINMUX_IPSR_GPSR(IP3_15_12, BS_N),
	PINMUX_IPSR_GPSR(IP3_19_16, QSPI0_MISO_QSPI0_IO1),
	PINMUX_IPSR_GPSR(IP3_19_16, RD_WR_N),
	PINMUX_IPSR_GPSR(IP3_23_20, QSPI0_IO2),
	PINMUX_IPSR_GPSR(IP3_23_20, CS0_N),
	PINMUX_IPSR_GPSR(IP3_27_24, QSPI0_IO3),
	PINMUX_IPSR_GPSR(IP3_27_24, RD_N),
	PINMUX_IPSR_GPSR(IP3_31_28, QSPI0_SSL),
	PINMUX_IPSR_GPSR(IP3_31_28, WE1_N),

	/* IPSR4 */
	PINMUX_IPSR_GPSR(IP4_3_0, EX_WAIT0),
	PINMUX_IPSR_MSEL(IP4_3_0, CAN_CLK_B, SEL_CANCLK_1),
	PINMUX_IPSR_MSEL(IP4_3_0, SCIF_CLK_A, SEL_SCIFCLK_0),
	PINMUX_IPSR_GPSR(IP4_7_4, DU0_DR0),
	PINMUX_IPSR_MSEL(IP4_7_4, RX5_C, SEL_SCIF5_2),
	PINMUX_IPSR_MSEL(IP4_7_4, SCL2_D, SEL_I2C02_3),
	PINMUX_IPSR_GPSR(IP4_7_4, A0),
	PINMUX_IPSR_GPSR(IP4_11_8, DU0_DR1),
	PINMUX_IPSR_MSEL(IP4_11_8, TX5_C, SEL_SCIF5_2),
	PINMUX_IPSR_MSEL(IP4_11_8, SDA2_D, SEL_I2C02_3),
	PINMUX_IPSR_GPSR(IP4_11_8, A1),
	PINMUX_IPSR_GPSR(IP4_15_12, DU0_DR2),
	PINMUX_IPSR_MSEL(IP4_15_12, RX0_D, SEL_SCIF0_3),
	PINMUX_IPSR_MSEL(IP4_15_12, SCL0_E, SEL_I2C00_4),
	PINMUX_IPSR_GPSR(IP4_15_12, A2),
	PINMUX_IPSR_GPSR(IP4_19_16, DU0_DR3),
	PINMUX_IPSR_MSEL(IP4_19_16, TX0_D, SEL_SCIF0_3),
	PINMUX_IPSR_MSEL(IP4_19_16, SDA0_E, SEL_I2C00_4),
	PINMUX_IPSR_GPSR(IP4_19_16, PWM0_B),
	PINMUX_IPSR_GPSR(IP4_19_16, A3),
	PINMUX_IPSR_GPSR(IP4_23_20, DU0_DR4),
	PINMUX_IPSR_MSEL(IP4_23_20, RX1_D, SEL_SCIF1_3),
	PINMUX_IPSR_GPSR(IP4_23_20, A4),
	PINMUX_IPSR_GPSR(IP4_27_24, DU0_DR5),
	PINMUX_IPSR_MSEL(IP4_27_24, TX1_D, SEL_SCIF1_3),
	PINMUX_IPSR_GPSR(IP4_27_24, PWM1_B),
	PINMUX_IPSR_GPSR(IP4_27_24, A5),
	PINMUX_IPSR_GPSR(IP4_31_28, DU0_DR6),
	PINMUX_IPSR_MSEL(IP4_31_28, RX2_C, SEL_SCIF2_2),
	PINMUX_IPSR_GPSR(IP4_31_28, A6),

	/* IPSR5 */
	PINMUX_IPSR_GPSR(IP5_3_0, DU0_DR7),
	PINMUX_IPSR_MSEL(IP5_3_0, TX2_C, SEL_SCIF2_2),
	PINMUX_IPSR_GPSR(IP5_3_0, PWM2_B),
	PINMUX_IPSR_GPSR(IP5_3_0, A7),
	PINMUX_IPSR_GPSR(IP5_7_4, DU0_DG0),
	PINMUX_IPSR_MSEL(IP5_7_4, RX3_B, SEL_SCIF3_1),
	PINMUX_IPSR_MSEL(IP5_7_4, SCL3_D, SEL_I2C03_3),
	PINMUX_IPSR_GPSR(IP5_7_4, A8),
	PINMUX_IPSR_GPSR(IP5_11_8, DU0_DG1),
	PINMUX_IPSR_MSEL(IP5_11_8, TX3_B, SEL_SCIF3_1),
	PINMUX_IPSR_MSEL(IP5_11_8, SDA3_D, SEL_I2C03_3),
	PINMUX_IPSR_GPSR(IP5_11_8, PWM3_B),
	PINMUX_IPSR_GPSR(IP5_11_8, A9),
	PINMUX_IPSR_GPSR(IP5_15_12, DU0_DG2),
	PINMUX_IPSR_MSEL(IP5_15_12, RX4_D, SEL_SCIF4_3),
	PINMUX_IPSR_GPSR(IP5_15_12, A10),
	PINMUX_IPSR_GPSR(IP5_19_16, DU0_DG3),
	PINMUX_IPSR_MSEL(IP5_19_16, TX4_D, SEL_SCIF4_3),
	PINMUX_IPSR_GPSR(IP5_19_16, PWM4_B),
	PINMUX_IPSR_GPSR(IP5_19_16, A11),
	PINMUX_IPSR_GPSR(IP5_23_20, DU0_DG4),
	PINMUX_IPSR_MSEL(IP5_23_20, HRX0_A, SEL_HSCIF0_0),
	PINMUX_IPSR_GPSR(IP5_23_20, A12),
	PINMUX_IPSR_GPSR(IP5_27_24, DU0_DG5),
	PINMUX_IPSR_MSEL(IP5_27_24, HTX0_A, SEL_HSCIF0_0),
	PINMUX_IPSR_GPSR(IP5_27_24, PWM5_B),
	PINMUX_IPSR_GPSR(IP5_27_24, A13),
	PINMUX_IPSR_GPSR(IP5_31_28, DU0_DG6),
	PINMUX_IPSR_MSEL(IP5_31_28, HRX1_C, SEL_HSCIF1_2),
	PINMUX_IPSR_GPSR(IP5_31_28, A14),

	/* IPSR6 */
	PINMUX_IPSR_GPSR(IP6_3_0, DU0_DG7),
	PINMUX_IPSR_MSEL(IP6_3_0, HTX1_C, SEL_HSCIF1_2),
	PINMUX_IPSR_GPSR(IP6_3_0, PWM6_B),
	PINMUX_IPSR_GPSR(IP6_3_0, A15),
	PINMUX_IPSR_GPSR(IP6_7_4, DU0_DB0),
	PINMUX_IPSR_MSEL(IP6_7_4, SCL4_D, SEL_I2C04_3),
	PINMUX_IPSR_MSEL(IP6_7_4, CAN0_RX_C, SEL_CAN0_2),
	PINMUX_IPSR_GPSR(IP6_7_4, A16),
	PINMUX_IPSR_GPSR(IP6_11_8, DU0_DB1),
	PINMUX_IPSR_MSEL(IP6_11_8, SDA4_D, SEL_I2C04_3),
	PINMUX_IPSR_MSEL(IP6_11_8, CAN0_TX_C, SEL_CAN0_2),
	PINMUX_IPSR_GPSR(IP6_11_8, A17),
	PINMUX_IPSR_GPSR(IP6_15_12, DU0_DB2),
	PINMUX_IPSR_GPSR(IP6_15_12, HCTS0_N),
	PINMUX_IPSR_GPSR(IP6_15_12, A18),
	PINMUX_IPSR_GPSR(IP6_19_16, DU0_DB3),
	PINMUX_IPSR_GPSR(IP6_19_16, HRTS0_N),
	PINMUX_IPSR_GPSR(IP6_19_16, A19),
	PINMUX_IPSR_GPSR(IP6_23_20, DU0_DB4),
	PINMUX_IPSR_MSEL(IP6_23_20, HCTS1_N_C, SEL_HSCIF1_2),
	PINMUX_IPSR_GPSR(IP6_23_20, A20),
	PINMUX_IPSR_GPSR(IP6_27_24, DU0_DB5),
	PINMUX_IPSR_MSEL(IP6_27_24, HRTS1_N_C, SEL_HSCIF1_2),
	PINMUX_IPSR_GPSR(IP6_27_24, A21),
	PINMUX_IPSR_GPSR(IP6_31_28, DU0_DB6),
	PINMUX_IPSR_GPSR(IP6_31_28, A22),

	/* IPSR7 */
	PINMUX_IPSR_GPSR(IP7_3_0, DU0_DB7),
	PINMUX_IPSR_GPSR(IP7_3_0, A23),
	PINMUX_IPSR_GPSR(IP7_7_4, DU0_DOTCLKIN),
	PINMUX_IPSR_GPSR(IP7_7_4, A24),
	PINMUX_IPSR_GPSR(IP7_11_8, DU0_DOTCLKOUT0),
	PINMUX_IPSR_GPSR(IP7_11_8, A25),
	PINMUX_IPSR_GPSR(IP7_15_12, DU0_DOTCLKOUT1),
	PINMUX_IPSR_MSEL(IP7_15_12, MSIOF2_RXD_B, SEL_MSIOF2_1),
	PINMUX_IPSR_GPSR(IP7_15_12, CS1_N_A26),
	PINMUX_IPSR_GPSR(IP7_19_16, DU0_EXHSYNC_DU0_HSYNC),
	PINMUX_IPSR_MSEL(IP7_19_16, MSIOF2_TXD_B, SEL_MSIOF2_1),
	PINMUX_IPSR_GPSR(IP7_19_16, DREQ0_N),
	PINMUX_IPSR_GPSR(IP7_23_20, DU0_EXVSYNC_DU0_VSYNC),
	PINMUX_IPSR_MSEL(IP7_23_20, MSIOF2_SYNC_B, SEL_MSIOF2_1),
	PINMUX_IPSR_GPSR(IP7_23_20, DACK0),
	PINMUX_IPSR_GPSR(IP7_27_24, DU0_EXODDF_DU0_ODDF_DISP_CDE),
	PINMUX_IPSR_MSEL(IP7_27_24, MSIOF2_SCK_B, SEL_MSIOF2_1),
	PINMUX_IPSR_GPSR(IP7_27_24, DRACK0),
	PINMUX_IPSR_GPSR(IP7_31_28, DU0_DISP),
	PINMUX_IPSR_MSEL(IP7_31_28, CAN1_RX_C, SEL_CAN1_2),

	/* IPSR8 */
	PINMUX_IPSR_GPSR(IP8_3_0, DU0_CDE),
	PINMUX_IPSR_MSEL(IP8_3_0, CAN1_TX_C, SEL_CAN1_2),
	PINMUX_IPSR_GPSR(IP8_7_4, VI1_CLK),
	PINMUX_IPSR_GPSR(IP8_7_4, AVB_RX_CLK),
	PINMUX_IPSR_GPSR(IP8_7_4, ETH_REF_CLK),
	PINMUX_IPSR_GPSR(IP8_11_8, VI1_DATA0),
	PINMUX_IPSR_GPSR(IP8_11_8, AVB_RX_DV),
	PINMUX_IPSR_GPSR(IP8_11_8, ETH_CRS_DV),
	PINMUX_IPSR_GPSR(IP8_15_12, VI1_DATA1),
	PINMUX_IPSR_GPSR(IP8_15_12, AVB_RXD0),
	PINMUX_IPSR_GPSR(IP8_15_12, ETH_RXD0),
	PINMUX_IPSR_GPSR(IP8_19_16, VI1_DATA2),
	PINMUX_IPSR_GPSR(IP8_19_16, AVB_RXD1),
	PINMUX_IPSR_GPSR(IP8_19_16, ETH_RXD1),
	PINMUX_IPSR_GPSR(IP8_23_20, VI1_DATA3),
	PINMUX_IPSR_GPSR(IP8_23_20, AVB_RXD2),
	PINMUX_IPSR_GPSR(IP8_23_20, ETH_MDIO),
	PINMUX_IPSR_GPSR(IP8_27_24, VI1_DATA4),
	PINMUX_IPSR_GPSR(IP8_27_24, AVB_RXD3),
	PINMUX_IPSR_GPSR(IP8_27_24, ETH_RX_ER),
	PINMUX_IPSR_GPSR(IP8_31_28, VI1_DATA5),
	PINMUX_IPSR_GPSR(IP8_31_28, AVB_RXD4),
	PINMUX_IPSR_GPSR(IP8_31_28, ETH_LINK),

	/* IPSR9 */
	PINMUX_IPSR_GPSR(IP9_3_0, VI1_DATA6),
	PINMUX_IPSR_GPSR(IP9_3_0, AVB_RXD5),
	PINMUX_IPSR_GPSR(IP9_3_0, ETH_TXD1),
	PINMUX_IPSR_GPSR(IP9_7_4, VI1_DATA7),
	PINMUX_IPSR_GPSR(IP9_7_4, AVB_RXD6),
	PINMUX_IPSR_GPSR(IP9_7_4, ETH_TX_EN),
	PINMUX_IPSR_GPSR(IP9_11_8, VI1_CLKENB),
	PINMUX_IPSR_MSEL(IP9_11_8, SCL3_A, SEL_I2C03_0),
	PINMUX_IPSR_GPSR(IP9_11_8, AVB_RXD7),
	PINMUX_IPSR_GPSR(IP9_11_8, ETH_MAGIC),
	PINMUX_IPSR_GPSR(IP9_15_12, VI1_FIELD),
	PINMUX_IPSR_MSEL(IP9_15_12, SDA3_A, SEL_I2C03_0),
	PINMUX_IPSR_GPSR(IP9_15_12, AVB_RX_ER),
	PINMUX_IPSR_GPSR(IP9_15_12, ETH_TXD0),
	PINMUX_IPSR_GPSR(IP9_19_16, VI1_HSYNC_N),
	PINMUX_IPSR_MSEL(IP9_19_16, RX0_B, SEL_SCIF0_1),
	PINMUX_IPSR_MSEL(IP9_19_16, SCL0_C, SEL_I2C00_2),
	PINMUX_IPSR_GPSR(IP9_19_16, AVB_GTXREFCLK),
	PINMUX_IPSR_GPSR(IP9_19_16, ETH_MDC),
	PINMUX_IPSR_GPSR(IP9_23_20, VI1_VSYNC_N),
	PINMUX_IPSR_MSEL(IP9_23_20, TX0_B, SEL_SCIF0_1),
	PINMUX_IPSR_MSEL(IP9_23_20, SDA0_C, SEL_I2C00_2),
	PINMUX_IPSR_GPSR(IP9_23_20, AUDIO_CLKOUT_B),
	PINMUX_IPSR_GPSR(IP9_23_20, AVB_TX_CLK),
	PINMUX_IPSR_GPSR(IP9_27_24, VI1_DATA8),
	PINMUX_IPSR_MSEL(IP9_27_24, SCL2_B, SEL_I2C02_1),
	PINMUX_IPSR_GPSR(IP9_27_24, AVB_TX_EN),
	PINMUX_IPSR_GPSR(IP9_31_28, VI1_DATA9),
	PINMUX_IPSR_MSEL(IP9_31_28, SDA2_B, SEL_I2C02_1),
	PINMUX_IPSR_GPSR(IP9_31_28, AVB_TXD0),

	/* IPSR10 */
	PINMUX_IPSR_GPSR(IP10_3_0, VI1_DATA10),
	PINMUX_IPSR_MSEL(IP10_3_0, CAN0_RX_B, SEL_CAN0_1),
	PINMUX_IPSR_GPSR(IP10_3_0, AVB_TXD1),
	PINMUX_IPSR_GPSR(IP10_7_4, VI1_DATA11),
	PINMUX_IPSR_MSEL(IP10_7_4, CAN0_TX_B, SEL_CAN0_1),
	PINMUX_IPSR_GPSR(IP10_7_4, AVB_TXD2),
	PINMUX_IPSR_GPSR(IP10_11_8, AVB_TXD3),
	PINMUX_IPSR_MSEL(IP10_11_8, AUDIO_CLKA_B, SEL_ADGA_1),
	PINMUX_IPSR_MSEL(IP10_11_8, SSI_SCK1_D, SEL_SSI1_3),
	PINMUX_IPSR_MSEL(IP10_11_8, RX5_F, SEL_SCIF5_5),
	PINMUX_IPSR_MSEL(IP10_11_8, MSIOF0_RXD_B, SEL_MSIOF0_1),
	PINMUX_IPSR_GPSR(IP10_15_12, AVB_TXD4),
	PINMUX_IPSR_MSEL(IP10_15_12, AUDIO_CLKB_B, SEL_ADGB_1),
	PINMUX_IPSR_MSEL(IP10_15_12, SSI_WS1_D, SEL_SSI1_3),
	PINMUX_IPSR_MSEL(IP10_15_12, TX5_F, SEL_SCIF5_5),
	PINMUX_IPSR_MSEL(IP10_15_12, MSIOF0_TXD_B, SEL_MSIOF0_1),
	PINMUX_IPSR_GPSR(IP10_19_16, AVB_TXD5),
	PINMUX_IPSR_MSEL(IP10_19_16, SCIF_CLK_B, SEL_SCIFCLK_1),
	PINMUX_IPSR_MSEL(IP10_19_16, AUDIO_CLKC_B, SEL_ADGC_1),
	PINMUX_IPSR_MSEL(IP10_19_16, SSI_SDATA1_D, SEL_SSI1_3),
	PINMUX_IPSR_MSEL(IP10_19_16, MSIOF0_SCK_B, SEL_MSIOF0_1),
	PINMUX_IPSR_MSEL(IP10_23_20, SCL0_A, SEL_I2C00_0),
	PINMUX_IPSR_MSEL(IP10_23_20, RX0_C, SEL_SCIF0_2),
	PINMUX_IPSR_GPSR(IP10_23_20, PWM5_A),
	PINMUX_IPSR_MSEL(IP10_23_20, TCLK1_B, SEL_TMU1_1),
	PINMUX_IPSR_GPSR(IP10_23_20, AVB_TXD6),
	PINMUX_IPSR_MSEL(IP10_23_20, CAN1_RX_D, SEL_CAN1_3),
	PINMUX_IPSR_MSEL(IP10_23_20, MSIOF0_SYNC_B, SEL_MSIOF0_1),
	PINMUX_IPSR_MSEL(IP10_27_24, SDA0_A, SEL_I2C00_0),
	PINMUX_IPSR_MSEL(IP10_27_24, TX0_C, SEL_SCIF0_2),
	PINMUX_IPSR_GPSR(IP10_27_24, IRQ5),
	PINMUX_IPSR_MSEL(IP10_27_24, CAN_CLK_A, SEL_CANCLK_0),
	PINMUX_IPSR_GPSR(IP10_27_24, AVB_GTX_CLK),
	PINMUX_IPSR_MSEL(IP10_27_24, CAN1_TX_D, SEL_CAN1_3),
	PINMUX_IPSR_GPSR(IP10_27_24, DVC_MUTE),
	PINMUX_IPSR_MSEL(IP10_31_28, SCL1_A, SEL_I2C01_0),
	PINMUX_IPSR_MSEL(IP10_31_28, RX4_A, SEL_SCIF4_0),
	PINMUX_IPSR_GPSR(IP10_31_28, PWM5_D),
	PINMUX_IPSR_GPSR(IP10_31_28, DU1_DR0),
	PINMUX_IPSR_MSEL(IP10_31_28, SSI_SCK6_B, SEL_SSI6_1),
	PINMUX_IPSR_GPSR(IP10_31_28, VI0_G0),

	/* IPSR11 */
	PINMUX_IPSR_MSEL(IP11_3_0, SDA1_A, SEL_I2C01_0),
	PINMUX_IPSR_MSEL(IP11_3_0, TX4_A, SEL_SCIF4_0),
	PINMUX_IPSR_GPSR(IP11_3_0, DU1_DR1),
	PINMUX_IPSR_MSEL(IP11_3_0, SSI_WS6_B, SEL_SSI6_1),
	PINMUX_IPSR_GPSR(IP11_3_0, VI0_G1),
	PINMUX_IPSR_MSEL(IP11_7_4, MSIOF0_RXD_A, SEL_MSIOF0_0),
	PINMUX_IPSR_MSEL(IP11_7_4, RX5_A, SEL_SCIF5_0),
	PINMUX_IPSR_MSEL(IP11_7_4, SCL2_C, SEL_I2C02_2),
	PINMUX_IPSR_GPSR(IP11_7_4, DU1_DR2),
	PINMUX_IPSR_GPSR(IP11_7_4, QSPI1_MOSI_QSPI1_IO0),
	PINMUX_IPSR_MSEL(IP11_7_4, SSI_SDATA6_B, SEL_SSI6_1),
	PINMUX_IPSR_GPSR(IP11_7_4, VI0_G2),
	PINMUX_IPSR_MSEL(IP11_11_8, MSIOF0_TXD_A, SEL_MSIOF0_0),
	PINMUX_IPSR_MSEL(IP11_11_8, TX5_A, SEL_SCIF5_0),
	PINMUX_IPSR_MSEL(IP11_11_8, SDA2_C, SEL_I2C02_2),
	PINMUX_IPSR_GPSR(IP11_11_8, DU1_DR3),
	PINMUX_IPSR_GPSR(IP11_11_8, QSPI1_MISO_QSPI1_IO1),
	PINMUX_IPSR_MSEL(IP11_11_8, SSI_WS78_B, SEL_SSI7_1),
	PINMUX_IPSR_GPSR(IP11_11_8, VI0_G3),
	PINMUX_IPSR_MSEL(IP11_15_12, MSIOF0_SCK_A, SEL_MSIOF0_0),
	PINMUX_IPSR_GPSR(IP11_15_12, IRQ0),
	PINMUX_IPSR_GPSR(IP11_15_12, DU1_DR4),
	PINMUX_IPSR_GPSR(IP11_15_12, QSPI1_SPCLK),
	PINMUX_IPSR_MSEL(IP11_15_12, SSI_SCK78_B, SEL_SSI7_1),
	PINMUX_IPSR_GPSR(IP11_15_12, VI0_G4),
	PINMUX_IPSR_MSEL(IP11_19_16, MSIOF0_SYNC_A, SEL_MSIOF0_0),
	PINMUX_IPSR_GPSR(IP11_19_16, PWM1_A),
	PINMUX_IPSR_GPSR(IP11_19_16, DU1_DR5),
	PINMUX_IPSR_GPSR(IP11_19_16, QSPI1_IO2),
	PINMUX_IPSR_MSEL(IP11_19_16, SSI_SDATA7_B, SEL_SSI7_1),
	PINMUX_IPSR_MSEL(IP11_23_20, MSIOF0_SS1_A, SEL_MSIOF0_0),
	PINMUX_IPSR_GPSR(IP11_23_20, DU1_DR6),
	PINMUX_IPSR_GPSR(IP11_23_20, QSPI1_IO3),
	PINMUX_IPSR_MSEL(IP11_23_20, SSI_SDATA8_B, SEL_SSI8_1),
	PINMUX_IPSR_MSEL(IP11_27_24, MSIOF0_SS2_A, SEL_MSIOF0_0),
	PINMUX_IPSR_GPSR(IP11_27_24, DU1_DR7),
	PINMUX_IPSR_GPSR(IP11_27_24, QSPI1_SSL),
	PINMUX_IPSR_MSEL(IP11_31_28, HRX1_A, SEL_HSCIF1_0),
	PINMUX_IPSR_MSEL(IP11_31_28, SCL4_A, SEL_I2C04_0),
	PINMUX_IPSR_GPSR(IP11_31_28, PWM6_A),
	PINMUX_IPSR_GPSR(IP11_31_28, DU1_DG0),
	PINMUX_IPSR_MSEL(IP11_31_28, RX0_A, SEL_SCIF0_0),

	/* IPSR12 */
	PINMUX_IPSR_MSEL(IP12_3_0, HTX1_A, SEL_HSCIF1_0),
	PINMUX_IPSR_MSEL(IP12_3_0, SDA4_A, SEL_I2C04_0),
	PINMUX_IPSR_GPSR(IP12_3_0, DU1_DG1),
	PINMUX_IPSR_MSEL(IP12_3_0, TX0_A, SEL_SCIF0_0),
	PINMUX_IPSR_MSEL(IP12_7_4, HCTS1_N_A, SEL_HSCIF1_0),
	PINMUX_IPSR_GPSR(IP12_7_4, PWM2_A),
	PINMUX_IPSR_GPSR(IP12_7_4, DU1_DG2),
	PINMUX_IPSR_MSEL(IP12_7_4, REMOCON_B, SEL_RCN_1),
	PINMUX_IPSR_MSEL(IP12_11_8, HRTS1_N_A, SEL_HSCIF1_0),
	PINMUX_IPSR_GPSR(IP12_11_8, DU1_DG3),
	PINMUX_IPSR_MSEL(IP12_11_8, SSI_WS1_B, SEL_SSI1_1),
	PINMUX_IPSR_GPSR(IP12_11_8, IRQ1),
	PINMUX_IPSR_GPSR(IP12_15_12, SD2_CLK),
	PINMUX_IPSR_GPSR(IP12_15_12, HSCK1),
	PINMUX_IPSR_GPSR(IP12_15_12, DU1_DG4),
	PINMUX_IPSR_MSEL(IP12_15_12, SSI_SCK1_B, SEL_SSI1_1),
	PINMUX_IPSR_GPSR(IP12_19_16, SD2_CMD),
	PINMUX_IPSR_MSEL(IP12_19_16, SCIF1_SCK_A, SEL_SCIF1_0),
	PINMUX_IPSR_MSEL(IP12_19_16, TCLK2_A, SEL_TMU2_0),
	PINMUX_IPSR_GPSR(IP12_19_16, DU1_DG5),
	PINMUX_IPSR_MSEL(IP12_19_16, SSI_SCK2_B, SEL_SSI2_1),
	PINMUX_IPSR_GPSR(IP12_19_16, PWM3_A),
	PINMUX_IPSR_GPSR(IP12_23_20, SD2_DAT0),
	PINMUX_IPSR_MSEL(IP12_23_20, RX1_A, SEL_SCIF1_0),
	PINMUX_IPSR_MSEL(IP12_23_20, SCL1_E, SEL_I2C01_4),
	PINMUX_IPSR_GPSR(IP12_23_20, DU1_DG6),
	PINMUX_IPSR_MSEL(IP12_23_20, SSI_SDATA1_B, SEL_SSI1_1),
	PINMUX_IPSR_GPSR(IP12_27_24, SD2_DAT1),
	PINMUX_IPSR_MSEL(IP12_27_24, TX1_A, SEL_SCIF1_0),
	PINMUX_IPSR_MSEL(IP12_27_24, SDA1_E, SEL_I2C01_4),
	PINMUX_IPSR_GPSR(IP12_27_24, DU1_DG7),
	PINMUX_IPSR_MSEL(IP12_27_24, SSI_WS2_B, SEL_SSI2_1),
	PINMUX_IPSR_GPSR(IP12_31_28, SD2_DAT2),
	PINMUX_IPSR_MSEL(IP12_31_28, RX2_A, SEL_SCIF2_0),
	PINMUX_IPSR_GPSR(IP12_31_28, DU1_DB0),
	PINMUX_IPSR_MSEL(IP12_31_28, SSI_SDATA2_B, SEL_SSI2_1),

	/* IPSR13 */
	PINMUX_IPSR_GPSR(IP13_3_0, SD2_DAT3),
	PINMUX_IPSR_MSEL(IP13_3_0, TX2_A, SEL_SCIF2_0),
	PINMUX_IPSR_GPSR(IP13_3_0, DU1_DB1),
	PINMUX_IPSR_MSEL(IP13_3_0, SSI_WS9_B, SEL_SSI9_1),
	PINMUX_IPSR_GPSR(IP13_7_4, SD2_CD),
	PINMUX_IPSR_MSEL(IP13_7_4, SCIF2_SCK_A, SEL_SCIF2_CLK_0),
	PINMUX_IPSR_GPSR(IP13_7_4, DU1_DB2),
	PINMUX_IPSR_MSEL(IP13_7_4, SSI_SCK9_B, SEL_SSI9_1),
	PINMUX_IPSR_GPSR(IP13_11_8, SD2_WP),
	PINMUX_IPSR_GPSR(IP13_11_8, SCIF3_SCK),
	PINMUX_IPSR_GPSR(IP13_11_8, DU1_DB3),
	PINMUX_IPSR_MSEL(IP13_11_8, SSI_SDATA9_B, SEL_SSI9_1),
	PINMUX_IPSR_MSEL(IP13_15_12, RX3_A, SEL_SCIF3_0),
	PINMUX_IPSR_MSEL(IP13_15_12, SCL1_C, SEL_I2C01_2),
	PINMUX_IPSR_MSEL(IP13_15_12, MSIOF1_RXD_B, SEL_MSIOF1_1),
	PINMUX_IPSR_GPSR(IP13_15_12, DU1_DB4),
	PINMUX_IPSR_MSEL(IP13_15_12, AUDIO_CLKA_C, SEL_ADGA_2),
	PINMUX_IPSR_MSEL(IP13_15_12, SSI_SDATA4_B, SEL_SSI4_1),
	PINMUX_IPSR_MSEL(IP13_19_16, TX3_A, SEL_SCIF3_0),
	PINMUX_IPSR_MSEL(IP13_19_16, SDA1_C, SEL_I2C01_2),
	PINMUX_IPSR_MSEL(IP13_19_16, MSIOF1_TXD_B, SEL_MSIOF1_1),
	PINMUX_IPSR_GPSR(IP13_19_16, DU1_DB5),
	PINMUX_IPSR_MSEL(IP13_19_16, AUDIO_CLKB_C, SEL_ADGB_2),
	PINMUX_IPSR_MSEL(IP13_19_16, SSI_WS4_B, SEL_SSI4_1),
	PINMUX_IPSR_MSEL(IP13_23_20, SCL2_A, SEL_I2C02_0),
	PINMUX_IPSR_MSEL(IP13_23_20, MSIOF1_SCK_B, SEL_MSIOF1_1),
	PINMUX_IPSR_GPSR(IP13_23_20, DU1_DB6),
	PINMUX_IPSR_MSEL(IP13_23_20, AUDIO_CLKC_C, SEL_ADGC_2),
	PINMUX_IPSR_MSEL(IP13_23_20, SSI_SCK4_B, SEL_SSI4_1),
	PINMUX_IPSR_MSEL(IP13_27_24, SDA2_A, SEL_I2C02_0),
	PINMUX_IPSR_MSEL(IP13_27_24, MSIOF1_SYNC_B, SEL_MSIOF1_1),
	PINMUX_IPSR_GPSR(IP13_27_24, DU1_DB7),
	PINMUX_IPSR_GPSR(IP13_27_24, AUDIO_CLKOUT_C),
	PINMUX_IPSR_MSEL(IP13_31_28, SSI_SCK5_A, SEL_SSI5_0),
	PINMUX_IPSR_GPSR(IP13_31_28, DU1_DOTCLKOUT1),

	/* IPSR14 */
	PINMUX_IPSR_MSEL(IP14_3_0, SSI_WS5_A, SEL_SSI5_0),
	PINMUX_IPSR_MSEL(IP14_3_0, SCL3_C, SEL_I2C03_2),
	PINMUX_IPSR_GPSR(IP14_3_0, DU1_DOTCLKIN),
	PINMUX_IPSR_MSEL(IP14_7_4, SSI_SDATA5_A, SEL_SSI5_0),
	PINMUX_IPSR_MSEL(IP14_7_4, SDA3_C, SEL_I2C03_2),
	PINMUX_IPSR_GPSR(IP14_7_4, DU1_DOTCLKOUT0),
	PINMUX_IPSR_MSEL(IP14_11_8, SSI_SCK6_A, SEL_SSI6_0),
	PINMUX_IPSR_GPSR(IP14_11_8, DU1_EXODDF_DU1_ODDF_DISP_CDE),
	PINMUX_IPSR_MSEL(IP14_15_12, SSI_WS6_A, SEL_SSI6_0),
	PINMUX_IPSR_MSEL(IP14_15_12, SCL4_C, SEL_I2C04_2),
	PINMUX_IPSR_GPSR(IP14_15_12, DU1_EXHSYNC_DU1_HSYNC),
	PINMUX_IPSR_MSEL(IP14_19_16, SSI_SDATA6_A, SEL_SSI6_0),
	PINMUX_IPSR_MSEL(IP14_19_16, SDA4_C, SEL_I2C04_2),
	PINMUX_IPSR_GPSR(IP14_19_16, DU1_EXVSYNC_DU1_VSYNC),
	PINMUX_IPSR_MSEL(IP14_23_20, SSI_SCK78_A, SEL_SSI7_0),
	PINMUX_IPSR_MSEL(IP14_23_20, SDA4_E, SEL_I2C04_4),
	PINMUX_IPSR_GPSR(IP14_23_20, DU1_DISP),
	PINMUX_IPSR_MSEL(IP14_27_24, SSI_WS78_A, SEL_SSI7_0),
	PINMUX_IPSR_MSEL(IP14_27_24, SCL4_E, SEL_I2C04_4),
	PINMUX_IPSR_GPSR(IP14_27_24, DU1_CDE),
	PINMUX_IPSR_MSEL(IP14_31_28, SSI_SDATA7_A, SEL_SSI7_0),
	PINMUX_IPSR_GPSR(IP14_31_28, IRQ8),
	PINMUX_IPSR_MSEL(IP14_31_28, AUDIO_CLKA_D, SEL_ADGA_3),
	PINMUX_IPSR_MSEL(IP14_31_28, CAN_CLK_D, SEL_CANCLK_3),
	PINMUX_IPSR_GPSR(IP14_31_28, VI0_G5),

	/* IPSR15 */
	PINMUX_IPSR_MSEL(IP15_3_0, SSI_SCK0129_A, SEL_SSI0_0),
	PINMUX_IPSR_MSEL(IP15_3_0, MSIOF1_RXD_A, SEL_MSIOF1_0),
	PINMUX_IPSR_MSEL(IP15_3_0, RX5_D, SEL_SCIF5_3),
	PINMUX_IPSR_GPSR(IP15_3_0, VI0_G6),
	PINMUX_IPSR_MSEL(IP15_7_4, SSI_WS0129_A, SEL_SSI0_0),
	PINMUX_IPSR_MSEL(IP15_7_4, MSIOF1_TXD_A, SEL_MSIOF1_0),
	PINMUX_IPSR_MSEL(IP15_7_4, TX5_D, SEL_SCIF5_3),
	PINMUX_IPSR_GPSR(IP15_7_4, VI0_G7),
	PINMUX_IPSR_MSEL(IP15_11_8, SSI_SDATA0_A, SEL_SSI0_0),
	PINMUX_IPSR_MSEL(IP15_11_8, MSIOF1_SYNC_A, SEL_MSIOF1_0),
	PINMUX_IPSR_GPSR(IP15_11_8, PWM0_C),
	PINMUX_IPSR_GPSR(IP15_11_8, VI0_R0),
	PINMUX_IPSR_GPSR(IP15_15_12, SSI_SCK34),
	PINMUX_IPSR_MSEL(IP15_15_12, MSIOF1_SCK_A, SEL_MSIOF1_0),
	PINMUX_IPSR_GPSR(IP15_15_12, AVB_MDC),
	PINMUX_IPSR_GPSR(IP15_15_12, DACK1),
	PINMUX_IPSR_GPSR(IP15_15_12, VI0_R1),
	PINMUX_IPSR_GPSR(IP15_19_16, SSI_WS34),
	PINMUX_IPSR_MSEL(IP15_19_16, MSIOF1_SS1_A, SEL_MSIOF1_0),
	PINMUX_IPSR_GPSR(IP15_19_16, AVB_MDIO),
	PINMUX_IPSR_MSEL(IP15_19_16, CAN1_RX_A, SEL_CAN1_0),
	PINMUX_IPSR_GPSR(IP15_19_16, DREQ1_N),
	PINMUX_IPSR_GPSR(IP15_19_16, VI0_R2),
	PINMUX_IPSR_GPSR(IP15_23_20, SSI_SDATA3),
	PINMUX_IPSR_MSEL(IP15_23_20, MSIOF1_SS2_A, SEL_MSIOF1_0),
	PINMUX_IPSR_GPSR(IP15_23_20, AVB_LINK),
	PINMUX_IPSR_MSEL(IP15_23_20, CAN1_TX_A, SEL_CAN1_0),
	PINMUX_IPSR_GPSR(IP15_23_20, DREQ2_N),
	PINMUX_IPSR_GPSR(IP15_23_20, VI0_R3),
	PINMUX_IPSR_MSEL(IP15_27_24, SSI_SCK4_A, SEL_SSI4_0),
	PINMUX_IPSR_GPSR(IP15_27_24, AVB_MAGIC),
	PINMUX_IPSR_GPSR(IP15_27_24, VI0_R4),
	PINMUX_IPSR_MSEL(IP15_31_28, SSI_WS4_A, SEL_SSI4_0),
	PINMUX_IPSR_GPSR(IP15_31_28, AVB_PHY_INT),
	PINMUX_IPSR_GPSR(IP15_31_28, VI0_R5),

	/* IPSR16 */
	PINMUX_IPSR_MSEL(IP16_3_0, SSI_SDATA4_A, SEL_SSI4_0),
	PINMUX_IPSR_GPSR(IP16_3_0, AVB_CRS),
	PINMUX_IPSR_GPSR(IP16_3_0, VI0_R6),
	PINMUX_IPSR_MSEL(IP16_7_4, SSI_SCK1_A, SEL_SSI1_0),
	PINMUX_IPSR_MSEL(IP16_7_4, SCIF1_SCK_B, SEL_SCIF1_1),
	PINMUX_IPSR_GPSR(IP16_7_4, PWM1_D),
	PINMUX_IPSR_GPSR(IP16_7_4, IRQ9),
	PINMUX_IPSR_MSEL(IP16_7_4, REMOCON_A, SEL_RCN_0),
	PINMUX_IPSR_GPSR(IP16_7_4, DACK2),
	PINMUX_IPSR_GPSR(IP16_7_4, VI0_CLK),
	PINMUX_IPSR_GPSR(IP16_7_4, AVB_COL),
	PINMUX_IPSR_MSEL(IP16_11_8, SSI_SDATA8_A, SEL_SSI8_0),
	PINMUX_IPSR_MSEL(IP16_11_8, RX1_B, SEL_SCIF1_1),
	PINMUX_IPSR_MSEL(IP16_11_8, CAN0_RX_D, SEL_CAN0_3),
	PINMUX_IPSR_MSEL(IP16_11_8, AVB_AVTP_CAPTURE_B, SEL_AVB_1),
	PINMUX_IPSR_GPSR(IP16_11_8, VI0_R7),
	PINMUX_IPSR_MSEL(IP16_15_12, SSI_WS1_A, SEL_SSI1_0),
	PINMUX_IPSR_MSEL(IP16_15_12, TX1_B, SEL_SCIF1_1),
	PINMUX_IPSR_MSEL(IP16_15_12, CAN0_TX_D, SEL_CAN0_3),
	PINMUX_IPSR_MSEL(IP16_15_12, AVB_AVTP_MATCH_B, SEL_AVB_1),
	PINMUX_IPSR_GPSR(IP16_15_12, VI0_DATA0_VI0_B0),
	PINMUX_IPSR_MSEL(IP16_19_16, SSI_SDATA1_A, SEL_SSI1_0),
	PINMUX_IPSR_MSEL(IP16_19_16, HRX1_B, SEL_HSCIF1_1),
	PINMUX_IPSR_GPSR(IP16_19_16, VI0_DATA1_VI0_B1),
	PINMUX_IPSR_MSEL(IP16_23_20, SSI_SCK2_A, SEL_SSI2_0),
	PINMUX_IPSR_MSEL(IP16_23_20, HTX1_B, SEL_HSCIF1_1),
	PINMUX_IPSR_GPSR(IP16_23_20, AVB_TXD7),
	PINMUX_IPSR_GPSR(IP16_23_20, VI0_DATA2_VI0_B2),
	PINMUX_IPSR_MSEL(IP16_27_24, SSI_WS2_A, SEL_SSI2_0),
	PINMUX_IPSR_MSEL(IP16_27_24, HCTS1_N_B, SEL_HSCIF1_1),
	PINMUX_IPSR_GPSR(IP16_27_24, AVB_TX_ER),
	PINMUX_IPSR_GPSR(IP16_27_24, VI0_DATA3_VI0_B3),
	PINMUX_IPSR_MSEL(IP16_31_28, SSI_SDATA2_A, SEL_SSI2_0),
	PINMUX_IPSR_MSEL(IP16_31_28, HRTS1_N_B, SEL_HSCIF1_1),
	PINMUX_IPSR_GPSR(IP16_31_28, VI0_DATA4_VI0_B4),

	/* IPSR17 */
	PINMUX_IPSR_MSEL(IP17_3_0, SSI_SCK9_A, SEL_SSI9_0),
	PINMUX_IPSR_MSEL(IP17_3_0, RX2_B, SEL_SCIF2_1),
	PINMUX_IPSR_MSEL(IP17_3_0, SCL3_E, SEL_I2C03_4),
	PINMUX_IPSR_GPSR(IP17_3_0, EX_WAIT1),
	PINMUX_IPSR_GPSR(IP17_3_0, VI0_DATA5_VI0_B5),
	PINMUX_IPSR_MSEL(IP17_7_4, SSI_WS9_A, SEL_SSI9_0),
	PINMUX_IPSR_MSEL(IP17_7_4, TX2_B, SEL_SCIF2_1),
	PINMUX_IPSR_MSEL(IP17_7_4, SDA3_E, SEL_I2C03_4),
	PINMUX_IPSR_GPSR(IP17_7_4, VI0_DATA6_VI0_B6),
	PINMUX_IPSR_MSEL(IP17_11_8, SSI_SDATA9_A, SEL_SSI9_0),
	PINMUX_IPSR_GPSR(IP17_11_8, SCIF2_SCK_B),
	PINMUX_IPSR_GPSR(IP17_11_8, PWM2_D),
	PINMUX_IPSR_GPSR(IP17_11_8, VI0_DATA7_VI0_B7),
	PINMUX_IPSR_MSEL(IP17_15_12, AUDIO_CLKA_A, SEL_ADGA_0),
	PINMUX_IPSR_MSEL(IP17_15_12, SCL0_B, SEL_I2C00_1),
	PINMUX_IPSR_GPSR(IP17_15_12, VI0_CLKENB),
	PINMUX_IPSR_MSEL(IP17_19_16, AUDIO_CLKB_A, SEL_ADGB_0),
	PINMUX_IPSR_MSEL(IP17_19_16, SDA0_B, SEL_I2C00_1),
	PINMUX_IPSR_GPSR(IP17_19_16, VI0_FIELD),
	PINMUX_IPSR_MSEL(IP17_23_20, AUDIO_CLKC_A, SEL_ADGC_0),
	PINMUX_IPSR_MSEL(IP17_23_20, SCL4_B, SEL_I2C04_1),
	PINMUX_IPSR_GPSR(IP17_23_20, VI0_HSYNC_N),
	PINMUX_IPSR_GPSR(IP17_27_24, AUDIO_CLKOUT_A),
	PINMUX_IPSR_MSEL(IP17_27_24, SDA4_B, SEL_I2C04_1),
	PINMUX_IPSR_GPSR(IP17_27_24, VI0_VSYNC_N),
};

static const struct sh_pfc_pin pinmux_pins[] = {
	PINMUX_GPIO_GP_ALL(),
};

/* - AVB -------------------------------------------------------------------- */
static const unsigned int avb_col_pins[] = {
	RCAR_GP_PIN(5, 18),
};
static const unsigned int avb_col_mux[] = {
	AVB_COL_MARK,
};
static const unsigned int avb_crs_pins[] = {
	RCAR_GP_PIN(5, 17),
};
static const unsigned int avb_crs_mux[] = {
	AVB_CRS_MARK,
};
static const unsigned int avb_link_pins[] = {
	RCAR_GP_PIN(5, 14),
};
static const unsigned int avb_link_mux[] = {
	AVB_LINK_MARK,
};
static const unsigned int avb_magic_pins[] = {
	RCAR_GP_PIN(5, 15),
};
static const unsigned int avb_magic_mux[] = {
	AVB_MAGIC_MARK,
};
static const unsigned int avb_phy_int_pins[] = {
	RCAR_GP_PIN(5, 16),
};
static const unsigned int avb_phy_int_mux[] = {
	AVB_PHY_INT_MARK,
};
static const unsigned int avb_mdio_pins[] = {
	RCAR_GP_PIN(5, 12), RCAR_GP_PIN(5, 13),
};
static const unsigned int avb_mdio_mux[] = {
	AVB_MDC_MARK, AVB_MDIO_MARK,
};
static const unsigned int avb_mii_tx_rx_pins[] = {
	RCAR_GP_PIN(3, 12), RCAR_GP_PIN(3, 14), RCAR_GP_PIN(3, 15),
	RCAR_GP_PIN(3, 16), RCAR_GP_PIN(3, 27), RCAR_GP_PIN(3, 13),

	RCAR_GP_PIN(3, 0), RCAR_GP_PIN(3, 2), RCAR_GP_PIN(3, 3),
	RCAR_GP_PIN(3, 4), RCAR_GP_PIN(3, 5), RCAR_GP_PIN(3, 1),
	RCAR_GP_PIN(3, 10),
};
static const unsigned int avb_mii_tx_rx_mux[] = {
	AVB_TX_CLK_MARK, AVB_TXD0_MARK, AVB_TXD1_MARK, AVB_TXD2_MARK,
	AVB_TXD3_MARK, AVB_TX_EN_MARK,

	AVB_RX_CLK_MARK, AVB_RXD0_MARK, AVB_RXD1_MARK, AVB_RXD2_MARK,
	AVB_RXD3_MARK, AVB_RX_DV_MARK, AVB_RX_ER_MARK,
};
static const unsigned int avb_mii_tx_er_pins[] = {
	RCAR_GP_PIN(5, 23),
};
static const unsigned int avb_mii_tx_er_mux[] = {
	AVB_TX_ER_MARK,
};
static const unsigned int avb_gmii_tx_rx_pins[] = {
	RCAR_GP_PIN(4, 1), RCAR_GP_PIN(3, 11), RCAR_GP_PIN(3, 12),
	RCAR_GP_PIN(3, 14), RCAR_GP_PIN(3, 15), RCAR_GP_PIN(3, 16),
	RCAR_GP_PIN(3, 27), RCAR_GP_PIN(3, 28), RCAR_GP_PIN(3, 29),
	RCAR_GP_PIN(4, 0), RCAR_GP_PIN(5, 22), RCAR_GP_PIN(3, 13),
	RCAR_GP_PIN(5, 23),

	RCAR_GP_PIN(3, 0), RCAR_GP_PIN(3, 2), RCAR_GP_PIN(3, 3),
	RCAR_GP_PIN(3, 4), RCAR_GP_PIN(3, 5), RCAR_GP_PIN(3, 6),
	RCAR_GP_PIN(3, 7), RCAR_GP_PIN(3, 8), RCAR_GP_PIN(3, 9),
	RCAR_GP_PIN(3, 1), RCAR_GP_PIN(3, 10),
};
static const unsigned int avb_gmii_tx_rx_mux[] = {
	AVB_GTX_CLK_MARK, AVB_GTXREFCLK_MARK, AVB_TX_CLK_MARK, AVB_TXD0_MARK,
	AVB_TXD1_MARK, AVB_TXD2_MARK, AVB_TXD3_MARK, AVB_TXD4_MARK,
	AVB_TXD5_MARK, AVB_TXD6_MARK, AVB_TXD7_MARK, AVB_TX_EN_MARK,
	AVB_TX_ER_MARK,

	AVB_RX_CLK_MARK, AVB_RXD0_MARK, AVB_RXD1_MARK, AVB_RXD2_MARK,
	AVB_RXD3_MARK, AVB_RXD4_MARK, AVB_RXD5_MARK, AVB_RXD6_MARK,
	AVB_RXD7_MARK, AVB_RX_DV_MARK, AVB_RX_ER_MARK,
};
static const unsigned int avb_avtp_match_a_pins[] = {
	RCAR_GP_PIN(1, 15),
};
static const unsigned int avb_avtp_match_a_mux[] = {
	AVB_AVTP_MATCH_A_MARK,
};
static const unsigned int avb_avtp_capture_a_pins[] = {
	RCAR_GP_PIN(1, 14),
};
static const unsigned int avb_avtp_capture_a_mux[] = {
	AVB_AVTP_CAPTURE_A_MARK,
};
static const unsigned int avb_avtp_match_b_pins[] = {
	RCAR_GP_PIN(5, 20),
};
static const unsigned int avb_avtp_match_b_mux[] = {
	AVB_AVTP_MATCH_B_MARK,
};
static const unsigned int avb_avtp_capture_b_pins[] = {
	RCAR_GP_PIN(5, 19),
};
static const unsigned int avb_avtp_capture_b_mux[] = {
	AVB_AVTP_CAPTURE_B_MARK,
};
/* - DU --------------------------------------------------------------------- */
static const unsigned int du0_rgb666_pins[] = {
	/* R[7:2], G[7:2], B[7:2] */
	RCAR_GP_PIN(2, 7),  RCAR_GP_PIN(2, 6),  RCAR_GP_PIN(2, 5),
	RCAR_GP_PIN(2, 4),  RCAR_GP_PIN(2, 3),  RCAR_GP_PIN(2, 2),
	RCAR_GP_PIN(2, 15), RCAR_GP_PIN(2, 14), RCAR_GP_PIN(2, 13),
	RCAR_GP_PIN(2, 12), RCAR_GP_PIN(2, 11), RCAR_GP_PIN(2, 10),
	RCAR_GP_PIN(2, 23), RCAR_GP_PIN(2, 22), RCAR_GP_PIN(2, 21),
	RCAR_GP_PIN(2, 20), RCAR_GP_PIN(2, 19), RCAR_GP_PIN(2, 18),
};
static const unsigned int du0_rgb666_mux[] = {
	DU0_DR7_MARK, DU0_DR6_MARK, DU0_DR5_MARK, DU0_DR4_MARK,
	DU0_DR3_MARK, DU0_DR2_MARK,
	DU0_DG7_MARK, DU0_DG6_MARK, DU0_DG5_MARK, DU0_DG4_MARK,
	DU0_DG3_MARK, DU0_DG2_MARK,
	DU0_DB7_MARK, DU0_DB6_MARK, DU0_DB5_MARK, DU0_DB4_MARK,
	DU0_DB3_MARK, DU0_DB2_MARK,
};
static const unsigned int du0_rgb888_pins[] = {
	/* R[7:0], G[7:0], B[7:0] */
	RCAR_GP_PIN(2, 7),  RCAR_GP_PIN(2, 6),  RCAR_GP_PIN(2, 5),
	RCAR_GP_PIN(2, 4),  RCAR_GP_PIN(2, 3),  RCAR_GP_PIN(2, 2),
	RCAR_GP_PIN(2, 1),  RCAR_GP_PIN(2, 0),
	RCAR_GP_PIN(2, 15), RCAR_GP_PIN(2, 14), RCAR_GP_PIN(2, 13),
	RCAR_GP_PIN(2, 12), RCAR_GP_PIN(2, 11), RCAR_GP_PIN(2, 10),
	RCAR_GP_PIN(2, 9),  RCAR_GP_PIN(2, 8),
	RCAR_GP_PIN(2, 23), RCAR_GP_PIN(2, 22), RCAR_GP_PIN(2, 21),
	RCAR_GP_PIN(2, 20), RCAR_GP_PIN(2, 19), RCAR_GP_PIN(2, 18),
	RCAR_GP_PIN(2, 17), RCAR_GP_PIN(2, 16),
};
static const unsigned int du0_rgb888_mux[] = {
	DU0_DR7_MARK, DU0_DR6_MARK, DU0_DR5_MARK, DU0_DR4_MARK,
	DU0_DR3_MARK, DU0_DR2_MARK, DU0_DR1_MARK, DU0_DR0_MARK,
	DU0_DG7_MARK, DU0_DG6_MARK, DU0_DG5_MARK, DU0_DG4_MARK,
	DU0_DG3_MARK, DU0_DG2_MARK, DU0_DG1_MARK, DU0_DG0_MARK,
	DU0_DB7_MARK, DU0_DB6_MARK, DU0_DB5_MARK, DU0_DB4_MARK,
	DU0_DB3_MARK, DU0_DB2_MARK, DU0_DB1_MARK, DU0_DB0_MARK,
};
static const unsigned int du0_clk0_out_pins[] = {
	/* DOTCLKOUT0 */
	RCAR_GP_PIN(2, 25),
};
static const unsigned int du0_clk0_out_mux[] = {
	DU0_DOTCLKOUT0_MARK
};
static const unsigned int du0_clk1_out_pins[] = {
	/* DOTCLKOUT1 */
	RCAR_GP_PIN(2, 26),
};
static const unsigned int du0_clk1_out_mux[] = {
	DU0_DOTCLKOUT1_MARK
};
static const unsigned int du0_clk_in_pins[] = {
	/* CLKIN */
	RCAR_GP_PIN(2, 24),
};
static const unsigned int du0_clk_in_mux[] = {
	DU0_DOTCLKIN_MARK
};
static const unsigned int du0_sync_pins[] = {
	/* EXVSYNC/VSYNC, EXHSYNC/HSYNC */
	RCAR_GP_PIN(2, 28), RCAR_GP_PIN(2, 27),
};
static const unsigned int du0_sync_mux[] = {
	DU0_EXVSYNC_DU0_VSYNC_MARK, DU0_EXHSYNC_DU0_HSYNC_MARK
};
static const unsigned int du0_oddf_pins[] = {
	/* EXODDF/ODDF/DISP/CDE */
	RCAR_GP_PIN(2, 29),
};
static const unsigned int du0_oddf_mux[] = {
	DU0_EXODDF_DU0_ODDF_DISP_CDE_MARK,
};
static const unsigned int du0_cde_pins[] = {
	/* CDE */
	RCAR_GP_PIN(2, 31),
};
static const unsigned int du0_cde_mux[] = {
	DU0_CDE_MARK,
};
static const unsigned int du0_disp_pins[] = {
	/* DISP */
	RCAR_GP_PIN(2, 30),
};
static const unsigned int du0_disp_mux[] = {
	DU0_DISP_MARK
};
static const unsigned int du1_rgb666_pins[] = {
	/* R[7:2], G[7:2], B[7:2] */
	RCAR_GP_PIN(4, 9),  RCAR_GP_PIN(4, 8),  RCAR_GP_PIN(4, 7),
	RCAR_GP_PIN(4, 6),  RCAR_GP_PIN(4, 5),  RCAR_GP_PIN(4, 4),
	RCAR_GP_PIN(4, 17), RCAR_GP_PIN(4, 16), RCAR_GP_PIN(4, 15),
	RCAR_GP_PIN(4, 14), RCAR_GP_PIN(4, 13), RCAR_GP_PIN(4, 12),
	RCAR_GP_PIN(4, 25), RCAR_GP_PIN(4, 24), RCAR_GP_PIN(4, 23),
	RCAR_GP_PIN(4, 22), RCAR_GP_PIN(4, 21), RCAR_GP_PIN(4, 20),
};
static const unsigned int du1_rgb666_mux[] = {
	DU1_DR7_MARK, DU1_DR6_MARK, DU1_DR5_MARK, DU1_DR4_MARK,
	DU1_DR3_MARK, DU1_DR2_MARK,
	DU1_DG7_MARK, DU1_DG6_MARK, DU1_DG5_MARK, DU1_DG4_MARK,
	DU1_DG3_MARK, DU1_DG2_MARK,
	DU1_DB7_MARK, DU1_DB6_MARK, DU1_DB5_MARK, DU1_DB4_MARK,
	DU1_DB3_MARK, DU1_DB2_MARK,
};
static const unsigned int du1_rgb888_pins[] = {
	/* R[7:0], G[7:0], B[7:0] */
	RCAR_GP_PIN(4, 9),  RCAR_GP_PIN(4, 8),  RCAR_GP_PIN(4, 7),
	RCAR_GP_PIN(4, 6),  RCAR_GP_PIN(4, 5),  RCAR_GP_PIN(4, 4),
	RCAR_GP_PIN(4, 3),  RCAR_GP_PIN(4, 2),
	RCAR_GP_PIN(4, 17), RCAR_GP_PIN(4, 16), RCAR_GP_PIN(4, 15),
	RCAR_GP_PIN(4, 14), RCAR_GP_PIN(4, 13), RCAR_GP_PIN(4, 12),
	RCAR_GP_PIN(4, 11), RCAR_GP_PIN(4, 10),
	RCAR_GP_PIN(4, 25), RCAR_GP_PIN(4, 24), RCAR_GP_PIN(4, 23),
	RCAR_GP_PIN(4, 22), RCAR_GP_PIN(4, 21), RCAR_GP_PIN(4, 20),
	RCAR_GP_PIN(4, 19), RCAR_GP_PIN(4, 18),
};
static const unsigned int du1_rgb888_mux[] = {
	DU1_DR7_MARK, DU1_DR6_MARK, DU1_DR5_MARK, DU1_DR4_MARK,
	DU1_DR3_MARK, DU1_DR2_MARK, DU1_DR1_MARK, DU1_DR0_MARK,
	DU1_DG7_MARK, DU1_DG6_MARK, DU1_DG5_MARK, DU1_DG4_MARK,
	DU1_DG3_MARK, DU1_DG2_MARK, DU1_DG1_MARK, DU1_DG0_MARK,
	DU1_DB7_MARK, DU1_DB6_MARK, DU1_DB5_MARK, DU1_DB4_MARK,
	DU1_DB3_MARK, DU1_DB2_MARK, DU1_DB1_MARK, DU1_DB0_MARK,
};
static const unsigned int du1_clk0_out_pins[] = {
	/* DOTCLKOUT0 */
	RCAR_GP_PIN(5, 2),
};
static const unsigned int du1_clk0_out_mux[] = {
	DU1_DOTCLKOUT0_MARK
};
static const unsigned int du1_clk1_out_pins[] = {
	/* DOTCLKOUT1 */
	RCAR_GP_PIN(5, 0),
};
static const unsigned int du1_clk1_out_mux[] = {
	DU1_DOTCLKOUT1_MARK
};
static const unsigned int du1_clk_in_pins[] = {
	/* DOTCLKIN */
	RCAR_GP_PIN(5, 1),
};
static const unsigned int du1_clk_in_mux[] = {
	DU1_DOTCLKIN_MARK
};
static const unsigned int du1_sync_pins[] = {
	/* EXVSYNC/VSYNC, EXHSYNC/HSYNC */
	RCAR_GP_PIN(5, 5), RCAR_GP_PIN(5, 4),
};
static const unsigned int du1_sync_mux[] = {
	DU1_EXVSYNC_DU1_VSYNC_MARK, DU1_EXHSYNC_DU1_HSYNC_MARK
};
static const unsigned int du1_oddf_pins[] = {
	/* EXODDF/ODDF/DISP/CDE */
	RCAR_GP_PIN(5, 3),
};
static const unsigned int du1_oddf_mux[] = {
	DU1_EXODDF_DU1_ODDF_DISP_CDE_MARK,
};
static const unsigned int du1_cde_pins[] = {
	/* CDE */
	RCAR_GP_PIN(5, 7),
};
static const unsigned int du1_cde_mux[] = {
	DU1_CDE_MARK
};
static const unsigned int du1_disp_pins[] = {
	/* DISP */
	RCAR_GP_PIN(5, 6),
};
static const unsigned int du1_disp_mux[] = {
	DU1_DISP_MARK
};
/* - I2C0 ------------------------------------------------------------------- */
static const unsigned int i2c0_a_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 0), RCAR_GP_PIN(4, 1),
};
static const unsigned int i2c0_a_mux[] = {
	SCL0_A_MARK, SDA0_A_MARK,
};
static const unsigned int i2c0_b_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(5, 28), RCAR_GP_PIN(5, 29),
};
static const unsigned int i2c0_b_mux[] = {
	SCL0_B_MARK, SDA0_B_MARK,
};
static const unsigned int i2c0_c_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(3, 11), RCAR_GP_PIN(3, 12),
};
static const unsigned int i2c0_c_mux[] = {
	SCL0_C_MARK, SDA0_C_MARK,
};
static const unsigned int i2c0_d_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(1, 2), RCAR_GP_PIN(1, 3),
};
static const unsigned int i2c0_d_mux[] = {
	SCL0_D_MARK, SDA0_D_MARK,
};
static const unsigned int i2c0_e_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(2, 2), RCAR_GP_PIN(2, 3),
};
static const unsigned int i2c0_e_mux[] = {
	SCL0_E_MARK, SDA0_E_MARK,
};
/* - I2C1 ------------------------------------------------------------------- */
static const unsigned int i2c1_a_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 2), RCAR_GP_PIN(4, 3),
};
static const unsigned int i2c1_a_mux[] = {
	SCL1_A_MARK, SDA1_A_MARK,
};
static const unsigned int i2c1_b_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(1, 5), RCAR_GP_PIN(1, 6),
};
static const unsigned int i2c1_b_mux[] = {
	SCL1_B_MARK, SDA1_B_MARK,
};
static const unsigned int i2c1_c_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 22), RCAR_GP_PIN(4, 23),
};
static const unsigned int i2c1_c_mux[] = {
	SCL1_C_MARK, SDA1_C_MARK,
};
static const unsigned int i2c1_d_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(1, 8), RCAR_GP_PIN(1, 9),
};
static const unsigned int i2c1_d_mux[] = {
	SCL1_D_MARK, SDA1_D_MARK,
};
static const unsigned int i2c1_e_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 16), RCAR_GP_PIN(4, 17),
};
static const unsigned int i2c1_e_mux[] = {
	SCL1_E_MARK, SDA1_E_MARK,
};
/* - I2C2 ------------------------------------------------------------------- */
static const unsigned int i2c2_a_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 24), RCAR_GP_PIN(4, 25),
};
static const unsigned int i2c2_a_mux[] = {
	SCL2_A_MARK, SDA2_A_MARK,
};
static const unsigned int i2c2_b_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(3, 13), RCAR_GP_PIN(3, 14),
};
static const unsigned int i2c2_b_mux[] = {
	SCL2_B_MARK, SDA2_B_MARK,
};
static const unsigned int i2c2_c_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 4), RCAR_GP_PIN(4, 5),
};
static const unsigned int i2c2_c_mux[] = {
	SCL2_C_MARK, SDA2_C_MARK,
};
static const unsigned int i2c2_d_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(2, 0), RCAR_GP_PIN(2, 1),
};
static const unsigned int i2c2_d_mux[] = {
	SCL2_D_MARK, SDA2_D_MARK,
};
/* - I2C3 ------------------------------------------------------------------- */
static const unsigned int i2c3_a_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(3, 9), RCAR_GP_PIN(3, 10),
};
static const unsigned int i2c3_a_mux[] = {
	SCL3_A_MARK, SDA3_A_MARK,
};
static const unsigned int i2c3_b_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(1, 0), RCAR_GP_PIN(1, 1),
};
static const unsigned int i2c3_b_mux[] = {
	SCL3_B_MARK, SDA3_B_MARK,
};
static const unsigned int i2c3_c_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(5, 1), RCAR_GP_PIN(5, 2),
};
static const unsigned int i2c3_c_mux[] = {
	SCL3_C_MARK, SDA3_C_MARK,
};
static const unsigned int i2c3_d_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(2, 8), RCAR_GP_PIN(2, 9),
};
static const unsigned int i2c3_d_mux[] = {
	SCL3_D_MARK, SDA3_D_MARK,
};
static const unsigned int i2c3_e_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(5, 25), RCAR_GP_PIN(5, 26),
};
static const unsigned int i2c3_e_mux[] = {
	SCL3_E_MARK, SDA3_E_MARK,
};
/* - I2C4 ------------------------------------------------------------------- */
static const unsigned int i2c4_a_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(4, 10), RCAR_GP_PIN(4, 11),
};
static const unsigned int i2c4_a_mux[] = {
	SCL4_A_MARK, SDA4_A_MARK,
};
static const unsigned int i2c4_b_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(5, 30), RCAR_GP_PIN(5, 31),
};
static const unsigned int i2c4_b_mux[] = {
	SCL4_B_MARK, SDA4_B_MARK,
};
static const unsigned int i2c4_c_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(5, 4), RCAR_GP_PIN(5, 5),
};
static const unsigned int i2c4_c_mux[] = {
	SCL4_C_MARK, SDA4_C_MARK,
};
static const unsigned int i2c4_d_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(2, 16), RCAR_GP_PIN(2, 17),
};
static const unsigned int i2c4_d_mux[] = {
	SCL4_D_MARK, SDA4_D_MARK,
};
static const unsigned int i2c4_e_pins[] = {
	/* SCL, SDA */
	RCAR_GP_PIN(5, 7), RCAR_GP_PIN(5, 6),
};
static const unsigned int i2c4_e_mux[] = {
	SCL4_E_MARK, SDA4_E_MARK,
};
/* - MMC -------------------------------------------------------------------- */
static const unsigned int mmc_data1_pins[] = {
	/* D0 */
	RCAR_GP_PIN(0, 15),
};
static const unsigned int mmc_data1_mux[] = {
	MMC0_D0_SDHI1_D0_MARK,
};
static const unsigned int mmc_data4_pins[] = {
	/* D[0:3] */
	RCAR_GP_PIN(0, 15), RCAR_GP_PIN(0, 16),
	RCAR_GP_PIN(0, 17), RCAR_GP_PIN(0, 18),
};
static const unsigned int mmc_data4_mux[] = {
	MMC0_D0_SDHI1_D0_MARK, MMC0_D1_SDHI1_D1_MARK,
	MMC0_D2_SDHI1_D2_MARK, MMC0_D3_SDHI1_D3_MARK,
};
static const unsigned int mmc_data8_pins[] = {
	/* D[0:3] */
	RCAR_GP_PIN(0, 15), RCAR_GP_PIN(0, 16),
	RCAR_GP_PIN(0, 17), RCAR_GP_PIN(0, 18),
	RCAR_GP_PIN(0, 19), RCAR_GP_PIN(0, 20),
	RCAR_GP_PIN(0, 21), RCAR_GP_PIN(0, 22),
};
static const unsigned int mmc_data8_mux[] = {
	MMC0_D0_SDHI1_D0_MARK, MMC0_D1_SDHI1_D1_MARK,
	MMC0_D2_SDHI1_D2_MARK, MMC0_D3_SDHI1_D3_MARK,
	MMC0_D4_MARK, MMC0_D5_MARK,
	MMC0_D6_MARK, MMC0_D7_MARK,
};
static const unsigned int mmc_ctrl_pins[] = {
	/* CLK, CMD */
	RCAR_GP_PIN(0, 13), RCAR_GP_PIN(0, 14),
};
static const unsigned int mmc_ctrl_mux[] = {
	MMC0_CLK_SDHI1_CLK_MARK, MMC0_CMD_SDHI1_CMD_MARK,
};
/* - QSPI ------------------------------------------------------------------- */
static const unsigned int qspi0_ctrl_pins[] = {
	/* SPCLK, SSL */
	RCAR_GP_PIN(1, 16), RCAR_GP_PIN(1, 21),
};
static const unsigned int qspi0_ctrl_mux[] = {
	QSPI0_SPCLK_MARK, QSPI0_SSL_MARK,
};
static const unsigned int qspi0_data2_pins[] = {
	/* MOSI_IO0, MISO_IO1 */
	RCAR_GP_PIN(1, 17), RCAR_GP_PIN(1, 18),
};
static const unsigned int qspi0_data2_mux[] = {
	QSPI0_MOSI_QSPI0_IO0_MARK, QSPI0_MISO_QSPI0_IO1_MARK,
};
static const unsigned int qspi0_data4_pins[] = {
	/* MOSI_IO0, MISO_IO1, IO2, IO3 */
	RCAR_GP_PIN(1, 17), RCAR_GP_PIN(1, 18), RCAR_GP_PIN(1, 19),
	RCAR_GP_PIN(1, 20),
};
static const unsigned int qspi0_data4_mux[] = {
	QSPI0_MOSI_QSPI0_IO0_MARK, QSPI0_MISO_QSPI0_IO1_MARK,
	QSPI0_IO2_MARK, QSPI0_IO3_MARK,
};
static const unsigned int qspi1_ctrl_pins[] = {
	/* SPCLK, SSL */
	RCAR_GP_PIN(4, 6), RCAR_GP_PIN(4, 9),
};
static const unsigned int qspi1_ctrl_mux[] = {
	QSPI1_SPCLK_MARK, QSPI1_SSL_MARK,
};
static const unsigned int qspi1_data2_pins[] = {
	/* MOSI_IO0, MISO_IO1 */
	RCAR_GP_PIN(4, 4), RCAR_GP_PIN(4, 5),
};
static const unsigned int qspi1_data2_mux[] = {
	QSPI1_MOSI_QSPI1_IO0_MARK, QSPI1_MISO_QSPI1_IO1_MARK,
};
static const unsigned int qspi1_data4_pins[] = {
	/* MOSI_IO0, MISO_IO1, IO2, IO3 */
	RCAR_GP_PIN(4, 4), RCAR_GP_PIN(4, 5), RCAR_GP_PIN(4, 7),
	RCAR_GP_PIN(4, 8),
};
static const unsigned int qspi1_data4_mux[] = {
	QSPI1_MOSI_QSPI1_IO0_MARK, QSPI1_MISO_QSPI1_IO1_MARK,
	QSPI1_IO2_MARK, QSPI1_IO3_MARK,
};
/* - SCIF0 ------------------------------------------------------------------ */
static const unsigned int scif0_data_a_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 10), RCAR_GP_PIN(4, 11),
};
static const unsigned int scif0_data_a_mux[] = {
	RX0_A_MARK, TX0_A_MARK,
};
static const unsigned int scif0_data_b_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(3, 11), RCAR_GP_PIN(3, 12),
};
static const unsigned int scif0_data_b_mux[] = {
	RX0_B_MARK, TX0_B_MARK,
};
static const unsigned int scif0_data_c_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 0), RCAR_GP_PIN(4, 1),
};
static const unsigned int scif0_data_c_mux[] = {
	RX0_C_MARK, TX0_C_MARK,
};
static const unsigned int scif0_data_d_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(2, 2), RCAR_GP_PIN(2, 3),
};
static const unsigned int scif0_data_d_mux[] = {
	RX0_D_MARK, TX0_D_MARK,
};
/* - SCIF1 ------------------------------------------------------------------ */
static const unsigned int scif1_data_a_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 16), RCAR_GP_PIN(4, 17),
};
static const unsigned int scif1_data_a_mux[] = {
	RX1_A_MARK, TX1_A_MARK,
};
static const unsigned int scif1_clk_a_pins[] = {
	/* SCK */
	RCAR_GP_PIN(4, 15),
};
static const unsigned int scif1_clk_a_mux[] = {
	SCIF1_SCK_A_MARK,
};
static const unsigned int scif1_data_b_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(5, 19), RCAR_GP_PIN(5, 20),
};
static const unsigned int scif1_data_b_mux[] = {
	RX1_B_MARK, TX1_B_MARK,
};
static const unsigned int scif1_clk_b_pins[] = {
	/* SCK */
	RCAR_GP_PIN(5, 18),
};
static const unsigned int scif1_clk_b_mux[] = {
	SCIF1_SCK_B_MARK,
};
static const unsigned int scif1_data_c_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(1, 8), RCAR_GP_PIN(1, 9),
};
static const unsigned int scif1_data_c_mux[] = {
	RX1_C_MARK, TX1_C_MARK,
};
static const unsigned int scif1_clk_c_pins[] = {
	/* SCK */
	RCAR_GP_PIN(1, 7),
};
static const unsigned int scif1_clk_c_mux[] = {
	SCIF1_SCK_C_MARK,
};
static const unsigned int scif1_data_d_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(2, 4), RCAR_GP_PIN(2, 5),
};
static const unsigned int scif1_data_d_mux[] = {
	RX1_D_MARK, TX1_D_MARK,
};
/* - SCIF2 ------------------------------------------------------------------ */
static const unsigned int scif2_data_a_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 18), RCAR_GP_PIN(4, 19),
};
static const unsigned int scif2_data_a_mux[] = {
	RX2_A_MARK, TX2_A_MARK,
};
static const unsigned int scif2_clk_a_pins[] = {
	/* SCK */
	RCAR_GP_PIN(4, 20),
};
static const unsigned int scif2_clk_a_mux[] = {
	SCIF2_SCK_A_MARK,
};
static const unsigned int scif2_data_b_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(5, 25), RCAR_GP_PIN(5, 26),
};
static const unsigned int scif2_data_b_mux[] = {
	RX2_B_MARK, TX2_B_MARK,
};
static const unsigned int scif2_clk_b_pins[] = {
	/* SCK */
	RCAR_GP_PIN(5, 27),
};
static const unsigned int scif2_clk_b_mux[] = {
	SCIF2_SCK_B_MARK,
};
static const unsigned int scif2_data_c_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(2, 6), RCAR_GP_PIN(2, 7),
};
static const unsigned int scif2_data_c_mux[] = {
	RX2_C_MARK, TX2_C_MARK,
};
/* - SCIF3 ------------------------------------------------------------------ */
static const unsigned int scif3_data_a_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 22), RCAR_GP_PIN(4, 23),
};
static const unsigned int scif3_data_a_mux[] = {
	RX3_A_MARK, TX3_A_MARK,
};
static const unsigned int scif3_clk_pins[] = {
	/* SCK */
	RCAR_GP_PIN(4, 21),
};
static const unsigned int scif3_clk_mux[] = {
	SCIF3_SCK_MARK,
};
static const unsigned int scif3_data_b_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(2, 8), RCAR_GP_PIN(2, 9),
};
static const unsigned int scif3_data_b_mux[] = {
	RX3_B_MARK, TX3_B_MARK,
};
static const unsigned int scif3_data_c_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(0, 5), RCAR_GP_PIN(0, 6),
};
static const unsigned int scif3_data_c_mux[] = {
	RX3_C_MARK, TX3_C_MARK,
};
/* - SCIF4 ------------------------------------------------------------------ */
static const unsigned int scif4_data_a_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 2), RCAR_GP_PIN(4, 3),
};
static const unsigned int scif4_data_a_mux[] = {
	RX4_A_MARK, TX4_A_MARK,
};
static const unsigned int scif4_data_b_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(1, 2), RCAR_GP_PIN(1, 3),
};
static const unsigned int scif4_data_b_mux[] = {
	RX4_B_MARK, TX4_B_MARK,
};
static const unsigned int scif4_data_c_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(1, 13), RCAR_GP_PIN(1, 14),
};
static const unsigned int scif4_data_c_mux[] = {
	RX4_C_MARK, TX4_C_MARK,
};
static const unsigned int scif4_data_d_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(2, 10), RCAR_GP_PIN(2, 11),
};
static const unsigned int scif4_data_d_mux[] = {
	RX4_D_MARK, TX4_D_MARK,
};
static const unsigned int scif4_data_e_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(0, 7), RCAR_GP_PIN(0, 8),
};
static const unsigned int scif4_data_e_mux[] = {
	RX4_E_MARK, TX4_E_MARK,
};
/* - SCIF5 ------------------------------------------------------------------ */
static const unsigned int scif5_data_a_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(4, 4), RCAR_GP_PIN(4, 5),
};
static const unsigned int scif5_data_a_mux[] = {
	RX5_A_MARK, TX5_A_MARK,
};
static const unsigned int scif5_data_b_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(1, 0), RCAR_GP_PIN(1, 1),
};
static const unsigned int scif5_data_b_mux[] = {
	RX5_B_MARK, TX5_B_MARK,
};
static const unsigned int scif5_data_c_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(2, 0), RCAR_GP_PIN(2, 1),
};
static const unsigned int scif5_data_c_mux[] = {
	RX5_C_MARK, TX5_C_MARK,
};
static const unsigned int scif5_data_d_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(5, 9), RCAR_GP_PIN(5, 10),
};
static const unsigned int scif5_data_d_mux[] = {
	RX5_D_MARK, TX5_D_MARK,
};
static const unsigned int scif5_data_e_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(0, 9), RCAR_GP_PIN(0, 10),
};
static const unsigned int scif5_data_e_mux[] = {
	RX5_E_MARK, TX5_E_MARK,
};
static const unsigned int scif5_data_f_pins[] = {
	/* RX, TX */
	RCAR_GP_PIN(3, 27), RCAR_GP_PIN(3, 28),
};
static const unsigned int scif5_data_f_mux[] = {
	RX5_F_MARK, TX5_F_MARK,
};
/* - SCIF Clock ------------------------------------------------------------- */
static const unsigned int scif_clk_a_pins[] = {
	/* SCIF_CLK */
	RCAR_GP_PIN(1, 22),
};
static const unsigned int scif_clk_a_mux[] = {
	SCIF_CLK_A_MARK,
};
static const unsigned int scif_clk_b_pins[] = {
	/* SCIF_CLK */
	RCAR_GP_PIN(3, 29),
};
static const unsigned int scif_clk_b_mux[] = {
	SCIF_CLK_B_MARK,
};
/* - SDHI2 ------------------------------------------------------------------ */
static const unsigned int sdhi2_data1_pins[] = {
	/* D0 */
	RCAR_GP_PIN(4, 16),
};
static const unsigned int sdhi2_data1_mux[] = {
	SD2_DAT0_MARK,
};
static const unsigned int sdhi2_data4_pins[] = {
	/* D[0:3] */
	RCAR_GP_PIN(4, 16), RCAR_GP_PIN(4, 17),
	RCAR_GP_PIN(4, 18), RCAR_GP_PIN(4, 19),
};
static const unsigned int sdhi2_data4_mux[] = {
	SD2_DAT0_MARK, SD2_DAT1_MARK, SD2_DAT2_MARK, SD2_DAT3_MARK,
};
static const unsigned int sdhi2_ctrl_pins[] = {
	/* CLK, CMD */
	RCAR_GP_PIN(4, 14), RCAR_GP_PIN(4, 15),
};
static const unsigned int sdhi2_ctrl_mux[] = {
	SD2_CLK_MARK, SD2_CMD_MARK,
};
static const unsigned int sdhi2_cd_pins[] = {
	/* CD */
	RCAR_GP_PIN(4, 20),
};
static const unsigned int sdhi2_cd_mux[] = {
	SD2_CD_MARK,
};
static const unsigned int sdhi2_wp_pins[] = {
	/* WP */
	RCAR_GP_PIN(4, 21),
};
static const unsigned int sdhi2_wp_mux[] = {
	SD2_WP_MARK,
};
/* - USB0 ------------------------------------------------------------------- */
static const unsigned int usb0_pins[] = {
	RCAR_GP_PIN(0, 0), /* PWEN */
	RCAR_GP_PIN(0, 1), /* OVC */
};
static const unsigned int usb0_mux[] = {
	USB0_PWEN_MARK,
	USB0_OVC_MARK,
};
/* - USB1 ------------------------------------------------------------------- */
static const unsigned int usb1_pins[] = {
	RCAR_GP_PIN(0, 2), /* PWEN */
	RCAR_GP_PIN(0, 3), /* OVC */
};
static const unsigned int usb1_mux[] = {
	USB1_PWEN_MARK,
	USB1_OVC_MARK,
};
/* - VIN0 ------------------------------------------------------------------- */
static const union vin_data vin0_data_pins = {
	.data24 = {
		/* B */
		RCAR_GP_PIN(5, 20), RCAR_GP_PIN(5, 21),
		RCAR_GP_PIN(5, 22), RCAR_GP_PIN(5, 23),
		RCAR_GP_PIN(5, 24), RCAR_GP_PIN(5, 25),
		RCAR_GP_PIN(5, 26), RCAR_GP_PIN(5, 27),
		/* G */
		RCAR_GP_PIN(4, 2), RCAR_GP_PIN(4, 3),
		RCAR_GP_PIN(4, 4), RCAR_GP_PIN(4, 5),
		RCAR_GP_PIN(4, 6), RCAR_GP_PIN(5, 8),
		RCAR_GP_PIN(5, 9), RCAR_GP_PIN(5, 10),
		/* R */
		RCAR_GP_PIN(5, 11), RCAR_GP_PIN(5, 12),
		RCAR_GP_PIN(5, 13), RCAR_GP_PIN(5, 14),
		RCAR_GP_PIN(5, 15), RCAR_GP_PIN(5, 16),
		RCAR_GP_PIN(5, 17), RCAR_GP_PIN(5, 19),
	},
};
static const union vin_data vin0_data_mux = {
	.data24 = {
		/* B */
		VI0_DATA0_VI0_B0_MARK, VI0_DATA1_VI0_B1_MARK,
		VI0_DATA2_VI0_B2_MARK, VI0_DATA3_VI0_B3_MARK,
		VI0_DATA4_VI0_B4_MARK, VI0_DATA5_VI0_B5_MARK,
		VI0_DATA6_VI0_B6_MARK, VI0_DATA7_VI0_B7_MARK,
		/* G */
		VI0_G0_MARK, VI0_G1_MARK,
		VI0_G2_MARK, VI0_G3_MARK,
		VI0_G4_MARK, VI0_G5_MARK,
		VI0_G6_MARK, VI0_G7_MARK,
		/* R */
		VI0_R0_MARK, VI0_R1_MARK,
		VI0_R2_MARK, VI0_R3_MARK,
		VI0_R4_MARK, VI0_R5_MARK,
		VI0_R6_MARK, VI0_R7_MARK,
	},
};
static const unsigned int vin0_data18_pins[] = {
	/* B */
	RCAR_GP_PIN(5, 22), RCAR_GP_PIN(5, 23),
	RCAR_GP_PIN(5, 24), RCAR_GP_PIN(5, 25),
	RCAR_GP_PIN(5, 26), RCAR_GP_PIN(5, 27),
	/* G */
	RCAR_GP_PIN(4, 4), RCAR_GP_PIN(4, 5),
	RCAR_GP_PIN(4, 6), RCAR_GP_PIN(5, 8),
	RCAR_GP_PIN(5, 9), RCAR_GP_PIN(5, 10),
	/* R */
	RCAR_GP_PIN(5, 13), RCAR_GP_PIN(5, 14),
	RCAR_GP_PIN(5, 15), RCAR_GP_PIN(5, 16),
	RCAR_GP_PIN(5, 17), RCAR_GP_PIN(5, 19),
};
static const unsigned int vin0_data18_mux[] = {
	/* B */
	VI0_DATA2_VI0_B2_MARK, VI0_DATA3_VI0_B3_MARK,
	VI0_DATA4_VI0_B4_MARK, VI0_DATA5_VI0_B5_MARK,
	VI0_DATA6_VI0_B6_MARK, VI0_DATA7_VI0_B7_MARK,
	/* G */
	VI0_G2_MARK, VI0_G3_MARK,
	VI0_G4_MARK, VI0_G5_MARK,
	VI0_G6_MARK, VI0_G7_MARK,
	/* R */
	VI0_R2_MARK, VI0_R3_MARK,
	VI0_R4_MARK, VI0_R5_MARK,
	VI0_R6_MARK, VI0_R7_MARK,
};
static const unsigned int vin0_sync_pins[] = {
	RCAR_GP_PIN(5, 30), /* HSYNC */
	RCAR_GP_PIN(5, 31), /* VSYNC */
};
static const unsigned int vin0_sync_mux[] = {
	VI0_HSYNC_N_MARK,
	VI0_VSYNC_N_MARK,
};
static const unsigned int vin0_field_pins[] = {
	RCAR_GP_PIN(5, 29),
};
static const unsigned int vin0_field_mux[] = {
	VI0_FIELD_MARK,
};
static const unsigned int vin0_clkenb_pins[] = {
	RCAR_GP_PIN(5, 28),
};
static const unsigned int vin0_clkenb_mux[] = {
	VI0_CLKENB_MARK,
};
static const unsigned int vin0_clk_pins[] = {
	RCAR_GP_PIN(5, 18),
};
static const unsigned int vin0_clk_mux[] = {
	VI0_CLK_MARK,
};
/* - VIN1 ------------------------------------------------------------------- */
static const union vin_data vin1_data_pins = {
	.data12 = {
		RCAR_GP_PIN(3,  1), RCAR_GP_PIN(3, 2),
		RCAR_GP_PIN(3,  3), RCAR_GP_PIN(3, 4),
		RCAR_GP_PIN(3,  5), RCAR_GP_PIN(3, 6),
		RCAR_GP_PIN(3,  7), RCAR_GP_PIN(3, 8),
		RCAR_GP_PIN(3, 13), RCAR_GP_PIN(3, 14),
		RCAR_GP_PIN(3, 15), RCAR_GP_PIN(3, 16),
	},
};
static const union vin_data vin1_data_mux = {
	.data12 = {
		VI1_DATA0_MARK, VI1_DATA1_MARK,
		VI1_DATA2_MARK, VI1_DATA3_MARK,
		VI1_DATA4_MARK, VI1_DATA5_MARK,
		VI1_DATA6_MARK, VI1_DATA7_MARK,
		VI1_DATA8_MARK, VI1_DATA9_MARK,
		VI1_DATA10_MARK, VI1_DATA11_MARK,
	},
};
static const unsigned int vin1_sync_pins[] = {
	RCAR_GP_PIN(3, 11), /* HSYNC */
	RCAR_GP_PIN(3, 12), /* VSYNC */
};
static const unsigned int vin1_sync_mux[] = {
	VI1_HSYNC_N_MARK,
	VI1_VSYNC_N_MARK,
};
static const unsigned int vin1_field_pins[] = {
	RCAR_GP_PIN(3, 10),
};
static const unsigned int vin1_field_mux[] = {
	VI1_FIELD_MARK,
};
static const unsigned int vin1_clkenb_pins[] = {
	RCAR_GP_PIN(3, 9),
};
static const unsigned int vin1_clkenb_mux[] = {
	VI1_CLKENB_MARK,
};
static const unsigned int vin1_clk_pins[] = {
	RCAR_GP_PIN(3, 0),
};
static const unsigned int vin1_clk_mux[] = {
	VI1_CLK_MARK,
};

static const struct sh_pfc_pin_group pinmux_groups[] = {
	SH_PFC_PIN_GROUP(avb_col),
	SH_PFC_PIN_GROUP(avb_crs),
	SH_PFC_PIN_GROUP(avb_link),
	SH_PFC_PIN_GROUP(avb_magic),
	SH_PFC_PIN_GROUP(avb_phy_int),
	SH_PFC_PIN_GROUP(avb_mdio),
	SH_PFC_PIN_GROUP(avb_mii_tx_rx),
	SH_PFC_PIN_GROUP(avb_mii_tx_er),
	SH_PFC_PIN_GROUP(avb_gmii_tx_rx),
	SH_PFC_PIN_GROUP(avb_avtp_match_a),
	SH_PFC_PIN_GROUP(avb_avtp_capture_a),
	SH_PFC_PIN_GROUP(avb_avtp_match_b),
	SH_PFC_PIN_GROUP(avb_avtp_capture_b),
	SH_PFC_PIN_GROUP(du0_rgb666),
	SH_PFC_PIN_GROUP(du0_rgb888),
	SH_PFC_PIN_GROUP(du0_clk0_out),
	SH_PFC_PIN_GROUP(du0_clk1_out),
	SH_PFC_PIN_GROUP(du0_clk_in),
	SH_PFC_PIN_GROUP(du0_sync),
	SH_PFC_PIN_GROUP(du0_oddf),
	SH_PFC_PIN_GROUP(du0_cde),
	SH_PFC_PIN_GROUP(du0_disp),
	SH_PFC_PIN_GROUP(du1_rgb666),
	SH_PFC_PIN_GROUP(du1_rgb888),
	SH_PFC_PIN_GROUP(du1_clk0_out),
	SH_PFC_PIN_GROUP(du1_clk1_out),
	SH_PFC_PIN_GROUP(du1_clk_in),
	SH_PFC_PIN_GROUP(du1_sync),
	SH_PFC_PIN_GROUP(du1_oddf),
	SH_PFC_PIN_GROUP(du1_cde),
	SH_PFC_PIN_GROUP(du1_disp),
	SH_PFC_PIN_GROUP(i2c0_a),
	SH_PFC_PIN_GROUP(i2c0_b),
	SH_PFC_PIN_GROUP(i2c0_c),
	SH_PFC_PIN_GROUP(i2c0_d),
	SH_PFC_PIN_GROUP(i2c0_e),
	SH_PFC_PIN_GROUP(i2c1_a),
	SH_PFC_PIN_GROUP(i2c1_b),
	SH_PFC_PIN_GROUP(i2c1_c),
	SH_PFC_PIN_GROUP(i2c1_d),
	SH_PFC_PIN_GROUP(i2c1_e),
	SH_PFC_PIN_GROUP(i2c2_a),
	SH_PFC_PIN_GROUP(i2c2_b),
	SH_PFC_PIN_GROUP(i2c2_c),
	SH_PFC_PIN_GROUP(i2c2_d),
	SH_PFC_PIN_GROUP(i2c3_a),
	SH_PFC_PIN_GROUP(i2c3_b),
	SH_PFC_PIN_GROUP(i2c3_c),
	SH_PFC_PIN_GROUP(i2c3_d),
	SH_PFC_PIN_GROUP(i2c3_e),
	SH_PFC_PIN_GROUP(i2c4_a),
	SH_PFC_PIN_GROUP(i2c4_b),
	SH_PFC_PIN_GROUP(i2c4_c),
	SH_PFC_PIN_GROUP(i2c4_d),
	SH_PFC_PIN_GROUP(i2c4_e),
	SH_PFC_PIN_GROUP(mmc_data1),
	SH_PFC_PIN_GROUP(mmc_data4),
	SH_PFC_PIN_GROUP(mmc_data8),
	SH_PFC_PIN_GROUP(mmc_ctrl),
	SH_PFC_PIN_GROUP(qspi0_ctrl),
	SH_PFC_PIN_GROUP(qspi0_data2),
	SH_PFC_PIN_GROUP(qspi0_data4),
	SH_PFC_PIN_GROUP(qspi1_ctrl),
	SH_PFC_PIN_GROUP(qspi1_data2),
	SH_PFC_PIN_GROUP(qspi1_data4),
	SH_PFC_PIN_GROUP(scif0_data_a),
	SH_PFC_PIN_GROUP(scif0_data_b),
	SH_PFC_PIN_GROUP(scif0_data_c),
	SH_PFC_PIN_GROUP(scif0_data_d),
	SH_PFC_PIN_GROUP(scif1_data_a),
	SH_PFC_PIN_GROUP(scif1_clk_a),
	SH_PFC_PIN_GROUP(scif1_data_b),
	SH_PFC_PIN_GROUP(scif1_clk_b),
	SH_PFC_PIN_GROUP(scif1_data_c),
	SH_PFC_PIN_GROUP(scif1_clk_c),
	SH_PFC_PIN_GROUP(scif1_data_d),
	SH_PFC_PIN_GROUP(scif2_data_a),
	SH_PFC_PIN_GROUP(scif2_clk_a),
	SH_PFC_PIN_GROUP(scif2_data_b),
	SH_PFC_PIN_GROUP(scif2_clk_b),
	SH_PFC_PIN_GROUP(scif2_data_c),
	SH_PFC_PIN_GROUP(scif3_data_a),
	SH_PFC_PIN_GROUP(scif3_clk),
	SH_PFC_PIN_GROUP(scif3_data_b),
	SH_PFC_PIN_GROUP(scif3_data_c),
	SH_PFC_PIN_GROUP(scif4_data_a),
	SH_PFC_PIN_GROUP(scif4_data_b),
	SH_PFC_PIN_GROUP(scif4_data_c),
	SH_PFC_PIN_GROUP(scif4_data_d),
	SH_PFC_PIN_GROUP(scif4_data_e),
	SH_PFC_PIN_GROUP(scif5_data_a),
	SH_PFC_PIN_GROUP(scif5_data_b),
	SH_PFC_PIN_GROUP(scif5_data_c),
	SH_PFC_PIN_GROUP(scif5_data_d),
	SH_PFC_PIN_GROUP(scif5_data_e),
	SH_PFC_PIN_GROUP(scif5_data_f),
	SH_PFC_PIN_GROUP(scif_clk_a),
	SH_PFC_PIN_GROUP(scif_clk_b),
	SH_PFC_PIN_GROUP(sdhi2_data1),
	SH_PFC_PIN_GROUP(sdhi2_data4),
	SH_PFC_PIN_GROUP(sdhi2_ctrl),
	SH_PFC_PIN_GROUP(sdhi2_cd),
	SH_PFC_PIN_GROUP(sdhi2_wp),
	SH_PFC_PIN_GROUP(usb0),
	SH_PFC_PIN_GROUP(usb1),
	VIN_DATA_PIN_GROUP(vin0_data, 24),
	VIN_DATA_PIN_GROUP(vin0_data, 20),
	SH_PFC_PIN_GROUP(vin0_data18),
	VIN_DATA_PIN_GROUP(vin0_data, 16),
	VIN_DATA_PIN_GROUP(vin0_data, 12),
	VIN_DATA_PIN_GROUP(vin0_data, 10),
	VIN_DATA_PIN_GROUP(vin0_data, 8),
	SH_PFC_PIN_GROUP(vin0_sync),
	SH_PFC_PIN_GROUP(vin0_field),
	SH_PFC_PIN_GROUP(vin0_clkenb),
	SH_PFC_PIN_GROUP(vin0_clk),
	VIN_DATA_PIN_GROUP(vin1_data, 12),
	VIN_DATA_PIN_GROUP(vin1_data, 10),
	VIN_DATA_PIN_GROUP(vin1_data, 8),
	SH_PFC_PIN_GROUP(vin1_sync),
	SH_PFC_PIN_GROUP(vin1_field),
	SH_PFC_PIN_GROUP(vin1_clkenb),
	SH_PFC_PIN_GROUP(vin1_clk),
};

static const char * const avb_groups[] = {
	"avb_col",
	"avb_crs",
	"avb_link",
	"avb_magic",
	"avb_phy_int",
	"avb_mdio",
	"avb_mii_tx_rx",
	"avb_mii_tx_er",
	"avb_gmii_tx_rx",
	"avb_avtp_match_a",
	"avb_avtp_capture_a",
	"avb_avtp_match_b",
	"avb_avtp_capture_b",
};

static const char * const du0_groups[] = {
	"du0_rgb666",
	"du0_rgb888",
	"du0_clk0_out",
	"du0_clk1_out",
	"du0_clk_in",
	"du0_sync",
	"du0_oddf",
	"du0_cde",
	"du0_disp",
};

static const char * const du1_groups[] = {
	"du1_rgb666",
	"du1_rgb888",
	"du1_clk0_out",
	"du1_clk1_out",
	"du1_clk_in",
	"du1_sync",
	"du1_oddf",
	"du1_cde",
	"du1_disp",
};

static const char * const i2c0_groups[] = {
	"i2c0_a",
	"i2c0_b",
	"i2c0_c",
	"i2c0_d",
	"i2c0_e",
};

static const char * const i2c1_groups[] = {
	"i2c1_a",
	"i2c1_b",
	"i2c1_c",
	"i2c1_d",
	"i2c1_e",
};

static const char * const i2c2_groups[] = {
	"i2c2_a",
	"i2c2_b",
	"i2c2_c",
	"i2c2_d",
};

static const char * const i2c3_groups[] = {
	"i2c3_a",
	"i2c3_b",
	"i2c3_c",
	"i2c3_d",
	"i2c3_e",
};

static const char * const i2c4_groups[] = {
	"i2c4_a",
	"i2c4_b",
	"i2c4_c",
	"i2c4_d",
	"i2c4_e",
};

static const char * const mmc_groups[] = {
	"mmc_data1",
	"mmc_data4",
	"mmc_data8",
	"mmc_ctrl",
};

static const char * const qspi0_groups[] = {
	"qspi0_ctrl",
	"qspi0_data2",
	"qspi0_data4",
};

static const char * const qspi1_groups[] = {
	"qspi1_ctrl",
	"qspi1_data2",
	"qspi1_data4",
};

static const char * const scif0_groups[] = {
	"scif0_data_a",
	"scif0_data_b",
	"scif0_data_c",
	"scif0_data_d",
};

static const char * const scif1_groups[] = {
	"scif1_data_a",
	"scif1_clk_a",
	"scif1_data_b",
	"scif1_clk_b",
	"scif1_data_c",
	"scif1_clk_c",
	"scif1_data_d",
};

static const char * const scif2_groups[] = {
	"scif2_data_a",
	"scif2_clk_a",
	"scif2_data_b",
	"scif2_clk_b",
	"scif2_data_c",
};

static const char * const scif3_groups[] = {
	"scif3_data_a",
	"scif3_clk",
	"scif3_data_b",
	"scif3_data_c",
};

static const char * const scif4_groups[] = {
	"scif4_data_a",
	"scif4_data_b",
	"scif4_data_c",
	"scif4_data_d",
	"scif4_data_e",
};

static const char * const scif5_groups[] = {
	"scif5_data_a",
	"scif5_data_b",
	"scif5_data_c",
	"scif5_data_d",
	"scif5_data_e",
	"scif5_data_f",
};

static const char * const scif_clk_groups[] = {
	"scif_clk_a",
	"scif_clk_b",
};

static const char * const sdhi2_groups[] = {
	"sdhi2_data1",
	"sdhi2_data4",
	"sdhi2_ctrl",
	"sdhi2_cd",
	"sdhi2_wp",
};

static const char * const usb0_groups[] = {
	"usb0",
};

static const char * const usb1_groups[] = {
	"usb1",
};

static const char * const vin0_groups[] = {
	"vin0_data24",
	"vin0_data20",
	"vin0_data18",
	"vin0_data16",
	"vin0_data12",
	"vin0_data10",
	"vin0_data8",
	"vin0_sync",
	"vin0_field",
	"vin0_clkenb",
	"vin0_clk",
};

static const char * const vin1_groups[] = {
	"vin1_data12",
	"vin1_data10",
	"vin1_data8",
	"vin1_sync",
	"vin1_field",
	"vin1_clkenb",
	"vin1_clk",
};

static const struct sh_pfc_function pinmux_functions[] = {
	SH_PFC_FUNCTION(avb),
	SH_PFC_FUNCTION(du0),
	SH_PFC_FUNCTION(du1),
	SH_PFC_FUNCTION(i2c0),
	SH_PFC_FUNCTION(i2c1),
	SH_PFC_FUNCTION(i2c2),
	SH_PFC_FUNCTION(i2c3),
	SH_PFC_FUNCTION(i2c4),
	SH_PFC_FUNCTION(mmc),
	SH_PFC_FUNCTION(qspi0),
	SH_PFC_FUNCTION(qspi1),
	SH_PFC_FUNCTION(scif0),
	SH_PFC_FUNCTION(scif1),
	SH_PFC_FUNCTION(scif2),
	SH_PFC_FUNCTION(scif3),
	SH_PFC_FUNCTION(scif4),
	SH_PFC_FUNCTION(scif5),
	SH_PFC_FUNCTION(scif_clk),
	SH_PFC_FUNCTION(sdhi2),
	SH_PFC_FUNCTION(usb0),
	SH_PFC_FUNCTION(usb1),
	SH_PFC_FUNCTION(vin0),
	SH_PFC_FUNCTION(vin1),
};

static const struct pinmux_cfg_reg pinmux_config_regs[] = {
	{ PINMUX_CFG_REG("GPSR0", 0xE6060004, 32, 1) {
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		GP_0_22_FN, FN_MMC0_D7,
		GP_0_21_FN, FN_MMC0_D6,
		GP_0_20_FN, FN_IP1_7_4,
		GP_0_19_FN, FN_IP1_3_0,
		GP_0_18_FN, FN_MMC0_D3_SDHI1_D3,
		GP_0_17_FN, FN_MMC0_D2_SDHI1_D2,
		GP_0_16_FN, FN_MMC0_D1_SDHI1_D1,
		GP_0_15_FN, FN_MMC0_D0_SDHI1_D0,
		GP_0_14_FN, FN_MMC0_CMD_SDHI1_CMD,
		GP_0_13_FN, FN_MMC0_CLK_SDHI1_CLK,
		GP_0_12_FN, FN_IP0_31_28,
		GP_0_11_FN, FN_IP0_27_24,
		GP_0_10_FN, FN_IP0_23_20,
		GP_0_9_FN, FN_IP0_19_16,
		GP_0_8_FN, FN_IP0_15_12,
		GP_0_7_FN, FN_IP0_11_8,
		GP_0_6_FN, FN_IP0_7_4,
		GP_0_5_FN, FN_IP0_3_0,
		GP_0_4_FN, FN_CLKOUT,
		GP_0_3_FN, FN_USB1_OVC,
		GP_0_2_FN, FN_USB1_PWEN,
		GP_0_1_FN, FN_USB0_OVC,
		GP_0_0_FN, FN_USB0_PWEN, }
	},
	{ PINMUX_CFG_REG("GPSR1", 0xE6060008, 32, 1) {
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		GP_1_22_FN, FN_IP4_3_0,
		GP_1_21_FN, FN_IP3_31_28,
		GP_1_20_FN, FN_IP3_27_24,
		GP_1_19_FN, FN_IP3_23_20,
		GP_1_18_FN, FN_IP3_19_16,
		GP_1_17_FN, FN_IP3_15_12,
		GP_1_16_FN, FN_IP3_11_8,
		GP_1_15_FN, FN_IP3_7_4,
		GP_1_14_FN, FN_IP3_3_0,
		GP_1_13_FN, FN_IP2_31_28,
		GP_1_12_FN, FN_IP2_27_24,
		GP_1_11_FN, FN_IP2_23_20,
		GP_1_10_FN, FN_IP2_19_16,
		GP_1_9_FN, FN_IP2_15_12,
		GP_1_8_FN, FN_IP2_11_8,
		GP_1_7_FN, FN_IP2_7_4,
		GP_1_6_FN, FN_IP2_3_0,
		GP_1_5_FN, FN_IP1_31_28,
		GP_1_4_FN, FN_IP1_27_24,
		GP_1_3_FN, FN_IP1_23_20,
		GP_1_2_FN, FN_IP1_19_16,
		GP_1_1_FN, FN_IP1_15_12,
		GP_1_0_FN, FN_IP1_11_8, }
	},
	{ PINMUX_CFG_REG("GPSR2", 0xE606000C, 32, 1) {
		GP_2_31_FN, FN_IP8_3_0,
		GP_2_30_FN, FN_IP7_31_28,
		GP_2_29_FN, FN_IP7_27_24,
		GP_2_28_FN, FN_IP7_23_20,
		GP_2_27_FN, FN_IP7_19_16,
		GP_2_26_FN, FN_IP7_15_12,
		GP_2_25_FN, FN_IP7_11_8,
		GP_2_24_FN, FN_IP7_7_4,
		GP_2_23_FN, FN_IP7_3_0,
		GP_2_22_FN, FN_IP6_31_28,
		GP_2_21_FN, FN_IP6_27_24,
		GP_2_20_FN, FN_IP6_23_20,
		GP_2_19_FN, FN_IP6_19_16,
		GP_2_18_FN, FN_IP6_15_12,
		GP_2_17_FN, FN_IP6_11_8,
		GP_2_16_FN, FN_IP6_7_4,
		GP_2_15_FN, FN_IP6_3_0,
		GP_2_14_FN, FN_IP5_31_28,
		GP_2_13_FN, FN_IP5_27_24,
		GP_2_12_FN, FN_IP5_23_20,
		GP_2_11_FN, FN_IP5_19_16,
		GP_2_10_FN, FN_IP5_15_12,
		GP_2_9_FN, FN_IP5_11_8,
		GP_2_8_FN, FN_IP5_7_4,
		GP_2_7_FN, FN_IP5_3_0,
		GP_2_6_FN, FN_IP4_31_28,
		GP_2_5_FN, FN_IP4_27_24,
		GP_2_4_FN, FN_IP4_23_20,
		GP_2_3_FN, FN_IP4_19_16,
		GP_2_2_FN, FN_IP4_15_12,
		GP_2_1_FN, FN_IP4_11_8,
		GP_2_0_FN, FN_IP4_7_4, }
	},
	{ PINMUX_CFG_REG("GPSR3", 0xE6060010, 32, 1) {
		0, 0,
		0, 0,
		GP_3_29_FN, FN_IP10_19_16,
		GP_3_28_FN, FN_IP10_15_12,
		GP_3_27_FN, FN_IP10_11_8,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		GP_3_16_FN, FN_IP10_7_4,
		GP_3_15_FN, FN_IP10_3_0,
		GP_3_14_FN, FN_IP9_31_28,
		GP_3_13_FN, FN_IP9_27_24,
		GP_3_12_FN, FN_IP9_23_20,
		GP_3_11_FN, FN_IP9_19_16,
		GP_3_10_FN, FN_IP9_15_12,
		GP_3_9_FN, FN_IP9_11_8,
		GP_3_8_FN, FN_IP9_7_4,
		GP_3_7_FN, FN_IP9_3_0,
		GP_3_6_FN, FN_IP8_31_28,
		GP_3_5_FN, FN_IP8_27_24,
		GP_3_4_FN, FN_IP8_23_20,
		GP_3_3_FN, FN_IP8_19_16,
		GP_3_2_FN, FN_IP8_15_12,
		GP_3_1_FN, FN_IP8_11_8,
		GP_3_0_FN, FN_IP8_7_4, }
	},
	{ PINMUX_CFG_REG("GPSR4", 0xE6060014, 32, 1) {
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		0, 0,
		GP_4_25_FN, FN_IP13_27_24,
		GP_4_24_FN, FN_IP13_23_20,
		GP_4_23_FN, FN_IP13_19_16,
		GP_4_22_FN, FN_IP13_15_12,
		GP_4_21_FN, FN_IP13_11_8,
		GP_4_20_FN, FN_IP13_7_4,
		GP_4_19_FN, FN_IP13_3_0,
		GP_4_18_FN, FN_IP12_31_28,
		GP_4_17_FN, FN_IP12_27_24,
		GP_4_16_FN, FN_IP12_23_20,
		GP_4_15_FN, FN_IP12_19_16,
		GP_4_14_FN, FN_IP12_15_12,
		GP_4_13_FN, FN_IP12_11_8,
		GP_4_12_FN, FN_IP12_7_4,
		GP_4_11_FN, FN_IP12_3_0,
		GP_4_10_FN, FN_IP11_31_28,
		GP_4_9_FN, FN_IP11_27_24,
		GP_4_8_FN, FN_IP11_23_20,
		GP_4_7_FN, FN_IP11_19_16,
		GP_4_6_FN, FN_IP11_15_12,
		GP_4_5_FN, FN_IP11_11_8,
		GP_4_4_FN, FN_IP11_7_4,
		GP_4_3_FN, FN_IP11_3_0,
		GP_4_2_FN, FN_IP10_31_28,
		GP_4_1_FN, FN_IP10_27_24,
		GP_4_0_FN, FN_IP10_23_20, }
	},
	{ PINMUX_CFG_REG("GPSR5", 0xE6060018, 32, 1) {
		GP_5_31_FN, FN_IP17_27_24,
		GP_5_30_FN, FN_IP17_23_20,
		GP_5_29_FN, FN_IP17_19_16,
		GP_5_28_FN, FN_IP17_15_12,
		GP_5_27_FN, FN_IP17_11_8,
		GP_5_26_FN, FN_IP17_7_4,
		GP_5_25_FN, FN_IP17_3_0,
		GP_5_24_FN, FN_IP16_31_28,
		GP_5_23_FN, FN_IP16_27_24,
		GP_5_22_FN, FN_IP16_23_20,
		GP_5_21_FN, FN_IP16_19_16,
		GP_5_20_FN, FN_IP16_15_12,
		GP_5_19_FN, FN_IP16_11_8,
		GP_5_18_FN, FN_IP16_7_4,
		GP_5_17_FN, FN_IP16_3_0,
		GP_5_16_FN, FN_IP15_31_28,
		GP_5_15_FN, FN_IP15_27_24,
		GP_5_14_FN, FN_IP15_23_20,
		GP_5_13_FN, FN_IP15_19_16,
		GP_5_12_FN, FN_IP15_15_12,
		GP_5_11_FN, FN_IP15_11_8,
		GP_5_10_FN, FN_IP15_7_4,
		GP_5_9_FN, FN_IP15_3_0,
		GP_5_8_FN, FN_IP14_31_28,
		GP_5_7_FN, FN_IP14_27_24,
		GP_5_6_FN, FN_IP14_23_20,
		GP_5_5_FN, FN_IP14_19_16,
		GP_5_4_FN, FN_IP14_15_12,
		GP_5_3_FN, FN_IP14_11_8,
		GP_5_2_FN, FN_IP14_7_4,
		GP_5_1_FN, FN_IP14_3_0,
		GP_5_0_FN, FN_IP13_31_28, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR0", 0xE6060040, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP0_31_28 [4] */
		FN_SD0_WP, FN_IRQ7, FN_CAN0_TX_A, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_27_24 [4] */
		FN_SD0_CD, 0, FN_CAN0_RX_A, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_23_20 [4] */
		FN_SD0_DAT3, 0, 0, FN_SSI_SDATA0_B, FN_TX5_E, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_19_16 [4] */
		FN_SD0_DAT2, 0, 0, FN_SSI_WS0129_B, FN_RX5_E, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_15_12 [4] */
		FN_SD0_DAT1, 0, 0, FN_SSI_SCK0129_B, FN_TX4_E, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_11_8 [4] */
		FN_SD0_DAT0, 0, 0, FN_SSI_SDATA1_C, FN_RX4_E, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_7_4 [4] */
		FN_SD0_CMD, 0, 0, FN_SSI_WS1_C, FN_TX3_C, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP0_3_0 [4] */
		FN_SD0_CLK, 0, 0, FN_SSI_SCK1_C, FN_RX3_C, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR1", 0xE6060044, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP1_31_28 [4] */
		FN_D5, FN_HRX2, FN_SCL1_B, FN_PWM2_C, FN_TCLK2_B, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_27_24 [4] */
		FN_D4, 0, FN_IRQ3, FN_TCLK1_A, FN_PWM6_C, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_23_20 [4] */
		FN_D3, 0, FN_TX4_B, FN_SDA0_D, FN_PWM0_A,
		FN_MSIOF2_SYNC_C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_19_16 [4] */
		FN_D2, 0, FN_RX4_B, FN_SCL0_D, FN_PWM1_C,
		FN_MSIOF2_SCK_C, FN_SSI_SCK5_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_15_12 [4] */
		FN_D1, 0, FN_SDA3_B, FN_TX5_B, 0, FN_MSIOF2_TXD_C,
		FN_SSI_WS5_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_11_8 [4] */
		FN_D0, 0, FN_SCL3_B, FN_RX5_B, FN_IRQ4,
		FN_MSIOF2_RXD_C, FN_SSI_SDATA5_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_7_4 [4] */
		FN_MMC0_D5, FN_SD1_WP, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP1_3_0 [4] */
		FN_MMC0_D4, FN_SD1_CD, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR2", 0xE6060048, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP2_31_28 [4] */
		FN_D13, FN_MSIOF2_SYNC_A, 0, FN_RX4_C, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP2_27_24 [4] */
		FN_D12, FN_MSIOF2_SCK_A, FN_HSCK0, 0, FN_CAN_CLK_C,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP2_23_20 [4] */
		FN_D11, FN_MSIOF2_TXD_A, FN_HTX0_B, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP2_19_16 [4] */
		FN_D10, FN_MSIOF2_RXD_A, FN_HRX0_B, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP2_15_12 [4] */
		FN_D9, FN_HRTS2_N, FN_TX1_C, FN_SDA1_D, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP2_11_8 [4] */
		FN_D8, FN_HCTS2_N, FN_RX1_C, FN_SCL1_D, FN_PWM3_C, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP2_7_4 [4] */
		FN_D7, FN_HSCK2, FN_SCIF1_SCK_C, FN_IRQ6, FN_PWM5_C,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP2_3_0 [4] */
		FN_D6, FN_HTX2, FN_SDA1_B, FN_PWM4_C, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR3", 0xE606004C, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP3_31_28 [4] */
		FN_QSPI0_SSL, FN_WE1_N, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0,
		/* IP3_27_24 [4] */
		FN_QSPI0_IO3, FN_RD_N, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0,
		/* IP3_23_20 [4] */
		FN_QSPI0_IO2, FN_CS0_N, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0,
		/* IP3_19_16 [4] */
		FN_QSPI0_MISO_QSPI0_IO1, FN_RD_WR_N, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0,
		/* IP3_15_12 [4] */
		FN_QSPI0_MOSI_QSPI0_IO0, FN_BS_N, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0,
		/* IP3_11_8 [4] */
		FN_QSPI0_SPCLK, FN_WE0_N, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0,
		/* IP3_7_4 [4] */
		FN_D15, FN_MSIOF2_SS2, FN_PWM4_A, 0, FN_CAN1_TX_B, FN_IRQ2,
		FN_AVB_AVTP_MATCH_A, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP3_3_0 [4] */
		FN_D14, FN_MSIOF2_SS1, 0, FN_TX4_C, FN_CAN1_RX_B,
		0, FN_AVB_AVTP_CAPTURE_A,
		0, 0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR4", 0xE6060050, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP4_31_28 [4] */
		FN_DU0_DR6, 0, FN_RX2_C, 0, 0, 0, FN_A6, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP4_27_24 [4] */
		FN_DU0_DR5, 0, FN_TX1_D, 0, FN_PWM1_B, 0, FN_A5, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP4_23_20 [4] */
		FN_DU0_DR4, 0, FN_RX1_D, 0, 0, 0, FN_A4, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP4_19_16 [4] */
		FN_DU0_DR3, 0, FN_TX0_D, FN_SDA0_E, FN_PWM0_B, 0,
		FN_A3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP4_15_12 [4] */
		FN_DU0_DR2, 0, FN_RX0_D, FN_SCL0_E, 0, 0, FN_A2, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP4_11_8 [4] */
		FN_DU0_DR1, 0, FN_TX5_C, FN_SDA2_D, 0, 0, FN_A1, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP4_7_4 [4] */
		FN_DU0_DR0, 0, FN_RX5_C, FN_SCL2_D, 0, 0, FN_A0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP4_3_0 [4] */
		FN_EX_WAIT0, FN_CAN_CLK_B, FN_SCIF_CLK_A, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR5", 0xE6060054, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP5_31_28 [4] */
		FN_DU0_DG6, 0, FN_HRX1_C, 0, 0, 0, FN_A14,  0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP5_27_24 [4] */
		FN_DU0_DG5, 0, FN_HTX0_A, 0, FN_PWM5_B, 0, FN_A13,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP5_23_20 [4] */
		FN_DU0_DG4, 0, FN_HRX0_A, 0, 0, 0, FN_A12, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP5_19_16 [4] */
		FN_DU0_DG3, 0, FN_TX4_D, 0, FN_PWM4_B, 0, FN_A11, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP5_15_12 [4] */
		FN_DU0_DG2, 0, FN_RX4_D, 0, 0, 0, FN_A10, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP5_11_8 [4] */
		FN_DU0_DG1, 0, FN_TX3_B, FN_SDA3_D, FN_PWM3_B, 0,
		FN_A9, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP5_7_4 [4] */
		FN_DU0_DG0, 0, FN_RX3_B, FN_SCL3_D, 0, 0, FN_A8, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP5_3_0 [4] */
		FN_DU0_DR7, 0, FN_TX2_C, 0, FN_PWM2_B, 0, FN_A7, 0,
		0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR6", 0xE6060058, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP6_31_28 [4] */
		FN_DU0_DB6, 0, 0, 0, 0, 0, FN_A22, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP6_27_24 [4] */
		FN_DU0_DB5, 0, FN_HRTS1_N_C, 0, 0, 0,
		FN_A21, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP6_23_20 [4] */
		FN_DU0_DB4, 0, FN_HCTS1_N_C, 0, 0, 0,
		FN_A20, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP6_19_16 [4] */
		FN_DU0_DB3, 0, FN_HRTS0_N, 0, 0, 0, FN_A19, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP6_15_12 [4] */
		FN_DU0_DB2, 0, FN_HCTS0_N, 0, 0, 0, FN_A18, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP6_11_8 [4] */
		FN_DU0_DB1, 0, 0, FN_SDA4_D, FN_CAN0_TX_C, 0, FN_A17,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP6_7_4 [4] */
		FN_DU0_DB0, 0, 0, FN_SCL4_D, FN_CAN0_RX_C, 0, FN_A16,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP6_3_0 [4] */
		FN_DU0_DG7, 0, FN_HTX1_C, 0,  FN_PWM6_B, 0, FN_A15,
		0, 0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR7", 0xE606005C, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP7_31_28 [4] */
		FN_DU0_DISP, 0, 0, 0, FN_CAN1_RX_C, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP7_27_24 [4] */
		FN_DU0_EXODDF_DU0_ODDF_DISP_CDE, 0, FN_MSIOF2_SCK_B,
		0, 0, 0, FN_DRACK0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP7_23_20 [4] */
		FN_DU0_EXVSYNC_DU0_VSYNC, 0, FN_MSIOF2_SYNC_B, 0,
		0, 0, FN_DACK0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP7_19_16 [4] */
		FN_DU0_EXHSYNC_DU0_HSYNC, 0, FN_MSIOF2_TXD_B, 0,
		0, 0, FN_DREQ0_N, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP7_15_12 [4] */
		FN_DU0_DOTCLKOUT1, 0, FN_MSIOF2_RXD_B, 0, 0, 0,
		FN_CS1_N_A26, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP7_11_8 [4] */
		FN_DU0_DOTCLKOUT0, 0, 0, 0, 0, 0, FN_A25, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP7_7_4 [4] */
		FN_DU0_DOTCLKIN, 0, 0, 0, 0, 0, FN_A24, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP7_3_0 [4] */
		FN_DU0_DB7, 0, 0, 0, 0, 0, FN_A23, 0, 0,
		0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR8", 0xE6060060, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP8_31_28 [4] */
		FN_VI1_DATA5, 0, 0, 0, FN_AVB_RXD4, FN_ETH_LINK, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP8_27_24 [4] */
		FN_VI1_DATA4, 0, 0, 0, FN_AVB_RXD3, FN_ETH_RX_ER, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP8_23_20 [4] */
		FN_VI1_DATA3, 0, 0, 0, FN_AVB_RXD2, FN_ETH_MDIO, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP8_19_16 [4] */
		FN_VI1_DATA2, 0, 0, 0, FN_AVB_RXD1, FN_ETH_RXD1, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP8_15_12 [4] */
		FN_VI1_DATA1, 0, 0, 0, FN_AVB_RXD0, FN_ETH_RXD0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP8_11_8 [4] */
		FN_VI1_DATA0, 0, 0, 0, FN_AVB_RX_DV, FN_ETH_CRS_DV, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP8_7_4 [4] */
		FN_VI1_CLK, 0, 0, 0, FN_AVB_RX_CLK, FN_ETH_REF_CLK, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP8_3_0 [4] */
		FN_DU0_CDE, 0, 0, 0, FN_CAN1_TX_C, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR9", 0xE6060064, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP9_31_28 [4] */
		FN_VI1_DATA9, 0, 0, FN_SDA2_B, FN_AVB_TXD0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP9_27_24 [4] */
		FN_VI1_DATA8, 0, 0, FN_SCL2_B, FN_AVB_TX_EN, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP9_23_20 [4] */
		FN_VI1_VSYNC_N, FN_TX0_B, FN_SDA0_C, FN_AUDIO_CLKOUT_B,
		FN_AVB_TX_CLK, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP9_19_16 [4] */
		FN_VI1_HSYNC_N, FN_RX0_B, FN_SCL0_C, 0, FN_AVB_GTXREFCLK,
		FN_ETH_MDC, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP9_15_12 [4] */
		FN_VI1_FIELD, FN_SDA3_A, 0, 0, FN_AVB_RX_ER, FN_ETH_TXD0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP9_11_8 [4] */
		FN_VI1_CLKENB, FN_SCL3_A, 0, 0, FN_AVB_RXD7, FN_ETH_MAGIC, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP9_7_4 [4] */
		FN_VI1_DATA7, 0, 0, 0, FN_AVB_RXD6, FN_ETH_TX_EN, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP9_3_0 [4] */
		FN_VI1_DATA6, 0, 0, 0, FN_AVB_RXD5, FN_ETH_TXD1, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR10", 0xE6060068, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP10_31_28 [4] */
		FN_SCL1_A, FN_RX4_A, FN_PWM5_D, FN_DU1_DR0, 0, 0,
		FN_SSI_SCK6_B, FN_VI0_G0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP10_27_24 [4] */
		FN_SDA0_A, FN_TX0_C, FN_IRQ5, FN_CAN_CLK_A, FN_AVB_GTX_CLK,
		FN_CAN1_TX_D, FN_DVC_MUTE, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP10_23_20 [4] */
		FN_SCL0_A, FN_RX0_C, FN_PWM5_A, FN_TCLK1_B, FN_AVB_TXD6,
		FN_CAN1_RX_D, FN_MSIOF0_SYNC_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP10_19_16 [4] */
		FN_AVB_TXD5, FN_SCIF_CLK_B, FN_AUDIO_CLKC_B, 0,
		FN_SSI_SDATA1_D, 0, FN_MSIOF0_SCK_B, 0, 0, 0, 0, 0, 0, 0,
		0, 0,
		/* IP10_15_12 [4] */
		FN_AVB_TXD4, 0, FN_AUDIO_CLKB_B, 0, FN_SSI_WS1_D, FN_TX5_F,
		FN_MSIOF0_TXD_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP10_11_8 [4] */
		FN_AVB_TXD3, 0, FN_AUDIO_CLKA_B, 0, FN_SSI_SCK1_D, FN_RX5_F,
		FN_MSIOF0_RXD_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP10_7_4 [4] */
		FN_VI1_DATA11, 0, 0, FN_CAN0_TX_B, FN_AVB_TXD2, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0,
		/* IP10_3_0 [4] */
		FN_VI1_DATA10, 0, 0, FN_CAN0_RX_B, FN_AVB_TXD1, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR11", 0xE606006C, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP11_31_28 [4] */
		FN_HRX1_A, FN_SCL4_A, FN_PWM6_A, FN_DU1_DG0, FN_RX0_A, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP11_27_24 [4] */
		FN_MSIOF0_SS2_A, 0, 0, FN_DU1_DR7, 0,
		FN_QSPI1_SSL, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP11_23_20 [4] */
		FN_MSIOF0_SS1_A, 0, 0, FN_DU1_DR6, 0,
		FN_QSPI1_IO3, FN_SSI_SDATA8_B, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP11_19_16 [4] */
		FN_MSIOF0_SYNC_A, FN_PWM1_A, 0, FN_DU1_DR5,
		0, FN_QSPI1_IO2, FN_SSI_SDATA7_B, 0, 0, 0, 0, 0,
		0, 0, 0, 0,
		/* IP11_15_12 [4] */
		FN_MSIOF0_SCK_A, FN_IRQ0, 0, FN_DU1_DR4,
		0, FN_QSPI1_SPCLK, FN_SSI_SCK78_B, FN_VI0_G4,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP11_11_8 [4] */
		FN_MSIOF0_TXD_A, FN_TX5_A, FN_SDA2_C, FN_DU1_DR3, 0,
		FN_QSPI1_MISO_QSPI1_IO1, FN_SSI_WS78_B, FN_VI0_G3,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP11_7_4 [4] */
		FN_MSIOF0_RXD_A, FN_RX5_A, FN_SCL2_C, FN_DU1_DR2, 0,
		FN_QSPI1_MOSI_QSPI1_IO0, FN_SSI_SDATA6_B, FN_VI0_G2,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP11_3_0 [4] */
		FN_SDA1_A, FN_TX4_A, 0, FN_DU1_DR1, 0, 0, FN_SSI_WS6_B,
		FN_VI0_G1, 0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR12", 0xE6060070, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP12_31_28 [4] */
		FN_SD2_DAT2, FN_RX2_A, 0, FN_DU1_DB0, FN_SSI_SDATA2_B, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_27_24 [4] */
		FN_SD2_DAT1, FN_TX1_A, FN_SDA1_E, FN_DU1_DG7, FN_SSI_WS2_B,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_23_20 [4] */
		FN_SD2_DAT0, FN_RX1_A, FN_SCL1_E, FN_DU1_DG6,
		FN_SSI_SDATA1_B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_19_16 [4] */
		FN_SD2_CMD, FN_SCIF1_SCK_A, FN_TCLK2_A, FN_DU1_DG5,
		FN_SSI_SCK2_B, FN_PWM3_A, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_15_12 [4] */
		FN_SD2_CLK, FN_HSCK1, 0, FN_DU1_DG4, FN_SSI_SCK1_B, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_11_8 [4] */
		FN_HRTS1_N_A, 0, 0, FN_DU1_DG3, FN_SSI_WS1_B, FN_IRQ1, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_7_4 [4] */
		FN_HCTS1_N_A, FN_PWM2_A, 0, FN_DU1_DG2, FN_REMOCON_B,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP12_3_0 [4] */
		FN_HTX1_A, FN_SDA4_A, 0, FN_DU1_DG1, FN_TX0_A, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR13", 0xE6060074, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP13_31_28 [4] */
		FN_SSI_SCK5_A, 0, 0, FN_DU1_DOTCLKOUT1, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP13_27_24 [4] */
		FN_SDA2_A, 0, FN_MSIOF1_SYNC_B, FN_DU1_DB7, FN_AUDIO_CLKOUT_C,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP13_23_20 [4] */
		FN_SCL2_A, 0, FN_MSIOF1_SCK_B, FN_DU1_DB6, FN_AUDIO_CLKC_C,
		FN_SSI_SCK4_B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP13_19_16 [4] */
		FN_TX3_A, FN_SDA1_C, FN_MSIOF1_TXD_B, FN_DU1_DB5,
		FN_AUDIO_CLKB_C, FN_SSI_WS4_B, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP13_15_12 [4] */
		FN_RX3_A, FN_SCL1_C, FN_MSIOF1_RXD_B, FN_DU1_DB4,
		FN_AUDIO_CLKA_C, FN_SSI_SDATA4_B, 0, 0, 0, 0, 0, 0, 0, 0,
		0, 0,
		/* IP13_11_8 [4] */
		FN_SD2_WP, FN_SCIF3_SCK, 0, FN_DU1_DB3, FN_SSI_SDATA9_B, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP13_7_4 [4] */
		FN_SD2_CD, FN_SCIF2_SCK_A, 0, FN_DU1_DB2, FN_SSI_SCK9_B, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP13_3_0 [4] */
		FN_SD2_DAT3, FN_TX2_A, 0, FN_DU1_DB1, FN_SSI_WS9_B, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR14", 0xE6060078, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP14_31_28 [4] */
		FN_SSI_SDATA7_A, 0, 0, FN_IRQ8, FN_AUDIO_CLKA_D, FN_CAN_CLK_D,
		FN_VI0_G5, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP14_27_24 [4] */
		FN_SSI_WS78_A, 0, FN_SCL4_E, FN_DU1_CDE, 0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		/* IP14_23_20 [4] */
		FN_SSI_SCK78_A, 0, FN_SDA4_E, FN_DU1_DISP, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP14_19_16 [4] */
		FN_SSI_SDATA6_A, 0, FN_SDA4_C, FN_DU1_EXVSYNC_DU1_VSYNC, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP14_15_12 [4] */
		FN_SSI_WS6_A, 0, FN_SCL4_C, FN_DU1_EXHSYNC_DU1_HSYNC, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP14_11_8 [4] */
		FN_SSI_SCK6_A, 0, 0, FN_DU1_EXODDF_DU1_ODDF_DISP_CDE, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP14_7_4 [4] */
		FN_SSI_SDATA5_A, 0, FN_SDA3_C, FN_DU1_DOTCLKOUT0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP14_3_0 [4] */
		FN_SSI_WS5_A, 0, FN_SCL3_C, FN_DU1_DOTCLKIN, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR15", 0xE606007C, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP15_31_28 [4] */
		FN_SSI_WS4_A, 0, FN_AVB_PHY_INT, 0, 0, 0, FN_VI0_R5, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP15_27_24 [4] */
		FN_SSI_SCK4_A, 0, FN_AVB_MAGIC, 0, 0, 0, FN_VI0_R4, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		/* IP15_23_20 [4] */
		FN_SSI_SDATA3, FN_MSIOF1_SS2_A, FN_AVB_LINK, 0, FN_CAN1_TX_A,
		FN_DREQ2_N, FN_VI0_R3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP15_19_16 [4] */
		FN_SSI_WS34, FN_MSIOF1_SS1_A, FN_AVB_MDIO, 0, FN_CAN1_RX_A,
		FN_DREQ1_N, FN_VI0_R2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP15_15_12 [4] */
		FN_SSI_SCK34, FN_MSIOF1_SCK_A, FN_AVB_MDC, 0, 0, FN_DACK1,
		FN_VI0_R1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP15_11_8 [4] */
		FN_SSI_SDATA0_A, FN_MSIOF1_SYNC_A, FN_PWM0_C, 0, 0, 0,
		FN_VI0_R0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP15_7_4 [4] */
		FN_SSI_WS0129_A, FN_MSIOF1_TXD_A, FN_TX5_D, 0, 0, 0,
		FN_VI0_G7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP15_3_0 [4] */
		FN_SSI_SCK0129_A, FN_MSIOF1_RXD_A, FN_RX5_D, 0, 0, 0,
		FN_VI0_G6, 0, 0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR16", 0xE6060080, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP16_31_28 [4] */
		FN_SSI_SDATA2_A, FN_HRTS1_N_B, 0, 0, 0, 0,
		FN_VI0_DATA4_VI0_B4, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP16_27_24 [4] */
		FN_SSI_WS2_A, FN_HCTS1_N_B, 0, 0, 0, FN_AVB_TX_ER,
		FN_VI0_DATA3_VI0_B3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP16_23_20 [4] */
		FN_SSI_SCK2_A, FN_HTX1_B, 0, 0, 0, FN_AVB_TXD7,
		FN_VI0_DATA2_VI0_B2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP16_19_16 [4] */
		FN_SSI_SDATA1_A, FN_HRX1_B, 0, 0, 0, 0, FN_VI0_DATA1_VI0_B1,
		0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP16_15_12 [4] */
		FN_SSI_WS1_A, FN_TX1_B, 0, 0, FN_CAN0_TX_D,
		FN_AVB_AVTP_MATCH_B, FN_VI0_DATA0_VI0_B0, 0, 0, 0, 0, 0, 0,
		0, 0, 0,
		/* IP16_11_8 [4] */
		FN_SSI_SDATA8_A, FN_RX1_B, 0, 0, FN_CAN0_RX_D,
		FN_AVB_AVTP_CAPTURE_B, FN_VI0_R7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP16_7_4 [4] */
		FN_SSI_SCK1_A, FN_SCIF1_SCK_B, FN_PWM1_D, FN_IRQ9, FN_REMOCON_A,
		FN_DACK2, FN_VI0_CLK, FN_AVB_COL, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP16_3_0 [4] */
		FN_SSI_SDATA4_A, 0, FN_AVB_CRS, 0, 0, 0, FN_VI0_R6, 0, 0, 0,
		0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("IPSR17", 0xE6060084, 32,
			     4, 4, 4, 4, 4, 4, 4, 4) {
		/* IP17_31_28 [4] */
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_27_24 [4] */
		FN_AUDIO_CLKOUT_A, FN_SDA4_B, 0, 0, 0, 0,
		FN_VI0_VSYNC_N, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_23_20 [4] */
		FN_AUDIO_CLKC_A, FN_SCL4_B, 0, 0, 0, 0,
		FN_VI0_HSYNC_N, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_19_16 [4] */
		FN_AUDIO_CLKB_A, FN_SDA0_B, 0, 0, 0, 0,
		FN_VI0_FIELD, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_15_12 [4] */
		FN_AUDIO_CLKA_A, FN_SCL0_B, 0, 0, 0, 0,
		FN_VI0_CLKENB, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_11_8 [4] */
		FN_SSI_SDATA9_A, FN_SCIF2_SCK_B, FN_PWM2_D, 0, 0, 0,
		FN_VI0_DATA7_VI0_B7, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_7_4 [4] */
		FN_SSI_WS9_A, FN_TX2_B, FN_SDA3_E, 0, 0, 0,
		FN_VI0_DATA6_VI0_B6, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		/* IP17_3_0 [4] */
		FN_SSI_SCK9_A, FN_RX2_B, FN_SCL3_E, 0, 0, FN_EX_WAIT1,
		FN_VI0_DATA5_VI0_B5, 0, 0, 0, 0, 0, 0, 0, 0, 0, }
	},
	{ PINMUX_CFG_REG_VAR("MOD_SEL0", 0xE60600C0, 32,
			     1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 2, 1, 3, 3,
				 1, 2, 3, 3, 1) {
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* SEL_ADGA [2] */
		FN_SEL_ADGA_0, FN_SEL_ADGA_1, FN_SEL_ADGA_2, FN_SEL_ADGA_3,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* SEL_CANCLK [2] */
		FN_SEL_CANCLK_0, FN_SEL_CANCLK_1, FN_SEL_CANCLK_2,
		FN_SEL_CANCLK_3,
		/* SEL_CAN1 [2] */
		FN_SEL_CAN1_0, FN_SEL_CAN1_1, FN_SEL_CAN1_2, FN_SEL_CAN1_3,
		/* SEL_CAN0 [2] */
		FN_SEL_CAN0_0, FN_SEL_CAN0_1, FN_SEL_CAN0_2, FN_SEL_CAN0_3,
		/* RESERVED [1] */
		0, 0,
		/* SEL_I2C04 [3] */
		FN_SEL_I2C04_0, FN_SEL_I2C04_1, FN_SEL_I2C04_2, FN_SEL_I2C04_3,
		FN_SEL_I2C04_4, 0, 0, 0,
		/* SEL_I2C03 [3] */
		FN_SEL_I2C03_0, FN_SEL_I2C03_1, FN_SEL_I2C03_2, FN_SEL_I2C03_3,
		FN_SEL_I2C03_4, 0, 0, 0,
		/* RESERVED [1] */
		0, 0,
		/* SEL_I2C02 [2] */
		FN_SEL_I2C02_0, FN_SEL_I2C02_1, FN_SEL_I2C02_2, FN_SEL_I2C02_3,
		/* SEL_I2C01 [3] */
		FN_SEL_I2C01_0, FN_SEL_I2C01_1, FN_SEL_I2C01_2, FN_SEL_I2C01_3,
		FN_SEL_I2C01_4, 0, 0, 0,
		/* SEL_I2C00 [3] */
		FN_SEL_I2C00_0, FN_SEL_I2C00_1, FN_SEL_I2C00_2, FN_SEL_I2C00_3,
		FN_SEL_I2C00_4, 0, 0, 0,
		/* SEL_AVB [1] */
		FN_SEL_AVB_0, FN_SEL_AVB_1, }
	},
	{ PINMUX_CFG_REG_VAR("MOD_SEL1", 0xE60600C4, 32,
			     1, 3, 3, 2, 2, 1, 2, 2,
			     2, 1, 1, 1, 1, 1, 2, 1, 1, 2, 2, 1) {
		/* SEL_SCIFCLK [1] */
		FN_SEL_SCIFCLK_0, FN_SEL_SCIFCLK_1,
		/* SEL_SCIF5 [3] */
		FN_SEL_SCIF5_0, FN_SEL_SCIF5_1, FN_SEL_SCIF5_2, FN_SEL_SCIF5_3,
		FN_SEL_SCIF5_4, FN_SEL_SCIF5_5, 0, 0,
		/* SEL_SCIF4 [3] */
		FN_SEL_SCIF4_0, FN_SEL_SCIF4_1, FN_SEL_SCIF4_2, FN_SEL_SCIF4_3,
		FN_SEL_SCIF4_4, 0, 0, 0,
		/* SEL_SCIF3 [2] */
		FN_SEL_SCIF3_0, FN_SEL_SCIF3_1, FN_SEL_SCIF3_2, 0,
		/* SEL_SCIF2 [2] */
		FN_SEL_SCIF2_0, FN_SEL_SCIF2_1, FN_SEL_SCIF2_2, 0,
		/* SEL_SCIF2_CLK [1] */
		FN_SEL_SCIF2_CLK_0, FN_SEL_SCIF2_CLK_1,
		/* SEL_SCIF1 [2] */
		FN_SEL_SCIF1_0, FN_SEL_SCIF1_1, FN_SEL_SCIF1_2, FN_SEL_SCIF1_3,
		/* SEL_SCIF0 [2] */
		FN_SEL_SCIF0_0, FN_SEL_SCIF0_1, FN_SEL_SCIF0_2, FN_SEL_SCIF0_3,
		/* SEL_MSIOF2 [2] */
		FN_SEL_MSIOF2_0, FN_SEL_MSIOF2_1, FN_SEL_MSIOF2_2, 0,
		/* RESERVED [1] */
		0, 0,
		/* SEL_MSIOF1 [1] */
		FN_SEL_MSIOF1_0, FN_SEL_MSIOF1_1,
		/* RESERVED [1] */
		0, 0,
		/* SEL_MSIOF0 [1] */
		FN_SEL_MSIOF0_0, FN_SEL_MSIOF0_1,
		/* SEL_RCN [1] */
		FN_SEL_RCN_0, FN_SEL_RCN_1,
		/* RESERVED [2] */
		0, 0, 0, 0,
		/* SEL_TMU2 [1] */
		FN_SEL_TMU2_0, FN_SEL_TMU2_1,
		/* SEL_TMU1 [1] */
		FN_SEL_TMU1_0, FN_SEL_TMU1_1,
		/* RESERVED [2] */
		0, 0, 0, 0,
		/* SEL_HSCIF1 [2] */
		FN_SEL_HSCIF1_0, FN_SEL_HSCIF1_1, FN_SEL_HSCIF1_2, 0,
		/* SEL_HSCIF0 [1] */
		FN_SEL_HSCIF0_0, FN_SEL_HSCIF0_1,}
	},
	{ PINMUX_CFG_REG_VAR("MOD_SEL2", 0xE60600C8, 32,
			     1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
				 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2) {
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* RESERVED [1] */
		0, 0,
		/* SEL_ADGB [2] */
		FN_SEL_ADGB_0, FN_SEL_ADGB_1, FN_SEL_ADGB_2, 0,
		/* SEL_ADGC [2] */
		FN_SEL_ADGC_0, FN_SEL_ADGC_1, FN_SEL_ADGC_2, 0,
		/* SEL_SSI9 [2] */
		FN_SEL_SSI9_0, FN_SEL_SSI9_1, 0, 0,
		/* SEL_SSI8 [2] */
		FN_SEL_SSI8_0, FN_SEL_SSI8_1, 0, 0,
		/* SEL_SSI7 [2] */
		FN_SEL_SSI7_0, FN_SEL_SSI7_1, 0, 0,
		/* SEL_SSI6 [2] */
		FN_SEL_SSI6_0, FN_SEL_SSI6_1, 0, 0,
		/* SEL_SSI5 [2] */
		FN_SEL_SSI5_0, FN_SEL_SSI5_1, 0, 0,
		/* SEL_SSI4 [2] */
		FN_SEL_SSI4_0, FN_SEL_SSI4_1, 0, 0,
		/* SEL_SSI2 [2] */
		FN_SEL_SSI2_0, FN_SEL_SSI2_1, 0, 0,
		/* SEL_SSI1 [2] */
		FN_SEL_SSI1_0, FN_SEL_SSI1_1, FN_SEL_SSI1_2, FN_SEL_SSI1_3,
		/* SEL_SSI0 [2] */
		FN_SEL_SSI0_0, FN_SEL_SSI0_1, 0, 0, }
	},
	{ },
};

#ifdef CONFIG_PINCTRL_PFC_R8A77470
const struct sh_pfc_soc_info r8a77470_pinmux_info = {
	.name = "r8a77470_pfc",
	.unlock_reg = 0xe6060000, /* PMMR */

	.function = { PINMUX_FUNCTION_BEGIN, PINMUX_FUNCTION_END },

	.pins = pinmux_pins,
	.nr_pins = ARRAY_SIZE(pinmux_pins),
	.groups = pinmux_groups,
	.nr_groups = ARRAY_SIZE(pinmux_groups),
	.functions = pinmux_functions,
	.nr_functions = ARRAY_SIZE(pinmux_functions),

	.cfg_regs = pinmux_config_regs,

	.pinmux_data = pinmux_data,
	.pinmux_data_size = ARRAY_SIZE(pinmux_data),
};
#endif
