package gnu.crypto.sasl.srp;

// ----------------------------------------------------------------------------
// $Id: CALG.java,v 1.2 2003/05/30 13:05:57 raif Exp $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import gnu.crypto.Registry;
import gnu.crypto.assembly.Assembly;
import gnu.crypto.assembly.Cascade;
import gnu.crypto.assembly.Direction;
import gnu.crypto.assembly.Stage;
import gnu.crypto.assembly.Transformer;
import gnu.crypto.assembly.TransformerException;
import gnu.crypto.cipher.CipherFactory;
import gnu.crypto.cipher.IBlockCipher;
import gnu.crypto.mode.IMode;
import gnu.crypto.mode.ModeFactory;
import gnu.crypto.pad.IPad;
import gnu.crypto.pad.PadFactory;
import gnu.crypto.sasl.ConfidentialityException;

import java.util.HashMap;

import javax.security.sasl.SaslException;

/**
 * <p>A Factory class that returns CALG (Confidentiality Algorithm) instances
 * that operate as described in the draft-burdis-cat-sasl-srp-08.</p>
 *
 * <p>The designated CALG block cipher should be used in OFB (Output Feedback
 * Block) mode in the ISO variant, as described in <i>The Handbook of Applied
 * Cryptography</i>, algorithm 7.20.</p>
 *
 * <p>Let <code>k</code> be the block size of the chosen symmetric key block
 * cipher algorithm; e.g. for AES this is <code>128</code> bits or <code>16</code>
 * octets.  The OFB mode used shall be of length/size <code>k</code>.</p>
 *
 * <p>It is recommended that block ciphers operating in OFB mode be used with an
 * Initial Vector (the mode's IV). In such a mode of operation - OFB with key
 * re-use - the IV need not be secret. For the mechanism in question the IVs
 * shall be a random octet sequence of <code>k</code> bytes.</p>
 *
 * The input data to the confidentiality protection algorithm shall be
 * a multiple of the symmetric cipher block size <code>k</code>. When the input
 * length is not a multiple of <code>k</code> octets, the data shall be padded
 * according to the following scheme:</p>
 *
 * <p>Assuming the length of the input is <code>l</code> octets,
 * <code>(k - (l mod k))</code> octets, all having the value
 * <code>(k - (l mod k))</code>, shall be appended to the original data. In
 * other words, the input is padded at the trailing end with one of the
 * following sequences:</p>
 *
 * <pre>
 *
 *                    01 -- if l mod k = k-1
 *                   02 02 -- if l mod k = k-2
 *                             ...
 *                             ...
 *                             ...
 *                 k k ... k k -- if l mod k = 0
 *</pre>
 *
 * <p>The padding can be removed unambiguously since all input is padded and no
 * padding sequence is a suffix of another. This padding method is well-defined
 * if and only if <code>k &lt; 256</code> octets, which is the case with
 * symmetric key block ciphers today, and in the forseeable future.</p>
 *
 * @version $Revision: 1.2 $
 */
public final class CALG {

	// Constants and variables
	// --------------------------------------------------------------------------

   private Assembly assembly;
   private Object modeNdx; // initialisation key of the cascade's attributes
   private int blockSize; // the underlying cipher's blocksize == IV length

	// Constructor(s)
	// --------------------------------------------------------------------------

	/** Private constructor to enforce instantiation through Factory method. */
	private CALG(int blockSize, Object modeNdx, Assembly assembly) {
		super();

      this.blockSize = blockSize;
      this.modeNdx = modeNdx;
		this.assembly = assembly;
	}

	// Class methods
	// -------------------------------------------------------------------------

   /**
    * <p>Returns an instance of a SASL-SRP CALG implementation.</p>
    *
    * @param algorithm the name of the symmetric cipher algorithm.
    * @return an instance of this object.
    */
   public static synchronized CALG getInstance(String algorithm) {
      IBlockCipher cipher = CipherFactory.getInstance(algorithm);
      int blockSize = cipher.defaultBlockSize();
      Cascade ofbCipher = new Cascade();
      Object modeNdx = ofbCipher.append(
            Stage.getInstance(
                  ModeFactory.getInstance(Registry.OFB_MODE, cipher, blockSize),
                  Direction.FORWARD));
      IPad pkcs7 = PadFactory.getInstance(Registry.PKCS7_PAD);
      // the passed IV may be longer that what we need. ensure correct length
//      byte[] realIV = null;
//      if (iv.length == blockSize) {
//         realIV = iv;
//      } else {
//         realIV = new byte[blockSize];
//         if (iv.length > blockSize) {
//            System.arraycopy(iv, 0, realIV, 0, blockSize);
//         } else { // shouldnt happen
//            System.arraycopy(iv, 0, realIV, 0, iv.length);
//         }
//      }

//      HashMap modeAttributes = new HashMap();
//      modeAttributes.put(IBlockCipher.KEY_MATERIAL, K.clone());
//      modeAttributes.put(IMode.IV, realIV);

      Assembly asm = new Assembly();
      asm.addPreTransformer(Transformer.getCascadeTransformer(ofbCipher));
      asm.addPreTransformer(Transformer.getPaddingTransformer(pkcs7));

//      HashMap attributes = new HashMap();
//      attributes.put(Assembly.DIRECTION, dir);
//      attributes.put(modeNdx, modeAttributes);
//      try {
//         asm.init(attributes);
//      } catch (TransformerException x) {
//         throw new SaslException("getInstance()", x);
//      }

      return new CALG(blockSize, modeNdx, asm);
   }

	// Instance methods
	// -------------------------------------------------------------------------

   /**
    * <p>Initialises a SASL-SRP CALG implementation.</p>
    *
    * @param K the shared secret.
    * @param iv the initial vector value to use.
    * @param dir whether this CALG is used for encryption or decryption.
    */
   public void init(byte[] K, byte[] iv, Direction dir) throws SaslException {
//      IBlockCipher cipher = CipherFactory.getInstance(algorithm);
//      int blockSize = cipher.defaultBlockSize();
//      Cascade ofbCipher = new Cascade();
//      Object modeNdx = ofbCipher.append(
//            Stage.getInstace(
//                  ModeFactory.getInstance(Registry.OFB_MODE, cipher, blockSize),
//                  Direction.FORWARD));
//      IPad pkcs7 = PadFactory.getInstance(Registry.PKCS7_PAD);
      // the passed IV may be longer that what we need. ensure correct length
      byte[] realIV = null;
      if (iv.length == blockSize) {
         realIV = iv;
      } else {
         realIV = new byte[blockSize];
         if (iv.length > blockSize) {
            System.arraycopy(iv, 0, realIV, 0, blockSize);
         } else { // shouldnt happen
            System.arraycopy(iv, 0, realIV, 0, iv.length);
         }
      }

      // TODO: generate key material from K through the KDF

      HashMap modeAttributes = new HashMap();
      modeAttributes.put(IBlockCipher.KEY_MATERIAL, K.clone());
      modeAttributes.put(IMode.IV, realIV);

//      Assembly asm = new Assembly();
//      asm.addPreTransformer(Transformer.getCascadeTransformer(ofbCipher));
//      asm.addPreTransformer(Transformer.getPaddingTransformer(pkcs7));

      HashMap attributes = new HashMap();
      attributes.put(Assembly.DIRECTION, dir);
      attributes.put(modeNdx, modeAttributes);
      try {
//         asm.init(attributes);
         assembly.init(attributes);
      } catch (TransformerException x) {
         throw new SaslException("getInstance()", x);
      }

//      return new CALG(asm);
   }

	/**
	 * <p>Encrypts or decrypts, depending on the mode already set, a designated
	 * array of bytes and returns the result.</p>
	 *
	 * @param data the data to encrypt/decrypt
	 * @return the decrypted/encrypted result.
	 * @throws ConfidentialityException if an exception occurs duirng the process.
	 */
	public byte[] doFinal(byte[] data) throws ConfidentialityException {
      byte[] result = null;
      try {
         result = assembly.lastUpdate(data);
      } catch (TransformerException x) {
         throw new ConfidentialityException("doFinal()", x);
      }
		return result;
	}
}