/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.db.commitlog.CommitLog;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractCommitLogService {
    static final long DEFAULT_MARKER_INTERVAL_MILLIS = 100L;
    private volatile Thread thread;
    private volatile boolean shutdown = false;
    protected volatile long lastSyncedAt = System.currentTimeMillis();
    private final AtomicLong written = new AtomicLong(0L);
    protected final AtomicLong pending = new AtomicLong(0L);
    protected final WaitQueue syncComplete = new WaitQueue();
    protected final Semaphore haveWork = new Semaphore(1);
    final CommitLog commitLog;
    private final String name;
    final long syncIntervalMillis;
    final long markerIntervalMillis;
    private volatile boolean syncRequested;
    private static final Logger logger = LoggerFactory.getLogger(AbstractCommitLogService.class);

    AbstractCommitLogService(CommitLog commitLog, String name, long syncIntervalMillis) {
        this(commitLog, name, syncIntervalMillis, false);
    }

    AbstractCommitLogService(CommitLog commitLog, String name, long syncIntervalMillis, boolean markHeadersFaster) {
        this.commitLog = commitLog;
        this.name = name;
        if (markHeadersFaster && syncIntervalMillis > 100L) {
            this.markerIntervalMillis = 100L;
            long modulo = syncIntervalMillis % this.markerIntervalMillis;
            if (modulo != 0L) {
                syncIntervalMillis -= modulo;
                if (modulo >= this.markerIntervalMillis / 2L) {
                    syncIntervalMillis += this.markerIntervalMillis;
                }
            }
            logger.debug("Will update the commitlog markers every {}ms and flush every {}ms", (Object)this.markerIntervalMillis, (Object)syncIntervalMillis);
        } else {
            this.markerIntervalMillis = syncIntervalMillis;
        }
        assert (syncIntervalMillis % this.markerIntervalMillis == 0L);
        this.syncIntervalMillis = syncIntervalMillis;
    }

    void start() {
        if (this.syncIntervalMillis < 1L) {
            throw new IllegalArgumentException(String.format("Commit log flush interval must be positive: %dms", this.syncIntervalMillis));
        }
        this.shutdown = false;
        SyncRunnable runnable = new SyncRunnable(new Clock());
        this.thread = new Thread(NamedThreadFactory.threadLocalDeallocator(runnable), this.name);
        this.thread.start();
    }

    public void finishWriteFor(CommitLogSegment.Allocation alloc) {
        this.maybeWaitForSync(alloc);
        this.written.incrementAndGet();
    }

    protected abstract void maybeWaitForSync(CommitLogSegment.Allocation var1);

    public WaitQueue.Signal requestExtraSync() {
        WaitQueue.Signal signal = this.syncComplete.register();
        this.requestSync();
        return signal;
    }

    protected void requestSync() {
        this.syncRequested = true;
        this.haveWork.release(1);
    }

    public void shutdown() {
        this.shutdown = true;
        this.haveWork.release(1);
    }

    public void restartUnsafe() {
        while (this.haveWork.availablePermits() < 1) {
            this.haveWork.release();
        }
        while (this.haveWork.availablePermits() > 1) {
            try {
                this.haveWork.acquire();
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        this.shutdown = false;
        this.start();
    }

    public void awaitTermination() throws InterruptedException {
        this.thread.join();
    }

    public long getCompletedTasks() {
        return this.written.get();
    }

    public long getPendingTasks() {
        return this.pending.get();
    }

    class SyncRunnable
    implements Runnable {
        final Clock clock;
        long firstLagAt = 0L;
        long totalSyncDuration = 0L;
        long syncExceededIntervalBy = 0L;
        int lagCount = 0;
        int syncCount = 0;

        SyncRunnable(Clock clock) {
            this.clock = clock;
        }

        @Override
        public void run() {
            while (this.sync()) {
            }
        }

        boolean sync() {
            try {
                boolean flushToDisk;
                boolean run = !AbstractCommitLogService.this.shutdown;
                long pollStarted = this.clock.currentTimeMillis();
                boolean bl = flushToDisk = AbstractCommitLogService.this.lastSyncedAt + AbstractCommitLogService.this.syncIntervalMillis <= pollStarted || AbstractCommitLogService.this.shutdown || AbstractCommitLogService.this.syncRequested;
                if (flushToDisk) {
                    AbstractCommitLogService.this.syncRequested = false;
                    AbstractCommitLogService.this.commitLog.sync(AbstractCommitLogService.this.shutdown, true);
                    AbstractCommitLogService.this.lastSyncedAt = pollStarted;
                    AbstractCommitLogService.this.syncComplete.signalAll();
                    ++this.syncCount;
                } else {
                    AbstractCommitLogService.this.commitLog.sync(false, false);
                }
                long now = this.clock.currentTimeMillis();
                if (flushToDisk) {
                    this.maybeLogFlushLag(pollStarted, now);
                }
                if (!run) {
                    return false;
                }
                long sleep = pollStarted + AbstractCommitLogService.this.markerIntervalMillis - now;
                if (sleep < 0L) {
                    return true;
                }
                try {
                    AbstractCommitLogService.this.haveWork.tryAcquire(sleep, TimeUnit.MILLISECONDS);
                    AbstractCommitLogService.this.haveWork.drainPermits();
                }
                catch (InterruptedException e) {
                    throw new AssertionError();
                }
            }
            catch (Throwable t) {
                if (!CommitLog.handleCommitError("Failed to persist commits to disk", t)) {
                    return false;
                }
                try {
                    AbstractCommitLogService.this.haveWork.tryAcquire(AbstractCommitLogService.this.markerIntervalMillis, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    throw new AssertionError();
                }
            }
            return true;
        }

        @VisibleForTesting
        boolean maybeLogFlushLag(long pollStarted, long now) {
            boolean logged;
            long flushDuration = now - pollStarted;
            this.totalSyncDuration += flushDuration;
            long maxFlushTimestamp = pollStarted + AbstractCommitLogService.this.syncIntervalMillis;
            if (maxFlushTimestamp > now) {
                return false;
            }
            if (this.firstLagAt == 0L) {
                this.firstLagAt = now;
                this.lagCount = 0;
                this.syncExceededIntervalBy = 0;
                this.syncCount = 1;
                this.totalSyncDuration = flushDuration;
            }
            this.syncExceededIntervalBy += now - maxFlushTimestamp;
            ++this.lagCount;
            if (this.firstLagAt > 0L && (logged = NoSpamLogger.log(logger, NoSpamLogger.Level.WARN, 5L, TimeUnit.MINUTES, "Out of {} commit log syncs over the past {}s with average duration of {}ms, {} have exceeded the configured commit interval by an average of {}ms", this.syncCount, (now - this.firstLagAt) / 1000L, String.format("%.2f", (double)this.totalSyncDuration / (double)this.syncCount), this.lagCount, String.format("%.2f", (double)this.syncExceededIntervalBy / (double)this.lagCount)))) {
                this.firstLagAt = 0L;
            }
            return true;
        }

        @VisibleForTesting
        long getTotalSyncDuration() {
            return this.totalSyncDuration;
        }
    }
}

