/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package javaobject;

import org.apache.geode.security.AuthenticationFailedException;
import org.apache.geode.security.SecurityManager;

import java.util.Properties;

import javaobject.UserPasswordAuthInit;
import javaobject.UsernamePrincipal;

/**
 * This Security manager only Authenticates - and allows any operations.
 */
public class SimpleSecurityManager implements SecurityManager {


    /**
     * Verify the credentials provided in the properties
     * <p>
     * Your security manager needs to validate credentials coming from all communication channels.
     * If you use AuthInitialize to generate your client/peer credentials, then the input of this
     * method is the output of your AuthInitialize.getCredentials method. But remember that this
     * method will also need to validate credentials coming from gfsh/jmx/rest client, the framework
     * is putting the username/password under security-username and security-password keys in the
     * property, so your securityManager implementation needs to validate these kind of properties
     * as well.
     *
     * @param credentials it contains the security-username and security-password as keys of the
     *                    properties, also the properties generated by your AuthInitialize interface
     * @return a serializable principal object
     */
    @Override
    public Object authenticate(Properties props) throws AuthenticationFailedException {
        String userName = props.getProperty(UserPasswordAuthInit.USER_NAME);
        if (userName == null) {
            throw new AuthenticationFailedException(
                    "SimpleSecurityManager: user name property ["
                            + UserPasswordAuthInit.USER_NAME + "] not provided");
        }
        String password = props.getProperty(UserPasswordAuthInit.PASSWORD);
        if (password == null) {
            throw new AuthenticationFailedException(
                    "SimpleSecurityManager: password property ["
                            + UserPasswordAuthInit.PASSWORD + "] not provided");
        }

        if (userName.equals(password) && testValidName(userName)) {
            return new UsernamePrincipal(userName);
        } else {
            throw new AuthenticationFailedException(
                    "SimpleSecurityManager: Invalid user name [" + userName
                            + "], password supplied.");
        }
    }

    private boolean testValidName(String userName) {

        return (userName.startsWith("server") || userName.startsWith("user")
                || userName.startsWith("reader") || userName.startsWith("writer")
                || userName.equals("admin")  || userName.equals("root")
                || userName.equals("administrator"));
    }
}
