/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Optional;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.metastore.auth.HttpAuthenticationException;
import org.apache.hadoop.hive.metastore.auth.jwt.JWTValidator;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hadoop.security.UserGroupInformation;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.pac4j.core.context.JEEContext;
import org.pac4j.core.context.WebContext;
import org.pac4j.core.credentials.TokenCredentials;
import org.pac4j.core.credentials.extractor.BearerAuthExtractor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ServletSecurity {
    private static final Logger LOG = LoggerFactory.getLogger(ServletSecurity.class);
    static final String X_USER = "x-actor-username";
    private final boolean isSecurityEnabled;
    private final AuthType authType;
    private final Configuration conf;
    private JWTValidator jwtValidator = null;

    public ServletSecurity(AuthType authType, Configuration conf) {
        this.conf = conf;
        this.isSecurityEnabled = UserGroupInformation.isSecurityEnabled();
        this.authType = authType;
    }

    public void init() throws ServletException {
        if (this.authType == AuthType.JWT && this.jwtValidator == null) {
            try {
                this.jwtValidator = new JWTValidator(this.conf);
            }
            catch (Exception e) {
                throw new ServletException("Failed to initialize ServletSecurity.", (Throwable)e);
            }
        }
    }

    public HttpServlet proxy(HttpServlet servlet) {
        if (this.authType == AuthType.NONE) {
            LOG.warn("Servlet security is disabled for {}", (Object)servlet);
            return servlet;
        }
        try {
            this.init();
        }
        catch (ServletException e) {
            LOG.error("Unable to proxy security for servlet {}", (Object)servlet, (Object)e);
            return null;
        }
        return new ProxyServlet(servlet);
    }

    public void execute(HttpServletRequest request, HttpServletResponse response, MethodExecutor executor) throws IOException {
        UserGroupInformation clientUgi;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Logging headers in {} request", (Object)request.getMethod());
            Enumeration headerNames = request.getHeaderNames();
            while (headerNames.hasMoreElements()) {
                String headerName = (String)headerNames.nextElement();
                LOG.debug("Header: [{}], Value: [{}]", (Object)headerName, (Object)request.getHeader(headerName));
            }
        }
        try {
            String userFromHeader = this.extractUserName(request, response);
            if (this.isSecurityEnabled || this.authType == AuthType.JWT) {
                LOG.info("Creating proxy user for: {}", (Object)userFromHeader);
                clientUgi = UserGroupInformation.createProxyUser((String)userFromHeader, (UserGroupInformation)UserGroupInformation.getLoginUser());
            } else {
                Preconditions.checkState((this.authType == AuthType.SIMPLE ? 1 : 0) != 0);
                LOG.info("Creating remote user for: {}", (Object)userFromHeader);
                clientUgi = UserGroupInformation.createRemoteUser((String)userFromHeader);
            }
        }
        catch (HttpAuthenticationException e) {
            response.setStatus(401);
            response.getWriter().printf("Authentication error: %s", e.getMessage());
            LOG.error("Authentication error: ", (Throwable)e);
            return;
        }
        PrivilegedExceptionAction<Void> action = () -> {
            executor.execute(request, response);
            return null;
        };
        try {
            clientUgi.doAs(action);
        }
        catch (InterruptedException e) {
            LOG.info("Interrupted when executing http request as user: {}", (Object)clientUgi.getUserName(), (Object)e);
            Thread.currentThread().interrupt();
        }
        catch (RuntimeException e) {
            throw new IOException("Exception when executing http request as user: " + clientUgi.getUserName(), e);
        }
    }

    private String extractUserName(HttpServletRequest request, HttpServletResponse response) throws HttpAuthenticationException {
        String user;
        if (this.authType == AuthType.SIMPLE) {
            String userFromHeader = request.getHeader(X_USER);
            if (userFromHeader == null || userFromHeader.isEmpty()) {
                throw new HttpAuthenticationException("User header x-actor-username missing in request");
            }
            return userFromHeader;
        }
        Preconditions.checkState((this.authType == AuthType.JWT ? 1 : 0) != 0);
        String signedJwt = this.extractBearerToken(request, response);
        if (signedJwt == null) {
            throw new HttpAuthenticationException("Couldn't find bearer token in the auth header in the request");
        }
        try {
            user = this.jwtValidator.validateJWTAndExtractUser(signedJwt);
            Preconditions.checkNotNull((Object)user, (Object)"JWT needs to contain the user name as subject");
            Preconditions.checkState((!user.isEmpty() ? 1 : 0) != 0, (Object)"User name should not be empty in JWT");
            LOG.info("Successfully validated and extracted user name {} from JWT in Auth header in the request", (Object)user);
        }
        catch (Exception e) {
            throw new HttpAuthenticationException("Failed to validate JWT from Bearer token in Authentication header", e);
        }
        return user;
    }

    private String extractBearerToken(HttpServletRequest request, HttpServletResponse response) {
        BearerAuthExtractor extractor = new BearerAuthExtractor();
        Optional tokenCredentials = extractor.extract((WebContext)new JEEContext(request, response));
        return tokenCredentials.map(TokenCredentials::getToken).orElse(null);
    }

    static void loginServerPrincipal(Configuration conf) throws IOException {
        LOG.info(" Checking if security is enabled");
        if (UserGroupInformation.isSecurityEnabled()) {
            LOG.info("Logging in via keytab while starting HTTP metastore");
            String kerberosName = SecurityUtil.getServerPrincipal((String)MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.KERBEROS_PRINCIPAL), (String)"0.0.0.0");
            String keyTabFile = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.KERBEROS_KEYTAB_FILE);
            UserGroupInformation.loginUserFromKeytab((String)kerberosName, (String)keyTabFile);
        } else {
            LOG.info("Security is not enabled. Not logging in via keytab");
        }
    }

    static SslContextFactory createSslContextFactory(Configuration conf) throws IOException {
        boolean useSsl = MetastoreConf.getBoolVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.USE_SSL);
        if (!useSsl) {
            return null;
        }
        String keyStorePath = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYSTORE_PATH).trim();
        if (keyStorePath.isEmpty()) {
            throw new IllegalArgumentException(MetastoreConf.ConfVars.SSL_KEYSTORE_PATH + " Not configured for SSL connection");
        }
        String keyStorePassword = MetastoreConf.getPassword((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYSTORE_PASSWORD);
        String keyStoreType = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYSTORE_TYPE).trim();
        String keyStoreAlgorithm = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_KEYMANAGERFACTORY_ALGORITHM).trim();
        Object[] excludedProtocols = MetastoreConf.getVar((Configuration)conf, (MetastoreConf.ConfVars)MetastoreConf.ConfVars.SSL_PROTOCOL_BLACKLIST).split(",");
        if (LOG.isInfoEnabled()) {
            LOG.info("HTTP Server SSL: adding excluded protocols: {}", (Object)Arrays.toString(excludedProtocols));
        }
        SslContextFactory.Server factory = new SslContextFactory.Server();
        factory.addExcludeProtocols((String[])excludedProtocols);
        if (LOG.isInfoEnabled()) {
            LOG.info("HTTP Server SSL: SslContextFactory.getExcludeProtocols = {}", (Object)Arrays.toString(factory.getExcludeProtocols()));
        }
        factory.setKeyStorePath(keyStorePath);
        factory.setKeyStorePassword(keyStorePassword);
        factory.setKeyStoreType(keyStoreType);
        factory.setKeyManagerFactoryAlgorithm(keyStoreAlgorithm);
        return factory;
    }

    public static enum AuthType {
        NONE,
        SIMPLE,
        JWT;


        public static AuthType fromString(String type) {
            return AuthType.valueOf(type.toUpperCase());
        }
    }

    public class ProxyServlet
    extends HttpServlet {
        private final HttpServlet delegate;

        ProxyServlet(HttpServlet delegate) {
            this.delegate = delegate;
        }

        public void init() throws ServletException {
            ServletSecurity.this.init();
            this.delegate.init();
        }

        public void service(HttpServletRequest request, HttpServletResponse response) throws IOException {
            ServletSecurity.this.execute(request, response, (arg_0, arg_1) -> ((HttpServlet)this.delegate).service(arg_0, arg_1));
        }

        public String getServletName() {
            try {
                return this.delegate.getServletName();
            }
            catch (IllegalStateException ill) {
                return this.delegate.toString();
            }
        }

        public String getServletInfo() {
            return this.delegate.getServletInfo();
        }
    }

    @FunctionalInterface
    public static interface MethodExecutor {
        public void execute(HttpServletRequest var1, HttpServletResponse var2) throws ServletException, IOException;
    }
}

