/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.store.kv;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.rocketmq.common.ThreadFactoryImpl;
import org.apache.rocketmq.common.TopicConfig;
import org.apache.rocketmq.common.attribute.CleanupPolicy;
import org.apache.rocketmq.common.utils.CleanupPolicyUtils;
import org.apache.rocketmq.common.utils.ThreadUtils;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.store.DefaultMessageStore;
import org.apache.rocketmq.store.DispatchRequest;
import org.apache.rocketmq.store.GetMessageResult;
import org.apache.rocketmq.store.SelectMappedBufferResult;
import org.apache.rocketmq.store.config.MessageStoreConfig;
import org.apache.rocketmq.store.kv.CompactionLog;
import org.apache.rocketmq.store.kv.CompactionPositionMgr;

public class CompactionStore {
    public static final String COMPACTION_DIR = "compaction";
    public static final String COMPACTION_LOG_DIR = "compactionLog";
    public static final String COMPACTION_CQ_DIR = "compactionCq";
    private final String compactionPath;
    private final String compactionLogPath;
    private final String compactionCqPath;
    private final DefaultMessageStore defaultMessageStore;
    private final CompactionPositionMgr positionMgr;
    private final ConcurrentHashMap<String, CompactionLog> compactionLogTable;
    private final ScheduledExecutorService compactionSchedule;
    private final int scanInterval = 30000;
    private final int compactionInterval;
    private final int compactionThreadNum;
    private final int offsetMapSize;
    private String masterAddr;
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqStore");

    public CompactionStore(DefaultMessageStore defaultMessageStore) {
        this.defaultMessageStore = defaultMessageStore;
        this.compactionLogTable = new ConcurrentHashMap();
        MessageStoreConfig config = defaultMessageStore.getMessageStoreConfig();
        String storeRootPath = config.getStorePathRootDir();
        this.compactionPath = Paths.get(storeRootPath, COMPACTION_DIR).toString();
        this.compactionLogPath = Paths.get(this.compactionPath, COMPACTION_LOG_DIR).toString();
        this.compactionCqPath = Paths.get(this.compactionPath, COMPACTION_CQ_DIR).toString();
        this.positionMgr = new CompactionPositionMgr(this.compactionPath);
        this.compactionThreadNum = Math.min(Runtime.getRuntime().availableProcessors(), Math.max(1, config.getCompactionThreadNum()));
        this.compactionSchedule = ThreadUtils.newScheduledThreadPool((int)this.compactionThreadNum, (ThreadFactory)new ThreadFactoryImpl("compactionSchedule_"));
        this.offsetMapSize = config.getMaxOffsetMapSize() / this.compactionThreadNum;
        this.compactionInterval = defaultMessageStore.getMessageStoreConfig().getCompactionScheduleInternal();
    }

    public void load(boolean exitOk) throws Exception {
        File logRoot = new File(this.compactionLogPath);
        File[] fileTopicList = logRoot.listFiles();
        if (fileTopicList != null) {
            for (File fileTopic : fileTopicList) {
                File[] fileQueueIdList;
                if (!fileTopic.isDirectory() || (fileQueueIdList = fileTopic.listFiles()) == null) continue;
                for (File fileQueueId : fileQueueIdList) {
                    if (!fileQueueId.isDirectory()) continue;
                    try {
                        String topic = fileTopic.getName();
                        int queueId = Integer.parseInt(fileQueueId.getName());
                        if (Files.isDirectory(Paths.get(this.compactionCqPath, topic, String.valueOf(queueId)), new LinkOption[0])) {
                            this.loadAndGetClog(topic, queueId);
                            continue;
                        }
                        log.error("{}:{} compactionLog mismatch with compactionCq", (Object)topic, (Object)queueId);
                    }
                    catch (Exception e) {
                        log.error("load compactionLog {}:{} exception: ", new Object[]{fileTopic.getName(), fileQueueId.getName(), e});
                        throw new Exception("load compactionLog " + fileTopic.getName() + ":" + fileQueueId.getName() + " exception: " + e.getMessage());
                    }
                }
            }
        }
        log.info("compactionStore {}:{} load completed.", (Object)this.compactionLogPath, (Object)this.compactionCqPath);
        this.compactionSchedule.scheduleWithFixedDelay(this::scanAllTopicConfig, 30000L, 30000L, TimeUnit.MILLISECONDS);
        log.info("loop to scan all topicConfig with fixed delay {}ms", (Object)30000);
    }

    private void scanAllTopicConfig() {
        log.info("start to scan all topicConfig");
        try {
            for (Map.Entry it : this.defaultMessageStore.getTopicConfigs().entrySet()) {
                TopicConfig topicConfig = (TopicConfig)it.getValue();
                CleanupPolicy policy = CleanupPolicyUtils.getDeletePolicy(Optional.ofNullable(topicConfig));
                if (!Objects.equals(policy, CleanupPolicy.COMPACTION)) continue;
                for (int queueId = 0; queueId < topicConfig.getWriteQueueNums(); ++queueId) {
                    this.loadAndGetClog((String)it.getKey(), queueId);
                }
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        log.info("scan all topicConfig over");
    }

    private CompactionLog loadAndGetClog(String topic, int queueId) {
        CompactionLog clog = this.compactionLogTable.compute(topic + "_" + queueId, (k, v) -> {
            if (v == null) {
                try {
                    v = new CompactionLog(this.defaultMessageStore, this, topic, queueId);
                    v.load(true);
                    int randomDelay = 1000 + new Random(System.currentTimeMillis()).nextInt(this.compactionInterval);
                    this.compactionSchedule.scheduleWithFixedDelay(v::doCompaction, this.compactionInterval + randomDelay, this.compactionInterval + randomDelay, TimeUnit.MILLISECONDS);
                }
                catch (IOException e) {
                    log.error("create compactionLog exception: ", (Throwable)e);
                    return null;
                }
            }
            return v;
        });
        return clog;
    }

    public void putMessage(String topic, int queueId, SelectMappedBufferResult smr) throws Exception {
        CompactionLog clog = this.loadAndGetClog(topic, queueId);
        if (clog != null) {
            clog.asyncPutMessage(smr);
        }
    }

    public void doDispatch(DispatchRequest dispatchRequest, SelectMappedBufferResult smr) throws Exception {
        CompactionLog clog = this.loadAndGetClog(dispatchRequest.getTopic(), dispatchRequest.getQueueId());
        if (clog != null) {
            clog.asyncPutMessage(smr.getByteBuffer(), dispatchRequest);
        }
    }

    public GetMessageResult getMessage(String group, String topic, int queueId, long offset, int maxMsgNums, int maxTotalMsgSize) {
        CompactionLog log = this.compactionLogTable.get(topic + "_" + queueId);
        if (log == null) {
            return GetMessageResult.NO_MATCH_LOGIC_QUEUE;
        }
        return log.getMessage(group, topic, queueId, offset, maxMsgNums, maxTotalMsgSize);
    }

    public void flush(int flushLeastPages) {
        this.compactionLogTable.values().forEach(log -> log.flush(flushLeastPages));
    }

    public void flushLog(int flushLeastPages) {
        this.compactionLogTable.values().forEach(log -> log.flushLog(flushLeastPages));
    }

    public void flushCQ(int flushLeastPages) {
        this.compactionLogTable.values().forEach(log -> log.flushCQ(flushLeastPages));
    }

    public void updateMasterAddress(String addr) {
        this.masterAddr = addr;
    }

    public void shutdown() {
        this.compactionSchedule.shutdown();
        try {
            if (!this.compactionSchedule.awaitTermination(1000L, TimeUnit.MILLISECONDS)) {
                List<Runnable> droppedTasks = this.compactionSchedule.shutdownNow();
                log.warn("compactionSchedule was abruptly shutdown. {} tasks will not be executed.", (Object)droppedTasks.size());
            }
        }
        catch (InterruptedException e) {
            log.warn("wait compaction schedule shutdown interrupted. ");
        }
        this.flush(0);
        this.positionMgr.persist();
    }

    public ScheduledExecutorService getCompactionSchedule() {
        return this.compactionSchedule;
    }

    public String getCompactionLogPath() {
        return this.compactionLogPath;
    }

    public String getCompactionCqPath() {
        return this.compactionCqPath;
    }

    public CompactionPositionMgr getPositionMgr() {
        return this.positionMgr;
    }

    public int getOffsetMapSize() {
        return this.offsetMapSize;
    }

    public String getMasterAddr() {
        return this.masterAddr;
    }
}

