/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gtk;

import java.util.ArrayList;
import java.util.List;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;

/**
 * An action which can be triggered by a menu or toolbar item.
 * 
 * <p>Actions represent operations that the user can be perform, along with some 
 * information how it should be presented in the interface. Each action provides 
 * methods to create icons, menu items and toolbar items representing itself.
 * 
 * <p>As well as the callback that is called when the action gets activated, the 
 * following also gets associated with the action:
 *
 * <p>a name (not translated, for path lookup)
 * 
 * <p>a label (translated, for display)
 * 
 * <p>an accelerator
 *
 * <p>whether label indicates a stock id
 * 
 * <p>a tooltip (optional, translated)
 *	
 * <p>a toolbar label (optional, shorter than label)
 *
 * <p>The action will also have some state information:
 * 
 * <p>visible (shown/hidden)
 *
 * <p>sensitive (enabled/disabled)
 * 
 * <p>Apart from regular actions, there are {@link ToggleAction}, which can be toggled 
 * between two states and {@link RadioAction}, of which only one in a group can be in 
 * the "active" state. Other actions can be implemented as {@link Action} subclasses.
 * 
 * <p>Each action can have one or more proxy menu item, toolbar button or other proxy 
 * widgets. Proxies mirror the state of the action (text label, tooltip, icon, visible,
 * sensitive, etc), and should change when the action's state changes. When the proxy
 * is activated, it should activate its action. 
 */

public class Action extends GObject {
	
	/**
	 * Construct a new Action object.
	 */
	public Action(String name, String label, String tooltip, String stockId) {
		super(gtk_action_new(name, label, tooltip, stockId));
	}
	
    //	protected Action() {}
	
	Action(Handle handle) {
		super(handle);
	}
	
	/**
	 * Returns the name of the action.
	 * @return the name of the action
	 */
	public String getName() {
		return gtk_action_get_name(getHandle());
	}
	
	/**
	 * Emits the "activate" signal on the specified action, if it isn't insensitive. 
	 * This gets called by the proxy widgets when they get activated.
	 * 
	 * It can also be used to manually activate an action.
	 */
	public void activate() {
		gtk_action_activate(getHandle());
	}
	
	/**
	 * This function is intended for use by action implementations to create icons
	 *  displayed in the proxy widgets.
	 * @param iconSize the size of the icon that should be created
	 * @return a widget that displays the icon for this action
	 */
	public Widget createIcon(IconSize iconSize) {
		return new Widget(gtk_action_create_icon(getHandle(), iconSize.getValue()));
	}
	
	/**
	 * Creates a menu item widget that proxies for the given action.
	 * @return  a menu item connected to the action
	 */
	public Widget createMenuItem() {
		return new Widget(gtk_action_create_menu_item(getHandle()));
	}
	
	/**
	 * Creates a toolbar item widget that proxies for the given action.
	 * @return  a toolbar item connected to the action
	 */
	public Widget createToolItem() {
		return new Widget(gtk_action_create_tool_item(getHandle()));
	}
	
	/**
	 * Connects a widget to an action object as a proxy. Synchronises various 
	 * properties of the action with the widget (such as label text, icon, 
	 * tooltip, etc), and attaches a callback so that the action gets activated
	 * when the proxy widget does.
	 *
	 * If the widget is already connected to an action, it is disconnected first.
	 * @param proxy the proxy widget
	 */
	public void connectProxy(Widget proxy) {
		gtk_action_connect_proxy(getHandle(), proxy.getHandle());
	}
	
	/**
	 * Disconnects a proxy widget from an action. Does not destroy the widget, however.
	 * @param proxy the proxy widget
	 */
	public void disconnectProxy(Widget proxy) {
		gtk_action_disconnect_proxy(getHandle(), proxy.getHandle());
	}
	
	/**
	 * Returns the proxy widgets for an action.
	 * @return  a List of proxy widgets
	 */
	public List getProxies() {
		Handle[] proxies = gtk_action_get_proxies(getHandle());
		List objs = new ArrayList();
		for (int i = 0; i < proxies.length; i++) {
			GObject obj = getGObjectFromHandle(proxies[i]);
			if (null != obj)
				objs.add((Widget)obj);
			else
				objs.add(new Widget(proxies[i]));
		}
		return objs;
	}
	
	/**
	 * Installs the accelerator for action if action has an accel path and group.
	 * Since multiple proxies may independently trigger the installation of the 
	 * accelerator, the action counts the number of times this function has been 
	 * called and doesn't remove the accelerator until {@link Action#disconnectAccelerator()}
	 * has been called as many times.
	 */
	public void connectAccelerator() {
		gtk_action_connect_accelerator(getHandle());
	}
	
	/**
	 * Undoes the effect of one call to {@link Action#connectAccelerator()}.
	 */
	public void disconnectAccelerator() {
		gtk_action_disconnect_accelerator(getHandle());
	}
	
	public boolean isSensitive() {
		return gtk_action_is_sensitive(getHandle());
	}
	
	public boolean isVisible() {
		return gtk_action_is_visible(getHandle());
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static org.gnu.glib.Type getType() {
		return new org.gnu.glib.Type(gtk_action_get_type());
	}
	

	native static final protected int gtk_action_get_type ();
	native static final protected Handle gtk_action_new(String name, String label, String tooltip, String stockId);
	native static final protected String gtk_action_get_name(Handle action);
	native static final protected boolean gtk_action_is_sensitive(Handle action);
	native static final protected boolean gtk_action_get_sensitive(Handle action);
	native static final protected boolean gtk_action_is_visible(Handle action);
	native static final protected boolean gtk_action_get_visible(Handle action);
	native static final protected void gtk_action_activate(Handle action);
	native static final protected Handle gtk_action_create_icon(Handle action, int iconSize);
	native static final protected Handle gtk_action_create_menu_item(Handle action);
	native static final protected Handle gtk_action_create_tool_item(Handle action);
	native static final protected void gtk_action_connect_proxy(Handle action, Handle proxy);
	native static final protected void gtk_action_disconnect_proxy(Handle action, Handle proxy);
	native static final protected Handle[] gtk_action_get_proxies(Handle action);
	native static final protected void gtk_action_connect_accelerator(Handle action);
	native static final protected void gtk_action_disconnect_accelerator(Handle action);

}
