/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  gnome-print-module.c: A module loader
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useoful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Chema Celorio <chema@celorio.com>
 *
 *  Copyright 2002  Ximian, Inc. and authors
 *
 */

#include <config.h>
#include <glib.h>
#include <gmodule.h>
#include <libgnomeprint/gnome-print-module.h>
#include <libgnomeprint/gpa/gpa-list.h>
#include <libgnomeprint/gpa/gpa-printer.h>
#include <libgnomeprint/gpa/gpa-value.h>
#include <libgnomeprint/gpa/gpa-settings.h>
#include <libgnomeprint/gpa/gpa-reference.h>
#include <libgnomeprint/gpa/gpa-model.h>

#include <cups/cups.h>

xmlChar *model_xml_template = 
"<?xml version=\"1.0\"?>"
"<Model Id=\"%s\" Version=\"1.0\">"
"  <Name>Cups printer</Name>"
"  <Vendor>GNOME</Vendor>"
"  <ModelVersion>0.0.1</ModelVersion>"
"  <Options>"
"    <Option Id=\"Transport\">"
"      <Option Id=\"Backend\" Type=\"List\" Default=\"cups\">"
"        <Item Id=\"cups\">"
"          <Name>cups</Name>"
"          <Key Id=\"Module\" Value=\"libgnomeprintcups.so\"/>"
"          <Option Id=\"Printer\" Type=\"String\" Default=\"%s\"/>"
"        </Item>"
"      </Option>"
"    </Option>"
"  </Options>"
"</Model>";

static GPANode *
gnome_print_cups_model_new (const gchar *cups_printer)
{
	xmlNodePtr root;
	xmlDocPtr model_xml_doc;
	GPANode *model;
	gchar *model_id;
	gchar *model_xml;

	model_id = g_strdup_printf ("CupsModel-%s", cups_printer);
	model_xml = g_strdup_printf (model_xml_template, model_id, cups_printer);
	g_free (model_id);

	model_xml_doc = xmlParseDoc (model_xml);
	if (!model_xml_doc) {
		g_warning ("Could not parse model inside the cups module\n");
		return NULL;
	}
	g_free (model_xml);

	root = model_xml_doc->xmlRootNode;
	model = gpa_model_new_from_tree (root);
	if (!model) {
		g_warning ("Could not create model from tree inside the cups module\n");
		return NULL;
	}

	xmlFreeDoc (model_xml_doc);
	
	return model;
}

static gboolean
gnome_print_cups_printer_append (GPAList *printers_list, const gchar *cups_printer)
{
	GPAPrinter *printer;
	GPAList *settings_list;
	GPANode *settings;
	GPANode *model;
	gchar *printer_id;
	
	/* Printer */
	printer_id = g_strdup_printf ("CupsPrinter-%s", cups_printer);
	printer = gpa_printer_new_full (printer_id, cups_printer);
	g_free (printer_id);
	
	/* Model */
	model = gnome_print_cups_model_new (cups_printer);
	if (!model) {
		gpa_node_unref (GPA_NODE (printer));
		return FALSE;
	}
	printer->model = gpa_node_attach (GPA_NODE (printer), gpa_reference_new (model));
	gpa_node_unref (GPA_NODE (model));
	
	/* Attach a Settings List */
	settings_list = gpa_list_new (GPA_TYPE_SETTINGS, "SettingsList", TRUE);
	printer->settings = gpa_node_attach (GPA_NODE (printer), GPA_NODE (settings_list));

	/* Attach a Settings and make them the default */
	settings = gpa_settings_new_from_model_full (GPA_NODE (model), "Default", "SettIdFromCups");
	gpa_list_prepend     (settings_list, settings);
	gpa_list_set_default (settings_list, settings);

	/* Verify the printer that we've just created */
	if (!gpa_node_verify (GPA_NODE (printer))) {
		g_warning ("Can't load printer from cups pluging, _verify failed\n");
			 gpa_node_unref (GPA_NODE (printer));
			 return FALSE;
	}

	gpa_list_prepend (printers_list, GPA_NODE (printer));
	
	return TRUE;
}

static void
gnome_print_cups_printer_list_append (gpointer printers_list)
{
	/* start */
	int        i;                /* Looping var */
	int        num_printers;     /* Number of printers */
	char       **printers;       /* List of printers */
	
	g_return_if_fail (printers_list != NULL);
	g_return_if_fail (GPA_IS_LIST (printers_list));
	
	/* Get the list of printers */
	num_printers = cupsGetPrinters (&printers);
	
	if (num_printers < 1)
		return;
	
	for (i = 0; i < num_printers; i ++)
		gnome_print_cups_printer_append (GPA_LIST (printers_list), printers [i]);
	
	/* Free the printer list */
	for (i = 0; i < num_printers; i ++)
		free (printers[i]);
	free (printers);
	
	return;
}


/*  ------------- GPA init ------------- */
G_MODULE_EXPORT gboolean gpa_module_init (GpaModuleInfo *info);

G_MODULE_EXPORT gboolean
gpa_module_init (GpaModuleInfo *info)
{
	info->printer_list_append = gnome_print_cups_printer_list_append;
	return TRUE;
}
