/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  test-dialog.c:
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Chema Celorio <chema@ximian.com>
 *
 *  Copyright (C) 2002 Ximian Inc. and authors
 *
 */

#include "config.h"

#include <popt.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>

#include <gtk/gtk.h>
#include <libgnomeprint/gnome-print-config.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-dialog-private.h>
#include <libgnomeprintui/gnome-printer-dialog.h>
#include <libgnomeprintui/gpaui/gpa-tree-viewer.h>
#include <libgnomeprintui/gpaui/gpa-printer-selector.h>

#define MAGIC_NUMBER -100

gint     num_test = MAGIC_NUMBER;
gchar *  num_test_str = NULL;
gboolean debug = FALSE;
gboolean tree = FALSE;
gint     callback_count = 0;

typedef enum {
	TEST_RETVAL_CRASH = -3,
	TEST_RETVAL_ERROR = -2,
	TEST_RETVAL_BAD_PARAMETERS = -1,
	TEST_RETVAL_SUCCESS = 0,
	TEST_RETVAL_SUCCESS_LAST = 99,
} TestRetval;

typedef struct _TestCase TestCase;

struct _TestCase {
	TestRetval (*function) (void);
	const guchar *description;
};

#define max_num_test ((sizeof (test_cases) / sizeof (test_cases[0]))-1)

static struct poptOption options[] = {
	{ "num", '\0', POPT_ARG_STRING, &num_test_str,   0,
	  "Test number to run", "#"},
	{ "tree",     '\0', POPT_ARG_NONE, &tree,   0,
	  "Show the tree of GPANodes",  NULL},
	{ "debug",    '\0', POPT_ARG_NONE, &debug, 0,
	  "Print debugging output",          NULL},
	POPT_AUTOHELP
	{ NULL }
};

static TestRetval test_simple (void);
static TestRetval test_show (void);
static TestRetval test_dialog_flags (void);
static TestRetval test_paper_changes (void);
static TestRetval test_printer_changes (void);
static TestRetval test_multiple (void);

static const TestCase test_cases[] = {
	{ &test_simple,              "Simple checks."},
	{ &test_show,                "show a print dialog"},
	{ &test_dialog_flags,        "Shows a number of dialog with different flag convinationts"},
	{ &test_paper_changes,       "Change paper parameters and monitor dialogs for changes"},
	{ &test_printer_changes,     "Change the selected printer"},
	{ &test_multiple,            "Create and destroy muliple GnomePrintDialogs."},
};

/* Helper functions */
static gint
test_dialog_get_num_pages (GtkWidget *dialog)
{
	return g_list_length (GTK_NOTEBOOK (GNOME_PRINT_DIALOG (dialog)->notebook)->children);
}

static void
test_dialog_show_page (GtkWidget *dialog, gint page)
{
	gtk_notebook_set_current_page (GTK_NOTEBOOK (GNOME_PRINT_DIALOG (dialog)->notebook), page);
	
}

/**
 * test_pos_window:
 * @window: 
 * 
 * Position de window so that it doens't overlap with other windows
 * this allows us to see several windows on the screen
 **/
static void
test_pos_window (GtkWidget *window)
{
#define PANEL_HEIGHT 40
#define TOP_BOTTOM_FRAME 36
#define RIGHT_LEFT_FRAME 16
	static gint x = 0;
	static gint y = PANEL_HEIGHT;
	static gint new_x = 0;
	gint width;
	gint height;

	gtk_window_get_size (GTK_WINDOW (window), &width, &height);
	
	if ((y + height) > gdk_screen_height ()) {
		x += new_x + RIGHT_LEFT_FRAME;
		new_x = 0;
		y = PANEL_HEIGHT;
	}

	if (x + width > gdk_screen_width ())
		x = 0;
	if (y + height > gdk_screen_height ())
		y = 0;
	
	new_x = (width > new_x) ? width : new_x;
	
	gtk_window_move (GTK_WINDOW (window), x, y);
	y += height + TOP_BOTTOM_FRAME; 
}

static GtkWidget *
test_dialog_show_all_pages (GnomePrintConfig *config)
{
	GnomePrintJob *job;
	GtkWidget *dialog;
	
	job = gnome_print_job_new (config);
	dialog = gnome_print_dialog_new (job, "Sample Print Dialog",
					 GNOME_PRINT_DIALOG_RANGE | GNOME_PRINT_DIALOG_COPIES);
	gtk_widget_show (dialog);
	test_pos_window (dialog);
	if (test_dialog_get_num_pages (dialog) != 3) {
		g_print ("Invalid number of pages\n");
		exit (TEST_RETVAL_ERROR);
	}

	job = gnome_print_job_new (config);
	dialog = gnome_print_dialog_new (job, "Sample Print Dialog",
					 GNOME_PRINT_DIALOG_RANGE | GNOME_PRINT_DIALOG_COPIES);
	gtk_widget_show (dialog);
	test_dialog_show_page (dialog, 1);
	test_pos_window (dialog);
	if (test_dialog_get_num_pages (dialog) != 3) {
		g_print ("Invalid number of pages\n");
		exit (TEST_RETVAL_ERROR);
	}

	job = gnome_print_job_new (config);
	dialog = gnome_print_dialog_new (job, "Sample Print Dialog",
					 GNOME_PRINT_DIALOG_RANGE | GNOME_PRINT_DIALOG_COPIES);
	gtk_widget_show (dialog);
	test_pos_window (dialog);
	test_dialog_show_page (dialog, 2);	
	if (test_dialog_get_num_pages (dialog) != 3) {
		g_print ("Invalid number of pages\n");
		exit (TEST_RETVAL_ERROR);
	}
		
	return dialog;
}

guint tag = 0;

static gboolean
test_main_quit_real (void)
{
	gtk_main_quit ();
	return FALSE;
}

static gboolean
test_main_quit (void)
{
	gtk_timeout_remove (tag);
	gtk_idle_add ((GtkFunction) test_main_quit_real, NULL);
	return TRUE;
}

static void
test_run (gint msecs)
{
	tag = gtk_timeout_add (msecs, (GSourceFunc) test_main_quit, NULL);
	gtk_main ();
	return;
}

static void
increment_callback_count (void)
{
	callback_count++;
}

/* Tests */
static TestRetval
test_printer_changes (void)
{
	GnomePrintConfig *config;
	GPANode *node, *printers, *child;
	GSList *list, *l;
	GtkWidget *dialog;
	GtkWidget *option_menu;
	gint len;

	config = gnome_print_config_default ();
	
	dialog = test_dialog_show_all_pages (config);

	node = GNOME_PRINT_CONFIG_NODE (config);
	g_return_val_if_fail (GPA_IS_NODE (node), TEST_RETVAL_ERROR);
	printers = gpa_node_get_child_from_path (node, "Globals.Printers");
	g_return_val_if_fail (GPA_IS_NODE (printers), TEST_RETVAL_ERROR);	

	/* Make sure the gtkoptionmenu changes */
	callback_count = 0;
	option_menu = GPA_PRINTER_SELECTOR (GNOME_PRINTER_SELECTOR (GNOME_PRINT_DIALOG (dialog)->printer)->printers)->menu;
	g_return_val_if_fail (GTK_IS_OPTION_MENU (option_menu), TEST_RETVAL_ERROR);
	g_signal_connect (G_OBJECT (option_menu), "changed",
			  (GCallback) increment_callback_count, NULL);

	test_run (1000);
	
	/* Create list */
	list = NULL;
	child = gpa_node_get_child (printers, NULL);
	for (; child != NULL; child = gpa_node_get_child (printers, child))
		list = g_slist_prepend (list, child);
	
	l = list;
	while (l) {
		if (!gnome_print_config_set (config, "Printer", gpa_node_id (l->data))) {
			g_print ("Could not set the Printer to %s\n", gpa_node_id (l->data));
			return TEST_RETVAL_ERROR;
		}
		test_run (1000);
		l = l->next;
	}

	len = g_slist_length (list);
	
	if (callback_count < len) {
		g_warning ("The printers GtkOptionMenu didn't changed "
			   "as expected, expected: %d changed: %d\n",
			 len, callback_count);
		return TEST_RETVAL_ERROR;
	}
	if (callback_count > len) {
		g_warning ("The printers GtkOptionMenu didn't changed "
			   "as expected, expected: %d changed: %d\n",
			   len, callback_count);
		if (FALSE) /* FIXME: Ignore for now, known issue */
		return TEST_RETVAL_ERROR;
	}

	g_slist_free (list);
	
	g_print ("Callback count %d\n", callback_count);

	return TEST_RETVAL_SUCCESS;
}

static TestRetval
test_paper_changes (void)
{
	GnomePrintConfig *config;
	const gchar *papers [] = {
		"USLetter", "USLegal", "Executive", "A0", "A1", "A2", "A3", "A4", "A5",
		"A6", "A7", "A8", "A9", "A10", "B0", "B1", "B2", "B3", "B4", "B5", "B6",
		"B7", "B8", "B9", "B10", "C0", "C1", "C2", "C3", "C4", "C5", "C6", "C7",
		"C8", "C9", "C10", "A4_3", "A4_4", "A4_8", "A3_4", "A5_3", "DL", "C6_C5",
		"Envelope_No10", "Envelope_6x9", "USLetter"};
	const gchar *orientations [] = {"R0", "R90", "R180", "R270", "R0"};
	const gchar *layouts [] = {"Plain", "2_1", "4_1", "I2_1", "IM2_1", "Plain"};
	gint max, i;
		
	config = gnome_print_config_default ();
	test_dialog_show_all_pages (config);

	gnome_print_config_set (config, "Printer", "GENERIC");

	test_run (1000);

	max = sizeof (papers) / sizeof (gchar *);
	for (i = 0; i < max; i++) {
		if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAPER_SIZE, papers[i])) {
			g_print ("Could not set paper size to %s\n", papers[i]);
			return TEST_RETVAL_ERROR;
		}
		test_run (50);
	}

	max = sizeof (orientations) / sizeof (gchar *);
	for (i = 0; i < max; i++) {
		if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAGE_ORIENTATION, orientations[i])) {
			g_print ("Could not set the page orientation to %s\n", orientations[i]);
			return TEST_RETVAL_ERROR;
		}
		test_run (150);
	}
	for (i = 0; i < max; i++) {
		if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAPER_ORIENTATION, orientations[i])) {
			g_print ("Could not set the paper orientation to %s\n", orientations[i]);
			return TEST_RETVAL_ERROR;
		}
		test_run (150);
	}

	max = sizeof (layouts) / sizeof (gchar *);
	for (i = 0; i < max; i++) {
		if (!gnome_print_config_set (config, GNOME_PRINT_KEY_LAYOUT, layouts[i])) {
			g_print ("Could not set the Layout to %s\n", layouts[i]);
			return TEST_RETVAL_ERROR;
		}
		test_run (150);
	}

	
	test_run (1000);

	return TEST_RETVAL_SUCCESS;
}

static TestRetval
test_dialog_flags (void)
{
	GnomePrintConfig *config;
	GnomePrintJob *job;
	GtkWidget *dialog;

	config = gnome_print_config_default ();

	job = gnome_print_job_new (config);
	dialog = gnome_print_dialog_new (job, "Sample Print Dialog",
										0);
	gtk_widget_show (dialog);
	test_pos_window (dialog);
	if (test_dialog_get_num_pages (dialog) != 2)
		return TEST_RETVAL_ERROR;

	job = gnome_print_job_new (config);
	dialog = gnome_print_dialog_new (job, "Sample Print Dialog",
					 GNOME_PRINT_DIALOG_RANGE);
	gtk_widget_show (dialog);
	test_pos_window (dialog);
	if (test_dialog_get_num_pages (dialog) != 3)
		return TEST_RETVAL_ERROR;
	
	job = gnome_print_job_new (config);
	dialog = gnome_print_dialog_new (job, "Sample Print Dialog",
					 GNOME_PRINT_DIALOG_COPIES);
	gtk_widget_show (dialog);
	test_pos_window (dialog);
	if (test_dialog_get_num_pages (dialog) != 3)
		return TEST_RETVAL_ERROR;

	test_dialog_show_all_pages (config);

	test_run (1500);

	return TEST_RETVAL_SUCCESS;
}

static TestRetval
test_show (void)
{
	GtkWidget *dialog;

	dialog = gnome_print_dialog_new (NULL, "Sample Print Dialog", 0);

	gtk_widget_show (dialog);

	test_run (500);

	return TEST_RETVAL_SUCCESS;
}

static TestRetval
test_multiple (void)
{
	GnomePrintJob *job;
	GnomePrintConfig *config;
	GtkWidget *dialog;
	gint i;

	config = gnome_print_config_default ();
	if (!config)
		return TEST_RETVAL_ERROR;
	job = gnome_print_job_new (NULL);
	if (!job)
		return TEST_RETVAL_ERROR;
	job = gnome_print_job_new (config);
	if (!job)
		return TEST_RETVAL_ERROR;
	dialog = gnome_print_dialog_new (NULL, "Sample Print Dialog", 0);
	if (!dialog)
		return TEST_RETVAL_ERROR;

	dialog = gnome_print_dialog_new (job, "Sample Print Dialog", 0);
	if (!dialog)
		return TEST_RETVAL_ERROR;
	gtk_widget_show (dialog);

	g_print ("a\n");
	
	for (i = 0; i < 1; i++) {
		g_print ("b\n");
//		gtk_widget_destroy (dialog);
		/* This test needs love (Chema) */

		g_print ("c\n");
#if 1
		config = gnome_print_config_default ();
		job = gnome_print_job_new (config);
#endif
		g_print ("d\n");
		
		dialog = gnome_print_dialog_new (job, "Sample Print Dialog", 0);
		if (!dialog)
			return TEST_RETVAL_ERROR;

		g_print ("e\n");
		
		gtk_widget_show (dialog);
		test_run (1);
		g_print ("f\n");
	}
	g_print ("g\n");

	return TEST_RETVAL_SUCCESS;
	
	if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAPER_SIZE, "B3")) {
		g_warning ("Could not change the Paper Size\n");
		return TEST_RETVAL_ERROR;
	}
	if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAPER_ORIENTATION, "R270")) {
		g_warning ("Could not change the Paper Orientation\n");
		return TEST_RETVAL_ERROR;
	}
	if (!gnome_print_config_set (config, GNOME_PRINT_KEY_PAGE_ORIENTATION, "R90")) {
		g_warning ("Could not change the Page Orientation\n");
		return TEST_RETVAL_ERROR;
	}

	for (i = 0; i < 15; i++) {
		gtk_widget_destroy (dialog);
		dialog = gnome_print_dialog_new (job, "Sample Print Dialog", 0);
		if (!dialog)
			return TEST_RETVAL_ERROR;
		gtk_widget_show (dialog);
		test_run (100);
	}

	gtk_widget_destroy (dialog);
	
	return TEST_RETVAL_SUCCESS;
}

static TestRetval
test_simple (void)
{
	GnomePrintJob *job;
	GnomePrintConfig *config;
	GtkWidget *dialog;

	config = gnome_print_config_default ();
	if (!config)
		return TEST_RETVAL_ERROR;
	job = gnome_print_job_new (NULL);
	if (!job)
		return TEST_RETVAL_ERROR;
	job = gnome_print_job_new (config);
	if (!job)
		return TEST_RETVAL_ERROR;
	dialog = gnome_print_dialog_new (NULL, "Sample Print Dialog", 0);
	if (!dialog)
		return TEST_RETVAL_ERROR;
	
	return TEST_RETVAL_SUCCESS;
}

static TestRetval
test_dispatch (gint num_test)
{
	TestRetval ret;

	g_print ("Running num_test %d\n[%s]\n", num_test, test_cases[num_test].description);

	ret = (test_cases[num_test].function ());

	switch (ret) {
	case TEST_RETVAL_SUCCESS:
		if (num_test == max_num_test)
			ret = TEST_RETVAL_SUCCESS_LAST;
		g_print (" Pass..\n");
		break;
	case TEST_RETVAL_ERROR:
		g_print (" Fail..\n");
		break;
	default:
		g_assert_not_reached ();
		break;
			
	}

	return ret;
}

static void
test_dialog_tree ()
{
	GnomePrintConfig *config;
	GtkWidget *tree;

	config = gnome_print_config_default ();
	tree = gpa_tree_viewer_new (config);

	g_signal_connect (G_OBJECT (tree), "delete_event",
			  (GCallback) gtk_main_quit, NULL);
}

static void
usage (gchar *error)
{
	g_print ("Error: %s\n\n", error);
	g_print ("Usage: test-dialog --num=[num_test] [--tree]\n\n");
	exit (TEST_RETVAL_BAD_PARAMETERS);
}

static void
parse_command_line (int argc, const char ** argv, gint *num_test)
{
	poptContext popt;

	popt = poptGetContext ("test_gpa", argc, argv, options, 0);
	poptGetNextOpt (popt);

	if (!num_test_str || num_test_str[0] == '\0') {
		if (tree)
			return;
		usage ("num_test not specified");
	}

	*num_test = atoi (num_test_str);

	if (*num_test == -1) {
		GList *list = NULL;
		/* We crash, this is part of the sanity check that allows us verify that crashes
		 * are treated as errors when they happen (Chema)*/
		g_print ("Crashing ...\n");
		list->next = NULL;
	}
	
	if (*num_test < 0 || (*num_test > max_num_test)) {
		gchar *error;
		error = g_strdup_printf ("num_test number out of range. Valid range is -1 to %d",
					 max_num_test);
		usage (error);
	}
	
	if (debug)
		g_print ("num_test is %d\n", *num_test);
	
	poptFreeContext (popt);
}

static void
handle_sigsegv (int i)
{
	g_print ("\n\ntest-dialog crashed while running num_test %d [%s]\n",
		 num_test, test_cases[num_test].description);
	exit (TEST_RETVAL_CRASH);
}

int
main (int argc, const char * argv[])
{
	TestRetval ret = TEST_RETVAL_SUCCESS;
	struct sigaction sig;

	gtk_init (&argc, (char ***) &argv);

	parse_command_line (argc, argv, &num_test);

	/* Catch sigsegv signals */
	memset(&sig, 0, sizeof(sig));
	sig.sa_handler = handle_sigsegv;
	sigaction (SIGSEGV, &sig, NULL);

	if (tree)
		test_dialog_tree ();

	if (num_test != MAGIC_NUMBER)
		ret = test_dispatch (num_test);
	else
		gtk_main ();
		
	return ret;
}

