/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "prefs-dialog.h"
#include "general-prefs.h"
#include "appearance-prefs.h"
#include "ui-prefs.h"
#include "network-prefs.h"
#include "galeon-dialog.h"
#include "prefs-strings.h"
#include "galeon-embed-prefs.h"
#include "galeon-shell.h"
#include "gul-state.h"

#include <bonobo/bonobo-i18n.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkstock.h>

static void
prefs_dialog_class_init (PrefsDialogClass *klass);
static void
prefs_dialog_init (PrefsDialog *pd);
static void
prefs_dialog_finalize (GObject *object);

/* Filtering page */

enum
{
	ALLOW_POPUPS_PROP,
	ALLOW_JAVA_PROP,
	ALLOW_JS_PROP,
	ALLOW_STATUSBAR_PROP,
	REMEMBER_PASSWORDS_PROP,
	LOAD_IMAGES_PROP,
	ACCEPT_COOKIES_PROP,
	WARN_COOKIE_PROP,
	SESSION_COOKIE_PROP
};

int cookies_sg [] =
{
	SY_BEGIN_GROUP,
	SY_END_GROUP,

	SY_BEGIN_GROUP,	
	SY_END_GROUP,
	
	SY_BEGIN_GROUP_INVERSE,
	WARN_COOKIE_PROP,
	SESSION_COOKIE_PROP,
	SY_END_GROUP,

	SY_END
};

static const
GaleonDialogProperty filtering_properties [] =
{
	{ ALLOW_POPUPS_PROP, "allow_popups_checkbutton", CONF_FILTERING_ALLOW_POPUPS, PT_AUTOAPPLY, NULL },
	{ ALLOW_JAVA_PROP, "allow_java_checkbutton", CONF_FILTERING_JAVA_ENABLED, PT_AUTOAPPLY, NULL },
	{ ALLOW_JS_PROP, "allow_js_checkbutton", CONF_FILTERING_JAVASCRIPT_ENABLED, PT_AUTOAPPLY, NULL },
	{ ALLOW_STATUSBAR_PROP, "allow_statusbar_checkbutton", CONF_FILTERING_DEFAULT_STATUSBAR, PT_AUTOAPPLY, NULL },
	{ REMEMBER_PASSWORDS_PROP, "remember_passwords_checkbutton", CONF_PERSISTENT_PASSWORDS_SAVE, PT_AUTOAPPLY, NULL },
	{ LOAD_IMAGES_PROP, "images_radiobutton", CONF_FILTERING_IMAGE_LOADING_TYPE, PT_AUTOAPPLY, NULL },
	{ ACCEPT_COOKIES_PROP, "cookies_radiobutton", CONF_PERSISTENT_COOKIES_BEHAVIOR, PT_AUTOAPPLY, cookies_sg },
	{ WARN_COOKIE_PROP, "warn_cookie_checkbutton", CONF_PERSISTENT_COOKIE_WARN, PT_AUTOAPPLY },
	{ SESSION_COOKIE_PROP, "session_cookie_checkbutton", CONF_PERSISTENT_COOKIE_LIFETIME, PT_AUTOAPPLY },
	{ -1, NULL, NULL }
};

struct PrefsDialogPrivate
{
	GtkWidget *content;
	GaleonDialog *page;
	GHashTable *page_cache;
};

static GObjectClass *parent_class = NULL;

GType 
prefs_dialog_get_type (void)
{
        static GType prefs_dialog_type = 0;

        if (prefs_dialog_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (PrefsDialogClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) prefs_dialog_class_init,
                        NULL,
                        NULL, /* class_data */
                        sizeof (PrefsDialog),
                        0, /* n_preallocs */
                        (GInstanceInitFunc) prefs_dialog_init
                };

                prefs_dialog_type = g_type_register_static (GTK_TYPE_DIALOG,
							    "PrefsDialog",
							    &our_info, 0);
        }

        return prefs_dialog_type;

}

GtkDialog *
prefs_dialog_new (void)
{
        GtkDialog *dialog;
        
        dialog = GTK_DIALOG (g_object_new (PREFS_DIALOG_TYPE, NULL));

        return dialog;
}

static void
prefs_dialog_class_init (PrefsDialogClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        parent_class = g_type_class_peek_parent (klass);

        object_class->finalize = prefs_dialog_finalize;
}

static void
free_cached_dialog_cb (gpointer key,
                       gpointer value,
                       gpointer user_data)
{
	g_object_unref (G_OBJECT(value));
}

static void
prefs_dialog_finalize (GObject *object)
{
        PrefsDialog *pd;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_PREFS_DIALOG (object));

	pd = PREFS_DIALOG (object);

        g_return_if_fail (pd->priv != NULL);

	g_hash_table_foreach (pd->priv->page_cache,
			      free_cached_dialog_cb,
			      NULL);
	
	g_hash_table_destroy (pd->priv->page_cache);

        g_free (pd->priv);

        G_OBJECT_CLASS (parent_class)->finalize (object);
}

static GaleonDialog *
create_page (PrefsPageID id,
	     const char *page_widget,
	     const GaleonDialogProperty *prop)
{
	GaleonDialog *page = NULL;
	
	switch (id)
	{
	case PREFS_PAGE_GENERAL:
		page = general_prefs_new ();
		break;
	case PREFS_PAGE_APPEARANCE:
		page = appearance_prefs_new ();
		break;
	case PREFS_PAGE_UI:
		page = ui_prefs_new ();
		break;	
	case PREFS_PAGE_NETWORK:
		page = network_prefs_new ();
		break;
	case PREFS_PAGE_FILTERING:
		page = galeon_dialog_new ();
		galeon_dialog_construct (GALEON_DIALOG(page),
					 prop,
                	                 "prefs-dialog.glade", 
                        	         page_widget);
		break;
	}

	return page;
}

static GaleonDialog *
prefs_dialog_get_page (PrefsDialog *pd,
		       PrefsPageID id)
{
	const char *page_widget = NULL;
	GaleonDialog *page;
	const GaleonDialogProperty *prop = NULL;
	
	switch (id)
	{
	case PREFS_PAGE_APPEARANCE:
		page_widget = "appearance_page_box";
		break;
	case PREFS_PAGE_GENERAL:
		page_widget = "general_page_box";
		break;
	case PREFS_PAGE_UI:
		page_widget = "ui_page_box";
		break;
	case PREFS_PAGE_FILTERING:
		page_widget = "filtering_page_box";
		prop = filtering_properties;
		break;
	case PREFS_PAGE_NETWORK:
		page_widget = "network_page_box";
		break;
	}

	g_assert (page_widget != NULL);
	
	page = g_hash_table_lookup (pd->priv->page_cache, 
				    page_widget);

	if (!page)
	{
		page = create_page (id, page_widget, prop);
		g_assert (page != NULL);
		g_hash_table_insert (pd->priv->page_cache, 
				     g_strdup(page_widget), 
				     page); 
	}
	
	return page;
}
		       
void 
prefs_dialog_show_page (PrefsDialog *pd,
		        PrefsPageID id)
{
	GaleonDialog *old_page;
	old_page = pd->priv->page;

	pd->priv->page = prefs_dialog_get_page (pd, id);
	
	if (old_page)
	{
		galeon_dialog_remove_embedded (old_page);
		g_object_unref (old_page);
	}

	g_object_ref (pd->priv->page);
	galeon_dialog_show_embedded (pd->priv->page, 
				     pd->priv->content);
}

static void 
prefs_dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer data)
{
	if (response_id == GTK_RESPONSE_CLOSE)
	{
		gtk_widget_destroy (GTK_WIDGET(dialog));
	}
}

/**
 * prefs_sidebar_select_page: activate the selected (clicked) page
 */
static void 
prefs_sidebar_select_page_cb (GtkWidget *button,  PrefsDialog *pd)
{
	PrefsPageID id;	

	id = (int) g_object_get_data (G_OBJECT(button), "page_id");

	prefs_dialog_show_page (pd, id);
}

/*
 * preferences_build_sidebar: build the sidebar of the preferences dialog
 */
static void
prefs_build_sidebar (PrefsDialog *pd)
{
	int i;
	GtkWidget *hbox;
	GtkWidget *sidebar;
	GtkWidget *sidebar_frame;
	GtkWidget *content_frame;
	GSList *button_group = NULL;

	struct
	{
		const char *name;
		int id;
		const char *icon;
	}
	const icons[] =
	{
		{ N_("General"), PREFS_PAGE_GENERAL, SHARE_DIR "/general.png" },
		{ N_("Appearance"), PREFS_PAGE_APPEARANCE, SHARE_DIR "/fonts.png" },
		{ N_("User Interface"), PREFS_PAGE_UI, SHARE_DIR "/tabs.png" },
		{ N_("Filtering"), PREFS_PAGE_FILTERING, SHARE_DIR "/stop.png" },
		{ N_("Network"), PREFS_PAGE_NETWORK, SHARE_DIR "/network.png" },
		
		{ NULL, 0, NULL }
	};

	gtk_dialog_add_button (GTK_DIALOG (pd), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);

	g_signal_connect (pd, "response",
			  G_CALLBACK (prefs_dialog_response_cb),
			  NULL);
	
	hbox = gtk_hbox_new (FALSE, 0);
	sidebar = gtk_vbox_new (FALSE, 0);
	sidebar_frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (sidebar_frame),
				   GTK_SHADOW_IN);

	content_frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (content_frame),
				   GTK_SHADOW_NONE);

	gtk_box_pack_start (GTK_BOX (hbox), sidebar_frame, 
			    FALSE, FALSE, 5);
	gtk_box_pack_start (GTK_BOX (hbox), content_frame, 
			    TRUE, TRUE, 5);
	gtk_container_add (GTK_CONTAINER (sidebar_frame), sidebar);
	gtk_container_set_border_width (GTK_CONTAINER (sidebar_frame), 5);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (pd)->vbox), hbox);
	
	pd->priv->content = content_frame;
	
	gtk_widget_show_all (hbox);
	
	for (i = 0; icons[i].name != NULL; i++)
	{
		GtkWidget *label, *vbox, *page_button;
		GtkWidget *icon;
		gint id = icons[i].id;

		page_button = gtk_radio_button_new (button_group);
		gtk_button_set_relief (GTK_BUTTON (page_button),
				       GTK_RELIEF_NONE);
		gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON 
					    (page_button), FALSE);
		button_group = gtk_radio_button_get_group
					(GTK_RADIO_BUTTON (
	 				 page_button));
		g_object_set_data (G_OBJECT (page_button), 
		 		   "page_id", GINT_TO_POINTER (id));
		g_signal_connect (GTK_OBJECT (page_button), 
				  "clicked",
				  G_CALLBACK(prefs_sidebar_select_page_cb),
				  pd);
		gtk_box_pack_start (GTK_BOX (sidebar), page_button,
				    FALSE, FALSE, 0);
		gtk_widget_show (page_button);

		vbox = gtk_vbox_new (FALSE, 0);
		gtk_container_add (GTK_CONTAINER (page_button), vbox);
		gtk_widget_show (vbox);

		icon = gtk_image_new_from_file (icons[i].icon);

		if (icon != NULL)
		{
			gtk_box_pack_start (GTK_BOX (vbox), icon,
					    FALSE, FALSE, 0);
			gtk_widget_show (icon);
		}

		label = gtk_label_new (_(icons[i].name));
		gtk_misc_set_alignment (GTK_MISC (label), .5, .5);
		gtk_label_set_line_wrap (GTK_LABEL (label), FALSE);
		gtk_box_pack_start (GTK_BOX (vbox), label,
				    FALSE, FALSE, 0);
		gtk_widget_show (label);
	}
}

static gboolean
prefs_dialog_configure_event_cb (GtkWidget *widget,
				 GdkEventConfigure *event,
				 gpointer data)
{
	gul_state_save_window (widget, "prefs_dialog");

	return FALSE;
}

static void
prefs_dialog_init (PrefsDialog *pd)
{
	pd->priv = g_new0 (PrefsDialogPrivate, 1);
	pd->priv->content = NULL;
	pd->priv->page_cache = g_hash_table_new (g_str_hash, g_str_equal);

	gtk_window_set_title (GTK_WINDOW(pd), _("Preferences"));
	gtk_dialog_set_has_separator (GTK_DIALOG(pd), FALSE); 
	
	gul_state_load_window (GTK_WIDGET(pd), 
			       "prefs_dialog", -1, -1, FALSE);

	g_signal_connect (pd, 
			  "configure_event",
			  G_CALLBACK (prefs_dialog_configure_event_cb),
			  NULL);
	
	prefs_build_sidebar (pd);
}

