/* This is -*- C -*- */
/* $Id: guppi-barchart-state.c,v 1.20 2001/05/06 07:56:27 trow Exp $ */

/*
 * guppi-barchart-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-rgb.h>
#include <guppi-seq-scalar.h>
#include <guppi-seq-data.h>
#include <guppi-shared-hash.h>
#include <guppi-data-multiselect.h>
#include "guppi-barchart-state.h"
#include "guppi-barchart-view.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_DATA,
  ARG_ROW_LABELS,
  ARG_COLUMN_LABELS,
  ARG_COLUMN_STYLES,
  ARG_STACKED,
  ARG_NORMALIZE_STACKS,
  ARG_EDGE_THICKNESS,
  ARG_EDGE_COLOR,
  ARG_DEFAULT_COLOR,
  ARG_USE_STOCK_COLORS,
  ARG_VERTICAL_BARS,
  ARG_HORIZONTAL_BARS,
  ARG_BAR_BASE,
  ARG_CLUSTER_MARGIN,
  ARG_BAR_MARGIN,
  ARG_ENFORCE_PREFERRED_VIEW,
  ARG_BAR_POS_MIN,
  ARG_BAR_POS_MAX
};

static void
guppi_barchart_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiBarchartState *state = GUPPI_BARCHART_STATE (obj);

  switch (arg_id) {

  case ARG_DATA:
    GTK_VALUE_POINTER (*arg) = guppi_barchart_state_data (state);
    break;

  case ARG_ROW_LABELS:
    GTK_VALUE_POINTER (*arg) = guppi_barchart_state_row_labels (state);
    break;

  case ARG_COLUMN_LABELS:
    GTK_VALUE_POINTER (*arg) = guppi_barchart_state_column_labels (state);
    break;

  case ARG_COLUMN_STYLES:
    GTK_VALUE_POINTER (*arg) = guppi_barchart_state_column_styles (state);
    break;

  case ARG_STACKED:
    GTK_VALUE_BOOL (*arg) = guppi_barchart_state_stacked (state);
    break;

  case ARG_NORMALIZE_STACKS:
    GTK_VALUE_BOOL (*arg) = guppi_barchart_state_normalize_stacks (state);
    break;

  case ARG_EDGE_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_barchart_state_edge_thickness (state);
    break;

  case ARG_EDGE_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_barchart_state_edge_color (state);
    break;

  case ARG_DEFAULT_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_barchart_state_default_color (state);
    break;

  case ARG_USE_STOCK_COLORS:
    GTK_VALUE_UINT (*arg) = guppi_barchart_state_use_stock_colors (state);
    break;

  case ARG_VERTICAL_BARS:
    GTK_VALUE_BOOL (*arg) = guppi_barchart_state_vertical_bars (state);
    break;

  case ARG_HORIZONTAL_BARS:
    GTK_VALUE_BOOL (*arg) = guppi_barchart_state_horizontal_bars (state);
    break;

  case ARG_BAR_BASE:
    GTK_VALUE_DOUBLE (*arg) = guppi_barchart_state_bar_base (state);
    break;

  case ARG_CLUSTER_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_barchart_state_cluster_margin (state);
    break;

  case ARG_BAR_MARGIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_barchart_state_bar_margin (state);
    break;

  case ARG_ENFORCE_PREFERRED_VIEW:
    GTK_VALUE_BOOL (*arg) =
      guppi_barchart_state_enforce_preferred_view (state);
    break;

  case ARG_BAR_POS_MIN:
    GTK_VALUE_DOUBLE (*arg) = guppi_barchart_state_bar_pos_min (state);
    break;

  case ARG_BAR_POS_MAX:
    GTK_VALUE_DOUBLE (*arg) = guppi_barchart_state_bar_pos_max (state);
    break;

  default:
    break;
  };
}

static void
guppi_barchart_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiBarchartState *state = GUPPI_BARCHART_STATE (obj);

  switch (arg_id) {

  case ARG_DATA:
    guppi_barchart_state_set_data (state,
				   GUPPI_DATA0 (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_ROW_LABELS:
    guppi_barchart_state_set_row_labels (state,
					 GUPPI_SEQ_STRING0 (GTK_VALUE_POINTER
							    (*arg)));
    break;

  case ARG_COLUMN_LABELS:
    guppi_barchart_state_set_column_labels (state,
					    GUPPI_SEQ_STRING0
					    (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_COLUMN_STYLES:
    guppi_barchart_state_set_column_styles (state,
					    GUPPI_SEQ_STYLE0
					    (GTK_VALUE_POINTER (*arg)));
    break;

  case ARG_STACKED:
    guppi_barchart_state_set_stacked (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_NORMALIZE_STACKS:
    guppi_barchart_state_set_normalize_stacks (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_EDGE_THICKNESS:
    guppi_barchart_state_set_edge_thickness (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_VERTICAL_BARS:
    guppi_barchart_state_set_vertical_bars (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_HORIZONTAL_BARS:
    guppi_barchart_state_set_horizontal_bars (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_BAR_POS_MIN:
    guppi_barchart_state_set_bar_pos_min (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BAR_POS_MAX:
    guppi_barchart_state_set_bar_pos_max (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_DEFAULT_COLOR:
    guppi_barchart_state_set_default_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_USE_STOCK_COLORS:
    guppi_barchart_state_set_use_stock_colors (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_CLUSTER_MARGIN:
    guppi_barchart_state_set_cluster_margin (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_BAR_MARGIN:
    guppi_barchart_state_set_bar_margin (state, GTK_VALUE_DOUBLE (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_barchart_state_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_barchart_state_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/***************************************************************************/

static GtkWidget *
config_data_cb (gpointer user_data)
{
  GuppiBarchartState *state = GUPPI_BARCHART_STATE (user_data);
  GuppiData *data;
  GtkWidget *multi;

  data = guppi_barchart_state_data (state);

  multi = guppi_data_multiselect_new ();
  if (GUPPI_IS_SEQ_DATA (data))
    guppi_data_multiselect_connect (multi, GUPPI_SEQ_DATA (data));

  return multi;
}

static GuppiConfigItem *
config_tree (GuppiElementState * state)
{
  GuppiConfigItem *node1;
  GuppiConfigItem *node2;

  node1 = guppi_config_item_new (_("Bar Chart: Data"),
				 _("Data"), config_data_cb, state);

  node2 = guppi_config_item_new (_("Bar Chart: Appearance"),
				 _("Appearance"), NULL, state);

  guppi_config_item_append (node1, node2);

  return node1;
}

/***************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiBarchartState::" str, t, GTK_ARG_READWRITE, symb)

#define add_arg_w(str, t, symb) \
gtk_object_add_arg_type("GuppiBarchartState::" str, t, GTK_ARG_WRITABLE, symb)

static void
guppi_barchart_state_class_init (GuppiBarchartStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  object_class->get_arg = guppi_barchart_state_get_arg;
  object_class->set_arg = guppi_barchart_state_set_arg;
  object_class->destroy = guppi_barchart_state_destroy;
  object_class->finalize = guppi_barchart_state_finalize;

  state_class->name = _("Bar Chart");

  state_class->view_type = GUPPI_TYPE_BARCHART_VIEW;
  state_class->config_tree = config_tree;

  add_arg ("data", GTK_TYPE_POINTER, ARG_DATA);
  add_arg ("row_labels", GTK_TYPE_POINTER, ARG_ROW_LABELS);
  add_arg ("column_labels", GTK_TYPE_POINTER, ARG_COLUMN_LABELS);
  add_arg ("column_styles", GTK_TYPE_POINTER, ARG_COLUMN_STYLES);
  add_arg ("stacked", GTK_TYPE_BOOL, ARG_STACKED);
  add_arg ("normalize_stacks", GTK_TYPE_BOOL, ARG_NORMALIZE_STACKS);
  add_arg ("edge_thickness", GTK_TYPE_DOUBLE, ARG_EDGE_THICKNESS);
  add_arg ("edge_color_rgba", GTK_TYPE_UINT, ARG_EDGE_COLOR);
  add_arg ("default_color_rgba", GTK_TYPE_UINT, ARG_DEFAULT_COLOR);
  add_arg ("use_stock_colors", GTK_TYPE_UINT, ARG_USE_STOCK_COLORS);
  add_arg ("vertical_bars", GTK_TYPE_BOOL, ARG_VERTICAL_BARS);
  add_arg ("horizontal_bars", GTK_TYPE_BOOL, ARG_HORIZONTAL_BARS);
  add_arg ("bar_base", GTK_TYPE_DOUBLE, ARG_BAR_BASE);
  add_arg ("cluster_margin", GTK_TYPE_DOUBLE, ARG_CLUSTER_MARGIN);
  add_arg ("bar_margin", GTK_TYPE_DOUBLE, ARG_BAR_MARGIN);
  add_arg ("enforce_preferred_view", GTK_TYPE_BOOL,
	   ARG_ENFORCE_PREFERRED_VIEW);
  add_arg ("bar_pos_min", GTK_TYPE_DOUBLE, ARG_BAR_POS_MIN);
  add_arg ("bar_pos_max", GTK_TYPE_DOUBLE, ARG_BAR_POS_MAX);

}

static void
guppi_barchart_state_init (GuppiBarchartState * obj)
{
  GuppiElementState *es = GUPPI_ELEMENT_STATE (obj);

  guppi_element_state_add_shared (es, SHARED_DATA, guppi_shared_data ());
  guppi_element_state_add_shared (es, SHARED_LABEL_DATA,
				  guppi_shared_data ());
  guppi_element_state_add_shared (es, SHARED_ELEMENT_LABEL_DATA,
				  guppi_shared_data ());
  guppi_element_state_add_shared (es, SHARED_STYLE_DATA,
				  guppi_shared_data ());


  obj->stacked = FALSE;
  obj->normalize_stacks = FALSE;

  obj->edge_thickness = guppi_in2pt (1.0 / 48);

  obj->edge_color = RGBA_BLACK;
  obj->default_color = RGBA_RED;
  obj->use_stock_colors = 1;

  obj->vertical_bars = TRUE;
  obj->bar_base = 0;

  obj->cluster_margin = 0.20;
  obj->bar_margin = 0.06;

  obj->enforce_preferred_view = TRUE;
  obj->bar_pos_min = 0;
  obj->bar_pos_max = 1;
}

GtkType guppi_barchart_state_get_type (void)
{
  static GtkType guppi_barchart_state_type = 0;
  if (!guppi_barchart_state_type) {
    static const GtkTypeInfo guppi_barchart_state_info = {
      "GuppiBarchartState",
      sizeof (GuppiBarchartState),
      sizeof (GuppiBarchartStateClass),
      (GtkClassInitFunc) guppi_barchart_state_class_init,
      (GtkObjectInitFunc) guppi_barchart_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_barchart_state_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE, &guppi_barchart_state_info);
  }
  return guppi_barchart_state_type;
}

GuppiElementState *
guppi_barchart_state_new (void)
{
  return
    GUPPI_ELEMENT_STATE (guppi_type_new (guppi_barchart_state_get_type ()));
}

/**************************************************************************/

GuppiData *
guppi_barchart_state_data (GuppiBarchartState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_DATA);
  return GUPPI_DATA0 (obj);
}

GuppiSeqString *
guppi_barchart_state_column_labels (GuppiBarchartState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_LABEL_DATA);
  return GUPPI_SEQ_STRING0 (obj);
}

GuppiSeqString *
guppi_barchart_state_row_labels (GuppiBarchartState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_ELEMENT_LABEL_DATA);
  return GUPPI_SEQ_STRING0 (obj);
}

GuppiSeqStyle *
guppi_barchart_state_column_styles (GuppiBarchartState * state)
{
  GtkObject *obj;

  g_return_val_if_fail (state != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_STYLE_DATA);
  return GUPPI_SEQ_STYLE0 (obj);
}

void
guppi_barchart_state_set_data (GuppiBarchartState * state, GuppiData * data)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_DATA (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_DATA, data);
}

void
guppi_barchart_state_set_column_labels (GuppiBarchartState * state,
					GuppiSeqString * data)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_SEQ_STRING (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_LABEL_DATA, data);
}

void
guppi_barchart_state_set_row_labels (GuppiBarchartState * state,
				     GuppiSeqString * data)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_SEQ_STRING (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_ELEMENT_LABEL_DATA, data);
}

void
guppi_barchart_state_set_column_styles (GuppiBarchartState * state,
					GuppiSeqStyle * data)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_SEQ_STYLE (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_STYLE_DATA, data);
}

void
guppi_barchart_state_set_stacked (GuppiBarchartState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->stacked != x) {
    state->stacked = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_normalize_stacks (GuppiBarchartState * state,
					   gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->normalize_stacks != x) {
    state->normalize_stacks = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_edge_thickness (GuppiBarchartState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->edge_thickness != x) {
    state->edge_thickness = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_edge_color (GuppiBarchartState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->edge_color != x) {
    state->edge_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_default_color (GuppiBarchartState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->default_color != x) {
    state->default_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_use_stock_colors (GuppiBarchartState * state,
					   guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->use_stock_colors != x) {
    state->use_stock_colors = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_vertical_bars (GuppiBarchartState * state,
					gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->vertical_bars != x) {
    state->vertical_bars = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_horizontal_bars (GuppiBarchartState * state,
					  gboolean x)
{
  guppi_barchart_state_set_vertical_bars (state, !x);
}

void
guppi_barchart_state_set_bar_base (GuppiBarchartState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->bar_base != x) {
    state->bar_base = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_cluster_margin (GuppiBarchartState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->cluster_margin != x) {
    state->cluster_margin = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_bar_margin (GuppiBarchartState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->bar_margin != x) {
    state->bar_margin = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_enforce_preferred_view (GuppiBarchartState * state,
						 gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->enforce_preferred_view != x) {
    state->enforce_preferred_view = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_bar_pos_min (GuppiBarchartState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->bar_pos_min != x) {
    state->bar_pos_min = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_barchart_state_set_bar_pos_max (GuppiBarchartState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  if (state->bar_pos_max != x) {
    state->bar_pos_max = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}



/**************************************************************************/

/* Data Access Functions */

void
guppi_barchart_state_col_bounds (GuppiBarchartState * state, gint * c0,
				 gint * c1)
{
  GuppiData *data;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  data = guppi_barchart_state_data (state);

  if (data == NULL) {
    if (c0)
      *c0 = 0;
    if (c1)
      *c1 = -1;
  } else if (GUPPI_IS_SEQ_SCALAR (data)) {
    if (c0)
      *c0 = 0;
    if (c1)
      *c1 = 0;
  } else if (GUPPI_IS_SEQ_DATA (data)) {
    guppi_seq_bounds (GUPPI_SEQ (data), c0, c1);
  } else {
    g_assert_not_reached ();
  }
}

void
guppi_barchart_state_rows_in_col (GuppiBarchartState * state, gint c,
				  gint * r0, gint * r1)
{
  GuppiData *data;
  gint c0, c1;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  guppi_barchart_state_col_bounds (state, &c0, &c1);
  g_return_if_fail (c0 <= c && c <= c1);

  data = guppi_barchart_state_data (state);

  if (data == NULL) {
    if (r0)
      *r0 = 0;
    if (r1)
      *r1 = -1;
  } else if (GUPPI_IS_SEQ_SCALAR (data)) {
    guppi_seq_bounds (GUPPI_SEQ (data), r0, r1);
  } else if (GUPPI_IS_SEQ_DATA (data)) {
    data = guppi_seq_data_get (GUPPI_SEQ_DATA (data), c);
    guppi_seq_bounds (GUPPI_SEQ (data), r0, r1);
  } else {
    g_assert_not_reached ();
  }
}

double
guppi_barchart_state_data_by_rc (GuppiBarchartState * state, gint r, gint c)
{
  GuppiData *data;
  gint r0, r1, c0, c1;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), 0);

  guppi_barchart_state_col_bounds (state, &c0, &c1);
  g_return_val_if_fail (c0 <= c && c <= c1, 0);

  guppi_barchart_state_rows_in_col (state, c, &r0, &r1);
  g_return_val_if_fail (r0 <= r && r <= r1, 0);

  data = guppi_barchart_state_data (state);
  g_return_val_if_fail (data != NULL, 0);

  if (GUPPI_IS_SEQ_SCALAR (data)) {

    return guppi_seq_scalar_get (GUPPI_SEQ_SCALAR (data), r);

  } else if (GUPPI_IS_SEQ_DATA (data)) {

    data = guppi_seq_data_get (GUPPI_SEQ_DATA (data), c);
    g_assert (GUPPI_IS_SEQ_SCALAR (data));

    return guppi_seq_scalar_get (GUPPI_SEQ_SCALAR (data), r);

  } else {

    g_assert_not_reached ();

  }

  return 0;
}

/* Second-order accessors, defined in terms of our three basic operations. */

void
guppi_barchart_state_row_bounds (GuppiBarchartState * state, gint * r0,
				 gint * r1)
{
  gint c, c0, c1, rmin = 0, rmax = -1;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  guppi_barchart_state_col_bounds (state, &c0, &c1);

  for (c = c0; c <= c1; ++c) {
    gint rr0, rr1;

    guppi_barchart_state_rows_in_col (state, c, &rr0, &rr1);

    if (c == c0) {
      rmin = rr0;
      rmax = rr1;
    } else {
      if (rr0 < rmin)
	rmin = rr0;
      if (rr1 > rmax)
	rmax = rr1;
    }
  }

  if (r0)
    *r0 = rmin;
  if (r1)
    *r1 = rmax;
}

void
guppi_barchart_state_data_range (GuppiBarchartState * state,
				 double *min_ptr, double *max_ptr)
{
  gint c, c0, c1;
  double min = 0, max = -1;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_STATE (state));

  guppi_barchart_state_col_bounds (state, &c0, &c1);

  for (c = c0; c <= c1; ++c) {
    gint r, r0, r1;

    guppi_barchart_state_rows_in_col (state, c, &r0, &r1);

    if (c == c0) {
      min = max = guppi_barchart_state_data_by_rc (state, r0, c);
      ++r0;
    }

    for (r = r0; r <= r1; ++r) {
      double x = guppi_barchart_state_data_by_rc (state, r, c);
      if (x < min)
	min = x;
      if (x > max)
	max = x;
    }
  }

  if (min_ptr)
    *min_ptr = min;
  if (max_ptr)
    *max_ptr = max;
}

double
guppi_barchart_state_row_abs_sum (GuppiBarchartState * state, gint r)
{
  gint c1;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), 0);

  guppi_barchart_state_col_bounds (state, NULL, &c1);

  return guppi_barchart_state_row_abs_partial_sum (state, r, c1 + 1);
}

double
guppi_barchart_state_row_abs_partial_sum (GuppiBarchartState * state,
					  gint r, gint up_to_col)
{
  double sum = 0;
  gint c, c0, c1;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), 0);

  guppi_barchart_state_col_bounds (state, &c0, &c1);

  for (c = c0; c <= c1 && c < up_to_col; ++c) {
    gint r0, r1;
    guppi_barchart_state_rows_in_col (state, c, &r0, &r1);
    if (r0 <= r && r <= r1)
      sum += fabs (guppi_barchart_state_data_by_rc (state, r, c));
  }

  return sum;
}

double
guppi_barchart_state_max_row_abs_sum (GuppiBarchartState * state)
{
  gint r, r0, r1;
  double max = 0;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), 0);

  guppi_barchart_state_row_bounds (state, &r0, &r1);

  for (r = r0; r <= r1; ++r) {
    double x = guppi_barchart_state_row_abs_sum (state, r);
    if (r == r0)
      max = x;
    else if (x > max)
      max = x;
  }

  return max;
}

guint32 guppi_barchart_state_col_color (GuppiBarchartState * state, gint c)
{
  GuppiSeqStyle *data;
  guint32 stock;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_BARCHART_STATE (state), 0);

  data = guppi_barchart_state_column_styles (state);
  if (data != NULL) {
    GuppiStyle *sty = NULL;

    if (guppi_seq_in_bounds (GUPPI_SEQ (data), c) &&
	(sty = guppi_seq_style_get (data, c)) != NULL)
      return guppi_style_color (sty);
  }

  if (0 != (stock = guppi_barchart_state_use_stock_colors (state))) {
    GuppiStyle *sty = guppi_style_stock (c, stock);
    if (sty != NULL)
      return guppi_style_color (sty);
  }

  return guppi_barchart_state_default_color (state);
}


/* $Id: guppi-barchart-state.c,v 1.20 2001/05/06 07:56:27 trow Exp $ */
