/* This is -*- C -*- */
/* $Id: guppi-barchart-view.c,v 1.15 2001/05/06 07:56:27 trow Exp $ */

/*
 * guppi-barchart-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "guppi-barchart-view.h"
#include "guppi-barchart-state.h"
#include "guppi-barchart-item.h"
#include "guppi-barchart-print.h"
#include "guppi-barchart-tools.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_barchart_view_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_barchart_view_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_barchart_view_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_barchart_view_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/***************************************************************************/

static GuppiCanvasItem *
make_canvas_item (GuppiElementView * view, GnomeCanvas * canvas,
		  GnomeCanvasGroup * group)
{
  GnomeCanvasItem *gnoci;
  GuppiCanvasItem *gci;

  gnoci = gnome_canvas_item_new (group, GUPPI_TYPE_BARCHART_ITEM, NULL);

  gci = GUPPI_CANVAS_ITEM (gnoci);

  return gci;
}

static void
state_changed (GuppiElementView * view)
{
  GuppiBarchartState *state =
    GUPPI_BARCHART_STATE (guppi_element_view_state (view));

  if (guppi_barchart_state_enforce_preferred_view (state)) {
    guppi_element_view_set_preferred_view (view);
  }
}

static void
preferred_range (GuppiElementView * view, double *a, double *b)
{
  GuppiBarchartState *state;

  state = GUPPI_BARCHART_STATE (guppi_element_view_state (view));

  if (guppi_barchart_state_stacked (state)) {

    if (guppi_barchart_state_normalize_stacks (state)) {
      if (a)
	*a = 0;
      if (b)
	*b = 1;
    } else {

      /* Need to calc max row sum */
      if (a)
	*a = 0;
      if (b)
	*b = guppi_barchart_state_max_row_abs_sum (state);

    }

  } else {
    double bb = guppi_barchart_state_bar_base (state);
    double min, max, span;

    guppi_barchart_state_data_range (state, &min, &max);

    if (min > bb)
      min = bb;
    if (max < bb)
      max = bb;

    span = max - min;
    /* Add some margins.  It looks nicer this way. */
    /* That factor of 0.025 shouldn't be hard-wired... */
    if (max > bb)
      max += span * 0.025;
    if (min < bb)
      min -= span * 0.025;

    if (a)
      *a = min;
    if (b)
      *b = max;
  }
}

static gboolean
preferred_x_range (GuppiElementView * view, double *a, double *b)
{
  GuppiBarchartState *state;
  state = GUPPI_BARCHART_STATE (guppi_element_view_state (view));

  if (guppi_barchart_state_horizontal_bars (state)) {
    preferred_range (view, a, b);
  } else {
    if (a)
      *a = guppi_barchart_state_bar_pos_min (state);
    if (b)
      *b = guppi_barchart_state_bar_pos_max (state);
  }

  return TRUE;
}

static gboolean
preferred_y_range (GuppiElementView * view, double *a, double *b)
{
  GuppiBarchartState *state;
  state = GUPPI_BARCHART_STATE (guppi_element_view_state (view));

  if (guppi_barchart_state_vertical_bars (state)) {
    preferred_range (view, a, b);
  } else {
    if (a)
      *a = guppi_barchart_state_bar_pos_min (state);
    if (b)
      *b = guppi_barchart_state_bar_pos_max (state);
  }

  return TRUE;
}

static void
update_markers (GuppiElementView * ev,
		GuppiAxisMarkers * num_am,
		GuppiAxisMarkers * cat_am,
		double a, double b, gboolean is_x_axis)
{
  GuppiBarchartState *state;
  GuppiBarchartView *view;
  gboolean use_percent = FALSE;

  state = GUPPI_BARCHART_STATE0 (guppi_element_view_state (ev));
  view = GUPPI_BARCHART_VIEW (ev);

  if (state == NULL)
    return;

  if (num_am) {

    if (guppi_barchart_state_stacked (state) &&
	guppi_barchart_state_normalize_stacks (state))
      use_percent = TRUE;

    guppi_axis_markers_populate_scalar (num_am, a, b, 6, 10, use_percent);

  }

  if (cat_am) {
    GuppiSeqString *row_labels;

    row_labels = guppi_barchart_state_row_labels (state);

    guppi_axis_markers_freeze (cat_am);
    guppi_axis_markers_clear (cat_am);

    if (row_labels != NULL) {

      gint i, i0, i1, r0, r1, c0, c1;
      double x0, y0, x1, y1;

      guppi_seq_bounds (GUPPI_SEQ (row_labels), &i0, &i1);
      guppi_barchart_state_col_bounds (state, &c0, &c1);
      guppi_barchart_state_row_bounds (state, &r0, &r1);
      if (i0 < r0)
	i0 = r0;
      if (i1 > r1)
	i1 = r1;

      for (i = i0; i <= i1; ++i) {
	double p;

	guppi_barchart_view_bar_position (GUPPI_BARCHART_VIEW (ev),
					  i, c0, &x0, &y0, NULL, NULL);
	guppi_barchart_view_bar_position (GUPPI_BARCHART_VIEW (ev),
					  i, c1, NULL, NULL, &x1, &y1);
	p = is_x_axis ? (x0 + x1) / 2 : (y0 + y1) / 2;

	guppi_axis_markers_add (cat_am, p, GUPPI_TICK_NONE,
				guppi_seq_string_get (row_labels, i));
      }
    }

    guppi_axis_markers_thaw (cat_am);
  }
}

static void
update_x_axis_markers (GuppiElementView * ev, GuppiAxisMarkers * am,
		       double a, double b)
{
  GuppiBarchartState *state;
  GuppiBarchartView *view;

  state = GUPPI_BARCHART_STATE0 (guppi_element_view_state (ev));
  if (state == NULL)
    return;

  view = GUPPI_BARCHART_VIEW (ev);

  guppi_axis_markers_clear (am);

  if (guppi_barchart_state_horizontal_bars (state)) {
    update_markers (ev, am, NULL, a, b, TRUE);
  } else {
    update_markers (ev, NULL, am, a, b, TRUE);

  }
}

static void
update_y_axis_markers (GuppiElementView * ev, GuppiAxisMarkers * am,
		       double a, double b)
{
  GuppiBarchartState *state;
  GuppiBarchartView *view;

  state = GUPPI_BARCHART_STATE0 (guppi_element_view_state (ev));
  view = GUPPI_BARCHART_VIEW (ev);

  if (state == NULL)
    return;

  guppi_axis_markers_clear (am);

  if (guppi_barchart_state_vertical_bars (state)) {
    update_markers (ev, am, NULL, a, b, FALSE);
  } else {
    update_markers (ev, NULL, am, a, b, FALSE);
  }
}

/***************************************************************************/

static void
guppi_barchart_view_class_init (GuppiBarchartViewClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementViewClass *view_class = GUPPI_ELEMENT_VIEW_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_VIEW);

  object_class->get_arg = guppi_barchart_view_get_arg;
  object_class->set_arg = guppi_barchart_view_set_arg;
  object_class->destroy = guppi_barchart_view_destroy;
  object_class->finalize = guppi_barchart_view_finalize;

  view_class->make_canvas_item = make_canvas_item;
  view_class->print_type = GUPPI_TYPE_BARCHART_PRINT;
  view_class->state_changed = state_changed;

  view_class->preferred_x_range = preferred_x_range;
  view_class->preferred_y_range = preferred_y_range;

  view_class->update_x_axis_markers = update_x_axis_markers;
  view_class->update_y_axis_markers = update_y_axis_markers;
}

static void
guppi_barchart_view_init (GuppiBarchartView * obj)
{
  /* The key is to just have something non-zero here, to trigger
     axis construction. */
  guppi_element_view_set_x_axis_marker_type (GUPPI_ELEMENT_VIEW (obj),
					     GUPPI_AXIS_SCALAR);
  guppi_element_view_set_y_axis_marker_type (GUPPI_ELEMENT_VIEW (obj),
					     GUPPI_AXIS_SCALAR);
}

GtkType guppi_barchart_view_get_type (void)
{
  static GtkType guppi_barchart_view_type = 0;
  if (!guppi_barchart_view_type) {
    static const GtkTypeInfo guppi_barchart_view_info = {
      "GuppiBarchartView",
      sizeof (GuppiBarchartView),
      sizeof (GuppiBarchartViewClass),
      (GtkClassInitFunc) guppi_barchart_view_class_init,
      (GtkObjectInitFunc) guppi_barchart_view_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_barchart_view_type =
      gtk_type_unique (GUPPI_TYPE_ELEMENT_VIEW, &guppi_barchart_view_info);
  }
  return guppi_barchart_view_type;
}

void
guppi_barchart_view_bar_position (GuppiBarchartView * bc_view,
				  gint r, gint c,
				  double *bar_x0, double *bar_y0,
				  double *bar_x1, double *bar_y1)
{
  GuppiElementView *view;
  GuppiBarchartState *state;
  double x0, y0, x1, y1, val, w;
  gint r0, r1, R, c0, c1, C, rr, cc;

  g_return_if_fail (bc_view != NULL);
  g_return_if_fail (GUPPI_IS_BARCHART_VIEW (bc_view));

  view = GUPPI_ELEMENT_VIEW (bc_view);
  state = GUPPI_BARCHART_STATE (guppi_element_view_state (view));

  guppi_barchart_state_col_bounds (state, &c0, &c1);
  C = c1 - c0 + 1;
  g_return_if_fail (c0 <= c && c <= c1);

  guppi_barchart_state_rows_in_col (state, c, &r0, &r1);
  g_return_if_fail (r0 <= r && r <= r1);

  val = guppi_barchart_state_data_by_rc (state, r, c);

  guppi_barchart_state_row_bounds (state, &r0, &r1);
  R = r1 - r0 + 1;

  x0 = guppi_barchart_state_bar_pos_min (state);
  x1 = guppi_barchart_state_bar_pos_max (state);

  w = (x1 - x0) / R;
  rr = guppi_barchart_state_vertical_bars (state) ? r - r0 : r1 - r;
  x0 = x0 + rr * w;
  x1 = x0 + w;

  w = (x1 - x0) * guppi_barchart_state_cluster_margin (state);
  x0 += w / 2;
  x1 -= w / 2;


  if (guppi_barchart_state_stacked (state)) {

    y0 = guppi_barchart_state_row_abs_partial_sum (state, r, c);
    y1 = y0 + fabs (val);

    if (guppi_barchart_state_normalize_stacks (state)) {
      double S = guppi_barchart_state_row_abs_sum (state, r);
      y0 /= S;
      y1 /= S;
    }

  } else {

    w = (x1 - x0) / C;
    cc = guppi_barchart_state_vertical_bars (state) ? c - c0 : c1 - c;
    x0 = x0 + cc * w;
    x1 = x0 + w;

    w = (x1 - x0) * guppi_barchart_state_bar_margin (state);
    x0 += w / 2;
    x1 -= w / 2;

    y0 = guppi_barchart_state_bar_base (state);
    y1 = val;

  }

  if (guppi_barchart_state_vertical_bars (state)) {
    if (bar_x0)
      *bar_x0 = MIN (x0, x1);
    if (bar_x1)
      *bar_x1 = MAX (x0, x1);
    if (bar_y0)
      *bar_y0 = MIN (y0, y1);
    if (bar_y1)
      *bar_y1 = MAX (y0, y1);
  } else {
    /* Swap coordinates if we are horizontal */
    if (bar_y0)
      *bar_y0 = MIN (x0, x1);
    if (bar_y1)
      *bar_y1 = MAX (x0, x1);
    if (bar_x0)
      *bar_x0 = MIN (y0, y1);
    if (bar_x1)
      *bar_x1 = MAX (y0, y1);
  }
}

gboolean
guppi_barchart_view_find_bar_at_position (GuppiBarchartView * view,
					  double x, double y,
					  gint * r, gint * c)
{
  GuppiBarchartState *state;
  gint r0, r1, c0, c1, rr, cc;
  double x0, y0, x1, y1;
  g_return_val_if_fail (view != NULL && GUPPI_IS_BARCHART_VIEW (view), FALSE);

  state =
    GUPPI_BARCHART_STATE (guppi_element_view_state
			  (GUPPI_ELEMENT_VIEW (view)));

  guppi_barchart_state_col_bounds (state, &c0, &c1);
  for (cc = c0; cc <= c1; ++cc) {

    guppi_barchart_state_rows_in_col (state, cc, &r0, &r1);
    for (rr = r0; rr <= r1; ++rr) {
      guppi_barchart_view_bar_position (view, rr, cc, &x0, &y0, &x1, &y1);

      if (guppi_between (x0, x, x1) && guppi_between (y0, y, y1)) {
	if (r)
	  *r = rr;
	if (c)
	  *c = cc;
	return TRUE;
      }
    }
  }

  return FALSE;
}

/* $Id: guppi-barchart-view.c,v 1.15 2001/05/06 07:56:27 trow Exp $ */
