/* This is -*- C -*- */
/* $Id: guppi-seq-string.c,v 1.13 2001/05/06 08:26:35 trow Exp $ */

/*
 * guppi-seq-string.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdlib.h>
/* #include <gnome.h> */

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-memory.h>
#include "guppi-seq-string.h"
#include "guppi-seq-string-impl.h"


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_string_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_string_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_string_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static xmlNodePtr
export_xml_element (GuppiSeq *seq, gint i, GuppiXMLDocument *doc)
{
  const gchar *str;
  xmlNodePtr node;

  str = guppi_seq_string_get (GUPPI_SEQ_STRING (seq), i);
  node = xmlNewNode (doc->ns, "string");
  xmlAddChild (node, xmlNewText (str));

  return node;
}

static void
import_xml_element (GuppiSeq *seq, GuppiXMLDocument *doc, xmlNodePtr node)
{
  gchar *str;
  
  g_return_if_fail (!strcmp (node->name, "string"));
  
  /* I have to copy the string, because GuppiSeqString will later try to g_free it. */
  str = xmlNodeListGetString (doc->doc, node->xmlChildrenNode, 1);
  guppi_seq_string_append ((GuppiSeqString *) seq, str);
  free (str);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static void
guppi_seq_string_class_init (GuppiSeqStringClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);
  GuppiSeqClass *seq_class = GUPPI_SEQ_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_SEQ);

  data_class->type_name = _("String Sequence");

  seq_class->export_xml_element = export_xml_element;
  seq_class->import_xml_element = import_xml_element;

  object_class->get_arg = guppi_seq_string_get_arg;
  object_class->set_arg = guppi_seq_string_set_arg;
  object_class->finalize = guppi_seq_string_finalize;

}

static void
guppi_seq_string_init (GuppiSeqString * obj)
{

}

GtkType guppi_seq_string_get_type (void)
{
  static GtkType guppi_seq_string_type = 0;
  if (!guppi_seq_string_type) {
    static const GtkTypeInfo guppi_seq_string_info = {
      "GuppiSeqString",
      sizeof (GuppiSeqString),
      sizeof (GuppiSeqStringClass),
      (GtkClassInitFunc) guppi_seq_string_class_init,
      (GtkObjectInitFunc) guppi_seq_string_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_string_type =
      gtk_type_unique (GUPPI_TYPE_SEQ, &guppi_seq_string_info);
  }
  return guppi_seq_string_type;
}

GuppiData *
guppi_seq_string_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_SEQ_STRING, NULL, 0, NULL);
}

/**************************************************************************/

typedef struct _GuppiDataOp_String GuppiDataOp_String;
struct _GuppiDataOp_String {
  GuppiDataOp op;
  gint i;
  const gchar *str;
  gboolean copy;
};

static void
op_set (GuppiData * data, GuppiDataOp * op)
{
  GuppiSeqStringImpl *impl;
  GuppiSeqStringImplClass *impl_class;
  GuppiDataOp_String *sop = (GuppiDataOp_String *) op;

  impl = GUPPI_SEQ_STRING_IMPL (guppi_data_impl (data));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  if (sop->copy)
    impl_class->set (impl, sop->i, guppi_strdup (sop->str));
  else
    impl_class->set (impl, sop->i, (gchar *) sop->str);
}

static void
op_insert (GuppiData * data, GuppiDataOp * op)
{
  GuppiSeqStringImpl *impl;
  GuppiSeqStringImplClass *impl_class;
  GuppiDataOp_String *sop = (GuppiDataOp_String *) op;

  impl = GUPPI_SEQ_STRING_IMPL (guppi_data_impl (data));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert);
  if (sop->copy)
    impl_class->insert (impl, sop->i, guppi_strdup (sop->str));
  else
    impl_class->insert (impl, sop->i, (gchar *) sop->str);
}

/**************************************************************************/

const gchar *
guppi_seq_string_get (const GuppiSeqString * seq, gint i)
{
  GuppiSeqStringImpl *impl;
  GuppiSeqStringImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, NULL);
  g_return_val_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i), NULL);

  impl = GUPPI_SEQ_STRING_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get);
  return (impl_class->get) (impl, i);
}

void
guppi_seq_string_set_nc (GuppiSeqString * seq, gint i, gchar * str)
{
  GuppiDataOp_String op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_STRING (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  op.op.op = op_set;
  op.i = i;
  op.str = str;
  op.copy = FALSE;

  /* NOTE : The index is ignored */
  guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) & op);
}

void
guppi_seq_string_set (GuppiSeqString * seq, gint i, const gchar * str)
{
  GuppiDataOp_String op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_STRING (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (GUPPI_SEQ (seq), i));

  op.op.op = op_set;
  op.i = i;
  op.str = str;
  op.copy = TRUE;

  /* NOTE : The index is ignored */
  guppi_seq_changed_set (GUPPI_SEQ (seq), i, i, (GuppiDataOp *) & op);
}

void
guppi_seq_string_prepend_nc (GuppiSeqString * seq, gchar * str)
{
  gint first;
  first = guppi_seq_min_index (GUPPI_SEQ (seq));
  guppi_seq_string_insert_nc (seq, first, str);
}

void
guppi_seq_string_prepend (GuppiSeqString * seq, const gchar * str)
{
  gint first;
  first = guppi_seq_min_index (GUPPI_SEQ (seq));
  guppi_seq_string_insert (seq, first, str);
}

void
guppi_seq_string_append_nc (GuppiSeqString * seq, gchar * str)
{
  gint last;
  last = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_string_insert_nc (seq, last + 1, str);
}

void
guppi_seq_string_append (GuppiSeqString * seq, const gchar * str)
{
  gint last;
  last = guppi_seq_max_index (GUPPI_SEQ (seq));
  guppi_seq_string_insert (seq, last + 1, str);
}

void
guppi_seq_string_insert_nc (GuppiSeqString * seq, gint i, gchar * str)
{
  GuppiDataOp_String op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_STRING (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (str != NULL);

  op.op.op = op_insert;
  op.i = i;
  op.str = str;
  op.copy = FALSE;

  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, 1, (GuppiDataOp *) & op);
}

void
guppi_seq_string_insert (GuppiSeqString * seq, gint i, const gchar * str)
{
  GuppiDataOp_String op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ_STRING (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (str != NULL);

  op.op.op = op_insert;
  op.i = i;
  op.str = str;
  op.copy = TRUE;

  guppi_seq_changed_insert (GUPPI_SEQ (seq), i, 1, (GuppiDataOp *) & op);
}

gint guppi_seq_string_lookup (const GuppiSeqString * seq, const gchar * str)
{
  const GuppiSeqStringImpl *impl;
  GuppiSeqStringImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, 0);

  impl = GUPPI_SEQ_STRING_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->lookup) {

    return (impl_class->lookup) ((GuppiSeqStringImpl *) impl, str);

  } else {
    gint i0 = guppi_seq_min_index (GUPPI_SEQ (seq));
    gint i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
    gint i;

    g_assert (impl_class->get);

    for (i = i0; i <= i1; ++i) {
      if (!strcmp (str, (impl_class->get) (impl, i)))
	return i;
    }
    return i1 + 1;
  }
}

gboolean
guppi_seq_string_contains (const GuppiSeqString * seq, const gchar * str)
{
  const GuppiSeqStringImpl *impl;
  GuppiSeqStringImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, 0);

  impl = GUPPI_SEQ_STRING_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->contains) {

    return (impl_class->contains) ((GuppiSeqStringImpl *) impl, str);

  } else {

    return guppi_seq_in_bounds (GUPPI_SEQ (seq),
				guppi_seq_string_lookup (seq, str));
  }
}

gsize guppi_seq_string_distinct_values (const GuppiSeqString * seq)
{
  const GuppiSeqStringImpl *impl;
  GuppiSeqStringImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, 0);

  impl = GUPPI_SEQ_STRING_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_STRING_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (impl_class->distinct) {

    return (impl_class->distinct) ((GuppiSeqStringImpl *) impl);

  } else {
    gsize count = 0;
    GHashTable *hash = g_hash_table_new (g_str_hash, g_str_equal);
    const gchar *str;
    gpointer data;
    gint i0 = guppi_seq_min_index (GUPPI_SEQ (seq));
    gint i1 = guppi_seq_max_index (GUPPI_SEQ (seq));
    gint i;

    g_assert (impl_class->get);

    for (i = i0; i <= i1; ++i) {
      str = (impl_class->get) (impl, i);
      data = g_hash_table_lookup (hash, str);
      if (data == NULL) {
	++count;
	g_hash_table_insert (hash, (gchar *) str, (gpointer) str);
      }
    }

    g_hash_table_destroy (hash);
    return count;
  }
}



/* $Id: guppi-seq-string.c,v 1.13 2001/05/06 08:26:35 trow Exp $ */
