/* This is -*- C -*- */
/* $Id: guppi-seq.c,v 1.19 2001/05/06 08:26:35 trow Exp $ */

/*
 * guppi-seq.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 * Copyright (C) 2001 The Free Software Foundation
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <gtk/gtksignal.h>

#include <guppi-useful.h>
#include "guppi-seq.h"
#include "guppi-seq-impl.h"

static GtkObjectClass *parent_class = NULL;

enum {
  CHANGED_SHIFT_INDICES,
  CHANGED_SET,
  CHANGED_INSERT,
  CHANGED_DELETE,
  LAST_SIGNAL
};

static guint seq_signals[LAST_SIGNAL] = { 0 };

enum {
  ARG_0
};

static void
guppi_seq_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
changed_shift_indices (GuppiSeq * seq, gint i)
{
  guppi_data_changed (GUPPI_DATA (seq));
}

static void
changed_set (GuppiSeq * seq, gint i0, gint i1)
{
  guppi_data_changed (GUPPI_DATA (seq));
}

static void
changed_insert (GuppiSeq * seq, gint i, gsize N)
{
  guppi_data_changed (GUPPI_DATA (seq));
}

static void
changed_delete (GuppiSeq * seq, gint i, gsize N)
{
  guppi_data_changed (GUPPI_DATA (seq));
}

static void
prepare_attributes (GuppiAttributes *ax)
{
  if (GUPPI_DATA_CLASS (parent_class)->prepare_attributes)
    GUPPI_DATA_CLASS (parent_class)->prepare_attributes (ax);

  guppi_attributes_declare_int (ax, "min_index");
  guppi_attributes_declare_int (ax, "max_index");
}

static void
set_attributes (GuppiData *d, GuppiAttributes *ax)
{
  gint i0, i1;

  if (GUPPI_DATA_CLASS (parent_class)->set_attributes)
    GUPPI_DATA_CLASS (parent_class)->set_attributes (d, ax);

  guppi_seq_bounds (GUPPI_SEQ (d), &i0, &i1);
  guppi_attributes_set_int (ax, "min_index", i0);
  guppi_attributes_set_int (ax, "max_index", i1);
}

static xmlNodePtr
export_xml_content (GuppiData *data, GuppiXMLDocument *doc)
{
  GuppiSeqClass *klass;
  gint i, i0, i1;
  xmlNodePtr content_node;

  klass = GUPPI_SEQ_CLASS (GTK_OBJECT (data)->klass);

  if (! klass->export_xml_element) {
    return xmlNewComment ("XML element format not defined.");
  }

  content_node = xmlNewNode (doc->ns, "Content");
  
  guppi_seq_bounds (GUPPI_SEQ (data), &i0, &i1);
  for (i = i0; i <= i1; ++i) {
    xmlNodePtr data_node;
    if (guppi_seq_available (GUPPI_SEQ (data), i))
      data_node = klass->export_xml_element (GUPPI_SEQ (data), i, doc);
    else
      data_node = xmlNewNode (doc->ns, "missing");

    xmlAddChild (content_node, data_node);
  }

  return content_node;
}

static void
init_base_object (GuppiData *d, GuppiAttributes *ax)
{
  GuppiSeq *seq = GUPPI_SEQ (d);
  gboolean have_min, have_max;
  gint i0=0, i1=0;

  if (GUPPI_DATA_CLASS (parent_class)->init_base_object)
    GUPPI_DATA_CLASS (parent_class)->init_base_object (d, ax);

  if ( (have_min = guppi_attributes_is_defined (ax, "min_index")) )
    i0 = guppi_attributes_get_int (ax, "min_index");

  if ( (have_max = guppi_attributes_is_defined (ax, "max_index")) )
    i1 = guppi_attributes_get_int (ax, "max_index");

  if (have_min)
    guppi_seq_set_min_index (seq, i0);
  if (have_min && have_max)
    guppi_seq_size_hint (seq, i1-i0+1);
}

static void
import_xml_content (GuppiData *d, GuppiXMLDocument *doc, xmlNodePtr node)
{
  GuppiSeq *seq;
  GuppiSeqClass *klass;

  if (strcmp (node->name, "Content")) {
    g_warning ("Passed non-Content node (%s)", node->name);
    return;
  }

  seq = GUPPI_SEQ (d);
  klass = GUPPI_SEQ_CLASS (GTK_OBJECT (d)->klass);
  if (klass->import_xml_element == NULL) {
    g_warning("import_xml_element not defined");
    return;
  }

  node = node->xmlChildrenNode;

  while (node) {
    if (!strcmp (node->name, "missing"))
      guppi_seq_append_missing (seq);
    else
      klass->import_xml_element (seq, doc, node);
    node = node->next;
  }
}

static void
guppi_seq_class_init (GuppiSeqClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_DATA);

  object_class->get_arg = guppi_seq_get_arg;
  object_class->set_arg = guppi_seq_set_arg;
  object_class->finalize = guppi_seq_finalize;

  data_class->prepare_attributes = prepare_attributes;
  data_class->set_attributes = set_attributes;
  data_class->export_xml_content = export_xml_content;
  data_class->init_base_object = init_base_object;
  data_class->import_xml_content = import_xml_content;

  klass->changed_shift_indices = changed_shift_indices;
  klass->changed_set = changed_set;
  klass->changed_insert = changed_insert;
  klass->changed_delete = changed_delete;

  seq_signals[CHANGED_SHIFT_INDICES] =
    gtk_signal_new ("changed_shift_indices",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSeqClass, changed_shift_indices),
		    gtk_marshal_NONE__INT, GTK_TYPE_NONE, 1, GTK_TYPE_INT);

  seq_signals[CHANGED_SET] =
    gtk_signal_new ("changed_set",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSeqClass, changed_set),
		    gtk_marshal_NONE__INT_INT,
		    GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_INT);

  /* I've used the NONE__INT_INT marshaller here, even though the
     correct thing would be do use NONE__INT_UINT.  I'm just being
     lazy and using the predefined marshallers, though.  This should
     work fine --- Famous last words :-) */

  seq_signals[CHANGED_INSERT] =
    gtk_signal_new ("changed_insert",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSeqClass, changed_insert),
		    gtk_marshal_NONE__INT_INT,
		    GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_UINT);

  seq_signals[CHANGED_DELETE] =
    gtk_signal_new ("changed_delete",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSeqClass, changed_delete),
		    gtk_marshal_NONE__INT_INT,
		    GTK_TYPE_NONE, 2, GTK_TYPE_INT, GTK_TYPE_UINT);

  gtk_object_class_add_signals (object_class, seq_signals, LAST_SIGNAL);

}

static void
guppi_seq_init (GuppiSeq * obj)
{

}

GtkType guppi_seq_get_type (void)
{
  static GtkType guppi_seq_type = 0;
  if (!guppi_seq_type) {
    static const GtkTypeInfo guppi_seq_info = {
      "GuppiSeq",
      sizeof (GuppiSeq),
      sizeof (GuppiSeqClass),
      (GtkClassInitFunc) guppi_seq_class_init,
      (GtkObjectInitFunc) guppi_seq_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_type = gtk_type_unique (GUPPI_TYPE_DATA, &guppi_seq_info);
  }
  return guppi_seq_type;
}

/***************************************************************************/

/* Possible Operations */

typedef struct _GuppiDataOp_Seq GuppiDataOp_Seq;
struct _GuppiDataOp_Seq {
  GuppiDataOp op;
  gint i;
  gsize N;
  const gchar *str;
};

static void
op_shift (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  GuppiDataOp_Seq *seq_op = (GuppiDataOp_Seq *) op;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->shift_indices);
  impl_class->shift_indices (impl, seq_op->i);
}

static void
op_set (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  GuppiDataOp_Seq *seq_op = (GuppiDataOp_Seq *) op;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  impl_class->set (impl, seq_op->i, seq_op->str);
}

static void
op_set_missing (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  GuppiDataOp_Seq *seq_op = (GuppiDataOp_Seq *) op;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set_missing);
  impl_class->set_missing (impl, seq_op->i);
}

static void
op_insert (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  GuppiDataOp_Seq *seq_op = (GuppiDataOp_Seq *) op;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert);
  impl_class->insert (impl, seq_op->i, seq_op->str);
}

static void
op_insert_missing (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  GuppiDataOp_Seq *seq_op = (GuppiDataOp_Seq *) op;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->insert_missing);
  impl_class->insert_missing (impl, seq_op->i);
}

static void
op_delete (GuppiData * d, GuppiDataOp * op)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  GuppiDataOp_Seq *seq_op = (GuppiDataOp_Seq *) op;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (d));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->delete_many);
  impl_class->delete_many (impl, seq_op->i, seq_op->N);
}

/***************************************************************************/

void
guppi_seq_size_hint (GuppiSeq * seq, gsize expected_size)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_if_fail (seq != NULL);

  /* Read-only data silently ignores size hints.  It likes the size it
     is just fine, thank you. */
  if (guppi_data_is_read_only (GUPPI_DATA (seq)))
    return;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->size_hint);
  (impl_class->size_hint) (impl, expected_size);

}

void
guppi_seq_indices (const GuppiSeq * seq, gint * min, gint * max)
{
  const GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_if_fail (seq != NULL);

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get_bounds);
  (impl_class->get_bounds) (impl, min, max);
}

gint guppi_seq_min_index (const GuppiSeq * seq)
{
  gint min = 0;
  g_return_val_if_fail (seq != NULL, 0);
  guppi_seq_indices (seq, &min, NULL);
  return min;
}

gint guppi_seq_max_index (const GuppiSeq * seq)
{
  gint max = -1;
  g_return_val_if_fail (seq != NULL, -1);
  guppi_seq_indices (seq, NULL, &max);
  return max;
}

gsize guppi_seq_size (const GuppiSeq * seq)
{
  gint min, max;
  g_return_val_if_fail (seq != NULL, 0);
  guppi_seq_indices (seq, &min, &max);
  g_assert (max + 1 >= min);
  return max + 1 - min;
}

gsize guppi_seq_count (const GuppiSeq * seq)
{
  gsize size, missing;
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ (seq), 0);

  size = guppi_seq_size (seq);
  missing = guppi_seq_missing_count ((GuppiSeq *) seq);
  g_return_val_if_fail (size >= missing, 0);

  return size - missing;
}

gboolean guppi_seq_empty (const GuppiSeq * seq)
{
  g_return_val_if_fail (seq != NULL, TRUE);
  return guppi_seq_size (seq) == 0;
}

gboolean guppi_seq_nonempty (const GuppiSeq * seq)
{
  g_return_val_if_fail (seq != NULL, FALSE);
  return guppi_seq_size (seq) > 0;
}

gboolean guppi_seq_absent (const GuppiSeq * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ (seq), TRUE);
  return guppi_seq_count (seq) == 0;
}

gboolean guppi_seq_present (const GuppiSeq * seq)
{
  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ (seq), FALSE);
  return guppi_seq_count (seq) > 0;
}

gboolean guppi_seq_in_bounds (const GuppiSeq * seq, gint i)
{
  gint min = 0, max = -1;
  g_return_val_if_fail (seq != NULL, FALSE);
  guppi_seq_indices (seq, &min, &max);
  return min <= i && i <= max;
}

gboolean guppi_seq_contains_bounds (const GuppiSeq * a, const GuppiSeq * b)
{
  gint amin = 0, amax = -1, bmin = 0, bmax = -1;
  g_return_val_if_fail (a != NULL, FALSE);
  g_return_val_if_fail (b != NULL, FALSE);
  guppi_seq_indices (a, &amin, &amax);
  guppi_seq_indices (b, &bmin, &bmax);

  return amin <= bmin && bmax <= amax;
}

gboolean guppi_seq_equal_bounds (const GuppiSeq * a, const GuppiSeq * b)
{
  gint amin = 0, amax = -1, bmin = 0, bmax = -1;
  g_return_val_if_fail (a != NULL, FALSE);
  g_return_val_if_fail (b != NULL, FALSE);
  guppi_seq_indices (a, &amin, &amax);
  guppi_seq_indices (b, &bmin, &bmax);

  return amin == bmin && bmax == amax;
}

void
guppi_seq_common_bounds (const GuppiSeq * a, const GuppiSeq * b,
			 gint * min, gint * max)
{
  gint amin = 0, amax = -1, bmin = 0, bmax = -1;
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);
  guppi_seq_indices (a, &amin, &amax);
  guppi_seq_indices (b, &bmin, &bmax);

  if (min)
    *min = MAX (amin, bmin);
  if (max)
    *max = MIN (amax, bmax);
}

void
guppi_seq_shift_indices (GuppiSeq * seq, gint x)
{
  GuppiDataOp_Seq op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (x == 0)
    return;

  /* I confess... I love typing op.op.op! */

  op.op.op = op_shift;
  op.i = x;

  guppi_seq_changed_shift_indices (seq, x, (GuppiDataOp *) & op);
}

void
guppi_seq_set_min_index (GuppiSeq * seq, gint x)
{
  gint min;
  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  min = guppi_seq_min_index (seq);
  guppi_seq_shift_indices (seq, x - min);
}

void
guppi_seq_set_max_index (GuppiSeq * seq, gint x)
{
  gint max;
  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  max = guppi_seq_max_index (seq);
  guppi_seq_shift_indices (seq, x - max);
}

gboolean guppi_seq_validate (const GuppiSeq * seq, const gchar * str)
{
  return guppi_seq_validate_with_feedback (seq, str, NULL, 0);
}

gboolean
guppi_seq_validate_with_feedback (const GuppiSeq * seq,
				  const gchar * str,
				  gchar * error_buf, gsize errbuf_len)
{
  const GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_val_if_fail (seq != NULL, FALSE);
  g_return_val_if_fail (str != NULL, FALSE);

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->validate);

  return (impl_class->validate) (impl, str, error_buf, errbuf_len);
}

gchar *
guppi_seq_get_string (const GuppiSeq * seq, gint i)
{
  gchar buffer[512];		/* arbitrary upper limit */
  buffer[0] = '\0';
  guppi_seq_get_to_string (seq, i, buffer, 512);
  return guppi_strdup (buffer);
}

void
guppi_seq_get_to_string (const GuppiSeq * seq, gint i,
			 gchar * sbuf, gsize sbuf_len)
{
  const GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (sbuf != NULL);

  if (sbuf_len == 0)
    return;

  g_return_if_fail (guppi_seq_in_bounds (seq, i));

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get);

  (impl_class->get) (impl, i, sbuf, sbuf_len);
}

void
guppi_seq_set_from_string (GuppiSeq * seq, gint i, const gchar * str)
{
  GuppiDataOp_Seq op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_seq_in_bounds (seq, i));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  op.op.op = op_set;
  op.i = i;
  op.str = str;

  guppi_seq_changed_set (seq, i, i, (GuppiDataOp *) & op);
}

void
guppi_seq_insert_from_string (GuppiSeq * seq, gint i, const gchar * str)
{
  GuppiDataOp_Seq op;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_seq_in_bounds (seq, i));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  op.op.op = op_insert;
  op.i = i;
  op.str = str;

  guppi_seq_changed_insert (seq, i, 1, (GuppiDataOp *) & op);
}

void
guppi_seq_prepend_from_string (GuppiSeq * seq, const gchar * str)
{
  gint first;
  first = guppi_seq_min_index (seq);
  guppi_seq_insert_from_string (seq, first, str);
}

void
guppi_seq_append_from_string (GuppiSeq * seq, const gchar * str)
{
  gint last;
  last = guppi_seq_max_index (seq);
  guppi_seq_insert_from_string (seq, last + 1, str);
}

void
guppi_seq_delete (GuppiSeq * seq, gint i)
{
  GuppiDataOp_Seq op;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (seq, i));

  op.op.op = op_delete;
  op.i = i;
  op.N = 1;

  guppi_seq_changed_delete (seq, i, 1, (GuppiDataOp *) & op);
}

void
guppi_seq_delete_many (GuppiSeq * seq, gint i, gsize N)
{
  GuppiDataOp_Seq op;
  gint i1;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (seq, i));

  if (N == 0)
    return;

  i1 = guppi_seq_max_index (seq);
  if (i + N - 1 > i1)
    N = i1 - i + 1;

  op.op.op = op_delete;
  op.i = i;
  op.N = N;

  guppi_seq_changed_delete (seq, i, N, (GuppiDataOp *) & op);
}

void
guppi_seq_delete_range (GuppiSeq * seq, gint i0, gint i1)
{
  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (seq, i0));
  g_return_if_fail (guppi_seq_in_bounds (seq, i1));

  guppi_2sort_i (&i0, &i1);

  guppi_seq_delete_many (seq, i0, i1 - i0 + 1);
}

void
guppi_seq_grow_to_include (GuppiSeq * seq, gint i)
{
  guppi_seq_grow_to_include_range (seq, i, i);
}

void
guppi_seq_grow_to_include_range (GuppiSeq * seq, gint i0, gint i1)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;
  gint j0, j1;

  g_return_if_fail (seq != NULL);
  g_return_if_fail (GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  if (guppi_seq_in_bounds (seq, i0) && guppi_seq_in_bounds (seq, i1))
    return;

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->grow_to_include);

  guppi_2sort_i (&i0, &i1);

  guppi_seq_bounds (GUPPI_SEQ (seq), &j0, &j1);

  /* Should emit more specific signals */

  impl_class->grow_to_include (impl, i0, i1);
}

void
guppi_seq_grow_to_overlap (GuppiSeq * seq, const GuppiSeq * seq2)
{
  gint i0, i1;

  g_return_if_fail (seq2 != NULL);
  g_return_if_fail (GUPPI_IS_SEQ (seq2));

  if (guppi_seq_size (seq2) > 0) {
    guppi_seq_indices (seq2, &i0, &i1);
    guppi_seq_grow_to_include_range (seq, i0, i1);
  }
}

/**************************************************************************/

gboolean guppi_seq_has_missing (GuppiSeq * seq)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ (seq), FALSE);

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  return impl_class->has_missing && impl_class->has_missing (impl);
}

gsize guppi_seq_missing_count (GuppiSeq * seq)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ (seq), FALSE);

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  return impl_class->missing_count ? impl_class->missing_count (impl) : 0;
}

gboolean guppi_seq_missing (GuppiSeq * seq, gint i)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *impl_class;

  g_return_val_if_fail (seq != NULL && GUPPI_IS_SEQ (seq), FALSE);

  impl = GUPPI_SEQ_IMPL (guppi_data_impl (GUPPI_DATA (seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  return impl_class->missing && impl_class->missing (impl, i);
}

gboolean guppi_seq_available (GuppiSeq * seq, gint i)
{
  return !guppi_seq_missing (seq, i);
}

void
guppi_seq_set_missing (GuppiSeq * seq, gint i)
{
  gboolean x;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));
  g_return_if_fail (guppi_seq_in_bounds (seq, i));

  x = guppi_seq_missing (seq, i);
  if (!x) {

    GuppiDataOp_Seq op;
    op.op.op = op_set_missing;
    op.i = i;

    guppi_seq_changed_set (seq, i, i, (GuppiDataOp *) & op);
  }
}

void
guppi_seq_insert_missing (GuppiSeq * seq, gint i)
{
  GuppiDataOp_Seq op;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  op.op.op = op_insert_missing;
  op.i = i;

  guppi_seq_changed_insert (seq, i, 1, (GuppiDataOp *) & op);
}

void
guppi_seq_prepend_missing (GuppiSeq * seq)
{
  gint first;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  first = guppi_seq_min_index (seq);
  guppi_seq_insert_missing (seq, first);
}

void
guppi_seq_append_missing (GuppiSeq * seq)
{
  gint last;

  g_return_if_fail (seq != NULL && GUPPI_IS_SEQ (seq));
  g_return_if_fail (guppi_data_can_change (GUPPI_DATA (seq)));

  last = guppi_seq_max_index (seq);
  guppi_seq_insert_missing (seq, last + 1);
}

/**************************************************************************/

void
guppi_seq_changed_shift_indices (GuppiSeq * s, gint i, GuppiDataOp * op)
{
  g_return_if_fail (s != NULL && GUPPI_IS_SEQ (s));
  g_return_if_fail (op != NULL);

  guppi_data_add_pending_op (GUPPI_DATA (s), op);
  gtk_signal_emit (GTK_OBJECT (s), seq_signals[CHANGED_SHIFT_INDICES], i);
}

void
guppi_seq_changed_set (GuppiSeq * s, gint i0, gint i1, GuppiDataOp * op)
{
  g_return_if_fail (s != NULL && GUPPI_IS_SEQ (s));
  g_return_if_fail (op != NULL);

  guppi_data_add_pending_op (GUPPI_DATA (s), op);
  gtk_signal_emit (GTK_OBJECT (s), seq_signals[CHANGED_SET], i0, i1);
}

void
guppi_seq_changed_insert (GuppiSeq * s, gint i, gsize N, GuppiDataOp * op)
{
  g_return_if_fail (s != NULL && GUPPI_IS_SEQ (s));
  g_return_if_fail (op != NULL);

  guppi_data_add_pending_op (GUPPI_DATA (s), op);
  gtk_signal_emit (GTK_OBJECT (s), seq_signals[CHANGED_INSERT], i, N);
}

void
guppi_seq_changed_delete (GuppiSeq * s, gint i, gsize N, GuppiDataOp * op)
{
  g_return_if_fail (s != NULL && GUPPI_IS_SEQ (s));
  g_return_if_fail (op != NULL);

  guppi_data_add_pending_op (GUPPI_DATA (s), op);
  gtk_signal_emit (GTK_OBJECT (s), seq_signals[CHANGED_DELETE], i, N);
}

/* $Id: guppi-seq.c,v 1.19 2001/05/06 08:26:35 trow Exp $ */
