/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-attributes.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 * Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include "guppi-rgb.h"
#include "guppi-convenient.h"
#include "guppi-string.h"
#include "guppi-attributes.h"

typedef struct _AttrItem AttrItem;
struct _AttrItem {
  gchar *type;
  gboolean undef;
  gpointer ptr;

  /* vtable */
  AttributeStringToPointerFn s2p;
  AttributePointerToStringFn p2s;
  AttributePointerDestroyFn  pdest;
  AttributeCustomExportFn    c_export;
  AttributeCustomImportFn    c_import;
};

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_attributes_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_attributes_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
g_hash_free (gpointer key, gpointer val, gpointer user_data)
{
  gchar *name = (gchar *)key;
  AttrItem *item = (AttrItem *)val;

  if (item->pdest)
    item->pdest (item->ptr);
  item->ptr = NULL;

  guppi_free (name);
  guppi_free (item->type);
  guppi_free (item);
}

static void
guppi_attributes_finalize (GtkObject *obj)
{
  GuppiAttributes *ax = GUPPI_ATTRIBUTES (obj);

  guppi_finalized (obj);

  g_hash_table_foreach (ax->attr_table, g_hash_free, NULL);

  g_hash_table_destroy (ax->attr_table);
  ax->attr_table = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_attributes_class_init (GuppiAttributesClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_attributes_get_arg;
  object_class->set_arg = guppi_attributes_set_arg;
  object_class->finalize = guppi_attributes_finalize;
}

static void
guppi_attributes_init (GuppiAttributes *obj)
{
  obj->attr_table = g_hash_table_new (g_str_hash, g_str_equal);
}

GtkType
guppi_attributes_get_type (void)
{
  static GtkType guppi_attributes_type = 0;
  if (!guppi_attributes_type) {
    static const GtkTypeInfo guppi_attributes_info = {
      "GuppiAttributes",
      sizeof(GuppiAttributes),
      sizeof(GuppiAttributesClass),
      (GtkClassInitFunc)guppi_attributes_class_init,
      (GtkObjectInitFunc)guppi_attributes_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_attributes_type = gtk_type_unique (GTK_TYPE_OBJECT,
						 &guppi_attributes_info);
  }
  return guppi_attributes_type;
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

GuppiAttributes *
guppi_attributes_new (void)
{
  return
    GUPPI_ATTRIBUTES (guppi_type_new (guppi_attributes_get_type ()));
}

static AttrItem *
guppi_attributes_get (GuppiAttributes *ax, const gchar *name)
{
  gpointer ptr;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), FALSE);
  g_return_val_if_fail (name, FALSE);
  
  if (ax->attr_table 
      && g_hash_table_lookup_extended (ax->attr_table, name, NULL, &ptr))
    return (AttrItem *)ptr;
  
  return NULL;
}

gboolean
guppi_attributes_check_type(GuppiAttributes *ax, const gchar *name, const gchar *type)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), FALSE);
  g_return_val_if_fail (name, FALSE);

  item = guppi_attributes_get (ax, name);
  return item ? !strcmp (item->type, type) : FALSE;
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

void
guppi_attributes_declare_full (GuppiAttributes *ax,
			       const gchar *name,
			       const gchar *type,
			       AttributeStringToPointerFn s2p,
			       AttributePointerToStringFn p2s,
			       AttributePointerDestroyFn  pdest)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (type);
  g_return_if_fail (s2p);
  g_return_if_fail (p2s);

  if (guppi_attributes_get (ax, name) != NULL) {
    g_warning ("Namespace collision: \"%s\"", name);
    return;
  }

  item = guppi_new0 (AttrItem, 1);
  item->type  = g_strdup (type);
  item->undef = TRUE;
  item->s2p   = s2p;
  item->p2s   = p2s;
  item->pdest = pdest;

  g_hash_table_insert (ax->attr_table, guppi_strdup (name), item);
}

void
guppi_attributes_declare_full_custom (GuppiAttributes *ax,
				      const gchar *name,
				      const gchar *type,
				      AttributeCustomExportFn   c_export,
				      AttributeCustomImportFn   c_import,
				      AttributePointerDestroyFn pdest)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (type);
  g_return_if_fail (c_export);
  g_return_if_fail (c_import);

  if (guppi_attributes_get (ax, name) != NULL) {
    g_warning ("Namespace collision: \"%s\"", name);
    return;
  }

  item = guppi_new0 (AttrItem, 1);
  item->type = g_strdup (type);
  item->undef    = TRUE;
  item->c_export = c_export;
  item->c_import = c_import;
  item->pdest    = pdest;

  g_hash_table_insert (ax->attr_table, guppi_strdup (name), item);
}

void
guppi_attributes_undeclare (GuppiAttributes *ax, const gchar *name)
{
  gpointer *orig_key = NULL;
  gpointer *value = NULL;
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  if (! g_hash_table_lookup_extended (ax->attr_table, name, orig_key, value))
    return;
  
  g_hash_table_remove (ax->attr_table, name);
  guppi_free (*orig_key);

  item = (AttrItem *) *value;
  if (item->pdest)
    item->pdest (item->ptr);
  guppi_free (item);
}

gboolean
guppi_attributes_declared (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), FALSE);
  g_return_val_if_fail (name != NULL, FALSE);

  return guppi_attributes_get (ax, name) != NULL;
}

gpointer
guppi_attributes_get_raw (GuppiAttributes *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), NULL);
  g_return_val_if_fail (name != NULL, NULL);

  item = guppi_attributes_get (ax, name);

  return item ? item->ptr : NULL;
}

void
guppi_attributes_set_raw (GuppiAttributes *ax, const gchar *name, gpointer ptr)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name != NULL);

  item = guppi_attributes_get (ax, name);

  if (item == NULL) {
    g_warning ("\"%s\" is undeclared", name);
    return;
  }

  if (item->pdest)
    item->pdest (item->ptr);
  item->ptr = ptr;

  item->undef = FALSE;
}

gboolean
guppi_attributes_is_undef (GuppiAttributes *ax, const gchar *name)
{
  AttrItem *item;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), FALSE);
  g_return_val_if_fail (name != NULL, FALSE);
  
  item = guppi_attributes_get (ax, name);
  if (item == NULL) {
    g_warning ("\"%s\" is undeclared", name);
    return FALSE;
  }
  
  return item->undef;
}

gboolean
guppi_attributes_is_defined (GuppiAttributes *ax, const gchar *name)
{
  return ! guppi_attributes_is_undef (ax, name);
}

void
guppi_attributes_set_undef (GuppiAttributes *ax, const gchar *name)
{
  AttrItem *item;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name != NULL);

  item = guppi_attributes_get (ax, name);
  if (item == NULL) {
    g_warning ("\"%s\" is undeclared", name);
    return;
  }

  item->undef = TRUE;
  
  if (item->pdest)
    item->pdest (item->ptr);
  item->ptr = NULL;
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
string_s2p (const gchar *str)
{
  return (gpointer) guppi_strdup (str);
}

static gchar *
string_p2s (gpointer ptr)
{
  return guppi_strdup ((gchar *) ptr);
}

static void
string_dest (gpointer ptr)
{
  guppi_free (ptr);
}

void
guppi_attributes_declare_string (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "string", string_s2p, string_p2s, string_dest);
}


const gchar *
guppi_attributes_get_string (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), NULL);
  g_return_val_if_fail (name, NULL);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "string"), NULL);

  return (const gchar *) guppi_attributes_get_raw (ax, name);
}

void
guppi_attributes_set_string (GuppiAttributes *ax, const gchar *name,
				 const gchar *str)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "string"));
  
  guppi_attributes_set_raw (ax, name, g_strdup (str));
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
int_s2p (const gchar *str)
{
  return GINT_TO_POINTER (atoi (str));
}

static gchar *
int_p2s (gpointer ptr)
{
  return g_strdup_printf ("%d", GPOINTER_TO_INT (ptr));
}

void
guppi_attributes_declare_int (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  
  guppi_attributes_declare_full (ax, name, "int", int_s2p, int_p2s, NULL);
}
	
gint
guppi_attributes_get_int (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), 0);
  g_return_val_if_fail (name, 0);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "int"), 0);

  return GPOINTER_TO_INT (guppi_attributes_get_raw (ax, name));
}

void
guppi_attributes_set_int (GuppiAttributes *ax, const gchar *name, gint n)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "int"));

  guppi_attributes_set_raw (ax, name, GINT_TO_POINTER (n));
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
uint32_s2p (const gchar *str)
{
  guint32 x;
  if (sscanf (str, "%u", &x) != 1)
    x = 0;
  return GUINT_TO_POINTER (x);
}

static gchar *
uint32_p2s (gpointer ptr)
{
  return g_strdup_printf ("%u", GPOINTER_TO_UINT (ptr));
}

#define UINT32_MAGIC 0xbb19

void
guppi_attributes_declare_uint32 (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  
  guppi_attributes_declare_full (ax, name, "uint32", uint32_s2p, uint32_p2s, NULL);
}
	
guint32
guppi_attributes_get_uint32 (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), 0);
  g_return_val_if_fail (name, 0);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "uint32"), 0);

  return GPOINTER_TO_UINT (guppi_attributes_get_raw (ax, name));
}

void
guppi_attributes_set_uint32 (GuppiAttributes *ax, const gchar *name, guint32 n)
{
   g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
   g_return_if_fail (name);
   g_return_if_fail (guppi_attributes_check_type (ax, name, "uint32"));

   guppi_attributes_set_raw (ax, name, GUINT_TO_POINTER (n));
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
double_s2p (const gchar *str)
{
  double *p = guppi_new0 (double, 1);
  *p = atof (str);
  return p;
}

static gchar *
double_p2s (gpointer ptr)
{
  return g_strdup_printf ("%g", ptr ? *(double *) ptr : 0);
}

static void
double_dest (gpointer ptr)
{
  guppi_free (ptr);
}

void
guppi_attributes_declare_double (GuppiAttributes *ax,
				     const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "double", double_s2p, double_p2s, double_dest);
}

double
guppi_attributes_get_double (GuppiAttributes *ax, const gchar *name)
{
  gpointer ptr;
  
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), 0);
  g_return_val_if_fail (name, 0);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "double"), 0);

  ptr = guppi_attributes_get_raw (ax, name);
  return ptr ? *(double *) ptr : 0;
}

void
guppi_attributes_set_double (GuppiAttributes *ax, const gchar *name,
				 double x)
{
  double *p;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "double"));

  p = guppi_new (double, 1);
  *p = x;

  guppi_attributes_set_raw (ax, name, p);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
boolean_s2p (const gchar *str)
{
  return GINT_TO_POINTER ( (gint) (str && (*str == 't' || *str == 'T')));

}

static gchar *
boolean_p2s (gpointer ptr)
{
  return g_strdup (((gboolean) GPOINTER_TO_INT (ptr)) ? "true" : "false");
}

void
guppi_attributes_declare_boolean (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "boolean", boolean_s2p, boolean_p2s, NULL);
}

gboolean
guppi_attributes_get_boolean (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), FALSE);
  g_return_val_if_fail (name, FALSE);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "boolean"), FALSE);

  return (gboolean) GPOINTER_TO_INT (guppi_attributes_get_raw (ax, name));
}

void
guppi_attributes_set_boolean (GuppiAttributes *ax, const gchar *name, gboolean x)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "boolean"));

  guppi_attributes_set_raw (ax, name, GINT_TO_POINTER ((gint) x));
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
date_s2p (const gchar *str)
{
  gint y = 0, m = 0, d = 0;
  GDate *dt;
  if (str == NULL || sscanf (str, "%d-%d-%d", &y, &m, &d) != 3)
    return NULL;
  dt = g_date_new ();
  g_date_set_dmy (dt, d, m, y);
  return dt;
}

static gchar *
date_p2s (gpointer ptr)
{
  GDateDay d;
  GDateMonth m;
  GDateYear y;
  
  d = g_date_day   ((GDate *) ptr);
  m = g_date_month ((GDate *) ptr);
  y = g_date_year  ((GDate *) ptr);

  return g_strdup_printf ("%d-%d-%d", (gint)y, (gint)m, (gint)d);
}

static void
date_dest (gpointer ptr)
{
  g_date_free ((GDate *) ptr);
}

void
guppi_attributes_declare_date (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "date", date_s2p, date_p2s, date_dest);
}

const GDate *
guppi_attributes_get_date (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), NULL);
  g_return_val_if_fail (name, NULL);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "date"), NULL);

  return (const GDate *) guppi_attributes_get_raw (ax, name);
}

void
guppi_attributes_set_date (GuppiAttributes *ax, const gchar *name,
			       const GDate *dt)
{
  GDate *cpy;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "date"));

  cpy = g_date_new ();
  memcpy (cpy, dt, sizeof (GDate));
  guppi_attributes_set_raw (ax, name, cpy);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
rgb_s2p (const gchar *str)
{
  gint r, g, b;
  guint32 c = 0;

  if (sscanf (str, "#%2X%2X%2X", &r, &g, &b) == 3)
    c = RGB_TO_UINT (r, g, b);

  return GUINT_TO_POINTER (c);
}

static gchar *
rgb_p2s (gpointer ptr)
{
  guint32 c = GPOINTER_TO_UINT (ptr);
  guint r, g, b;

  UINT_TO_RGB (c, &r, &g, &b);

  return g_strdup_printf ("#%02X%02X%02X", r, g, b);
			  
}

void
guppi_attributes_declare_rgb (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "rgb", rgb_s2p, rgb_p2s, NULL);
}

guint32
guppi_attributes_get_rgb (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), 0);
  g_return_val_if_fail (name, 0);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "rgb"), 0);

  return GPOINTER_TO_UINT (guppi_attributes_get_raw (ax, name));
}

void
guppi_attributes_set_rgb (GuppiAttributes *ax, const gchar *name, guint32 color)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "rgb"));

  guppi_attributes_set_raw (ax, name, GUINT_TO_POINTER (color));
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
rgba_s2p (const gchar *str)
{
  gint r, g, b, a, rv;
  guint32 c = 0;

  a = 0xff;
  rv = sscanf (str, "#%2X%2X%2X%2X", &r, &g, &b, &a);
  if (rv == 3 || rv == 4)
    c = RGBA_TO_UINT (r, g, b, a);

  return GUINT_TO_POINTER (c);
}

static gchar *
rgba_p2s (gpointer ptr)
{
  guint32 c = GPOINTER_TO_UINT (ptr);
  guint r, g, b, a;

  UINT_TO_RGBA (c, &r, &g, &b, &a);

  return g_strdup_printf ("#%02X%02X%02X%02X", r, g, b, a);
			  
}

void
guppi_attributes_declare_rgba (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "rgba", rgba_s2p, rgba_p2s, NULL);
}

guint32
guppi_attributes_get_rgba (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), 0);
  g_return_val_if_fail (name, 0);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "rgba"), 0);

  return GPOINTER_TO_UINT (guppi_attributes_get_raw (ax, name));
}

void
guppi_attributes_set_rgba (GuppiAttributes *ax, const gchar *name, guint32 color)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "rgba"));

  guppi_attributes_set_raw (ax, name, GUINT_TO_POINTER (color));
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
font_s2p (const gchar *str)
{
  return gnome_font_new_from_full_name (str);
}

static gchar *
font_p2s (gpointer ptr)
{
  return gnome_font_get_full_name (GNOME_FONT (ptr));
}

static void
font_dest (gpointer ptr)
{
  guppi_unref (ptr);
}

void
guppi_attributes_declare_font (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);

  guppi_attributes_declare_full (ax, name, "font", font_s2p, font_p2s, font_dest);
}

GnomeFont *
guppi_attributes_get_font (GuppiAttributes *ax, const gchar *name)
{
  gpointer ptr;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), NULL);
  g_return_val_if_fail (name, NULL);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "font"), NULL);

  ptr = guppi_attributes_get_raw (ax, name);

  return ptr ? GNOME_FONT (ptr) : NULL;
}

void
guppi_attributes_set_font (GuppiAttributes *ax, const gchar *name, GnomeFont *f)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (f && GNOME_IS_FONT (f));

  guppi_attributes_set_raw (ax, name, f);
  guppi_ref (f);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static gpointer
uid_s2p (const gchar *str)
{
  guppi_uniq_t *p = guppi_new (guppi_uniq_t, 1);
  *p = guppi_str2uniq (str);
  return p;
}

static gchar *
uid_p2s (gpointer ptr)
{
  guppi_uniq_t *p = (guppi_uniq_t *) ptr;
  gchar str[18];
  guppi_uniq2str (*p, str);
  return g_strdup (str);
}

static void
uid_dest (gpointer ptr)
{
  guppi_free (ptr);
}

void
guppi_attributes_declare_unique_id (GuppiAttributes *ax, const gchar *name)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  
  guppi_attributes_declare_full (ax, name, "unique_id", uid_s2p, uid_p2s, uid_dest);
}

guppi_uniq_t
guppi_attributes_get_unique_id (GuppiAttributes *ax, const gchar *name)
{
  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), (guppi_uniq_t) 0);
  g_return_val_if_fail (name, (guppi_uniq_t) 0);
  g_return_val_if_fail (guppi_attributes_check_type (ax, name, "unique_id"),
			(guppi_uniq_t) 0);

  return *(guppi_uniq_t *) guppi_attributes_get_raw (ax, name);
}

void
guppi_attributes_set_unique_id (GuppiAttributes *ax, const gchar *name,
				    guppi_uniq_t uid)
{
  guppi_uniq_t *p;

  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (name);
  g_return_if_fail (guppi_attributes_check_type (ax, name, "unique_id"));

  p = guppi_new (guppi_uniq_t, 1);
  *p = uid;
  guppi_attributes_set_raw (ax, name, p);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

struct OutPair {
  xmlNodePtr attributes_node;
  GuppiXMLDocument *doc;
};

static void
output_cb (gpointer name_ptr, gpointer value_ptr, gpointer user_data)
{
  gchar *name = (gchar *) name_ptr;
  AttrItem *item = (AttrItem *) value_ptr;
  struct OutPair *pair = (struct OutPair *) user_data;
  xmlNodePtr attribute_node;
  xmlNsPtr ns;

  if (item->undef)
    return;

  ns = pair->attributes_node->ns;
  
  attribute_node = xmlNewNode (ns, "Attribute");
  xmlAddChild (pair->attributes_node, attribute_node);
  
  xmlNewTextChild (attribute_node, ns, "name", name);
  xmlNewTextChild (attribute_node, ns, "type", item->type);
  
  if (item->p2s) {

    gchar *s = item->p2s (item->ptr);
    xmlNewTextChild (attribute_node, ns, "value", s);
    guppi_free (s);

  } else if (item->c_export) {
    xmlNodePtr value_node;
    xmlNodePtr data_node;

    value_node = xmlNewNode (ns, "value");
    data_node = item->c_export (pair->doc, item->ptr);
    xmlAddChild (value_node, data_node);
    xmlAddChild (attribute_node, value_node);

  } else {

    g_assert_not_reached ();

  }
}

xmlNodePtr
guppi_attributes_export_xml (GuppiAttributes *ax, GuppiXMLDocument *doc)
{
  struct OutPair pair;

  g_return_val_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax), NULL);
  g_return_val_if_fail (doc, NULL);

  pair.attributes_node = xmlNewNode (doc->ns, "Attributes");
  pair.doc = doc;

  if (ax->attr_table) {
    g_hash_table_foreach (ax->attr_table, output_cb, &pair);
  }

  return pair.attributes_node;
}

void
guppi_attributes_import_xml (GuppiAttributes *ax, GuppiXMLDocument *doc, xmlNodePtr node)
{
  g_return_if_fail (ax && GUPPI_IS_ATTRIBUTES (ax));
  g_return_if_fail (doc);
  g_return_if_fail (node);

  if (strcmp (node->name, "Attributes")) {
    g_warning ("guppi_attributes_read_xml() Passed non-Attributes node (%s)", node->name);
    return;
  }

  node = node->xmlChildrenNode;
  while (node != NULL) {
    gchar *name = NULL;
    gchar *type = NULL;
    xmlNodePtr value_node = NULL;
    AttrItem *item;
    xmlNodePtr subnode;
    
    subnode = node->xmlChildrenNode;
    while (subnode) {
      //g_assert (subnode->ns == doc->ns);

      if (!strcmp (subnode->name, "name")) {

	g_assert (name == NULL);
	name = xmlNodeListGetString (doc->doc, subnode->xmlChildrenNode, 1);

      } else if (!strcmp (subnode->name, "type")) {

	g_assert (type == NULL);
	type = xmlNodeListGetString (doc->doc, subnode->xmlChildrenNode, 1);

      } else if (!strcmp (subnode->name, "value")) {

	g_assert (value_node == NULL);
	value_node = subnode;

      }
      
      subnode = subnode->next;
    }

    if (name && type && value_node) {
      item = guppi_attributes_get (ax, name);
      
      if (item) {
	if (item->type && strcmp (item->type, type)) {
	  g_warning ("type mismatch in \"%s\" (%s vs. %s)",name, item->type, type);
	} else {

	  /* The types match. */

	  if (item->pdest)
	    item->pdest (item->ptr);

	  if (item->s2p) {

	    gchar *value = xmlNodeListGetString (doc->doc, value_node->xmlChildrenNode, 1);
	    item->ptr = item->s2p (value);
	    item->undef = FALSE;
	    free (value);

	  } else if (item->c_import) {

	    item->ptr = item->c_import (doc, value_node);
	    item->undef = FALSE;
	    
	  } else {
	    g_assert_not_reached ();
	  }
	}
	    
      } else {
	g_warning ("missing/undefined item \"%s\"", name);
      }

    } else {
      g_warning ("Incomplete attribute: missing%s%s%s",
		 name ? " name" : "\0",
		 type ? " type" : "\0",
		 value_node ? " value" : "\0");
    }

    if (name) free (name);
    if (type) free (type);
    
    node = node->next;
  }
  
}
