/* This is -*- C -*- */
/* $Id: guppi-axis-state.c,v 1.20 2001/05/06 07:56:26 trow Exp $ */

/*
 * guppi-axis-state.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <math.h>
#include <guppi-useful.h>
#include <guppi-shared-hash.h>

#include "guppi-axis-state.h"
#include "guppi-axis-view.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_POSITION,
  ARG_SHOW_EDGE,
  ARG_EDGE_COLOR,
  ARG_EDGE_THICKNESS,
  ARG_SHOW_LEGEND,
  ARG_LEGEND,
  ARG_LEGEND_COLOR,
  ARG_LEGEND_FONT,
  ARG_LEGEND_OFFSET,
  ARG_LABEL_OFFSET,
  ARG_ROTATE_LABELS,
  ARG_SHRINK_LABELS_TO_FIT,
  ARG_SHOW_MAJOR_TICKS,
  ARG_MAJOR_TICK_COLOR,
  ARG_MAJOR_TICK_THICKNESS,
  ARG_MAJOR_TICK_LENGTH,
  ARG_SHOW_MAJOR_LABELS,
  ARG_MAJOR_LABEL_COLOR,
  ARG_MAJOR_LABEL_FONT,
  ARG_SHOW_MINOR_TICKS,
  ARG_MINOR_TICK_COLOR,
  ARG_MINOR_TICK_THICKNESS,
  ARG_MINOR_TICK_LENGTH,
  ARG_SHOW_MINOR_LABELS,
  ARG_MINOR_LABEL_COLOR,
  ARG_MINOR_LABEL_FONT,
  ARG_SHOW_MICRO_TICKS,
  ARG_MICRO_TICK_COLOR,
  ARG_MICRO_TICK_THICKNESS,
  ARG_MICRO_TICK_LENGTH,
  ARG_SHOW_MICRO_LABELS,
  ARG_MICRO_LABEL_COLOR,
  ARG_MICRO_LABEL_FONT
};

static void
guppi_axis_state_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiAxisState *state = GUPPI_AXIS_STATE (obj);

  switch (arg_id) {

  case ARG_POSITION:
    GTK_VALUE_INT (*arg) = (gint) guppi_axis_state_position (state);
    break;

  case ARG_SHOW_EDGE:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_edge (state);
    break;

  case ARG_EDGE_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_edge_color (state);
    break;

  case ARG_EDGE_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_edge_thickness (state);
    break;

  case ARG_SHOW_LEGEND:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_legend (state);
    break;

  case ARG_LEGEND:
    GTK_VALUE_POINTER (*arg) = (gpointer) guppi_axis_state_legend (state);
    break;

  case ARG_LEGEND_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_legend_color (state);
    break;

  case ARG_LEGEND_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_axis_state_legend_font (state);
    break;

  case ARG_LEGEND_OFFSET:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_legend_offset (state);
    break;

  case ARG_LABEL_OFFSET:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_label_offset (state);
    break;

  case ARG_ROTATE_LABELS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_rotate_labels (state);
    break;

  case ARG_SHRINK_LABELS_TO_FIT:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_shrink_labels_to_fit (state);
    break;

  case ARG_SHOW_MAJOR_TICKS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_major_ticks (state);
    break;

  case ARG_MAJOR_TICK_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_major_tick_color (state);
    break;

  case ARG_MAJOR_TICK_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_major_tick_thickness (state);
    break;

  case ARG_MAJOR_TICK_LENGTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_major_tick_length (state);
    break;

  case ARG_SHOW_MAJOR_LABELS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_major_labels (state);
    break;

  case ARG_MAJOR_LABEL_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_major_label_color (state);
    break;

  case ARG_MAJOR_LABEL_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_axis_state_major_label_font (state);
    break;

  case ARG_SHOW_MINOR_TICKS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_minor_ticks (state);
    break;

  case ARG_MINOR_TICK_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_minor_tick_color (state);
    break;

  case ARG_MINOR_TICK_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_minor_tick_thickness (state);
    break;

  case ARG_MINOR_TICK_LENGTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_minor_tick_length (state);
    break;

  case ARG_SHOW_MINOR_LABELS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_minor_labels (state);
    break;

  case ARG_MINOR_LABEL_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_minor_label_color (state);
    break;

  case ARG_MINOR_LABEL_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_axis_state_minor_label_font (state);
    break;

  case ARG_SHOW_MICRO_TICKS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_micro_ticks (state);
    break;

  case ARG_MICRO_TICK_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_micro_tick_color (state);
    break;

  case ARG_MICRO_TICK_THICKNESS:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_micro_tick_thickness (state);
    break;

  case ARG_MICRO_TICK_LENGTH:
    GTK_VALUE_DOUBLE (*arg) = guppi_axis_state_micro_tick_length (state);
    break;

  case ARG_SHOW_MICRO_LABELS:
    GTK_VALUE_BOOL (*arg) = guppi_axis_state_show_micro_labels (state);
    break;

  case ARG_MICRO_LABEL_COLOR:
    GTK_VALUE_UINT (*arg) = guppi_axis_state_micro_label_color (state);
    break;

  case ARG_MICRO_LABEL_FONT:
    GTK_VALUE_POINTER (*arg) = guppi_axis_state_micro_label_font (state);
    break;

  default:
    break;
  };
}

static void
guppi_axis_state_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  GuppiAxisState *state = GUPPI_AXIS_STATE (obj);

  switch (arg_id) {

  case ARG_POSITION:
    guppi_axis_state_set_position (state,
				   (guppi_compass_t) GTK_VALUE_INT (*arg));
    break;

  case ARG_SHOW_EDGE:
    guppi_axis_state_set_show_edge (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_EDGE_COLOR:
    guppi_axis_state_set_edge_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_EDGE_THICKNESS:
    guppi_axis_state_set_edge_thickness (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_SHOW_LEGEND:
    guppi_axis_state_set_show_legend (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_LEGEND:
    guppi_axis_state_set_legend (state, (gchar *) GTK_VALUE_POINTER (*arg));
    break;

  case ARG_LEGEND_COLOR:
    guppi_axis_state_set_legend_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_LEGEND_FONT:
    guppi_axis_state_set_legend_font (state,
				      (GnomeFont *) GTK_VALUE_POINTER (*arg));
    break;

  case ARG_LEGEND_OFFSET:
    guppi_axis_state_set_legend_offset (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_LABEL_OFFSET:
    guppi_axis_state_set_label_offset (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_ROTATE_LABELS:
    guppi_axis_state_set_rotate_labels (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_SHRINK_LABELS_TO_FIT:
    guppi_axis_state_set_shrink_labels_to_fit (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_SHOW_MAJOR_TICKS:
    guppi_axis_state_set_show_major_ticks (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_MAJOR_TICK_COLOR:
    guppi_axis_state_set_major_tick_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_MAJOR_TICK_THICKNESS:
    guppi_axis_state_set_major_tick_thickness (state,
					       GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_MAJOR_TICK_LENGTH:
    guppi_axis_state_set_major_tick_length (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_SHOW_MAJOR_LABELS:
    guppi_axis_state_set_show_major_labels (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_MAJOR_LABEL_COLOR:
    guppi_axis_state_set_major_label_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_MAJOR_LABEL_FONT:
    guppi_axis_state_set_major_label_font (state, GTK_VALUE_POINTER (*arg));
    break;

  case ARG_SHOW_MINOR_TICKS:
    guppi_axis_state_set_show_minor_ticks (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_MINOR_TICK_COLOR:
    guppi_axis_state_set_minor_tick_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_MINOR_TICK_THICKNESS:
    guppi_axis_state_set_minor_tick_thickness (state,
					       GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_MINOR_TICK_LENGTH:
    guppi_axis_state_set_minor_tick_length (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_SHOW_MINOR_LABELS:
    guppi_axis_state_set_show_minor_labels (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_MINOR_LABEL_COLOR:
    guppi_axis_state_set_minor_label_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_MINOR_LABEL_FONT:
    guppi_axis_state_set_minor_label_font (state, GTK_VALUE_POINTER (*arg));
    break;

  case ARG_SHOW_MICRO_TICKS:
    guppi_axis_state_set_show_micro_ticks (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_MICRO_TICK_COLOR:
    guppi_axis_state_set_micro_tick_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_MICRO_TICK_THICKNESS:
    guppi_axis_state_set_micro_tick_thickness (state,
					       GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_MICRO_TICK_LENGTH:
    guppi_axis_state_set_micro_tick_length (state, GTK_VALUE_DOUBLE (*arg));
    break;

  case ARG_SHOW_MICRO_LABELS:
    guppi_axis_state_set_show_micro_labels (state, GTK_VALUE_BOOL (*arg));
    break;

  case ARG_MICRO_LABEL_COLOR:
    guppi_axis_state_set_micro_label_color (state, GTK_VALUE_UINT (*arg));
    break;

  case ARG_MICRO_LABEL_FONT:
    guppi_axis_state_set_micro_label_font (state, GTK_VALUE_POINTER (*arg));
    break;

  default:
    break;
  };
}

static void
guppi_axis_state_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_axis_state_finalize (GtkObject * obj)
{
  GuppiAxisState *state = GUPPI_AXIS_STATE (obj);

  guppi_unref0 (state->legend_font);
  guppi_unref0 (state->lone_label_font);
  guppi_unref0 (state->major_label_font);
  guppi_unref0 (state->minor_label_font);
  guppi_unref0 (state->micro_label_font);

  guppi_free (state->legend);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/***************************************************************************/

static GuppiElementView *
make_view (GuppiElementState * state)
{
  return GUPPI_ELEMENT_VIEW (guppi_type_new (GUPPI_TYPE_AXIS_VIEW));
}

/***************************************************************************/

/** Config Widgets **/

static GtkWidget *
config_cb (gpointer user_data)
{
  return gtk_label_new ("Axis State");
}

static GuppiConfigItem *
config_tree (GuppiElementState * state)
{
  GuppiConfigItem *node;

  node = guppi_config_item_new (_("Axis: Style"),
				_("Style"), config_cb, state);

  return node;
}

/***************************************************************************/

#define add_arg(str, t, symb) \
gtk_object_add_arg_type("GuppiAxisState::" str, t, GTK_ARG_READWRITE, symb)

static void
guppi_axis_state_class_init (GuppiAxisStateClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiElementStateClass *state_class = GUPPI_ELEMENT_STATE_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_STATE);

  object_class->get_arg = guppi_axis_state_get_arg;
  object_class->set_arg = guppi_axis_state_set_arg;
  object_class->destroy = guppi_axis_state_destroy;
  object_class->finalize = guppi_axis_state_finalize;

  state_class->name = _("Axis");

  state_class->make_view = make_view;
  state_class->config_tree = config_tree;

  add_arg ("position", GTK_TYPE_INT, ARG_POSITION);

  add_arg ("show_edge", GTK_TYPE_BOOL, ARG_SHOW_EDGE);
  add_arg ("edge_color_rgba", GTK_TYPE_UINT, ARG_EDGE_COLOR);
  add_arg ("edge_thickness", GTK_TYPE_DOUBLE, ARG_EDGE_THICKNESS);

  add_arg ("show_legend", GTK_TYPE_BOOL, ARG_SHOW_LEGEND);
  add_arg ("legend", GTK_TYPE_POINTER, ARG_LEGEND);
  add_arg ("legend_color", GTK_TYPE_UINT, ARG_LEGEND_COLOR);
  add_arg ("legend_font", GTK_TYPE_POINTER, ARG_LEGEND_FONT);
  add_arg ("legend_offset", GTK_TYPE_DOUBLE, ARG_LEGEND_OFFSET);
  add_arg ("label_offset", GTK_TYPE_DOUBLE, ARG_LABEL_OFFSET);
  add_arg ("rotate_labels", GTK_TYPE_BOOL, ARG_ROTATE_LABELS);

  add_arg ("shrink_labels_to_fit", GTK_TYPE_BOOL, ARG_EDGE_THICKNESS);

  add_arg ("show_major_ticks", GTK_TYPE_BOOL, ARG_SHOW_MAJOR_TICKS);
  add_arg ("major_tick_color", GTK_TYPE_UINT, ARG_MAJOR_TICK_COLOR);
  add_arg ("major_tick_thickness", GTK_TYPE_DOUBLE, ARG_MAJOR_TICK_THICKNESS);
  add_arg ("major_tick_length", GTK_TYPE_DOUBLE, ARG_MAJOR_TICK_LENGTH);
  add_arg ("show_major_labels", GTK_TYPE_BOOL, ARG_SHOW_MAJOR_LABELS);
  add_arg ("major_label_color", GTK_TYPE_UINT, ARG_MAJOR_LABEL_COLOR);
  add_arg ("major_label_font", GTK_TYPE_POINTER, ARG_MAJOR_LABEL_FONT);

  add_arg ("show_minor_ticks", GTK_TYPE_BOOL, ARG_SHOW_MINOR_TICKS);
  add_arg ("minor_tick_color", GTK_TYPE_UINT, ARG_MINOR_TICK_COLOR);
  add_arg ("minor_tick_thickness", GTK_TYPE_DOUBLE, ARG_MINOR_TICK_THICKNESS);
  add_arg ("minor_tick_length", GTK_TYPE_DOUBLE, ARG_MINOR_TICK_LENGTH);
  add_arg ("show_minor_labels", GTK_TYPE_BOOL, ARG_SHOW_MINOR_LABELS);
  add_arg ("minor_label_color", GTK_TYPE_UINT, ARG_MINOR_LABEL_COLOR);
  add_arg ("minor_label_font", GTK_TYPE_POINTER, ARG_MINOR_LABEL_FONT);

  add_arg ("show_micro_ticks", GTK_TYPE_BOOL, ARG_SHOW_MICRO_TICKS);
  add_arg ("micro_tick_color", GTK_TYPE_UINT, ARG_MICRO_TICK_COLOR);
  add_arg ("micro_tick_thickness", GTK_TYPE_DOUBLE, ARG_MICRO_TICK_THICKNESS);
  add_arg ("micro_tick_length", GTK_TYPE_DOUBLE, ARG_MICRO_TICK_LENGTH);
  add_arg ("show_micro_labels", GTK_TYPE_BOOL, ARG_SHOW_MICRO_LABELS);
  add_arg ("micro_label_color", GTK_TYPE_UINT, ARG_MICRO_LABEL_COLOR);
  add_arg ("micro_label_font", GTK_TYPE_POINTER, ARG_MICRO_LABEL_FONT);

}

static void
guppi_axis_state_init (GuppiAxisState * obj)
{
  guppi_element_state_add_shared (GUPPI_ELEMENT_STATE (obj),
				  SHARED_DATA, guppi_shared_data ());


  /* Helpful (?) defaults */

  obj->position = GUPPI_NORTH;

  obj->show_edge = TRUE;
  obj->edge_color = RGBA_BLACK;
  obj->edge_thickness = guppi_in2pt (1.0 / 64);

  obj->show_legend = TRUE;
  obj->legend_color = RGBA_BLACK;
  obj->legend_offset = guppi_in2pt (1.0 / 16);
  obj->legend_font = guppi_default_font ();
  guppi_ref (obj->legend_font);

  obj->label_offset = guppi_in2pt (1.0 / 32);
  obj->rotate_labels = FALSE;
  obj->shrink_labels_to_fit = TRUE;

  obj->show_lone_labels = TRUE;
  obj->extra_lone_label_offset = guppi_in2pt (1.0 / 32);
  obj->lone_label_color = RGBA_BLACK;
  obj->lone_label_font = guppi_default_font ();
  guppi_ref (obj->lone_label_font);

  obj->show_major_ticks = TRUE;
  obj->major_tick_color = RGBA_BLACK;
  obj->major_tick_thickness = guppi_in2pt (1.0 / 64);
  obj->major_tick_length = guppi_in2pt (1.0 / 12);
  obj->show_major_labels = TRUE;
  obj->major_label_color = RGBA_BLACK;
  obj->major_label_font = guppi_default_font ();
  guppi_ref (obj->major_label_font);

  obj->show_minor_ticks = TRUE;
  obj->minor_tick_color = RGBA_BLACK;
  obj->minor_tick_thickness = guppi_in2pt (1.0 / 96);
  obj->minor_tick_length = guppi_in2pt (1.0 / 16);
  obj->show_minor_labels = FALSE;
  obj->minor_label_color = RGBA_BLACK;
  obj->minor_label_font = NULL;

  obj->show_micro_ticks = TRUE;
  obj->micro_tick_color = RGBA_BLACK;
  obj->micro_tick_thickness = guppi_in2pt (1.0 / 128);
  obj->micro_tick_length = guppi_in2pt (1.0 / 24);
  obj->show_micro_labels = FALSE;
  obj->micro_label_color = RGBA_BLACK;
  obj->micro_label_font = NULL;

}

GtkType guppi_axis_state_get_type (void)
{
  static GtkType guppi_axis_state_type = 0;
  if (!guppi_axis_state_type) {
    static const GtkTypeInfo guppi_axis_state_info = {
      "GuppiAxisState",
      sizeof (GuppiAxisState),
      sizeof (GuppiAxisStateClass),
      (GtkClassInitFunc) guppi_axis_state_class_init,
      (GtkObjectInitFunc) guppi_axis_state_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_axis_state_type = gtk_type_unique (GUPPI_TYPE_ELEMENT_STATE,
					     &guppi_axis_state_info);
  }
  return guppi_axis_state_type;
}

GuppiElementState *
guppi_axis_state_new (void)
{
  return GUPPI_ELEMENT_STATE (guppi_type_new (guppi_axis_state_get_type ()));
}

GuppiData *
guppi_axis_state_data (GuppiAxisState * state)
{
  GtkObject *obj;
  g_return_val_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state), NULL);

  obj = guppi_element_state_get_shared (GUPPI_ELEMENT_STATE (state),
					SHARED_DATA);
  return GUPPI_DATA0 (obj);
}

void
guppi_axis_state_set_data (GuppiAxisState * state, GuppiData * data)
{
  g_return_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (data == NULL || GUPPI_IS_DATA (data));

  guppi_element_state_set_shared (GUPPI_ELEMENT_STATE (state),
				  SHARED_DATA, data);
}

void
guppi_axis_state_set_position (GuppiAxisState * state, guppi_compass_t c)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->position != c) {
    state->position = c;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_edge (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_edge != x) {
    state->show_edge = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_edge_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->edge_color != x) {
    state->edge_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_edge_thickness (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->edge_thickness != x) {
    state->edge_thickness = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_legend (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state));

  if (state->show_legend != x) {
    state->show_legend = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_legend (GuppiAxisState * state, const gchar * x)
{
  g_return_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state));

  if ((state->legend == NULL && x != NULL) ||
      (state->legend != NULL && x == NULL) ||
      (state->legend && x && strcmp (state->legend, x))) {
    guppi_free (state->legend);
    state->legend = guppi_strdup (x);
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_legend_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state));

  if (state->legend_color != x) {
    state->legend_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_legend_font (GuppiAxisState * state, GnomeFont * f)
{
  g_return_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (f == NULL || GNOME_IS_FONT (f));

  if (state->legend_font != f) {
    guppi_refcounting_assign (state->legend_font, f);
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_legend_offset (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->legend_offset != x) {
    state->legend_offset = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_label_offset (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->label_offset != x) {
    state->label_offset = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_rotate_labels (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->rotate_labels != x) {
    state->rotate_labels = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_shrink_labels_to_fit (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->shrink_labels_to_fit != x) {
    state->shrink_labels_to_fit = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_lone_labels (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_lone_labels != x) {
    state->show_lone_labels = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_lone_label_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->lone_label_color != x) {
    state->lone_label_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_lone_label_font (GuppiAxisState * state, GnomeFont * x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->lone_label_font != x) {

    guppi_refcounting_assign (state->lone_label_font, x);

    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_major_ticks (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_major_ticks != x) {
    state->show_major_ticks = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_major_tick_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->major_tick_color != x) {
    state->major_tick_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_major_tick_thickness (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->major_tick_thickness != x) {
    state->major_tick_thickness = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_major_tick_length (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->major_tick_length != x) {
    state->major_tick_length = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_major_labels (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_major_labels != x) {
    state->show_major_labels = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_major_label_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->major_label_color != x) {
    state->major_label_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_major_label_font (GuppiAxisState * state, GnomeFont * x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->major_label_font != x) {

    guppi_refcounting_assign (state->major_label_font, x);

    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_minor_ticks (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_minor_ticks != x) {
    state->show_minor_ticks = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_minor_tick_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->minor_tick_color != x) {
    state->minor_tick_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_minor_tick_thickness (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->minor_tick_thickness != x) {
    state->minor_tick_thickness = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_minor_tick_length (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->minor_tick_length != x) {
    state->minor_tick_length = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_minor_labels (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_minor_labels != x) {
    state->show_minor_labels = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_minor_label_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->minor_label_color != x) {
    state->minor_label_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_minor_label_font (GuppiAxisState * state, GnomeFont * x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->minor_label_font != x) {

    guppi_refcounting_assign (state->minor_label_font, x);

    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_micro_ticks (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_micro_ticks != x) {
    state->show_micro_ticks = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_micro_tick_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->micro_tick_color != x) {
    state->micro_tick_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_micro_tick_thickness (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->micro_tick_thickness != x) {
    state->micro_tick_thickness = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_micro_tick_length (GuppiAxisState * state, double x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));
  g_return_if_fail (x >= 0);

  if (state->micro_tick_length != x) {
    state->micro_tick_length = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_show_micro_labels (GuppiAxisState * state, gboolean x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->show_micro_labels != x) {
    state->show_micro_labels = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_micro_label_color (GuppiAxisState * state, guint32 x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->micro_label_color != x) {
    state->micro_label_color = x;
    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

void
guppi_axis_state_set_micro_label_font (GuppiAxisState * state, GnomeFont * x)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (state->micro_label_font != x) {

    guppi_refcounting_assign (state->micro_label_font, x);

    guppi_element_state_changed_delayed (GUPPI_ELEMENT_STATE (state));
  }
}

/***************************************************************************/

void
guppi_axis_state_tick_properties (const GuppiAxisState * state,
				  const GuppiTick * tick,
				  gboolean * show_tick,
				  guint32 * color,
				  double *thickness,
				  double *length,
				  gboolean * show_label,
				  double *label_offset,
				  guint32 * label_color,
				  GnomeFont ** label_font)
{
  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  if (show_tick)
    *show_tick = FALSE;

  if (show_label)
    *show_label = FALSE;

  g_return_if_fail (state != NULL);
  g_return_if_fail (GUPPI_IS_AXIS_STATE (state));

  g_return_if_fail (tick != NULL);

  if (label_offset)
    *label_offset = guppi_axis_state_label_offset (state);

  switch (guppi_tick_type (tick)) {

  case GUPPI_TICK_NONE:
    if (show_tick)
      *show_tick = FALSE;
    if (color)
      *color = 0;
    if (thickness)
      *thickness = 0;
    if (length)
      *length = 0;
    if (show_label)
      *show_label = guppi_axis_state_show_lone_labels (state);
    if (label_offset)
      *label_offset += guppi_axis_state_extra_lone_label_offset (state);
    if (label_color)
      *label_color = guppi_axis_state_lone_label_color (state);
    if (label_font)
      *label_font = guppi_axis_state_lone_label_font (state);
    break;

  case GUPPI_TICK_MAJOR:

    if (show_tick)
      *show_tick = guppi_axis_state_show_major_ticks (state);
    if (color)
      *color = guppi_axis_state_major_tick_color (state);
    if (thickness)
      *thickness = guppi_axis_state_major_tick_thickness (state);
    if (length)
      *length = guppi_axis_state_major_tick_length (state);
    if (show_label)
      *show_label = guppi_axis_state_show_major_labels (state);
    if (label_color)
      *label_color = guppi_axis_state_major_label_color (state);
    if (label_font)
      *label_font = guppi_axis_state_major_label_font (state);
    break;

  case GUPPI_TICK_MINOR:

    if (show_tick)
      *show_tick = guppi_axis_state_show_minor_ticks (state);
    if (color)
      *color = guppi_axis_state_minor_tick_color (state);
    if (thickness)
      *thickness = guppi_axis_state_minor_tick_thickness (state);
    if (length)
      *length = guppi_axis_state_minor_tick_length (state);
    if (show_label)
      *show_label = guppi_axis_state_show_minor_labels (state);
    if (label_color)
      *label_color = guppi_axis_state_minor_label_color (state);
    if (label_font)
      *label_font = guppi_axis_state_minor_label_font (state);
    break;

  case GUPPI_TICK_MICRO:

    if (show_tick)
      *show_tick = guppi_axis_state_show_micro_ticks (state);
    if (color)
      *color = guppi_axis_state_micro_tick_color (state);
    if (thickness)
      *thickness = guppi_axis_state_micro_tick_thickness (state);
    if (length)
      *length = guppi_axis_state_micro_tick_length (state);
    if (show_label)
      *show_label = guppi_axis_state_show_micro_labels (state);
    if (label_color)
      *label_color = guppi_axis_state_micro_label_color (state);
    if (label_font)
      *label_font = guppi_axis_state_micro_label_font (state);
    break;

  default:

  }
}

const gchar *
guppi_axis_state_displayed_legend (GuppiAxisState * state)
{
  g_return_val_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state), NULL);

  if (guppi_axis_state_show_legend (state)) {
    GuppiData *data;

    if (guppi_axis_state_legend (state))
      return guppi_axis_state_legend (state);

    data = guppi_axis_state_data (state);
    if (data && guppi_data_label (data))
      return guppi_data_label (data);

  }

  return NULL;
}

double
guppi_axis_state_legend_span (GuppiAxisState * state)
{
  const gchar *txt;
  GnomeFont *font;

  g_return_val_if_fail (state != NULL && GUPPI_IS_AXIS_STATE (state), 0);

  txt = guppi_axis_state_displayed_legend (state);
  font = guppi_axis_state_legend_font (state);

  if (txt && font) {

    return gnome_font_get_ascender (font) + gnome_font_get_descender (font) +
      guppi_axis_state_legend_offset (state);

  }

  return 0;
}

double
guppi_axis_state_maximum_span (GuppiAxisState * state,
			       GuppiAxisMarkers * marks)
{
  gint i;
  double offset, max_span = 0;
  gboolean care_about_width;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_AXIS_STATE (state), 0);

  g_return_val_if_fail (marks != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_AXIS_MARKERS (marks), 0);

  if (guppi_axis_state_show_edge (state)) {
    /* Make our span slightly larger than the edge thickness, which
       tends to be a very small number. */
    max_span = guppi_axis_state_edge_thickness (state) * 1.20;
  }

  care_about_width = guppi_axis_state_vertical (state);
  if (guppi_axis_state_rotate_labels (state))
    care_about_width = !care_about_width;

  offset = guppi_axis_state_label_offset (state);

  for (i = 0; i < guppi_axis_markers_size (marks); ++i) {
    const GuppiTick *tick = guppi_axis_markers_get (marks, i);
    gboolean show_tick, show_label;
    double length;
    GnomeFont *font;
    double span = 0;

    guppi_axis_state_tick_properties (state, tick,
				      &show_tick, NULL, NULL, &length,
				      &show_label, &offset, NULL, &font);

    if (show_tick) {
      span += length;
    }

    if (show_label) {
      span += offset;
      if (care_about_width) {
	span += gnome_font_get_width_string (font, guppi_tick_label (tick));
      } else {
	span +=
	  gnome_font_get_ascender (font) + gnome_font_get_descender (font);
      }
    }

    max_span = MAX (span, max_span);
  }

  max_span += guppi_axis_state_legend_span (state);

  /* This should become an optional parameter: */
  /* Round up to nearest eighth of an inch (72/8=9) */
  /* max_span = 9*ceil(max_span/9); */

  return max_span;
}

double
guppi_axis_state_label_shrink_to_fit_factor (GuppiAxisState * state,
					     GuppiAxisMarkers * marks,
					     double span)
{
  gint i;
  double offset, min_factor = 1;

  g_return_val_if_fail (state != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_AXIS_STATE (state), 0);

  g_return_val_if_fail (marks != NULL, 0);
  g_return_val_if_fail (GUPPI_IS_AXIS_MARKERS (marks), 0);

  /* This is not exactly the right thing to do... */
  if (!guppi_axis_state_rotate_labels (state))
    return 1.0;

  for (i = 0; i < guppi_axis_markers_size (marks); ++i) {
    const GuppiTick *tick = guppi_axis_markers_get (marks, i);
    gboolean show_tick, show_label;
    double length;
    GnomeFont *font;

    guppi_axis_state_tick_properties (state, tick,
				      &show_tick, NULL, NULL, &length,
				      &show_label, &offset, NULL, &font);

    if (show_label && guppi_tick_label (tick) != NULL && font != NULL) {

      if (!show_tick)
	length = 0;

      if (show_label) {
	double w =
	  gnome_font_get_width_string (font, guppi_tick_label (tick));
	double t = (span - length - offset) / w;
	min_factor = MIN (t, min_factor);
      }
    }
  }

  return min_factor;
}

/* $Id: guppi-axis-state.c,v 1.20 2001/05/06 07:56:26 trow Exp $ */
