/* This is -*- C -*- */
/* vim: set sw=2: */

/*
 * guppi-pricebars-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <guppi-convenient.h>
#include "guppi-pricebars-state.h"
#include "guppi-pricebars-view.h"
#include "guppi-pricebars-item.h"
#include "guppi-pricebars-print.h"

static GtkObjectClass * parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_pricebars_view_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_pricebars_view_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_pricebars_view_finalize (GtkObject *obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
fix_y_bounds (GuppiPricebarsView *pb_view)
{
  GuppiPriceSeries *ser;
  GuppiViewInterval *vix;
  GuppiViewInterval *viy;
  double a, b, m, M, slack;
  GDate sd, ed;

  g_return_if_fail (pb_view && GUPPI_IS_PRICEBARS_VIEW (pb_view));

    ser = guppi_pricebars_state_data (GUPPI_PRICEBARS_STATE (guppi_element_view_state (GUPPI_ELEMENT_VIEW (pb_view))));

  if (pb_view->enforce_preferred_y_view
      && ser
      && guppi_date_indexed_nonempty (GUPPI_DATE_INDEXED (ser))) {

    g_assert_not_reached ();

    vix = guppi_element_view_x_view_interval (GUPPI_ELEMENT_VIEW (pb_view));
    guppi_view_interval_range (vix, &a, &b);
    a = floor (a);
    b = ceil (b);

    g_date_set_julian (&sd, (gint)a);
    g_date_set_julian (&ed, (gint)b);
    
    if (g_date_invalid (&sd) || g_date_invalid (&ed))
      return;
    
    guppi_price_series_get_bounds (ser, PRICE_LOW, &sd, &ed, &m, NULL);
    guppi_price_series_get_bounds (ser, PRICE_HIGH, &sd, &ed, NULL, &M);

    viy = guppi_element_view_y_view_interval (GUPPI_ELEMENT_VIEW (pb_view));
    slack = (M-m) * pb_view->y_view_slack;
    guppi_view_interval_set (viy, m - slack, M + slack);
  }

  pb_view->changed_x = FALSE;
}

static void
changed (GuppiElementView *view)
{
  GuppiPricebarsView *pb_view = GUPPI_PRICEBARS_VIEW (view);

  if (pb_view->changed_x)
    fix_y_bounds (pb_view);
}

static void
state_changed (GuppiElementView *view)
{

}

static gboolean
preferred_x_range (GuppiElementView *view, double *a, double *b)
{
  GuppiPricebarsState *state;
  GuppiPriceSeries *ser;

  state = GUPPI_PRICEBARS_STATE (guppi_element_view_state (view));
  ser = guppi_pricebars_state_data (state);

  if (ser == NULL || guppi_date_indexed_empty (GUPPI_DATE_INDEXED (ser)))
    return FALSE;

  if (a) {
    *a = g_date_julian ((GDate *) guppi_date_indexed_start (GUPPI_DATE_INDEXED (ser)));
  }

  if (b) {
    *b = g_date_julian ((GDate *) guppi_date_indexed_end (GUPPI_DATE_INDEXED (ser)));
  }

  return TRUE;
}

static gboolean
preferred_y_range (GuppiElementView *view, double *a, double *b)
{
  GuppiPricebarsState *state;
  GuppiPriceSeries *ser;
  double m, M, slack;
  GDate sd, ed;
  double x0, x1;

  state = GUPPI_PRICEBARS_STATE (guppi_element_view_state (view));
  ser = guppi_pricebars_state_data (state);

  if (ser == NULL || guppi_date_indexed_empty (GUPPI_DATE_INDEXED (ser)))
    return FALSE;

  guppi_element_view_get_bbox_vp (view, &x0, NULL, &x1, NULL);
  x0 = floor (x0);
  x1 = ceil (x1);

  if (!g_date_valid_julian ((gint)x0) || !g_date_valid_julian ((gint)x1))
    return FALSE;

  g_date_set_julian (&sd, (gint)x0);
  g_date_set_julian (&ed, (gint)x1);

  if (guppi_price_series_get_bounds (ser, PRICE_LOW, &sd, &ed, &m, NULL) 
      && guppi_price_series_get_bounds (ser, PRICE_HIGH, &sd, &ed, NULL, &M)) {
    slack = (M - m) * GUPPI_PRICEBARS_VIEW (view)->y_view_slack;
    if (a) *a = m - slack;
    if (b) *b = M + slack;

    return TRUE;
  }

  return FALSE;
}

static void
guppi_pricebars_view_class_init (GuppiPricebarsViewClass *klass)
{
  GtkObjectClass* object_class = (GtkObjectClass *)klass;
  GuppiElementViewClass *view_class = GUPPI_ELEMENT_VIEW_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_ELEMENT_VIEW);

  object_class->get_arg = guppi_pricebars_view_get_arg;
  object_class->set_arg = guppi_pricebars_view_set_arg;
  object_class->finalize = guppi_pricebars_view_finalize;

  view_class->changed = changed;
  view_class->canvas_item_type = GUPPI_TYPE_PRICEBARS_ITEM;
  view_class->print_type = GUPPI_TYPE_PRICEBARS_PRINT;
  view_class->state_changed = state_changed;
  
  view_class->preferred_x_range = preferred_x_range;
  view_class->preferred_y_range = preferred_y_range;

}

static void
x_change_cb (GuppiViewInterval *vix, GuppiPricebarsView *pb_view)
{
  pb_view->changed_x = TRUE;
}

static void
y_change_cb (GuppiViewInterval *viy, GuppiPricebarsView *pb_view)
{
  double a, b;
  guppi_view_interval_range (viy, &a, &b);
}

static void
guppi_pricebars_view_init (GuppiPricebarsView *obj)
{
  GuppiViewInterval *vix;
  GuppiViewInterval *viy;

  obj->y_view_slack = 0.05;
  obj->enforce_preferred_y_view = FALSE;

  vix = guppi_element_view_x_view_interval (GUPPI_ELEMENT_VIEW (obj));
  viy = guppi_element_view_y_view_interval (GUPPI_ELEMENT_VIEW (obj));
  g_assert (vix);
  gtk_signal_connect (GTK_OBJECT (vix),
		      "changed",
		      GTK_SIGNAL_FUNC (x_change_cb),
		      obj);
  gtk_signal_connect (GTK_OBJECT (viy),
		      "changed",
		      GTK_SIGNAL_FUNC (y_change_cb),
		      obj);
}

GtkType
guppi_pricebars_view_get_type (void)
{
  static GtkType guppi_pricebars_view_type = 0;
  if (!guppi_pricebars_view_type) {
    static const GtkTypeInfo guppi_pricebars_view_info = {
      "GuppiPricebarsView",
      sizeof (GuppiPricebarsView),
      sizeof (GuppiPricebarsViewClass),
      (GtkClassInitFunc)guppi_pricebars_view_class_init,
      (GtkObjectInitFunc)guppi_pricebars_view_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_pricebars_view_type = gtk_type_unique (GUPPI_TYPE_ELEMENT_VIEW, &guppi_pricebars_view_info);
  }
  return guppi_pricebars_view_type;
}

gboolean
guppi_pricebars_view_date_range (GuppiPricebarsView *view,
				 GDate *start, GDate *end)
{
  GuppiPricebarsState *state;
  GuppiDateIndexed *ser;
  double x0, x1;
  guint32 min_jul_d, max_jul_d;

  g_return_val_if_fail (view && GUPPI_IS_PRICEBARS_VIEW (view), FALSE);

  state = GUPPI_PRICEBARS_STATE (guppi_element_view_state (GUPPI_ELEMENT_VIEW (view)));
  ser = GUPPI_DATE_INDEXED (guppi_pricebars_state_data (state));

  if (ser == NULL || guppi_date_indexed_empty (ser))
    return FALSE;

  guppi_element_view_get_bbox_vp (GUPPI_ELEMENT_VIEW (view),
				  &x0, NULL, &x1, NULL);

  min_jul_d = (guint32)floor (x0);
  max_jul_d = (guint32)ceil (x1);

  if (!g_date_valid_julian (min_jul_d) || !g_date_valid_julian (max_jul_d))
    return FALSE;

  if (start) {
    g_date_set_julian (start, min_jul_d);
    guppi_date_indexed_clamp (ser, start);
  }

  if (end) {
    g_date_set_julian (end, max_jul_d);
    guppi_date_indexed_clamp (ser, end);
  }
  
  return TRUE;
}
