/* This is -*- C -*- */
/* $Id: guppi-data.c,v 1.36 2001/05/08 16:29:04 trow Exp $ */

/*
 * guppi-data.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 * Copyright (C) 2001 The Free Software Foundation
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <string.h>

#include <zlib.h>
/* #include <gnome.h> */
#include <gtk/gtksignal.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhseparator.h>
#include <gtk/gtkvseparator.h>
#include <gtk/gtkfilesel.h>
#include <gtk/gtkmenuitem.h>

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>

#include <guppi-useful.h>
#include <guppi-dialogs.h>
#include "guppi-data-impl-plug-in.h"
#include "guppi-data.h"
#include "guppi-data-tree.h"
#include "guppi-data-transform.h"


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

enum {
  CHANGED,
  CHANGED_LABEL,
  CHANGED_SUBDATA,
  LAST_SIGNAL
};
static guint data_signals[LAST_SIGNAL] = { 0 };

static void
guppi_data_get_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_set_arg (GtkObject *obj, GtkArg *arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_finalize (GtkObject *obj)
{
  GuppiData *gd = GUPPI_DATA (obj);

  guppi_finalized (obj);

  /* We should never die with pending operations! */
  g_assert (gd->pending_ops == NULL);

  guppi_unref0 (gd->impl);

  guppi_free (gd->label);
  gd->label = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/*************************************************************************/

static void
process_pending_op (GuppiData *data)
{
  GList *op_list;
  GuppiDataOp *op;

  g_return_if_fail (data != NULL && GUPPI_IS_DATA (data));

  op_list = (GList *) data->pending_ops;
  g_return_if_fail (op_list != NULL);

  op = (GuppiDataOp *) op_list->data;
  g_return_if_fail (op != NULL);

  op->op (data, op);

  data->pending_ops = g_list_remove_link (op_list, op_list);
  g_list_free_1 (op_list);
}

static void
changed (GuppiData *data)
{
  process_pending_op (data);
}

static void
changed_label (GuppiData *data, const gchar *label)
{
  guppi_free (data->label);
  data->label = guppi_strdup (label);
}

static void
changed_subdata (GuppiData *data)
{
  process_pending_op (data);
}

/***************************************************************************/

static void
prepare_attributes (GuppiAttributes *ax)
{
  guppi_attributes_declare_string (ax, "label");
  guppi_attributes_declare_unique_id (ax, "unique_id");
}

static void
set_attributes (GuppiData *d, GuppiAttributes *ax)
{
  if (d->label)
    guppi_attributes_set_string (ax, "label", guppi_data_label (d));
  guppi_attributes_set_unique_id (ax, "unique_id", d->id);
}

static void
init_base_object (GuppiData *d, GuppiAttributes *ax)
{
  if (guppi_attributes_is_defined (ax,"label"))
    guppi_data_set_label (d, guppi_attributes_get_string (ax, "label"));

  if (guppi_attributes_is_defined (ax,  "unique_id"))
    d->id = guppi_attributes_get_unique_id (ax,"unique_id");
}


static void
guppi_data_class_init (GuppiDataClass *klass)
{
  static void build_menu (GuppiData *, GtkMenu *, gpointer);

  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  klass->type_name = "-* GuppiData *-";
  klass->changed = changed;
  klass->changed_label = changed_label;
  klass->changed_subdata = changed_subdata;

  klass->prepare_attributes = prepare_attributes;
  klass->set_attributes = set_attributes;
  klass->init_base_object = init_base_object;

  klass->build_menu = build_menu;

  data_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiDataClass, changed),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  data_signals[CHANGED_LABEL] =
    gtk_signal_new ("changed_label",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiDataClass, changed_label),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

  data_signals[CHANGED_SUBDATA] =
    gtk_signal_new ("changed_subdata",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiDataClass, changed_subdata),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, data_signals, LAST_SIGNAL);

  object_class->get_arg = guppi_data_get_arg;
  object_class->set_arg = guppi_data_set_arg;
  object_class->finalize = guppi_data_finalize;
}

static void
guppi_data_init (GuppiData *gd)
{
  gd->id = guppi_unique_id ();
  gd->read_only = FALSE;
  gd->impl = NULL;
}

GtkType guppi_data_get_type (void)
{
  static GtkType guppi_data_type = 0;
  if (!guppi_data_type) {
    static const GtkTypeInfo guppi_data_info = {
      "GuppiData",
      sizeof (GuppiData),
      sizeof (GuppiDataClass),
      (GtkClassInitFunc) guppi_data_class_init,
      (GtkObjectInitFunc) guppi_data_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_data_type = gtk_type_unique (GTK_TYPE_OBJECT, &guppi_data_info);
  }
  return guppi_data_type;
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static GuppiDataImpl *
build_impl_by_code (const gchar *code)
{
  GuppiPlugIn *pi;
  GuppiDataImplPlugIn *dimpi;
  GuppiDataImpl *impl;

  g_return_val_if_fail (code != NULL, NULL);
  pi = guppi_plug_in_lookup ("data_impl", code);

  g_return_val_if_fail (pi != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_DATA_IMPL_PLUG_IN (pi), NULL);
  dimpi = GUPPI_DATA_IMPL_PLUG_IN (pi);

  impl = guppi_data_impl_plug_in_construct_impl (dimpi);
  g_return_val_if_fail (impl != NULL, NULL);

  return impl;
}

static GuppiDataImpl *
build_impl_by_name (GtkType type, const gchar *impl_name)
{
  GuppiDataImpl *impl;
  const gchar *type_name;
  gchar *pi_name;
  GuppiDataClass *klass;

  g_return_val_if_fail (type != 0, NULL);
  g_return_val_if_fail (gtk_type_is_a (type, GUPPI_TYPE_DATA), NULL);

  klass = GUPPI_DATA_CLASS (gtk_type_class (type));

  if (klass->type_name_for_impl_lookup)
    type_name = klass->type_name_for_impl_lookup;
  else
    type_name = gtk_type_name (type);

  if (impl_name == NULL)
    impl_name = "core";

  if (strstr (impl_name, "::")) {
    pi_name = g_strdup (impl_name);
  } else { /* form "fully resolved" plug-in name */
    pi_name = guppi_strdup_printf ("%s::%s", type_name, impl_name);
  }
  impl = build_impl_by_code (pi_name);
  guppi_free (pi_name);

  g_return_val_if_fail (impl != NULL, NULL);

  if (klass->impl_type)
    g_return_val_if_fail (gtk_type_is_a (GTK_OBJECT_TYPE (impl),
					 klass->impl_type), NULL);

  return impl;
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

GuppiData *
guppi_data_new_by_type (GtkType type, GtkType impl_type, const gchar *arg1,
			...)
{
  GuppiData *data;

  GSList *arg_list;
  GSList *info_list;
  gchar *error;
  va_list var_args;

  data = guppi_data_newv_by_type (type, impl_type, 0, NULL);
  g_return_val_if_fail (data != NULL, NULL);

  va_start (var_args, arg1);
  error = gtk_object_args_collect (GTK_OBJECT_TYPE (data->impl),
				   &arg_list, &info_list, arg1, var_args);

  if (error) {
    g_warning ("guppi_data_new_by_type(): %s", error);
    guppi_free (error);
  } else {
    GSList *slist_arg;
    GSList *slist_info;

    slist_arg = arg_list;
    slist_info = info_list;
    while (slist_arg) {
      gtk_object_arg_set (GTK_OBJECT (data->impl),
			  slist_arg->data, slist_info->data);
      slist_arg = g_slist_next (slist_arg);
      slist_info = g_slist_next (slist_info);
    }
    gtk_args_collect_cleanup (arg_list, info_list);
  }

  return data;
}

GuppiData *
guppi_data_newv_by_type (GtkType type, GtkType impl_type,
			 guint nargs, GtkArg *args)
{
  GuppiData *data;
  GuppiDataClass *klass;

  g_return_val_if_fail (type != 0, NULL);
  g_return_val_if_fail (impl_type != 0, NULL);

  g_return_val_if_fail (gtk_type_is_a (type, GUPPI_TYPE_DATA), NULL);
  g_return_val_if_fail (gtk_type_is_a (impl_type, GUPPI_TYPE_DATA_IMPL),
			NULL);

  klass = GUPPI_DATA_CLASS (gtk_type_class (type));
  if (klass->impl_type)
    g_return_val_if_fail (gtk_type_is_a (impl_type, klass->impl_type), NULL);

  data = GUPPI_DATA (guppi_type_new (type));
  data->impl = GUPPI_DATA_IMPL (guppi_type_new (impl_type));

  gtk_object_setv (GTK_OBJECT (data->impl), nargs, args);

  return data;
}

GuppiData *
guppi_data_new (GtkType type, const gchar *impl_name,
		const gchar *arg1, ...)
{
  GuppiData *data;
  GuppiDataImpl *impl;

  GSList *arg_list;
  GSList *info_list;
  gchar *error;
  va_list var_args;

  g_return_val_if_fail (type != 0, NULL);
  g_return_val_if_fail (gtk_type_is_a (type, GUPPI_TYPE_DATA), NULL);

  impl = build_impl_by_name (type, impl_name);
  g_return_val_if_fail (impl != NULL, NULL);

  data = GUPPI_DATA (guppi_type_new (type));
  data->impl = impl;

  va_start (var_args, arg1);
  error = gtk_object_args_collect (GTK_OBJECT_TYPE (data->impl),
				   &arg_list, &info_list, arg1, var_args);

  if (error) {
    g_warning ("guppi_data_new(): %s", error);
    guppi_free (error);
  } else {
    GSList *slist_arg;
    GSList *slist_info;

    slist_arg = arg_list;
    slist_info = info_list;
    while (slist_arg) {
      gtk_object_arg_set (GTK_OBJECT (data->impl),
			  slist_arg->data, slist_info->data);
      slist_arg = g_slist_next (slist_arg);
      slist_info = g_slist_next (slist_info);
    }
    gtk_args_collect_cleanup (arg_list, info_list);
  }

  gtk_signal_connect_object_after (GTK_OBJECT (data->impl),
				   "changed",
				   GTK_SIGNAL_FUNC (guppi_data_touch),
				   GTK_OBJECT (data));

  gtk_signal_connect_object_after (GTK_OBJECT (data->impl),
				   "changed_subdata",
				   GTK_SIGNAL_FUNC (guppi_data_touch_subdata),
				   GTK_OBJECT (data));

  return data;
}

GuppiData *
guppi_data_newv (GtkType type, const gchar *impl_name,
		 guint nargs, GtkArg *args)
{
  GuppiData *data;
  GuppiDataImpl *impl;

  g_return_val_if_fail (type != 0, NULL);
  g_return_val_if_fail (gtk_type_is_a (type, GUPPI_TYPE_DATA), NULL);

  impl = build_impl_by_name (type, impl_name);
  g_return_val_if_fail (impl != NULL, NULL);

  data = GUPPI_DATA (guppi_type_new (type));
  data->impl = impl;

  gtk_object_setv (GTK_OBJECT (data->impl), nargs, args);

  gtk_signal_connect_object_after (GTK_OBJECT (data->impl),
				   "changed",
				   GTK_SIGNAL_FUNC (guppi_data_touch),
				   GTK_OBJECT (data));

  return data;
}

void
guppi_data_set (GuppiData *gd, const gchar *first_arg_name, ...)
{
  GSList *arg_list;
  GSList *info_list;
  gchar *error;
  va_list var_args;

  va_start (var_args, first_arg_name);
  error = gtk_object_args_collect (GTK_OBJECT_TYPE (gd->impl),
				   &arg_list, &info_list, first_arg_name, var_args);
  if (error) {
    g_warning ("guppi_data_set(): %s", error);
    guppi_free (error);
  } else {
    GSList *slist_arg = arg_list;
    GSList *slist_info = info_list;

    while (slist_arg) {
      gtk_object_arg_set (GTK_OBJECT (gd->impl), slist_arg->data, slist_info->data);
      slist_arg = g_slist_next (slist_arg);
      slist_info = g_slist_next (slist_info);
    }
    gtk_args_collect_cleanup (arg_list, info_list);
  }
}

void
guppi_data_setv (GuppiData *gd, guint nargs, GtkArg *args)
{
  g_return_if_fail (gd && GUPPI_IS_DATA (gd));
  if (nargs == 0)
    return;
  g_return_if_fail (args != NULL);

  /* Just forward the args to the implementation. */
  gtk_object_setv (GTK_OBJECT (gd->impl), nargs, args);
}


/***************************************************************************/

GuppiDataImpl *
guppi_data_impl (GuppiData *gd)
{
  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), NULL);

  /* We must always have an implementation installed by this point. */
  g_assert (gd->impl != NULL);

  return gd->impl;
}

const gchar *
guppi_data_get_type_name (const GuppiData *gd)
{
  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), NULL);
  return GUPPI_DATA_CLASS (GTK_OBJECT (gd)->klass)->type_name;
}

const gchar *
guppi_data_get_impl_name (const GuppiData *gd)
{
  GuppiDataImpl *impl;
  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), NULL);

  impl = guppi_data_impl ((GuppiData *) gd);
  g_return_val_if_fail (impl != NULL, NULL);
  return guppi_data_impl_get_impl_name (impl);
}

const gchar *
guppi_data_get_impl_code (const GuppiData *gd)
{
  GuppiDataImpl *impl;
  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), NULL);

  impl = guppi_data_impl ((GuppiData *) gd);
  g_return_val_if_fail (impl != NULL, NULL);
  
  return GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass)->plug_in_code;
}

guppi_uniq_t
guppi_data_unique_id (GuppiData *gd)
{
  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), 0);

  return gd->id;
}

const gchar *
guppi_data_get_label (const GuppiData *gd)
{
  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), NULL);

  return gd->label ? gd->label : _("Unlabeled");
}

void
guppi_data_set_label (GuppiData *gd, const gchar *str)
{
  g_return_if_fail (gd && GUPPI_IS_DATA (gd));
  g_return_if_fail (str != NULL);

  if (strcmp (str, guppi_data_get_label (gd)))
    gtk_signal_emit (GTK_OBJECT (gd), data_signals[CHANGED_LABEL], str);
}

gboolean
guppi_data_is_read_only (GuppiData *gd)
{
  GuppiDataImpl *impl;
  GuppiDataImplClass *klass;

  g_return_val_if_fail (gd && GUPPI_IS_DATA (gd), FALSE);

  impl = guppi_data_impl (gd);
  klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  return gd->read_only || klass->read_only;
}

gboolean 
guppi_data_can_change (GuppiData *gd)
{
  return !guppi_data_is_read_only (gd);
}

void
guppi_data_add_pending_op (GuppiData *data, GuppiDataOp *op)
{
  GList *op_list;

  g_return_if_fail (data != NULL && GUPPI_IS_DATA (data));
  g_return_if_fail (op != NULL);

  op_list = (GList *) data->pending_ops;
  data->pending_ops = g_list_prepend (op_list, op);
}

void
guppi_data_changed (GuppiData *gd)
{
  g_return_if_fail (gd != NULL && GUPPI_IS_DATA (gd));
  g_assert (guppi_data_can_change (gd));
  g_assert (gd->pending_ops != NULL);

  gtk_signal_emit (GTK_OBJECT (gd), data_signals[CHANGED]);
}

static void
op_nop (GuppiData *d, GuppiDataOp *op)
{

}

void
guppi_data_touch (GuppiData *gd)
{
  GuppiDataOp op;

  g_return_if_fail (gd != NULL && GUPPI_IS_DATA (gd));

  op.op = op_nop;

  guppi_data_add_pending_op (gd, &op);
  gtk_signal_emit (GTK_OBJECT (gd), data_signals[CHANGED]);
}

void
guppi_data_touch_subdata (GuppiData *gd)
{
  g_return_if_fail (gd && GUPPI_IS_DATA (gd));

  gtk_signal_emit (GTK_OBJECT (gd), data_signals[CHANGED_SUBDATA]);
}

GuppiData *
guppi_data_copy (const GuppiData *gd)
{
  GuppiDataImpl *impl;
  GuppiDataImplClass *klass;
  GuppiData *copy;

  g_return_val_if_fail (gd != NULL && GUPPI_IS_DATA (gd), NULL);

  impl = guppi_data_impl ((GuppiData *) gd);
  klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  g_assert (klass->copy);

  copy = GUPPI_DATA (guppi_type_new (GTK_OBJECT_TYPE (gd)));
  copy->impl = (klass->copy) (impl);

  copy->read_only = gd->read_only;
  copy->label = guppi_strdup_printf (_("Copy of %s"), guppi_data_get_label (gd));

  return copy;
}

gint
guppi_data_get_size_in_bytes (GuppiData *gd)
{
  GuppiDataImpl *impl;
  GuppiDataImplClass *klass;

  g_return_val_if_fail (gd != NULL && GUPPI_IS_DATA (gd), -1);

  impl = guppi_data_impl (gd);
  klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (klass->get_size_in_bytes)
    return (klass->get_size_in_bytes) (impl);
  else
    return -1;
}

void
guppi_data_get_size_info (GuppiData *gd, gchar *buf, gsize N)
{
  GuppiDataImpl *impl;
  GuppiDataImplClass *klass;

  g_return_if_fail (gd != NULL && GUPPI_IS_DATA (gd));
  g_return_if_fail (buf != NULL);

  if (N == 0)
    return;
  if (N == 1) {
    buf[0] = '\0';
    return;
  }

  impl = guppi_data_impl (gd);
  klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (klass->get_size_info)
    (klass->get_size_info) (impl, buf, N);
  else {
    buf[0] = '?';
    buf[1] = '\0';
  }
}

/**************************************************************************/

xmlNodePtr
guppi_data_export_xml (GuppiData *d, GuppiXMLDocument *doc)
{
  GuppiDataClass *data_class;
  GuppiDataImpl *impl;
  GuppiDataImplClass *impl_class;
  xmlNodePtr data_node = NULL;
  xmlNodePtr attr_node = NULL;
  xmlNodePtr content_node = NULL;
  GuppiAttributes *ax;

  g_return_val_if_fail (d && GUPPI_IS_DATA (d), NULL);
  g_return_val_if_fail (doc, NULL);

  data_class = GUPPI_DATA_CLASS (GTK_OBJECT (d)->klass);
  impl = guppi_data_impl (d);
  impl_class = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  ax = guppi_attributes_new ();

  if (data_class->prepare_attributes)
    data_class->prepare_attributes (ax);

  if (impl_class->prepare_attributes)
    impl_class->prepare_attributes (ax);

  if (data_class->set_attributes)
      data_class->set_attributes (d, ax);

  if (impl_class->set_attributes)
    impl_class->set_attributes (impl, ax);

  attr_node = guppi_attributes_export_xml (ax, doc);
  guppi_unref0 (ax);
  
  if (impl_class->export_xml_content) {

    /* The implementation method overrides the default method. */
    content_node = impl_class->export_xml_content (impl, d, doc);

  } else if (data_class->export_xml_content) {

    content_node = data_class->export_xml_content (d, doc);

  } else {

    g_warning ("export_xml_content not implemented.");

  }

  data_node = xmlNewNode (doc->ns, "Data");
  xmlNewProp (data_node, "Type", gtk_type_name (GTK_OBJECT_TYPE (d)));
  xmlNewProp (data_node, "Impl", guppi_data_get_impl_code (d));

  if (attr_node)
    xmlAddChild (data_node, attr_node);
  if (content_node)
    xmlAddChild (data_node, content_node);

  return data_node;
}

void
guppi_data_spew_xml (GuppiData *d)
{
  GuppiXMLDocument *doc;

  g_return_if_fail (d && GUPPI_IS_DATA (d));

  doc = guppi_xml_document_new ();
  guppi_xml_document_add_node (doc, guppi_data_export_xml (d, doc));
  guppi_xml_document_spew (doc);
  guppi_xml_document_free (doc);
}

/***************************************************************************/

GuppiData *
guppi_data_import_xml (GuppiXMLDocument *doc, xmlNodePtr node)
{
  gchar *type_name;
  gchar *impl_name;
  GtkType data_type;
  GuppiDataClass *data_class;
  GuppiDataImplClass *impl_class;
  GuppiAttributes *attr = NULL;
  GuppiData *data = NULL;
  GuppiDataImpl *impl = NULL;

  g_return_val_if_fail (doc != NULL, NULL);
  g_return_val_if_fail (node != NULL, NULL);

  if (strcmp (node->name, "Data")) {
    g_warning ("guppi_data_read_xml() passed non-Data node (%s)", node->name);
    return NULL;
  }

  /* Construct base object from type and impl information */

  type_name = xmlGetProp (node, "Type");
  if (type_name == NULL) {
    g_message ("Missing data Type property");
    return NULL;
  }

  data_type = gtk_type_from_name (type_name);
  if (data_type == 0) {
    g_message ("Unknown type \"%s\"", type_name);
    return NULL;
  }

  impl_name = xmlGetProp (node, "Impl");

  data = guppi_data_newv (data_type, impl_name, 0, NULL);
  impl = guppi_data_impl (data);
  data_class = GUPPI_DATA_CLASS (GTK_OBJECT (data)->klass);
  impl_class = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  
  node = node->xmlChildrenNode;
  while (node != NULL) {
    /*g_assert (node->ns == doc->ns);*/
    
    if (!strcmp (node->name, "Attributes")) {

      if (attr == NULL) {

	attr = guppi_attributes_new ();

	if (data_class->prepare_attributes)
	  data_class->prepare_attributes (attr);
	if (impl_class->prepare_attributes)
	  impl_class->prepare_attributes (attr);

	guppi_attributes_import_xml (attr, doc, node);

	if (data_class->init_base_object)
	  data_class->init_base_object (data, attr);
	if (impl_class->init_base_object)
	  impl_class->init_base_object (impl, attr);

      } else {
	g_warning ("Ignoring repeated attributes declarations.");
      }
	
    } else if (!strcmp (node->name, "Content")) {

      if (attr == NULL) {

	g_warning ("Missing attributes!");

      } else if (data == NULL) {

	g_warning ("Data object not properly constructed!");

      } else if (impl_class->import_xml_content) {

	/* Impl importer overrides tyep importer. */
	impl_class->import_xml_content (impl, data, doc, node);

      } else if (data_class->import_xml_content) {

	data_class->import_xml_content (data, doc, node);

      }

    } else {
      /* Silently ignore other stuff */
    }
    
    node = node->next;
  }
  
  guppi_unref (attr);
  free (type_name);
  free (impl_name);
  return data;
}

GuppiData *
guppi_data_read_xml_file (const gchar *filename)
{
  GuppiXMLDocument *doc;
  GuppiData *data;
  xmlNodePtr node;

  if (filename == NULL)
    return NULL;

  doc = guppi_xml_document_read_file (filename);
  if (doc == NULL)
    return NULL;

  node = guppi_xml_document_root (doc);

  if (strcmp (node->name, "Guppi")) {
    g_warning ("Bad Root Node (%s)", node->name);
    return NULL;
  }

  node = node->xmlChildrenNode;
  data = guppi_data_import_xml (doc, node);

  guppi_xml_document_free (doc);

  return data;
}

/* Right now, this is sort of a hard-wired hack... but it works. */
gboolean
guppi_data_identify_xml_file (const gchar *filename)
{
  gzFile zfd;
  gchar buf[256];
  gint len;

  g_return_val_if_fail (filename, FALSE);

  zfd = gzopen (filename, "r");
  g_return_val_if_fail (zfd, FALSE);
  
  memset (buf, 0, 256);
  len = gzread (zfd, buf, 255);
  
  gzclose (zfd);

  return strstr (buf, "<?xml version=") == buf && strstr (buf, "<gpi:");
}

void
guppi_data_write_xml_file (GuppiData *d, const gchar *filename)
{
  GuppiXMLDocument *doc;
  xmlNodePtr node;

  g_return_if_fail (d && GUPPI_IS_DATA (d));
  g_return_if_fail (filename != NULL);

  doc = guppi_xml_document_new ();
  node = guppi_data_export_xml (d, doc);
  guppi_xml_document_add_node (doc, node);
  guppi_xml_document_write_file (doc, filename);
  guppi_xml_document_free (doc);
}

/**************************************************************************/

/* Subdata handling */

gboolean
guppi_data_has_subdata (GuppiData *data)
{
  g_return_val_if_fail (data && GUPPI_IS_DATA (data), FALSE);
  
  return guppi_data_subdata_count (data) != 0;
}

static void
count_subdata_fn (GuppiData *d, gpointer user_data)
{
  ++*(gint *)user_data;
}

gint
guppi_data_subdata_count (GuppiData *data)
{
  gint count = 0;

  g_return_val_if_fail (data && GUPPI_IS_DATA (data), 0);
  
  guppi_data_foreach_subdata (data, count_subdata_fn, &count);

  return count;
}

void
guppi_data_foreach_subdata (GuppiData *data,
			    void (*fn) (GuppiData *, gpointer),
			    gpointer user_data)
{
  GuppiDataImpl *impl;
  GuppiDataImplClass *klass;

  g_return_if_fail (data && GUPPI_IS_DATA (data));
  g_return_if_fail (fn);

  impl = guppi_data_impl (data);
  klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  /* If foreach_subdata isn't defined in the implementation class,
     we assume that that type can never have subdata, and the
     guppi_data_foreach_subdata () call is just a NOP. */
  if (klass->foreach_subdata)
    klass->foreach_subdata (impl, fn, user_data);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

/*** Data Menu Items ***/

void
guppi_data_build_menu (GuppiData *data, GtkMenu *menu, gpointer data_tree)
{
  GuppiDataClass *klass;
  GuppiDataImpl *impl;
  GuppiDataImplClass *impl_klass;

  g_return_if_fail (data && GUPPI_IS_DATA (data));
  g_return_if_fail (menu && GTK_IS_MENU (menu));

  klass = GUPPI_DATA_CLASS (GTK_OBJECT (data)->klass);
  impl = guppi_data_impl (data);
  impl_klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (klass->build_menu)
    klass->build_menu (data, menu, data_tree);
  if (impl_klass->build_menu)
    impl_klass->build_menu (impl, menu, data_tree);
}

/*
 * Implement a well-behaved "rename data" dialog.
 */

struct rename_info {
  GuppiData *data;
  gchar *original_label;
  GtkWidget *entry;
};

static void
rename_info_free (struct rename_info *ri)
{
  gtk_signal_disconnect_by_data (GTK_OBJECT (ri->data), ri->entry);
  guppi_unref (ri->data);
  guppi_free (ri->original_label);
  guppi_free (ri);
}

static void
rename_clicked_cb (GtkWidget *w, gint num, gpointer user_data)
{
  struct rename_info *ri = (struct rename_info *)user_data;

  if (num == 0) { /* OK */
    /* We don't need to do anything in this case. */
  } else if (num == 1) { /* Cancel */
    guppi_data_set_label (ri->data, ri->original_label);
  } else {
    g_assert_not_reached ();
  }

  rename_info_free (ri);

  gtk_widget_destroy (w);
}

static void
rename_close_cb (GtkWidget *w, gpointer user_data)
{
  struct rename_info *ri = (struct rename_info *)user_data;

  /* Revert to our original label */
  guppi_data_set_label (ri->data, ri->original_label);

  rename_info_free (ri);
}

static void
rename_edit_cb (GtkWidget *w, gpointer user_data)
{
  struct rename_info *ri = (struct rename_info *)user_data;
  gchar *s;

  s = gtk_editable_get_chars (GTK_EDITABLE (w), 0, -1);
  guppi_outside_alloc (s);
  guppi_data_set_label (ri->data, s);
  guppi_free (s);
}

static void
rename_changed_cb (GuppiData *data, const gchar* label, gpointer user_data)
{
  GtkEntry *entry = GTK_ENTRY (user_data);
  gchar *s;

  s = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);

  if (strcmp (s, label))
    gtk_entry_set_text (entry, label);
}

static void
rename_cb (GtkWidget *w, GuppiData *data)
{
  GtkWidget *dialog;
  GtkWidget *entry;
  struct rename_info *ri;

  ri = guppi_new0 (struct rename_info, 1);
  ri->data = data;
  guppi_ref (ri->data);
  ri->original_label = guppi_strdup (guppi_data_label (data));

  dialog = gnome_dialog_new (_("Edit Data Label"),
			     GNOME_STOCK_BUTTON_OK,
			     GNOME_STOCK_BUTTON_CANCEL,
			     NULL);

  ri->entry = entry = gtk_entry_new ();
  gtk_entry_set_text (GTK_ENTRY (entry), ri->original_label);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
		     entry, TRUE, TRUE, 2);

  gtk_signal_connect (GTK_OBJECT (entry),
		      "changed",
		      GTK_SIGNAL_FUNC (rename_edit_cb),
		      ri);

  gtk_signal_connect (GTK_OBJECT (dialog),
		      "clicked",
		      GTK_SIGNAL_FUNC (rename_clicked_cb),
		      ri);

  gtk_signal_connect (GTK_OBJECT (dialog),
		      "close",
		      GTK_SIGNAL_FUNC (rename_close_cb),
		      ri);
  gtk_signal_connect (GTK_OBJECT (data),
		      "changed_label",
		      GTK_SIGNAL_FUNC (rename_changed_cb),
		      entry);

  gtk_widget_show_all (dialog);
}

static void
data_info_cb (GtkWidget *w, GuppiData *data)
{
  GtkWidget *win;
  GtkWidget *box;
  GtkWidget *info;
  GtkWidget *close_button;

  win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (win), "Data Information");

  info = guppi_data_info_display (data);
  if (info == NULL)
    info = gtk_label_new (_("No information available."));

  box = gtk_vbox_new (FALSE, 2);

  gtk_box_pack_start (GTK_BOX (box), info, TRUE, TRUE, 2);

  gtk_box_pack_start (GTK_BOX (box), gtk_hseparator_new (),
		      FALSE, TRUE, 3);

  close_button = gnome_stock_button (GNOME_STOCK_BUTTON_CLOSE);
  gtk_box_pack_start (GTK_BOX (box), close_button, FALSE, FALSE, 2);

  gtk_container_add (GTK_CONTAINER (win), box);

  gtk_signal_connect_object (GTK_OBJECT (close_button),
			     "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (win));
  
  gtk_widget_show_all (win);

}

struct save_info {
  GtkFileSelection *fs;
  GuppiData *data;
};

static void
save_info_free (struct save_info *si)
{
  gtk_widget_destroy (GTK_WIDGET (si->fs));
  guppi_unref (si->data);
  guppi_free (si);
}

static void
save_ok_cb (GtkWidget *w, gpointer user_data)
{
  struct save_info *si = (struct save_info *)user_data;
  gchar *filename;

  filename = gtk_file_selection_get_filename (si->fs);
  guppi_data_write_xml_file (si->data, filename);

  save_info_free (si);
}

static void
save_cancel_cb (GtkWidget *w, gpointer user_data)
{
  struct save_info *si = (struct save_info *)user_data;

  save_info_free (si);
}

static gboolean
save_delete_event_cb (GtkWidget *w, GdkEvent *ev, gpointer user_data)
{
  struct save_info *si = (struct save_info *)user_data;

  save_info_free (si);

  return TRUE;
}

static void
data_save_cb (GtkWidget *w, GuppiData *data)
{
  GtkWidget *fs;
  gchar *s;
  gchar *tmp;
  gchar *filename;
  struct save_info *si;

  s = guppi_strdup_printf (_("Save \"%s\""), guppi_data_label (data));
  fs = gtk_file_selection_new (s);
  guppi_free (s);

  si = guppi_new0 (struct save_info, 1);
  si->fs = GTK_FILE_SELECTION (fs);
  si->data = data;
  guppi_ref (data);

  tmp = guppi_string_canonize_filename (guppi_data_label (data));
  filename = guppi_strdup_printf ("%s.gpi", tmp);
  guppi_free (tmp);
  gtk_file_selection_set_filename (si->fs, filename);
  guppi_free (filename);

  gtk_signal_connect (GTK_OBJECT (si->fs->ok_button),
		      "clicked",
		      GTK_SIGNAL_FUNC (save_ok_cb),
		      si);

  gtk_signal_connect (GTK_OBJECT (si->fs->cancel_button),
		      "clicked",
		      GTK_SIGNAL_FUNC (save_cancel_cb),
		      si);

  gtk_signal_connect (GTK_OBJECT (fs),
		      "delete_event",
		      GTK_SIGNAL_FUNC (save_delete_event_cb),
		      si);

  gtk_widget_show_all (fs);
}

static void
data_delete_cb (GtkWidget *w, GuppiData *data)
{
  GuppiDataTree *tree;

  tree = GUPPI_DATA_TREE (gtk_object_get_data (GTK_OBJECT (w), "tree"));

  guppi_data_tree_remove (tree, data);
}

static void
data_copy_cb (GtkWidget *w, GuppiData *data)
{
  GuppiData *copy;
  GuppiDataTree *tree;

  copy = guppi_data_copy (data);
  if (copy == NULL) {
    guppi_warning_dialog ("Can't copy \"%s\"", guppi_data_label (data));
  } else {
    tree = GUPPI_DATA_TREE (gtk_object_get_data (GTK_OBJECT (w), "tree"));
    guppi_data_tree_add (tree, copy);
  }

}

static void
data_transform_cb (GtkWidget *w, GuppiData *data)
{
  GuppiDataTransform *xform;
  GuppiDataTree *tree;
  GuppiData *result;

  xform =
    GUPPI_DATA_TRANSFORM (gtk_object_get_data (GTK_OBJECT (w), "xform"));
  tree = GUPPI_DATA_TREE (gtk_object_get_data (GTK_OBJECT (w), "tree"));

  result = guppi_data_transform_process (xform, data);
  if (result)
    guppi_data_tree_add_beside (tree, data, result);
}

static void
build_menu (GuppiData *data, GtkMenu *menu, gpointer data_tree)
{
  GtkWidget *mi;
  GuppiDataClass *klass;
  GuppiDataImpl *impl;
  GuppiDataImplClass *impl_klass;
  GList *xforms;
  GList *iter;

  g_return_if_fail (data && GUPPI_IS_DATA (data));
  g_return_if_fail (menu && GTK_IS_MENU (menu));

  klass = GUPPI_DATA_CLASS (GTK_OBJECT (data)->klass);
  impl = guppi_data_impl (data);
  impl_klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  /* Need to hook up signals... */

  if (klass->info_display || impl_klass->info_display) {
    mi = gtk_menu_item_new_with_label (_("Information"));
    gtk_menu_append (menu, mi);
    gtk_signal_connect (GTK_OBJECT (mi),
			"activate",
			GTK_SIGNAL_FUNC (data_info_cb),
			data);
    gtk_widget_show (mi);
  }

  mi = gtk_menu_item_new_with_label (_("Rename"));
  gtk_menu_append (menu, mi);
  gtk_signal_connect (GTK_OBJECT (mi),
		      "activate",
		      GTK_SIGNAL_FUNC (rename_cb),
		      data);
  gtk_widget_show (mi);

  mi = gtk_menu_item_new_with_label (_("Save"));
  gtk_menu_append (menu, mi);
  gtk_signal_connect (GTK_OBJECT (mi),
		      "activate",
		      GTK_SIGNAL_FUNC (data_save_cb),
		      data);
  gtk_widget_show (mi);


  mi = gtk_menu_item_new_with_label (_("Delete"));
  gtk_object_set_data (GTK_OBJECT (mi), "tree", data_tree);
  gtk_menu_append (menu, mi);
  gtk_signal_connect (GTK_OBJECT (mi),
		      "activate",
		      GTK_SIGNAL_FUNC (data_delete_cb),
		      data);
  gtk_widget_show (mi);


  mi = gtk_menu_item_new_with_label (_("Copy"));
  gtk_object_set_data (GTK_OBJECT (mi), "tree", data_tree);
  gtk_menu_append (menu, mi);
  gtk_signal_connect (GTK_OBJECT (mi),
		      "activate",
		      GTK_SIGNAL_FUNC (data_copy_cb),
		      data);
  gtk_widget_show (mi);


  xforms = iter = guppi_data_transforms_find_supported (data);
  if (xforms) {

    GtkMenu *xmenu = GTK_MENU (gtk_menu_new ());
    
    while (iter) {
      GuppiDataTransform *xform = GUPPI_DATA_TRANSFORM (iter->data);
      mi = gtk_menu_item_new_with_label (xform->name);

      gtk_object_set_data (GTK_OBJECT (mi), "xform", xform);
      gtk_object_set_data (GTK_OBJECT (mi), "tree", data_tree);

      gtk_signal_connect (GTK_OBJECT (mi),
			  "activate",
			  GTK_SIGNAL_FUNC (data_transform_cb),
			  data);

      gtk_menu_append (xmenu, mi);
      gtk_widget_show (mi);
      iter = g_list_next (iter);
    }
    g_list_free (xforms);

    mi = gtk_menu_item_new_with_label (_("Copy & Transform"));
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mi), GTK_WIDGET (xmenu));
    gtk_widget_show (GTK_WIDGET (xmenu));

    gtk_menu_append (menu, mi);
    gtk_widget_show (mi);
  }

}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

GtkWidget *
guppi_data_info_display (GuppiData *data)
{
  GuppiDataClass *klass;
  GuppiDataImpl *impl;
  GuppiDataImplClass *impl_klass;
  GtkWidget *w = NULL;
  GtkWidget *impl_w = NULL;
  GtkWidget *frame;
  GtkWidget *box = NULL;

  g_return_val_if_fail (data && GUPPI_IS_DATA (data), NULL);

  klass = GUPPI_DATA_CLASS (GTK_OBJECT (data)->klass);
  impl = guppi_data_impl (data);
  impl_klass = GUPPI_DATA_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  if (klass->info_display)
    w = klass->info_display (data);

  if (impl_klass->info_display)
    impl_w = impl_klass->info_display (impl);

  if (w == NULL && impl_w == NULL)
    return NULL;

  frame = gtk_frame_new (guppi_data_label (data));
  box = gtk_vbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (frame), box);

  if (w)
    gtk_box_pack_start (GTK_BOX (box), w, TRUE, TRUE, 0);

  if (w && impl_w)
    gtk_box_pack_start (GTK_BOX (box), gtk_vseparator_new (), TRUE, FALSE, 2);

  if (impl_w)
    gtk_box_pack_start (GTK_BOX (box), impl_w, TRUE, TRUE, 0);

  return frame;
}

/* $Id: guppi-data.c,v 1.36 2001/05/08 16:29:04 trow Exp $ */

