/* This is -*- C -*- */
/* $Id: guppi-seq-impl.c,v 1.15 2001/05/06 08:26:35 trow Exp $ */

/*
 * guppi-seq-impl.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-convenient.h>
#include "guppi-seq-impl.h"
#include "guppi-seq-boolean.h"

static GtkObjectClass *parent_class = NULL;

static void
guppi_seq_impl_finalize (GtkObject * obj)
{
  guppi_unref (GUPPI_SEQ_IMPL (obj)->opaque_missing_values);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/* Default implementations */

static void
v_size_info (GuppiDataImpl * data_impl, gchar * buf, gsize N)
{
  GuppiSeqImpl *impl;
  GuppiSeqImplClass *klass;
  gint i0, i1;

  impl = GUPPI_SEQ_IMPL (data_impl);
  klass = GUPPI_SEQ_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (klass->get_bounds);
  (klass->get_bounds) (impl, &i0, &i1);

  if (i0 > i1)
    strncpy (buf, _("empty"), N);
  else {
    gint count = i1 - i0 + 1;
    if (klass->missing_count)
      count -= klass->missing_count (impl);
    g_snprintf (buf, N, "%d / %d:%d", count, i0, i1);
  }
}

static gboolean
v_seq_has_missing (GuppiSeqImpl * impl)
{
  return guppi_seq_impl_has_missing (impl);
}

static gsize
v_seq_missing_count (GuppiSeqImpl * impl)
{
  return guppi_seq_impl_missing_count (impl);
}

static gboolean
v_seq_missing (GuppiSeqImpl * impl, gint i)
{
  return guppi_seq_impl_missing (impl, i);
}

static void
v_seq_set_missing (GuppiSeqImpl * impl, gint i)
{
  guppi_seq_impl_set_missing (impl, i, TRUE);
}

static void
guppi_seq_impl_class_init (GuppiSeqImplClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataImplClass *data_class = GUPPI_DATA_IMPL_CLASS (klass);
  GuppiSeqImplClass *seq_class = GUPPI_SEQ_IMPL_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_DATA_IMPL);

  data_class->get_size_info = v_size_info;

  seq_class->has_missing = v_seq_has_missing;
  seq_class->missing_count = v_seq_missing_count;
  seq_class->missing = v_seq_missing;
  seq_class->set_missing = v_seq_set_missing;

  object_class->finalize = guppi_seq_impl_finalize;
}

static void
guppi_seq_impl_init (GuppiSeqImpl * obj)
{

}

GtkType guppi_seq_impl_get_type (void)
{
  static GtkType guppi_seq_impl_type = 0;
  if (!guppi_seq_impl_type) {
    static const GtkTypeInfo guppi_seq_impl_info = {
      "GuppiSeqImpl",
      sizeof (GuppiSeqImpl),
      sizeof (GuppiSeqImplClass),
      (GtkClassInitFunc) guppi_seq_impl_class_init,
      (GtkObjectInitFunc) guppi_seq_impl_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_seq_impl_type = gtk_type_unique (GUPPI_TYPE_DATA_IMPL,
					   &guppi_seq_impl_info);
  }
  return guppi_seq_impl_type;
}

GtkObject *
guppi_seq_impl_new (void)
{
  return GTK_OBJECT (guppi_type_new (guppi_seq_impl_get_type ()));
}

/** Default Missing Values Implementation **/

#define mv_seq(x) ((GuppiSeqBoolean*)((x)->opaque_missing_values))

gboolean guppi_seq_impl_has_missing (GuppiSeqImpl * impl)
{
  GuppiSeqBoolean *mv;

  g_return_val_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl), FALSE);

  mv = mv_seq (impl);

  if (mv == NULL)
    return FALSE;

  /*
   * Check if the boolean array contains nothing by false values.
   * If so, we don't actually have any missing values, despite the
   * presence of the array.
   */
  if (!guppi_seq_in_bounds
      (GUPPI_SEQ (mv), guppi_seq_boolean_first_true (mv))) return FALSE;

  return TRUE;
}

gsize guppi_seq_impl_missing_count (GuppiSeqImpl * impl)
{
  GuppiSeqBoolean *mv;

  g_return_val_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl), 0);

  mv = mv_seq (impl);

  return mv ? guppi_seq_boolean_true_count (mv) : 0;
}

gboolean guppi_seq_impl_missing (GuppiSeqImpl * impl, gint i)
{
  GuppiSeqBoolean *mv;

  g_return_val_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl), FALSE);

  mv = mv_seq (impl);

  if (mv == NULL)
    return FALSE;

  if (!guppi_seq_in_bounds (GUPPI_SEQ (mv), i))
    return FALSE;

  return guppi_seq_boolean_get (mv, i);
}

void
guppi_seq_impl_set_missing (GuppiSeqImpl * impl, gint i, gboolean x)
{
  GuppiSeqBoolean *mv;

  g_return_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl));

  mv = mv_seq (impl);

  if (x == FALSE && (mv == NULL || !guppi_seq_in_bounds (GUPPI_SEQ (mv), i)))
    return;

  if (mv == NULL) {
    mv = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());
    impl->opaque_missing_values = mv;
  }

  if (!guppi_seq_in_bounds (GUPPI_SEQ (mv), i))
    guppi_seq_grow_to_include (GUPPI_SEQ (mv), i);

  guppi_seq_boolean_set (mv, i, x);
}

void
guppi_seq_impl_set_many_missing (GuppiSeqImpl * impl, gint i, gint N,
				 gboolean x)
{
  GuppiSeqBoolean *mv;

  g_return_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl));

  mv = mv_seq (impl);

  if (x == FALSE
      && (mv == NULL || !guppi_seq_in_bounds (GUPPI_SEQ (mv), i + N - 1)))
    return;

  if (mv == NULL) {
    mv = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());
    impl->opaque_missing_values = mv;
  }

  if (!guppi_seq_in_bounds (GUPPI_SEQ (mv), i + N - 1))
    guppi_seq_grow_to_include (GUPPI_SEQ (mv), i + N - 1);

  while (N > 0) {
    guppi_seq_boolean_set (mv, i, x);
    --N;
  }
}

void
guppi_seq_impl_insert_missing (GuppiSeqImpl * impl, gint i, gboolean x,
			       gsize N)
{
  GuppiSeqBoolean *mv;
  g_return_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl));
  mv = mv_seq (impl);

  if (x && mv == NULL) {
    mv = GUPPI_SEQ_BOOLEAN (guppi_seq_boolean_new ());
    impl->opaque_missing_values = mv;
  }

  if (mv != NULL) {
    gint i0, i1;
    guppi_seq_bounds (GUPPI_SEQ (mv), &i0, &i1);
    if (i < i0)
      guppi_seq_grow_to_include (GUPPI_SEQ (mv), i);
    else if (i1 < i)
      guppi_seq_grow_to_include (GUPPI_SEQ (mv), i - 1);

    guppi_seq_boolean_insert_many (mv, i, x, N);

    guppi_seq_bounds (GUPPI_SEQ (mv), &i0, &i1);
  }
}

void
guppi_seq_impl_delete_missing (GuppiSeqImpl * impl, gint i, gsize N)
{
  GuppiSeqBoolean *mv;
  g_return_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl));
  mv = mv_seq (impl);

  if (N == 0)
    return;

  if (mv != NULL && !guppi_seq_empty (GUPPI_SEQ (mv))) {
    gint i0, i1, j0, j1;
    guppi_seq_bounds (GUPPI_SEQ (mv), &i0, &i1);
    j0 = MAX (i0, i);
    j1 = MIN (i1, (i - 1 + (gint) N));

    if (j0 <= j1) {
      guppi_seq_delete_many (GUPPI_SEQ (mv), j0, j1 - j0 + 1);
    }

    if (i < j0) {
      guppi_seq_shift_indices (GUPPI_SEQ (mv), i - j0);
    }
  }
}

void
guppi_seq_impl_clear_missing (GuppiSeqImpl * impl)
{
  g_return_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl));
  guppi_unref0 (mv_seq (impl));
}

void
guppi_seq_impl_debug_missing (GuppiSeqImpl * impl)
{
  GuppiSeqBoolean *mv;
  gint i, i0, i1;
  g_return_if_fail (impl != NULL && GUPPI_IS_SEQ_IMPL (impl));

  mv = mv_seq (impl);
  if (mv == NULL) {
    printf ("<null>\n");
    return;
  }

  guppi_seq_bounds (GUPPI_SEQ (mv), &i0, &i1);
  for (i = i0; i <= i1; ++i) {
    printf ("   %d: %s\n", i, guppi_seq_boolean_get (mv, i) ? "T" : "f");
  }
}

/* $Id: guppi-seq-impl.c,v 1.15 2001/05/06 08:26:35 trow Exp $ */
