/* This is -*- C -*- */
/* $Id: guppi-struct.c,v 1.13 2001/05/06 08:26:35 trow Exp $ */

/*
 * guppi-struct.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 * Copyright (C) 2001 The Free Software Foundation
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
/* #include <gnome.h> */
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include <guppi-memory.h>
#include "guppi-struct.h"
#include "guppi-struct-impl.h"

static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_struct_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_struct_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_struct_finalize (GtkObject * obj)
{
  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

struct xml_out {
  GuppiData *data;
  GuppiXMLDocument *doc;
  xmlNodePtr node;
};

static void
export_xml_subdata (const gchar *name, GuppiData *subdata, gpointer user_data)
{
  struct xml_out *xo = (struct xml_out *)user_data;
  GtkType type;
  xmlNodePtr subdata_node;

  type = guppi_struct_get_field_type (GUPPI_STRUCT (xo->data), name);

  subdata_node = xmlNewNode (xo->doc->ns, "SubData");
  xmlNewProp (subdata_node, "name", name);
  if (type)
    xmlNewProp (subdata_node, "type", gtk_type_name (type));

  xmlAddChild (subdata_node, guppi_data_export_xml (subdata, xo->doc));

  xmlAddChild (xo->node, subdata_node);
}

static xmlNodePtr
export_xml_content (GuppiData *data, GuppiXMLDocument *doc)
{
  struct xml_out xo;
  xo.data = data;
  xo.doc = doc;
  xo.node = xmlNewNode (doc->ns, "Content");

  guppi_struct_foreach_subdata (GUPPI_STRUCT (data), export_xml_subdata, &xo);

  return xo.node;
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

static void
guppi_struct_class_init (GuppiStructClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;
  GuppiDataClass *data_class = GUPPI_DATA_CLASS (klass);

  parent_class = gtk_type_class (GUPPI_TYPE_DATA);

  data_class->type_name = _("Struct");

  data_class->export_xml_content = export_xml_content;

  object_class->get_arg = guppi_struct_get_arg;
  object_class->set_arg = guppi_struct_set_arg;
  object_class->finalize = guppi_struct_finalize;
}

static void
guppi_struct_init (GuppiStruct * obj)
{

}

GtkType guppi_struct_get_type (void)
{
  static GtkType guppi_struct_type = 0;
  if (!guppi_struct_type) {
    static const GtkTypeInfo guppi_struct_info = {
      "GuppiStruct",
      sizeof (GuppiStruct),
      sizeof (GuppiStructClass),
      (GtkClassInitFunc) guppi_struct_class_init,
      (GtkObjectInitFunc) guppi_struct_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_struct_type = gtk_type_unique (GUPPI_TYPE_DATA, &guppi_struct_info);
  }
  return guppi_struct_type;
}

GuppiData *
guppi_struct_new (void)
{
  return guppi_data_newv (GUPPI_TYPE_STRUCT, NULL, 0, NULL);
}

/***************************************************************************/

GtkType
guppi_struct_get_field_type (const GuppiStruct * gs, const gchar * field)
{
  const GuppiStructImpl *impl;
  GuppiStructImplClass *impl_class;

  g_return_val_if_fail (gs && GUPPI_IS_STRUCT (gs), (GtkType) 0);
  g_return_val_if_fail (field != NULL, (GtkType) 0);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  impl_class = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->field_type);
  return (impl_class->field_type) (impl, field);
}

gchar **
guppi_struct_get_field_names (const GuppiStruct * gs)
{
  const GuppiStructImpl *impl;
  GuppiStructImplClass *impl_class;

  g_return_val_if_fail (gs && GUPPI_IS_STRUCT (gs), (GtkType) 0);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  impl_class = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get_names);
  return (impl_class->get_names) (impl);
}

gsize guppi_struct_size (const GuppiStruct * gs)
{
  const GuppiStructImpl *impl;
  GuppiStructImplClass *impl_class;

  g_return_val_if_fail (gs && GUPPI_IS_STRUCT (gs), (GtkType) 0);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  impl_class = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get_size);
  return (impl_class->get_size) (impl);
}

GuppiData *
guppi_struct_get (const GuppiStruct * gs, const gchar * field)
{
  const GuppiStructImpl *impl;
  GuppiStructImplClass *impl_class;

  g_return_val_if_fail (gs && GUPPI_IS_STRUCT (gs), NULL);
  g_return_val_if_fail (field != NULL, NULL);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  impl_class = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->get);
  return (impl_class->get) (impl, field);
}

void
guppi_struct_set (GuppiStruct * gs, const gchar * field, GuppiData * data)
{
  GuppiStructImpl *impl;
  GuppiStructImplClass *impl_class;
  gboolean success;

  g_return_if_fail (gs && GUPPI_IS_STRUCT (gs));
  g_return_if_fail (field != NULL);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  impl_class = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->set);
  success = (impl_class->set) (impl, field, data);
  if (!success)
    g_warning ("struct set failed - unknown field: %s", field);

#if 0
  gtk_signal_emit_by_name (GTK_OBJECT (gs), "changed");
  gtk_signal_emit_by_name (GTK_OBJECT (gs), "changed_subdata");
#endif
}

void
guppi_struct_add_field (GuppiStruct * gs, const gchar * field, GtkType type)
{
  GuppiStructImpl *impl;
  GuppiStructImplClass *impl_class;
  gboolean success;

  g_return_if_fail (gs && GUPPI_IS_STRUCT (gs));
  g_return_if_fail (field != NULL);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  impl_class = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);

  g_assert (impl_class->add_field);
  success = (impl_class->add_field) (impl, field, type);
  if (!success)
    g_warning ("struct add failed - field collision: %s", field);
}

void
guppi_struct_add_free_field (GuppiStruct * gs, const gchar * field)
{
  g_return_if_fail (gs && GUPPI_IS_STRUCT (gs));

  guppi_struct_add_field (gs, field, (GtkType) 0);
}

void
guppi_struct_foreach_subdata (GuppiStruct *gs, 
			      void (*fn) (const gchar *, GuppiData *,
					  gpointer),
			      gpointer user_data)
{
  GuppiStructImpl *impl;
  GuppiStructImplClass *klass;

  g_return_if_fail (gs && GUPPI_IS_STRUCT (gs));
  g_return_if_fail (fn);

  impl = GUPPI_STRUCT_IMPL (guppi_data_impl (GUPPI_DATA (gs)));
  klass = GUPPI_STRUCT_IMPL_CLASS (GTK_OBJECT (impl)->klass);
  
  g_assert (klass->foreach);
  klass->foreach (impl, fn, user_data);
}


/* $Id: guppi-struct.c,v 1.13 2001/05/06 08:26:35 trow Exp $ */
