/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: guppi-xml.c,v 1.3 2001/05/06 08:26:46 trow Exp $ */

/*
 * guppi-xml.c
 *
 * Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 */

#include <config.h>
#include <stdlib.h>
#include <gnome-xml/parser.h>
#include "guppi-memory.h"
#include "guppi-debug.h"
#include "guppi-xml.h"

GuppiXMLDocument *
guppi_xml_document_new (void)
{
  GuppiXMLDocument *doc = guppi_new0 (GuppiXMLDocument, 1);

  doc->doc  = xmlNewDoc ("1.0");
  doc->doc->root = xmlNewNode (NULL, "Guppi");

  doc->ns = xmlNewNs (doc->doc->root, "http://www.gnome.org/guppi/xml-version/1.0", "gpi");

  doc->version = GUPPI_XML_V1;

  return doc;
}

GuppiXMLDocument *
guppi_xml_document_read_file (const gchar *filename)
{
  GuppiXMLDocument *doc;

  if (filename == NULL)
    return NULL;

  doc = guppi_new0 (GuppiXMLDocument, 1);

  doc->doc = xmlParseFile (filename);
  if (doc->doc == NULL) {
    guppi_free (doc);
    return NULL;
  }

  /* This is very broken! */
  if (doc->doc->root)
    doc->ns   = doc->doc->root->ns;
  doc->version = GUPPI_XML_V1;

  return doc;
}

void
guppi_xml_document_write_file (GuppiXMLDocument *doc, const gchar *filename)
{
  g_return_if_fail (doc != NULL);
  g_return_if_fail (filename != NULL);
  xmlSaveFile(filename, doc->doc);
}

xmlNodePtr
guppi_xml_document_root (GuppiXMLDocument *doc)
{
  g_return_val_if_fail (doc != NULL, NULL);

  return doc->doc->root;
}

void
guppi_xml_document_add_node (GuppiXMLDocument *doc, xmlNodePtr node)
{
  g_return_if_fail (doc != NULL);

  if (node != NULL)
    xmlAddChild (doc->doc->root, node);
}

void
guppi_xml_document_free (GuppiXMLDocument *doc)
{
  if (doc) {
    xmlFreeDoc (doc->doc);
    doc->doc = NULL;
    guppi_free0 (doc);
  }
}

void
guppi_xml_document_spew (GuppiXMLDocument *doc)
{
  g_return_if_fail (doc);

  xmlDocDump (stdout, doc->doc);
}

/* ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** */

xmlNodePtr
guppi_xml_new_node (GuppiXMLDocument *doc, const gchar *node_name)
{
  g_return_val_if_fail (doc != NULL, NULL);
  g_return_val_if_fail (node_name != NULL, NULL);

  return xmlNewNode (doc->ns, node_name);
}

xmlNodePtr
guppi_xml_new_text_node (GuppiXMLDocument *doc, const gchar *node_name, const gchar *text)
{
  xmlNodePtr node;
  g_return_val_if_fail (doc != NULL, NULL);
  g_return_val_if_fail (node_name != NULL, NULL);

  node = guppi_xml_new_node (doc, node_name);
  if (text)
    xmlAddChild (node, xmlNewText (text));

  return node;
}

xmlNodePtr
guppi_xml_new_text_nodef (GuppiXMLDocument *doc, const gchar *node_name,
			  const gchar *text_format, ...)
{
  gchar *str = NULL;
  xmlNodePtr node;
  va_list args;

  g_return_val_if_fail (doc != NULL, NULL);
  g_return_val_if_fail (node_name != NULL, NULL);
  
  if (text_format) {
    va_start (args, text_format);
    str = guppi_strdup_vprintf (text_format, args);
    va_end (args);
  }
  
  node = guppi_xml_new_text_node (doc, node_name, str);
  
  guppi_free (str);

  return node;
}

void
guppi_xml_set_property (xmlNodePtr node, const gchar *name, const gchar *value)
{
  g_return_if_fail (node != NULL);
  g_return_if_fail (name != NULL);
  g_return_if_fail (value != NULL);

  xmlNewProp (node, name, value);
}

void
guppi_xml_set_propertyf (xmlNodePtr node, const gchar *name, const gchar *value_format, ...)
{
  gchar *str = NULL;
  va_list args;

  g_return_if_fail (node != NULL);
  g_return_if_fail (name != NULL);
  g_return_if_fail (value_format != NULL);
  
  va_start (args, value_format);
  str = guppi_strdup_vprintf (value_format, args);
  va_end (args);

  guppi_xml_set_property (node, name, str);

  guppi_free (str);
}

gchar *
guppi_xml_get_property (xmlNodePtr node, const gchar *name)
{
  gchar *str;
  gchar *strcpy;

  g_return_val_if_fail (node != NULL, NULL);
  g_return_val_if_fail (name != NULL, NULL);

  /* We do this to ensure that the returned string is g_free-able. */
  str = xmlGetProp (node, name);
  strcpy = g_strdup (str);
  if (str) free (str);
  return strcpy;
}

gint
guppi_xml_get_property_int (xmlNodePtr node, const gchar *name)
{
  gchar *str;
  gint x = 0;

  g_return_val_if_fail (node != NULL, 0);
  g_return_val_if_fail (name != NULL, 0);
  
  str = xmlGetProp (node, name);
  if (str) {
    x = atoi (str);
    free (str);
  }
  return x;
}

double
guppi_xml_get_property_double (xmlNodePtr node, const gchar *name)
{
  gchar *str;
  double x = 0;

  g_return_val_if_fail (node != NULL, 0);
  g_return_val_if_fail (name != NULL, 0);
  
  str = xmlGetProp (node, name);
  if (str) {
    x = atof (str);
    free (str);
  }
  return x;
}
