/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* NetworkManager -- Network link manager
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Copyright 2011 - 2012 Red Hat, Inc.
 */

#include "nm-default.h"

#include "nm-device-vlan.h"

#include <sys/socket.h>

#include "nm-manager.h"
#include "nm-utils.h"
#include "NetworkManagerUtils.h"
#include "nm-device-private.h"
#include "settings/nm-settings.h"
#include "nm-act-request.h"
#include "nm-ip4-config.h"
#include "platform/nm-platform.h"
#include "nm-device-factory.h"
#include "nm-manager.h"
#include "nm-core-internal.h"
#include "platform/nmp-object.h"

#include "introspection/org.freedesktop.NetworkManager.Device.Vlan.h"

#include "nm-device-logging.h"
_LOG_DECLARE_SELF(NMDeviceVlan);

/*****************************************************************************/

NM_GOBJECT_PROPERTIES_DEFINE (NMDeviceVlan,
	PROP_VLAN_ID,
);

typedef struct {
	gulong parent_state_id;
	gulong parent_hwaddr_id;
	guint vlan_id;
} NMDeviceVlanPrivate;

struct _NMDeviceVlan {
	NMDevice parent;
	NMDeviceVlanPrivate _priv;
};

struct _NMDeviceVlanClass {
	NMDeviceClass parent;
};

G_DEFINE_TYPE (NMDeviceVlan, nm_device_vlan, NM_TYPE_DEVICE)

#define NM_DEVICE_VLAN_GET_PRIVATE(self) _NM_GET_PRIVATE (self, NMDeviceVlan, NM_IS_DEVICE_VLAN)

/*****************************************************************************/

static void
parent_state_changed (NMDevice *parent,
                      NMDeviceState new_state,
                      NMDeviceState old_state,
                      NMDeviceStateReason reason,
                      gpointer user_data)
{
	NMDeviceVlan *self = NM_DEVICE_VLAN (user_data);

	/* We'll react to our own carrier state notifications. Ignore the parent's. */
	if (reason == NM_DEVICE_STATE_REASON_CARRIER)
		return;

	nm_device_set_unmanaged_by_flags (NM_DEVICE (self), NM_UNMANAGED_PARENT, !nm_device_get_managed (parent, FALSE), reason);
}

static void
parent_hwaddr_maybe_changed (NMDevice *parent,
                             GParamSpec *pspec,
                             gpointer user_data)
{
	NMDeviceVlan *self = NM_DEVICE_VLAN (user_data);
	NMConnection *connection;
	const char *new_mac, *old_mac;
	NMSettingIPConfig *s_ip6;

	/* Never touch assumed devices */
	if (nm_device_uses_assumed_connection ((NMDevice *) self))
		return;

	connection = nm_device_get_applied_connection ((NMDevice *) self);
	if (!connection)
		return;

	/* Update the VLAN MAC only if configuration does not specify one */
	if (nm_device_hw_addr_is_explict ((NMDevice *) self))
		return;

	old_mac = nm_device_get_hw_address ((NMDevice *) self);
	new_mac = nm_device_get_hw_address (parent);
	if (nm_streq0 (old_mac, new_mac))
		return;

	_LOGD (LOGD_VLAN, "parent hardware address changed to %s%s%s",
	       NM_PRINT_FMT_QUOTE_STRING (new_mac));
	if (new_mac) {
		nm_device_hw_addr_set ((NMDevice *) self, new_mac, "vlan-parent", TRUE);
		/* When changing the hw address the interface is taken down,
		 * removing the IPv6 configuration; reapply it.
		 */
		s_ip6 = nm_connection_get_setting_ip6_config (connection);
		if (s_ip6)
			nm_device_reactivate_ip6_config (NM_DEVICE (self), s_ip6, s_ip6);
	}
}

static void
parent_changed_notify (NMDevice *device,
                       int old_ifindex,
                       NMDevice *old_parent,
                       int new_ifindex,
                       NMDevice *new_parent)
{
	NMDeviceVlan *self = NM_DEVICE_VLAN (device);
	NMDeviceVlanPrivate *priv = NM_DEVICE_VLAN_GET_PRIVATE (self);

	NM_DEVICE_CLASS (nm_device_vlan_parent_class)->parent_changed_notify (device, old_ifindex, old_parent, new_ifindex, new_parent);

	/*  note that @self doesn't have to clear @parent_state_id on dispose,
	 *  because NMDevice's dispose() will unset the parent, which in turn calls
	 *  parent_changed_notify(). */
	nm_clear_g_signal_handler (old_parent, &priv->parent_state_id);
	nm_clear_g_signal_handler (old_parent, &priv->parent_hwaddr_id);

	if (new_parent) {
		priv->parent_state_id = g_signal_connect (new_parent,
		                                          NM_DEVICE_STATE_CHANGED,
		                                          G_CALLBACK (parent_state_changed),
		                                          device);

		priv->parent_hwaddr_id = g_signal_connect (new_parent, "notify::" NM_DEVICE_HW_ADDRESS,
		                                           G_CALLBACK (parent_hwaddr_maybe_changed), device);
		parent_hwaddr_maybe_changed (new_parent, NULL, self);

		/* Set parent-dependent unmanaged flag */
		nm_device_set_unmanaged_by_flags (device,
		                                  NM_UNMANAGED_PARENT,
		                                  !nm_device_get_managed (new_parent, FALSE),
		                                  NM_DEVICE_STATE_REASON_PARENT_MANAGED_CHANGED);
	}

	/* Recheck availability now that the parent has changed */
	if (new_ifindex > 0) {
		nm_device_queue_recheck_available (device,
		                                   NM_DEVICE_STATE_REASON_PARENT_CHANGED,
		                                   NM_DEVICE_STATE_REASON_PARENT_CHANGED);
	}
}

static void
update_properties (NMDevice *device)
{
	NMDeviceVlanPrivate *priv;
	const NMPlatformLink *plink = NULL;
	const NMPlatformLnkVlan *plnk = NULL;
	int ifindex;
	int parent_ifindex = 0;
	guint vlan_id;

	g_return_if_fail (NM_IS_DEVICE_VLAN (device));

	priv = NM_DEVICE_VLAN_GET_PRIVATE ((NMDeviceVlan *) device);

	ifindex = nm_device_get_ifindex (device);

	if (ifindex > 0)
		plnk = nm_platform_link_get_lnk_vlan (NM_PLATFORM_GET, ifindex, &plink);

	if (   plnk
	    && plink->parent > 0)
		parent_ifindex = plink->parent;

	g_object_freeze_notify ((GObject *) device);

	nm_device_parent_set_ifindex (device, parent_ifindex);

	vlan_id = plnk ? plnk->id : 0;
	if (vlan_id != priv->vlan_id) {
		priv->vlan_id = vlan_id;
		_notify ((NMDeviceVlan *) device, PROP_VLAN_ID);
	}

	g_object_thaw_notify ((GObject *) device);
}

static void
link_changed (NMDevice *device,
              const NMPlatformLink *pllink)
{
	NM_DEVICE_CLASS (nm_device_vlan_parent_class)->link_changed (device, pllink);
	update_properties (device);
}

static gboolean
create_and_realize (NMDevice *device,
                    NMConnection *connection,
                    NMDevice *parent,
                    const NMPlatformLink **out_plink,
                    GError **error)
{
	NMDeviceVlanPrivate *priv = NM_DEVICE_VLAN_GET_PRIVATE ((NMDeviceVlan *) device);
	const char *iface = nm_device_get_iface (device);
	NMSettingVlan *s_vlan;
	int parent_ifindex;
	guint vlan_id;
	NMPlatformError plerr;

	s_vlan = nm_connection_get_setting_vlan (connection);
	g_assert (s_vlan);

	if (!parent) {
		g_set_error (error, NM_DEVICE_ERROR, NM_DEVICE_ERROR_FAILED,
		             "VLAN devices can not be created without a parent interface");
		return FALSE;
	}

	if (!nm_device_supports_vlans (parent)) {
		g_set_error (error, NM_DEVICE_ERROR, NM_DEVICE_ERROR_FAILED,
		             "no support for VLANs on interface %s of type %s",
		             nm_device_get_iface (parent),
		             nm_device_get_type_desc (parent));
		return FALSE;
	}

	parent_ifindex = nm_device_get_ifindex (parent);
	g_warn_if_fail (parent_ifindex > 0);

	vlan_id = nm_setting_vlan_get_id (s_vlan);

	plerr = nm_platform_link_vlan_add (NM_PLATFORM_GET,
	                                   iface,
	                                   parent_ifindex,
	                                   vlan_id,
	                                   nm_setting_vlan_get_flags (s_vlan),
	                                   out_plink);
	if (plerr != NM_PLATFORM_ERROR_SUCCESS) {
		g_set_error (error, NM_DEVICE_ERROR, NM_DEVICE_ERROR_CREATION_FAILED,
		             "Failed to create VLAN interface '%s' for '%s': %s",
		             iface,
		             nm_connection_get_id (connection),
		             nm_platform_error_to_string (plerr));
		return FALSE;
	}

	nm_device_parent_set_ifindex (device, parent_ifindex);
	if (vlan_id != priv->vlan_id) {
		priv->vlan_id = vlan_id;
		_notify ((NMDeviceVlan *) device, PROP_VLAN_ID);
	}

	return TRUE;
}

static void
unrealize_notify (NMDevice *device)
{
	NMDeviceVlan *self = NM_DEVICE_VLAN (device);
	NMDeviceVlanPrivate *priv = NM_DEVICE_VLAN_GET_PRIVATE (self);

	NM_DEVICE_CLASS (nm_device_vlan_parent_class)->unrealize_notify (device);

	if (priv->vlan_id != 0) {
		priv->vlan_id = 0;
		_notify (self, PROP_VLAN_ID);
	}
}

/*****************************************************************************/

static NMDeviceCapabilities
get_generic_capabilities (NMDevice *device)
{
	/* We assume VLAN interfaces always support carrier detect */
	return NM_DEVICE_CAP_CARRIER_DETECT | NM_DEVICE_CAP_IS_SOFTWARE;
}

/*****************************************************************************/

static gboolean
is_available (NMDevice *device, NMDeviceCheckDevAvailableFlags flags)
{
	if (!nm_device_parent_get_device (device))
		return FALSE;
	return NM_DEVICE_CLASS (nm_device_vlan_parent_class)->is_available (device, flags);
}

/*****************************************************************************/

static gboolean
match_parent (NMDeviceVlan *self, const char *parent)
{
	NMDevice *parent_device;

	g_return_val_if_fail (parent != NULL, FALSE);

	parent_device = nm_device_parent_get_device (NM_DEVICE (self));
	if (!parent_device)
		return FALSE;

	if (nm_utils_is_uuid (parent)) {
		NMActRequest *parent_req;
		NMConnection *parent_connection;

		/* If the parent is a UUID, the connection matches if our parent
		 * device has that connection activated.
		 */

		parent_req = nm_device_get_act_request (parent_device);
		if (!parent_req)
			return FALSE;

		parent_connection = nm_active_connection_get_applied_connection (NM_ACTIVE_CONNECTION (parent_req));
		if (!parent_connection)
			return FALSE;

		if (g_strcmp0 (parent, nm_connection_get_uuid (parent_connection)) != 0)
			return FALSE;
	} else {
		/* interface name */
		if (g_strcmp0 (parent, nm_device_get_ip_iface (parent_device)) != 0)
			return FALSE;
	}

	return TRUE;
}

static gboolean
match_hwaddr (NMDevice *device, NMConnection *connection, gboolean fail_if_no_hwaddr)
{
	NMSettingWired *s_wired;
	NMDevice *parent_device;
	const char *setting_mac;
	const char *parent_mac;

	s_wired = nm_connection_get_setting_wired (connection);
	if (!s_wired)
		return !fail_if_no_hwaddr;

	setting_mac = nm_setting_wired_get_mac_address (s_wired);
	if (!setting_mac)
		return !fail_if_no_hwaddr;

	parent_device = nm_device_parent_get_device (device);
	if (!parent_device)
		return !fail_if_no_hwaddr;

	parent_mac = nm_device_get_permanent_hw_address (parent_device);
	return parent_mac && nm_utils_hwaddr_matches (setting_mac, -1, parent_mac, -1);
}

static gboolean
check_connection_compatible (NMDevice *device, NMConnection *connection)
{
	NMDeviceVlanPrivate *priv = NM_DEVICE_VLAN_GET_PRIVATE ((NMDeviceVlan *) device);
	NMSettingVlan *s_vlan;
	const char *parent = NULL;

	if (!NM_DEVICE_CLASS (nm_device_vlan_parent_class)->check_connection_compatible (device, connection))
		return FALSE;

	s_vlan = nm_connection_get_setting_vlan (connection);
	if (!s_vlan)
		return FALSE;

	/* Before the device is realized some properties will not be set */
	if (nm_device_is_real (device)) {
		if (nm_setting_vlan_get_id (s_vlan) != priv->vlan_id)
			return FALSE;

		/* Check parent interface; could be an interface name or a UUID */
		parent = nm_setting_vlan_get_parent (s_vlan);
		if (parent) {
			if (!match_parent (NM_DEVICE_VLAN (device), parent))
				return FALSE;
		} else {
			/* Parent could be a MAC address in an NMSettingWired */
			if (!match_hwaddr (device, connection, TRUE))
				return FALSE;
		}
	}

	return TRUE;
}

static gboolean
check_connection_available (NMDevice *device,
                            NMConnection *connection,
                            NMDeviceCheckConAvailableFlags flags,
                            const char *specific_object)
{
	if (!nm_device_is_real (device))
		return TRUE;

	return NM_DEVICE_CLASS (nm_device_vlan_parent_class)->check_connection_available (device, connection, flags, specific_object);
}

static gboolean
complete_connection (NMDevice *device,
                     NMConnection *connection,
                     const char *specific_object,
                     const GSList *existing_connections,
                     GError **error)
{
	NMSettingVlan *s_vlan;

	nm_utils_complete_generic (NM_PLATFORM_GET,
	                           connection,
	                           NM_SETTING_VLAN_SETTING_NAME,
	                           existing_connections,
	                           NULL,
	                           _("VLAN connection"),
	                           NULL,
	                           TRUE);

	s_vlan = nm_connection_get_setting_vlan (connection);
	if (!s_vlan) {
		g_set_error_literal (error, NM_DEVICE_ERROR, NM_DEVICE_ERROR_INVALID_CONNECTION,
		                     "A 'vlan' setting is required.");
		return FALSE;
	}

	/* If there's no VLAN interface, no parent, and no hardware address in the
	 * settings, then there's not enough information to complete the setting.
	 */
	if (   !nm_setting_vlan_get_parent (s_vlan)
	    && !match_hwaddr (device, connection, TRUE)) {
		g_set_error_literal (error, NM_DEVICE_ERROR, NM_DEVICE_ERROR_INVALID_CONNECTION,
		                     "The 'vlan' setting had no interface name, parent, or hardware address.");
		return FALSE;
	}

	return TRUE;
}

static void
update_connection (NMDevice *device, NMConnection *connection)
{
	NMDeviceVlanPrivate *priv = NM_DEVICE_VLAN_GET_PRIVATE ((NMDeviceVlan *) device);
	NMSettingVlan *s_vlan = nm_connection_get_setting_vlan (connection);
	int ifindex = nm_device_get_ifindex (device);
	const char *setting_parent, *new_parent;
	const NMPlatformLink *plink;
	const NMPObject *polnk;
	NMDevice *parent_device;
	guint vlan_id;
	guint vlan_flags;

	if (!s_vlan) {
		s_vlan = (NMSettingVlan *) nm_setting_vlan_new ();
		nm_connection_add_setting (connection, (NMSetting *) s_vlan);
	}

	polnk = nm_platform_link_get_lnk (NM_PLATFORM_GET, ifindex, NM_LINK_TYPE_VLAN, &plink);

	if (polnk)
		vlan_id = polnk->lnk_vlan.id;
	else
		vlan_id = priv->vlan_id;
	if (vlan_id != nm_setting_vlan_get_id (s_vlan))
		g_object_set (s_vlan, NM_SETTING_VLAN_ID, vlan_id, NULL);

	/* Update parent in the connection; default to parent's interface name */
	parent_device = nm_device_parent_get_device (device);
	if (   parent_device
	    && polnk
	    && plink->parent > 0
	    && nm_device_get_ifindex (parent_device) == plink->parent) {
		new_parent = nm_device_get_iface (parent_device);
		setting_parent = nm_setting_vlan_get_parent (s_vlan);
		if (setting_parent && nm_utils_is_uuid (setting_parent)) {
			NMConnection *parent_connection;

			/* Don't change a parent specified by UUID if it's still valid */
			parent_connection = (NMConnection *) nm_settings_get_connection_by_uuid (nm_device_get_settings (device), setting_parent);
			if (parent_connection && nm_device_check_connection_compatible (parent_device, parent_connection))
				new_parent = NULL;
		}
		if (new_parent)
			g_object_set (s_vlan, NM_SETTING_VLAN_PARENT, new_parent, NULL);
	} else
		g_object_set (s_vlan, NM_SETTING_VLAN_PARENT, NULL, NULL);

	if (polnk)
		vlan_flags = polnk->lnk_vlan.flags;
	else
		vlan_flags = NM_VLAN_FLAG_REORDER_HEADERS;
	if (vlan_flags != nm_setting_vlan_get_flags (s_vlan))
		g_object_set (s_vlan, NM_SETTING_VLAN_FLAGS, (NMVlanFlags) vlan_flags, NULL);

	if (polnk) {
		_nm_setting_vlan_set_priorities (s_vlan, NM_VLAN_INGRESS_MAP,
		                                 polnk->_lnk_vlan.ingress_qos_map,
		                                 polnk->_lnk_vlan.n_ingress_qos_map);
		_nm_setting_vlan_set_priorities (s_vlan, NM_VLAN_EGRESS_MAP,
		                                 polnk->_lnk_vlan.egress_qos_map,
		                                 polnk->_lnk_vlan.n_egress_qos_map);
	} else {
		_nm_setting_vlan_set_priorities (s_vlan, NM_VLAN_INGRESS_MAP, NULL, 0);
		_nm_setting_vlan_set_priorities (s_vlan, NM_VLAN_EGRESS_MAP, NULL, 0);
	}
}

static NMActStageReturn
act_stage1_prepare (NMDevice *device, NMDeviceStateReason *reason)
{
	NMDevice *parent_device;
	NMSettingVlan *s_vlan;
	NMActStageReturn ret;

	g_return_val_if_fail (reason != NULL, NM_ACT_STAGE_RETURN_FAILURE);

	ret = NM_DEVICE_CLASS (nm_device_vlan_parent_class)->act_stage1_prepare (device, reason);
	if (ret != NM_ACT_STAGE_RETURN_SUCCESS)
		return ret;

	if (!nm_device_hw_addr_set_cloned (device, nm_device_get_applied_connection (device), FALSE))
		return NM_ACT_STAGE_RETURN_FAILURE;

	/* Change MAC address to parent's one if needed */
	parent_device = nm_device_parent_get_device (device);
	if (parent_device)
		parent_hwaddr_maybe_changed (parent_device, NULL, device);

	s_vlan = (NMSettingVlan *) nm_device_get_applied_setting (device, NM_TYPE_SETTING_VLAN);
	if (s_vlan) {
		gs_free NMVlanQosMapping *ingress_map = NULL;
		gs_free NMVlanQosMapping *egress_map = NULL;
		guint n_ingress_map = 0, n_egress_map = 0;

		_nm_setting_vlan_get_priorities (s_vlan,
		                                 NM_VLAN_INGRESS_MAP,
		                                 &ingress_map,
		                                 &n_ingress_map);
		_nm_setting_vlan_get_priorities (s_vlan,
		                                 NM_VLAN_EGRESS_MAP,
		                                 &egress_map,
		                                 &n_egress_map);

		nm_platform_link_vlan_change (NM_PLATFORM_GET,
		                              nm_device_get_ifindex (device),
		                              NM_VLAN_FLAGS_ALL,
		                              nm_setting_vlan_get_flags (s_vlan),
		                              TRUE,
		                              ingress_map,
		                              n_ingress_map,
		                              TRUE,
		                              egress_map,
		                              n_egress_map);
	}

	return ret;
}

/*****************************************************************************/

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
	NMDeviceVlanPrivate *priv = NM_DEVICE_VLAN_GET_PRIVATE ((NMDeviceVlan *) object);

	switch (prop_id) {
	case PROP_VLAN_ID:
		g_value_set_uint (value, priv->vlan_id);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

/*****************************************************************************/

static void
nm_device_vlan_init (NMDeviceVlan * self)
{
}

static void
nm_device_vlan_class_init (NMDeviceVlanClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	NMDeviceClass *parent_class = NM_DEVICE_CLASS (klass);

	NM_DEVICE_CLASS_DECLARE_TYPES (klass, NM_SETTING_VLAN_SETTING_NAME, NM_LINK_TYPE_VLAN)

	object_class->get_property = get_property;

	parent_class->create_and_realize = create_and_realize;
	parent_class->link_changed = link_changed;
	parent_class->unrealize_notify = unrealize_notify;
	parent_class->get_generic_capabilities = get_generic_capabilities;
	parent_class->act_stage1_prepare = act_stage1_prepare;
	parent_class->get_configured_mtu = nm_device_get_configured_mtu_for_wired;
	parent_class->is_available = is_available;
	parent_class->parent_changed_notify = parent_changed_notify;

	parent_class->check_connection_compatible = check_connection_compatible;
	parent_class->check_connection_available = check_connection_available;
	parent_class->complete_connection = complete_connection;
	parent_class->update_connection = update_connection;

	obj_properties[PROP_VLAN_ID] =
	     g_param_spec_uint (NM_DEVICE_VLAN_ID, "", "",
	                        0, 4095, 0,
	                        G_PARAM_READABLE
	                        | G_PARAM_STATIC_STRINGS);

	g_object_class_install_properties (object_class, _PROPERTY_ENUMS_LAST, obj_properties);

	nm_exported_object_class_add_interface (NM_EXPORTED_OBJECT_CLASS (klass),
	                                        NMDBUS_TYPE_DEVICE_VLAN_SKELETON,
	                                        NULL);
}

/*****************************************************************************/

#define NM_TYPE_VLAN_DEVICE_FACTORY (nm_vlan_device_factory_get_type ())
#define NM_VLAN_DEVICE_FACTORY(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), NM_TYPE_VLAN_DEVICE_FACTORY, NMVlanDeviceFactory))

static NMDevice *
create_device (NMDeviceFactory *factory,
               const char *iface,
               const NMPlatformLink *plink,
               NMConnection *connection,
               gboolean *out_ignore)
{
	return (NMDevice *) g_object_new (NM_TYPE_DEVICE_VLAN,
	                                  NM_DEVICE_IFACE, iface,
	                                  NM_DEVICE_DRIVER, "8021q",
	                                  NM_DEVICE_TYPE_DESC, "VLAN",
	                                  NM_DEVICE_DEVICE_TYPE, NM_DEVICE_TYPE_VLAN,
	                                  NM_DEVICE_LINK_TYPE, NM_LINK_TYPE_VLAN,
	                                  NULL);
}

static const char *
get_connection_parent (NMDeviceFactory *factory, NMConnection *connection)
{
	NMSettingVlan *s_vlan;
	NMSettingWired *s_wired;
	const char *parent = NULL;

	g_return_val_if_fail (nm_connection_is_type (connection, NM_SETTING_VLAN_SETTING_NAME), NULL);

	s_vlan = nm_connection_get_setting_vlan (connection);
	g_assert (s_vlan);

	parent = nm_setting_vlan_get_parent (s_vlan);
	if (parent)
		return parent;

	/* Try the hardware address from the VLAN connection's hardware setting */
	s_wired = nm_connection_get_setting_wired (connection);
	if (s_wired)
		return nm_setting_wired_get_mac_address (s_wired);

	return NULL;
}

static char *
get_connection_iface (NMDeviceFactory *factory,
                      NMConnection *connection,
                      const char *parent_iface)
{
	const char *ifname;
	NMSettingVlan *s_vlan;

	g_return_val_if_fail (nm_connection_is_type (connection, NM_SETTING_VLAN_SETTING_NAME), NULL);

	s_vlan = nm_connection_get_setting_vlan (connection);
	g_assert (s_vlan);

	if (!parent_iface)
		return NULL;

	ifname = nm_connection_get_interface_name (connection);
	if (ifname)
		return g_strdup (ifname);

	/* If the connection doesn't specify the interface name for the VLAN
	 * device, we create one for it using the VLAN ID and the parent
	 * interface's name.
	 */
	return nm_utils_new_vlan_name (parent_iface, nm_setting_vlan_get_id (s_vlan));
}

NM_DEVICE_FACTORY_DEFINE_INTERNAL (VLAN, Vlan, vlan,
	NM_DEVICE_FACTORY_DECLARE_LINK_TYPES    (NM_LINK_TYPE_VLAN)
	NM_DEVICE_FACTORY_DECLARE_SETTING_TYPES (NM_SETTING_VLAN_SETTING_NAME),
	factory_class->create_device = create_device;
	factory_class->get_connection_parent = get_connection_parent;
	factory_class->get_connection_iface = get_connection_iface;
);
