/*
 * main.c: the main program / app list view
 *
 * Author:
 *    Michael Meeks
 *
 * Copyright 2002 Sun Microsystems, Inc.
 */

/*
 * TODO: listen for childen-changed on
 * the desktop object ... impl. a
 * 'Global state' object that we can sync
 *  stuff with
 */
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <cspi/spi.h>
#include <libgnomeui/gnome-ui-init.h>

#include "graphics.h"
#include "child-listener.h"

extern void poke (Accessible *accessible);

#define APP_COL_NAME   0
#define APP_COL_DESCR  1
#define APP_COL_HANDLE 2
#define APP_COL_LAST   3

typedef struct {
	GladeXML     *app_xml;

	GtkWidget    *window;
	
	GtkListStore *list_store;
	GtkTreeView  *list_view;

	GtkLabel     *toolkit_name;
	GtkLabel     *toolkit_version;
	GtkLabel     *app_id;

	Accessible   *selected;
} AppWindow;

static void
update_app_display (AppWindow  *app_window,
		    Accessible *accessible)
{
	char *txt;
	AccessibleApplication *application;

	if (!accessible ||
	    !(application = Accessible_getApplication (accessible))) {
		gtk_label_set_text (app_window->toolkit_name, "<invalid>");
		gtk_label_set_text (app_window->toolkit_version, "<invalid>");
		gtk_label_set_text (app_window->app_id, "<invalid>");
		return;
	}

	txt = AccessibleApplication_getToolkitName (application);
	gtk_label_set_text (app_window->toolkit_name, txt);
	SPI_freeString (txt);

	txt = AccessibleApplication_getVersion (application);
	gtk_label_set_text (app_window->toolkit_version, txt);
	SPI_freeString (txt);

	txt = g_strdup_printf ("%ld", AccessibleApplication_getID (application));
	gtk_label_set_text (app_window->app_id, txt);
	g_free (txt);

	AccessibleApplication_unref (application);
}

static void
app_list_selection_changed (GtkTreeSelection *selection,
			    AppWindow        *app_window)
{
	GtkTreeIter iter;
	Accessible *accessible;
	GtkTreeModel *model;
	
	if (!gtk_tree_selection_get_selected (selection, &model, &iter)) {
		return;
	}

	gtk_tree_model_get (model, &iter,
			    APP_COL_HANDLE, &accessible,
			    -1);
	update_app_display (app_window, accessible);

	app_window->selected = accessible;
}

static void
validate_up_down_linkage (Accessible *accessible)
{
	Accessible *child, *aa;
	Accessible *parent;

	aa = Accessible_queryInterface (accessible, "IDL:Accessibility/Accessible:1.0");
	g_assert (aa == accessible);

	child = Accessible_getChildAtIndex (accessible, 0);
	if (!child)
		return;

	parent = Accessible_getParent (child);

	g_assert (parent == accessible);
}

static void
app_list_row_activated (GtkTreeView       *tree_view,
			GtkTreePath       *path,
			GtkTreeViewColumn *column,
			AppWindow         *app_window)
{
	validate_up_down_linkage (app_window->selected);
	poke (app_window->selected);
}

static void
application_clicked (GtkButton *button,
		     AppWindow *app_window)
{
	validate_up_down_linkage (app_window->selected);
	poke (app_window->selected);
}

static void
populate_app_list (AppWindow *app_window)
{
	int i, apps;
	Accessible *desktop;
	GtkTreeIter iter;

	desktop = SPI_getDesktop (0);

	apps = Accessible_getChildCount (desktop);

	gtk_list_store_clear (app_window->list_store);
	for (i = 0; i < apps; i++) {
		char *name, *descr;
		Accessible *child;

		child = Accessible_getChildAtIndex (desktop, i);
		if (!child)
			continue;

		name = Accessible_getName (child);
		descr = Accessible_getDescription (child);
		gtk_list_store_append (app_window->list_store, &iter);
		gtk_list_store_set (app_window->list_store,
				    &iter,
				    APP_COL_NAME,   name,
				    APP_COL_DESCR,  descr,
				    APP_COL_HANDLE, child,
				    -1);
		SPI_freeString (descr);
		SPI_freeString (name);
	}

	if (gtk_tree_model_get_iter_root (
		GTK_TREE_MODEL (app_window->list_store), &iter))
		gtk_tree_selection_select_iter (
			gtk_tree_view_get_selection (app_window->list_view), &iter);

	Accessible_unref (desktop);
}

static void
app_list_changed_cb (ChildListener *listener,
		     AppWindow     *window)
{
	populate_app_list (window);
}

static void
create_app_list (AppWindow *app_window)
{
	app_window->list_store = 
		gtk_list_store_new (APP_COL_LAST,
				    G_TYPE_STRING,
				    G_TYPE_STRING,
				    G_TYPE_POINTER);

	app_window->list_view = GTK_TREE_VIEW (
		glade_xml_get_widget (
			app_window->app_xml, "application_list_view"));

	gtk_tree_view_set_model (app_window->list_view,
				 GTK_TREE_MODEL (app_window->list_store));

	gtk_tree_view_insert_column_with_attributes
		(app_window->list_view,
		 APP_COL_NAME, "Application",
		 gtk_cell_renderer_text_new (),
		 "text", 0,
		 NULL);
	gtk_tree_view_insert_column_with_attributes
		(app_window->list_view,
		 APP_COL_DESCR, "Description",
		 gtk_cell_renderer_text_new (),
		 "text", 0,
		 NULL);
	gtk_tree_view_columns_autosize (app_window->list_view);
	gtk_tree_selection_set_mode (
		gtk_tree_view_get_selection (
			app_window->list_view),
		GTK_SELECTION_BROWSE);

	g_signal_connect (
		gtk_tree_view_get_selection (
			app_window->list_view),
		"changed",
		G_CALLBACK (app_list_selection_changed),
		app_window);

	g_signal_connect (
		GTK_TREE_VIEW (app_window->list_view),
		"row_activated",
		G_CALLBACK (app_list_row_activated),
		app_window);
}

static void
application_window (void)
{
	GClosure  *closure;
	GladeXML  *app_xml;
	AppWindow *app_window = g_new0 (AppWindow, 1);

	app_xml = get_glade_xml ();
	app_window->app_xml = app_xml;

	if (!app_xml) {
		g_error ("Can't find at-poke.glade2");
	}

	app_window->window = glade_xml_get_widget (
		app_xml, "application_window");

	gtk_widget_show (app_window->window);

	create_app_list (app_window);

	app_window->toolkit_name = GTK_LABEL (
		glade_xml_get_widget (app_xml, "application_toolkit_name"));
	app_window->toolkit_version = GTK_LABEL (
		glade_xml_get_widget (app_xml, "application_toolkit_version"));
	app_window->app_id = GTK_LABEL (
		glade_xml_get_widget (app_xml, "application_id"));

	populate_app_list (app_window);

	g_signal_connect (
		glade_xml_get_widget (
			app_xml, "application_poke_button"),
		"clicked",
		G_CALLBACK (application_clicked),
		app_window);

	g_signal_connect_closure (
		glade_xml_get_widget (
			app_xml, "application_refresh_button"),
		"clicked",
		g_cclosure_new_swap (
			G_CALLBACK (populate_app_list),
			app_window, NULL),
		0);

	closure = g_cclosure_new (
		G_CALLBACK (app_list_changed_cb),
		app_window, NULL);
	g_object_watch_closure (
		G_OBJECT (app_window->window),
		closure);
	g_signal_connect_closure (
		child_listener_get (),
		"app_list_changed",
		closure, FALSE);

	gtk_main ();

	gtk_widget_destroy (app_window->window);

	g_object_unref (app_xml);
	g_free (app_window);
}

static gboolean
no_accessible_apps (void)
{
	int num_apps;
	Accessible *desktop;

	desktop = SPI_getDesktop (0);

	num_apps = Accessible_getChildCount (desktop);
	
	Accessible_unref (desktop);

	return (num_apps == 0);
}

int
main (int argc, char **argv)
{
	int leaked;

	g_assert (!SPI_init ());

	if (no_accessible_apps ()) { /* fork gtk-demo */
		if (!fork ()) {
			putenv ("GTK_MODULES=gail:atk-bridge");
			execlp ("gtk-demo", "gtk-demo", NULL);
			g_warning ("Foo");
			_exit (0);
		}
	}

	/* We don't want to show ourself */
	putenv ("GTK_MODULES=");
	putenv ("GNOME_ACCESSIBILITY=0");

	gnome_program_init ("at-poke", VERSION,
			    LIBGNOMEUI_MODULE,
			    argc, argv, NULL);

	application_window ();

	if ((leaked = SPI_exit ()))
		g_error ("Leaked %d SPI handles", leaked);

	g_assert (!SPI_exit ());

	if (g_getenv ("_MEMPROF_SOCKET")) {
		fprintf (stderr, "Waiting for memprof\n");
		gtk_main ();
	}

	putenv ("AT_BRIDGE_SHUTDOWN=1");

	return 0;
}
