/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "App_WithDoc.h"
#include "Dialog_OfferSave.h"
#include "Dialog_OverwriteFile.h"
#include "libgnome/gnome-config.h"
#include <fstream>
#include <algorithm>

namespace Bakery
{


//Initialize static member data:
App_WithDoc::type_vecStrings App_WithDoc::m_vecDocHistory;

App_WithDoc::App_WithDoc(const string& appname, const string& title)
: App(appname, title),
  m_FileSelection_Open(_("Open Document")),
  m_FileSelection_Save(_("Save Document"))
{
  m_pDocument = 0;
  m_bCloseAfterSave = false;
  m_bMarkedForDestroy = false;

  m_iMenuPos_File = 0;
  m_iMenuPos_File_Save = 0;
  m_pMenuItem_File_Save = 0;
  m_pToolbarItem_Save = 0;
  m_bMenuOpenRecentCreated = false;

  m_FileSelection_Open.set_modal(true);
  m_FileSelection_Save.set_modal(true);

  m_vecDocHistory.resize(4); //TODO: Make the size a user-defined preference.
  
  //If this is the first instance, then load the configuration:
  //if(m_listAppInstances.size() == 0)
    //session_load_document_history(); 
}

App_WithDoc::~App_WithDoc()
{
  //Delete the document:
  if(m_pDocument)
  {
    delete m_pDocument;
    m_pDocument = NULL;
  }
}

void App_WithDoc::init_toolbars()
{
  using namespace Gnome::UI;

  m_toolbar_UI_Infos.push_back(Gnome::UI::Item(Gnome::UI::Icon(GNOME_STOCK_PIXMAP_NEW),
    N_("New "),
    slot(this, &App_WithDoc::on_menu_File_New),
    N_("Create a new " + m_strAppName)));

  m_toolbar_UI_Infos.push_back(Gnome::UI::Item(Gnome::UI::Icon(GNOME_STOCK_PIXMAP_OPEN),
    N_("Open "),
    slot(this, &App_WithDoc::on_menu_File_Open),
    N_("Open a " + m_strAppName)));

  m_toolbar_UI_Infos.push_back(Gnome::UI::Item(Gnome::UI::Icon(GNOME_STOCK_PIXMAP_SAVE),
    N_("Save "),
    slot(this, &App_WithDoc::on_menu_File_Save),
    N_("Save this " + m_strAppName)));
  guint iToolbarPos_Save = m_toolbar_UI_Infos.size() - 1; //Remember for later.  

  Gnome::UI::Array<Gnome::UI::Info>& arrayInfp = create_toolbar(m_toolbar_UI_Infos);

  //Get Toolbar 'Save' widget:
  m_pToolbarItem_Save = arrayInfp[iToolbarPos_Save].get_widget();
}

void App_WithDoc::on_menu_File_Close()
{
  if(!m_bMarkedForDestroy) //Maybe this method has been called a second (unnecessary) time.
  {
    if(m_pDocument->get_modified())
    {
      //Offer to save changes:
      m_bCloseAfterSave = true; //Checked in FileSelection signal handler.
      offer_to_save_changes(); //destroy will be called if necessary.
    }
    else
    {
      //Nothing to save - go ahead and close:
      close_mark_or_destroy();
    }
  }
}

void App_WithDoc::init_menus_file()
{
  // File menu
  type_vecGnome_UI_Info menu_file;

  //Build menu:
  menu_file.push_back(Gnome::MenuItems::New(_("New Document"), _("Create a new document"), slot(this, &App_WithDoc::on_menu_File_New)));
  menu_file.push_back(Gnome::MenuItems::Open(slot(this, &App_WithDoc::on_menu_File_Open)));
  //'Open Recent' added later.
  menu_file.push_back(Gnome::MenuItems::Save(slot(this, &App_WithDoc::on_menu_File_Save)));
  m_iMenuPos_File_Save = menu_file.size() - 1; //Remember for later.  
  menu_file.push_back(Gnome::MenuItems::SaveAs(slot(this, &App_WithDoc::on_menu_File_SaveAs)));

  menu_file.push_back(Gnome::UI::Separator());
  menu_file.push_back(Gnome::MenuItems::Close(slot(this, &App_WithDoc::on_menu_File_Close)));
  menu_file.push_back(Gnome::MenuItems::Exit(slot(this, &App_WithDoc::on_menu_File_Exit)));

  //Add menu:
  m_menu_UI_Infos.push_back(Gnome::Menus::File(menu_file));
  m_iMenuPos_File = m_menu_UI_Infos.size() - 1; //Remember for later.
}

void App_WithDoc::init_menus()
{
  //Overriden to get widget pointers:

  init_menus_file();
  init_menus_edit();
  init_menus_help();

  Gnome::UI::Array<Gnome::UI::SubTree>& arraySubTrees = create_menus(m_menu_UI_Infos);

  Gnome::UI::Array<Gnome::UI::Info>& arrayFileMenu = arraySubTrees[m_iMenuPos_File].get_uitree();
  m_pMenuItem_File_Save = arrayFileMenu[m_iMenuPos_File_Save].get_widget();

  install_menu_hints();

  //Override this to add more menus.
}

bool App_WithDoc::open_document(const std::string& strFilepath)
{
  //Check whether it's already open:
  //It could even be open in this instance.
  bool bAlreadyOpen = false;
  App_WithDoc* pAppAlreadyOpen = 0;
  for(type_listAppInstances::iterator iter = m_listAppInstances.begin(); iter != m_listAppInstances.end(); iter++)
  {
    App_WithDoc* pApp = static_cast<App_WithDoc*>(*iter);
    if(pApp)
    {
      Document* pDoc = pApp->get_document();
      if(pDoc->get_filepath() == strFilepath)
      {
        bAlreadyOpen = true;
        pAppAlreadyOpen = pApp;
      }
    }
  }

  if(bAlreadyOpen)
  {
    //Bring it to the front:
    if(pAppAlreadyOpen)
    {
      Gdk_Window gdkWindow = pAppAlreadyOpen->get_window();
      gdkWindow.raise();
    }

    //Tell user that it's already open:
    Gnome::Dialog* pDialog = Gnome::Dialogs::warning(_("This document is already open."));
    pDialog->run_and_close(); //deletes it too.

    return true; //success.
  }
  else
  {
    //Open it:
  			
    //Load it into a new instance unless the current document is just a default new.
    App_WithDoc* pApp = 0;
    bool bUsingNewInstance = false;
    if(!(get_document()->get_is_new())) //if it's not new.
    {
      //New instance:
      pApp = static_cast<App_WithDoc*>(new_instance());
      pApp->init(); //It's shown too.
      bUsingNewInstance = true;
    }
    else
    {
      pApp = this; //Replace the default new document in this instance.
    }

    //Open it.
    pApp->m_pDocument->set_filepath(strFilepath);
    bool bTest = pApp->m_pDocument->load();
    m_FileSelection_Open.hide();

    if(!bTest) //if open failed.
    {
      Gnome::Dialog* pDialog = Gnome::Dialogs::warning(_("Open failed."));
      pDialog->run_and_close(); //deletes it too.

      if(bUsingNewInstance)
      {
        //Remove new instance:
        pApp->get_document()->set_modified(false); //avoid 'do you want to save' dialog.
        pApp->on_menu_File_Close();
      }
      else
      {
        //re-initialize document.
        delete pApp->m_pDocument;
        pApp->m_pDocument = 0;
        pApp->init_create_document();
      }

      return false; //failed.
    }
    else
    {
      //if open succeeded:
      pApp->on_Document_load();
      pApp->update_window_title();
      set_Document_modified(false); //disables menu and toolbar Save items.

      //Update document history list:
      document_history_add(strFilepath);

      return true; //success.
    }
  } //if already open.

  return false; //failed.
}

void App_WithDoc::on_menu_File_Open()
{
  //Display File Open dialog and respond to choice:

  //Bring document window to front, to make it clear which document is being changed:
  Gdk_Window  gdkWindow = get_window();
  gdkWindow.raise();
 
  //Ask user to choose file to open:
  bool bFileChosen = m_FileSelection_Open.run();
  if(bFileChosen)
  {
    //Get selected filepath:
    std::string strFilepath = m_FileSelection_Open.get_filename();
    open_document(strFilepath);   
  }
}

void  App_WithDoc::on_menu_File_OpenRecent(guint index)
{
  if(index < m_vecDocHistory.size())
  {
    const std::string& strFilepath = m_vecDocHistory[index];
    bool bTest = open_document(strFilepath);

    //Remove it from the recent documents list if it couldn't be opened.
    if(!bTest)
      document_history_remove(strFilepath);
  }
}

//This is for calling directly, use the other override for the signal handler:
void App_WithDoc::offer_SaveAs()
{
	on_menu_File_SaveAs();
}

void App_WithDoc::on_menu_File_SaveAs()
{
  //Display File Save dialog and respond to choice:
   
  //Bring document window to front, to make it clear which document is being saved:
  //This doesn't work: TODO.
  Gdk_Window  gdkWindow = get_window();
  gdkWindow.raise();

  //Make the save dialog show the existing filename, if any:
  const string& strFilepathOld = m_pDocument->get_filepath(); 
  if(strFilepathOld.size())
    m_FileSelection_Save.set_filename(strFilepathOld);

  //Show the save dialog:
  bool bFileChosen = m_FileSelection_Save.run();
  if(bFileChosen)
  {
    std::string strFilepath = m_FileSelection_Save.get_filename();

    bool bUseThisFilePath = true;

    //Check whether file exists already:
    {
      std::ifstream fStream(strFilepath.c_str());
      if(fStream.is_open())
      {
        //Ask user to confirm overwrite:
        Bakery::Dialog_OverwriteFile* pDialogQuestion = new Bakery::Dialog_OverwriteFile(strFilepath);
        Bakery::Dialog_OverwriteFile::enumButtons buttonClicked = (Bakery::Dialog_OverwriteFile::enumButtons)pDialogQuestion->run_and_close();
        delete pDialogQuestion;
        pDialogQuestion = 0;

        //Respond to button that was clicked:
        bUseThisFilePath = (buttonClicked == Dialog_OverwriteFile::BUTTON_Overwrite);
      }
    }

    //Save to this filepath:
    if(bUseThisFilePath)
    {
      m_pDocument->set_filepath(strFilepath, true); //true = enforce file path
      bool bTest  = m_pDocument->save();
      m_FileSelection_Save.hide();

      if(!bTest)
      {
        Gnome::Dialog* pDialog = Gnome::Dialogs::warning(_("Save failed."));
        pDialog->run_and_close(); //deletes it too.
      }
      else
      {
        //Disable Save and SaveAs menu items:
        after_successful_save();
      }

      update_window_title();


      //Close if this save was a result of a File|Close or File|Exit:.
      if(bTest && m_bCloseAfterSave) //Don't close if the save failed.
      {
        on_menu_File_Close(); //This could be the second time, but now there are no unsaved changes.
      }
    }
    else
    {
      //Let the user choose a different file path,
      //because he decided not to overwrite the 1st one.
      offer_SaveAs(); //recursive.
    }
  }
  else
  {
    cancel_close_or_exit();
  }
}

void App_WithDoc::on_menu_File_Save(Gtk::Widget* pWidget)
{
  if(m_pDocument)
  {
    //If there is already a filepath, then save to that location:
    if(m_pDocument->get_filepath().size())
    {
      bool bTest = m_pDocument->save();

      if(bTest)
      {
        //Disable Save and SaveAs menu items:
	after_successful_save();    
			   
        //Close the document if this save was in response to a 'Do you want to save before closing?':
        if(m_bCloseAfterSave) // || m_bExiting
          close_mark_or_destroy();
      }
      else
      {
        //The save failed. Tell the user and don't do anything else:
        Gnome::Dialog* pDialog = Gnome::Dialogs::warning(_("Save failed."));
        pDialog->run_and_close(); //deletes it too.
        
        cancel_close_or_exit();
      }
    }
    else
    {
      //If there is no filepath, then ask for one and save to that location:
      offer_SaveAs();
    }
  }

  if(!m_bCloseAfterSave) //Don't try to do anything after closing - this instance would not exist anymore.
  {
    update_window_title();
  }

}

void App_WithDoc::init()
{
   init_create_document();

   //Call base method:
   App::init();

   session_load_document_history(); //calls show_document_history() too.

   set_Document_modified(false); //Disable Save menu item.

  
  //If the app just started, then reload any previously-open documents.
  static bool bAlreadyReopening = false; //prevent endless loop when this instance is re-used.
  if(bAlreadyReopening == false)
  {
    if(m_listAppInstances.size() == 1)
    {
      bAlreadyReopening = true;
      session_load_documents_open();
    }
  }
}

void App_WithDoc::init_create_document()
{
  //Overrides should call this base method at the end.

  if(m_pDocument == NULL)
  {
    m_pDocument = new Document();
  }

  m_pDocument->set_is_new(true); //Can't be modified if it's just been created.
  
  m_pDocument->signal_modified.connect(slot(this, &App_WithDoc::on_Document_modified));

  update_window_title();
}

Document* App_WithDoc::get_document()
{
  return m_pDocument;
}

void App_WithDoc::offer_to_save_changes()
{
  if(m_pDocument)
  {
    if(m_pDocument->get_modified())
    {
      //The document has unsaved changes,
      //so ask the user whether the document should be saved:
      Bakery::Dialog_OfferSave* pDialogQuestion = new Bakery::Dialog_OfferSave(m_pDocument->get_filepath());
      Bakery::Dialog_OfferSave::enumButtons buttonClicked = (Bakery::Dialog_OfferSave::enumButtons)pDialogQuestion->run_and_close();
      delete pDialogQuestion;
      pDialogQuestion = 0;

      //Respond to button that was clicked:
      switch(buttonClicked)
      {
        case(Bakery::Dialog_OfferSave::BUTTON_Save):
        {
          on_menu_File_Save();
          break;
        }
	      
        case(Bakery::Dialog_OfferSave::BUTTON_Discard):
        {
          //Close if this save offer was a result of a FileClose (It probably always is):
          close_mark_or_destroy();
          break;
        }
	      
        case(Bakery::Dialog_OfferSave::BUTTON_Cancel): //Cancel.
        {
          cancel_close_or_exit();
          break;
        }
	      
        default:
        {
          break;
        }
      }   
    }
  }
}

void App_WithDoc::close_mark_or_destroy()
{
  //If this is part of a File|Exit then we should just mark the instance for destruction,
  //and then tell the next instance about the Exit.
  //Otherwise, just go ahead and destroy it.
  if(m_bExiting)
  {
    //This instance needs to tell the other instance to close *before* this instance deletes itelf:
          
    hide(); //Pretend that the instance has been closed immediately. 
    m_bMarkedForDestroy = true; //This instance will actually be destroyed after the others have been saved.   

    exit_close_next_instance(); //Check for unsaved data, maybe mark it for destroy, then ask the next instance. 
  }
  else
  {  
    destroy_and_remove_from_list();
  }
}

void App_WithDoc::cancel_close_or_exit()
{
  set_operation_cancelled();
  m_bCloseAfterSave = false;
  m_bExiting = false;

  exit_destroy_marked_instances(); //Clean up after an exit.
}
  
void App_WithDoc::on_Document_load()
{	      
  //Show document contents:
  if(m_pDocument)
  {
    Bakery::ViewBase* pView = m_pDocument->get_view();
    if(pView)
  	  pView->load_from_document();

    //Set document as unmodified (this could have been wrongly set during the load):
    set_Document_modified(false);
  }
  
  //If you are not using Views, then override this to fill your various windows with stuff according to the contents of the document.
}

void App_WithDoc::update_window_title()
{
  //Set application's main window title:
  
  std::string strTitle = m_strAppName;
  Document* pDoc = get_document();
  if(pDoc)
  {
    strTitle += " - " + pDoc->get_name();

    //Indicate unsaved changes:
    if(pDoc->get_modified())
      strTitle += " *";

    set_title(strTitle);
  }
}

void App_WithDoc::on_Document_modified()
{
  //Change the displayed 'modified' status.
  //This method could be overridden to e.g. enable a Save icon or enable the Save menu item.
  //TODO: enable/disable the Save menu item.

  //Enable Save and SaveAs menu items:
  if(m_pMenuItem_File_Save)
    m_pMenuItem_File_Save->set_sensitive(true);

  if(m_pToolbarItem_Save)
    m_pToolbarItem_Save->set_sensitive(true);

  //Change title bar:
  update_window_title();
}

void App_WithDoc::set_Document_modified(bool bModified /* = true */)
{
  m_pDocument->set_modified(bModified);

  //Enable/Disable Save menu item and toolbar item:
  if(m_pMenuItem_File_Save)
    m_pMenuItem_File_Save->set_sensitive(bModified);
   
  if(m_pToolbarItem_Save)
    m_pToolbarItem_Save->set_sensitive(bModified);
}

void App_WithDoc::on_menu_File_Exit()
{
  m_bExiting = true; //Make sure that exit continues after saving unsaved documents.

  //Close the first instance - it will tell the next instance to close too, when it is ready.
  type_listAppInstances::iterator iter = m_listAppInstances.begin();
  if(iter != m_listAppInstances.end()) //Should never happen
  {
    App_WithDoc* pApp = static_cast<App_WithDoc*>(*iter);
    if(pApp)
      pApp->on_menu_File_Close();
  }
}

void App_WithDoc::on_menu_Edit_Copy()
{
  Bakery::ViewBase* pView = m_pDocument->get_view();
  if(pView)
    pView->clipboard_copy();  
}

void App_WithDoc::on_menu_Edit_Paste()
{
  Bakery::ViewBase* pView = m_pDocument->get_view();
  if(pView)
    pView->clipboard_paste();  
}

void App_WithDoc::on_menu_Edit_Clear()
{
  Bakery::ViewBase* pView = m_pDocument->get_view();
  if(pView)
    pView->clipboard_clear();  
}

  

void App_WithDoc::exit_destroy_marked_instances()
{
  //Destroy any instances that have been flagged for destroy:
  type_listAppInstances::iterator iter = m_listAppInstances.begin();
  while (iter != m_listAppInstances.end())
  {
    App_WithDoc* pApp = static_cast<App_WithDoc*>(*iter);
    iter++;
    
    if(pApp)
    {
      if(pApp->m_bMarkedForDestroy)
      {
        pApp->destroy_and_remove_from_list();

        iter = m_listAppInstances.begin();              
      }
    }
  }
  
}

void App_WithDoc::exit_close_next_instance()
{
  //When exiting, save each instance's document then tell the next instance to do the same:

  //Find the position of this instance:
  type_listAppInstances::iterator iter = std::find(m_listAppInstances.begin(), m_listAppInstances.end(), this);
  if(iter != m_listAppInstances.end()) //Should never happen - it's always in the list
  {
    iter++; //Step to the next instance.
	  
    if(iter != m_listAppInstances.end()) //If this is not the last instance
    {
      //Tell the next instance to close itself.
      App_WithDoc* pApp = static_cast<App_WithDoc*>(*iter);
      if(pApp)
      {
        pApp->on_menu_File_Close();
      }
    }
    else
    {
      //Finish the File|Exit:
      exit_destroy_marked_instances();
    }
  }
}

void App_WithDoc::after_successful_save()
{
  set_Document_modified(false); //enables/disables menu and toolbar widgets.
    
  //Update document history list:
  document_history_add(m_pDocument->get_filepath());     
}

gint App_WithDoc::on_Session_save_yourself(gint phase, GnomeSaveStyle save_style, gint is_shutdown, GnomeInteractStyle interact_style, gint is_fast)
{
  session_save_document_history();
  session_save_documents_open();

  return App::on_Session_save_yourself(phase, save_style, is_shutdown, interact_style, is_fast);
}

void App_WithDoc::session_save_document_history()
{
  //Save document history:
  //Heavily influenced by gnumeric's.
  
  Gnome::Client* pClient = Gnome::Client::master_client(); //static method.
  if(pClient)
  {
    std::string strConfigPrefix = pClient->get_config_prefix();
    strConfigPrefix += "History";
  
    gnome_config_clean_section(strConfigPrefix.c_str());
    gnome_config_push_prefix(std::string(strConfigPrefix + "/").c_str());
  
    //Store count:
    guint iCount = m_vecDocHistory.size();
    gnome_config_set_int("count", iCount);

    //Store items:
    for(guint i = 0; i < iCount; i++)
    {
      if(m_vecDocHistory[i].size()) //setting an empty string confuses gnome_config.
      {
        gchar* pchKey = g_strdup_printf ("File%d", i);
        gnome_config_set_string (pchKey, m_vecDocHistory[i].c_str()); 
        gnome_config_sync();
     
        g_free(pchKey);
      }
    }
  
    gnome_config_sync();
    gnome_config_pop_prefix();
  }
}

void App_WithDoc::session_load_document_history()
{
  //Load document history:
  //Heavily influenced by gnumeric's
  
  guint iDefaultSize = m_vecDocHistory.size();
  m_vecDocHistory.clear();
  
  Gnome::Client* pClient = Gnome::Client::master_client(); //static method.
  if(pClient)
  {
    std::string strConfigPrefix = pClient->get_config_prefix();
    strConfigPrefix += "History";
  
    gnome_config_push_prefix(std::string(strConfigPrefix + "/").c_str());
  
    //Get count:
    
    gchar* pchDefaultSize = g_strdup_printf ("%d", iDefaultSize);
    std::string strKey = std::string("count=") + pchDefaultSize;
  
    gboolean bDoSet = FALSE;
    guint iCount = gnome_config_get_int_with_default (strKey.c_str(), &bDoSet);
    if(bDoSet)
      gnome_config_set_int("count", iDefaultSize);
  
  
    //Load items:
    for(guint i = 0; i < iCount; i++)
    {
      gchar* pchKey = g_strdup_printf ("File%d", i);
      std::string strKey = pchKey;
      g_free(pchKey);
    
      gchar* pchFilepath = gnome_config_get_string(strKey.c_str());
      if(pchFilepath == 0)
      {
        break;
      }
      else
      {
        m_vecDocHistory.push_back(pchFilepath);
        g_free(pchFilepath);
      }
    }
  
    gnome_config_pop_prefix();
  }
  
  //enforce size:
  m_vecDocHistory.resize(iDefaultSize);
  
  show_document_history();
  
}


void App_WithDoc::document_history_add(const std::string& strFilepath)
{
  //TODO synchronize this static data.

  //If it's not already there:
  type_vecStrings::iterator iterFind = std::find(m_vecDocHistory.begin(), m_vecDocHistory.end(), strFilepath);
  if(iterFind == m_vecDocHistory.end())
  {
    guint sizeHistory = m_vecDocHistory.size(); //Remember size so that we can keep it the same.

    //m_vecDocHistory.erase(iterFind);

    //Add new item:
    m_vecDocHistory.insert(m_vecDocHistory.begin(), strFilepath); //Inserts at the start, before begin().

    //Enforce size, possibly forgetting old history at the end:
    m_vecDocHistory.resize(sizeHistory);

    session_save_document_history();

    show_document_history();
  }
}

void App_WithDoc::document_history_remove(const std::string& strFilepath)
{
  //TODO synchronize this static data.

  guint sizeHistory = m_vecDocHistory.size(); //Remember size so that we can keep it the same.

  //Remove it if it's there:
  type_vecStrings::iterator iterFind = std::find(m_vecDocHistory.begin(), m_vecDocHistory.end(), strFilepath);
  if(iterFind != m_vecDocHistory.end())
  {
    m_vecDocHistory.erase(iterFind);

    //Enforce size:
    m_vecDocHistory.resize(sizeHistory);

    session_save_document_history();

    show_document_history();
  }
}


void App_WithDoc::show_document_history()
{
  //Update File menu in all instances:
  for(type_listAppInstances::iterator iter = m_listAppInstances.begin(); iter != m_listAppInstances.end(); iter++)
  {
    App_WithDoc* pApp = static_cast<App_WithDoc*>(*iter);
    if(pApp)
    {
      if(pApp->m_bMenuOpenRecentCreated)
      {
        //Remove current "Open Recent" sub-menu:
        pApp->remove_menus( "File/Open Recent", 1 );
      }

      //Add new sub menu:
      type_vecGnome_UI_Info vecUI_Info;
      guint index = 0;
      for(type_vecStrings::iterator iter = m_vecDocHistory.begin(); iter != m_vecDocHistory.end(); iter++)
      {
        std::string strName = *iter;
        if(strName.size())
        {
          strName = Document::util_filepath_get_name(strName);
          vecUI_Info.push_back(Gnome::UI::Item(strName,
                               SigC::bind<guint>(slot(pApp, &App_WithDoc::on_menu_File_OpenRecent), index),
                               _("Open ") + strName));
        }

        index++;
      }

      type_vecGnome_UI_Info vecUI_Info_SubTree;
      vecUI_Info_SubTree.push_back(Gnome::UI::SubTree(_("Open Recent"), vecUI_Info, _("Open a recently-used document.")));            ;
      pApp->insert_menus("File/Open...", vecUI_Info_SubTree);
      pApp->m_bMenuOpenRecentCreated = true;
    }
  }

}

void App_WithDoc::session_save_documents_open()
{
  //Save list of open documents:
   
  Gnome::Client* pClient = Gnome::Client::master_client(); //static method.
  if(pClient)
  {
    std::string strConfigPrefix = pClient->get_config_prefix();
    strConfigPrefix += "SessionDocs";
  
    gnome_config_clean_section(strConfigPrefix.c_str());
    gnome_config_push_prefix(std::string(strConfigPrefix + "/").c_str());

  
    //Store count:
    guint iCount = m_listAppInstances.size();
    gnome_config_set_int("count", iCount);

    guint i = 0;
    for(type_listAppInstances::iterator iter = m_listAppInstances.begin(); iter != m_listAppInstances.end(); iter++)
    {
      std::string strFilepath;
      App_WithDoc* pApp = static_cast<App_WithDoc*>(*iter);
      if(pApp)
        strFilepath = pApp->get_document()->get_filepath();
     
      if(strFilepath.size())
      {
        gchar* pchKey = g_strdup_printf ("File%d", i);
        gnome_config_set_string (pchKey, strFilepath.c_str());
        g_free(pchKey);
      }
     
      i++;
    }

    gnome_config_sync();
    gnome_config_pop_prefix();
  }
}

void App_WithDoc::session_load_documents_open()
{
  //Load previously-open documents:
  
  Gnome::Client* pClient = Gnome::Client::master_client(); //static method.
  if(pClient)
  {
    std::string strConfigPrefix = pClient->get_config_prefix();
    strConfigPrefix += "SessionDocs";
  
    gnome_config_push_prefix(std::string(strConfigPrefix + "/").c_str());
  
    //Get count:
    
    std::string strKey = std::string("count=0");
  
    gboolean bDoSet = FALSE;
    guint iCount = gnome_config_get_int_with_default (strKey.c_str(), &bDoSet);
    if(bDoSet)
      gnome_config_set_int("count", 0);
  
  
    //Load items:
    for(guint i = 0; i < iCount; i++)
    {
      gchar* pchKey = g_strdup_printf ("File%d", i);
      std::string strKey = pchKey;
      g_free(pchKey);
    
      gchar* pchFilepath = gnome_config_get_string(strKey.c_str());
      if(pchFilepath == 0)
      {
        break;
      }
      else
      {
        open_document(pchFilepath);
        g_free(pchFilepath);
      }
    }
  
    gnome_config_pop_prefix();
  }
  
}
  

} //namespace
