/*
 * Copyright 2002 The Bakery team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "bakery/Configuration/Association.h"
#include <gtkmm/togglebutton.h>
#include <gtkmm/entry.h>
#include <gtkmm/range.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/combo.h>
#include <gtkmm/optionmenu.h>

namespace Bakery
{
namespace Conf
{
  
//----------------------------------------------------------------------
// Association<T> specializations. These provide widget/key
// association behaviors that are specific to individual widget types.
//----------------------------------------------------------------------

void Association<Gtk::SpinButton>::connect_widget(Callback widget_changed)
{
  m_widget.signal_value_changed().connect(widget_changed);
}

void Association<Gtk::SpinButton>::load_widget()
{
  double val = get_conf_client()->get_float(get_key());
  if (m_widget.get_value() != val)
    m_widget.set_value(val);
}

void Association<Gtk::SpinButton>::save_widget()
{
  double val = m_widget.get_value();
  if (get_conf_client()->get_float(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------

void Association<Gtk::Entry>::connect_widget(Callback widget_changed)
{
  m_widget.signal_changed().connect(widget_changed);
}

void Association<Gtk::Entry>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = get_conf_client()->get_string(get_key());
  if (m_widget.get_text() != val)
    m_widget.set_text(val);
}

void Association<Gtk::Entry>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = m_widget.get_text();
  if (get_conf_client()->get_string(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------

void Association<Gtk::ToggleButton>::connect_widget(Callback widget_changed)
{
  m_widget.signal_toggled().connect(widget_changed);
}

void Association<Gtk::ToggleButton>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  bool val = get_conf_client()->get_bool(get_key());
  if (m_widget.get_active() != val)
    m_widget.set_active(val);
}

void Association<Gtk::ToggleButton>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  bool val = m_widget.get_active();
  if (get_conf_client()->get_bool(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------

void Association<Gtk::Range>::connect_widget(Callback widget_changed)
{
  m_widget.signal_value_changed().connect(widget_changed);
}

void Association<Gtk::Range>::load_widget()
{
  double val = get_conf_client()->get_float(get_key());
  if (m_widget.get_value() != val)
    m_widget.set_value(val);
}

void Association<Gtk::Range>::save_widget()
{
  double val = m_widget.get_value();
  if (get_conf_client()->get_float(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------

void Association<Gtk::OptionMenu>::connect_widget(Callback widget_changed)
{
  m_widget.signal_changed().connect(widget_changed);
}

void Association<Gtk::OptionMenu>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  int val = get_conf_client()->get_int(get_key());
  if (m_widget.get_history() != val)
    m_widget.set_history(guint(val));
}

void Association<Gtk::OptionMenu>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  int val = m_widget.get_history();
  if (get_conf_client()->get_int(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------

void Association<Gtk::Combo>::connect_widget(Callback widget_changed)
{
  m_widget.get_entry()->signal_changed().connect(widget_changed);
}

void Association<Gtk::Combo>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = get_conf_client()->get_string(get_key());
  if (m_widget.get_entry()->get_text() != val)
    m_widget.get_entry()->set_text(val);
}

void Association<Gtk::Combo>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = m_widget.get_entry()->get_text();
  if (get_conf_client()->get_string(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

} //namespace Conf

} //namespace Bakery
