/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

#include <stdlib.h>
#include <jni.h>
#include <cairo.h>
#include <stdlib.h>
#include <jg_jnu.h>

#ifndef _Included_org_freedesktop_cairo_Surface
#define _Included_org_freedesktop_cairo_Surface
#ifdef __cplusplus
extern "C" {
#endif

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_create_similar
 * Signature: (Lorg/gnu/glib/Handle;III)Lorg/gnu/glib/Handle;
 */
JNIEXPORT jobject JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1create_1similar
  (JNIEnv *env, jclass cls, jobject other, jint content, jint width, jint height)
{
	cairo_surface_t *other_g;
	cairo_surface_t *surface;
	
	other_g = (cairo_surface_t*)getPointerFromHandle(env, other);

	surface = cairo_surface_create_similar(other_g, (cairo_content_t)content,
			width, height);
		
	return getHandleFromPointer(env, surface);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_destroy
 * Signature: (Lorg/gnu/glib/Handle;)V
 */
JNIEXPORT void JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1destroy
  (JNIEnv *env, jclass cls, jobject obj)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_surface_destroy(surface);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_finish
 * Signature: (Lorg/gnu/glib/Handle;)I
 */
JNIEXPORT jint JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1finish
  (JNIEnv *env, jclass cls, jobject obj)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_surface_finish(surface);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_write_to_png
 * Signature: (Lorg/gnu/glib/Handle;Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1write_1to_1png
  (JNIEnv *env, jclass cls, jobject obj, jstring png)
{
	cairo_surface_t *surface;
	char *png_g;
	cairo_status_t status;
	
	surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	png_g = (char*)(*env)->GetStringUTFChars(env, png, NULL);
	
	status = cairo_surface_write_to_png(surface, png_g);
	
	(*env)->ReleaseStringUTFChars(env, png, png_g);
	return status;
}

typedef struct
{
	JNIEnv *env;
	jobject ostream;
	jmethodID methodID;
} write_to_png_closure_t;

static cairo_status_t
write_to_png_func (void                *closure,
		   const unsigned char *data,
		   unsigned int         length)
{
	write_to_png_closure_t *png_closure = closure;
	JNIEnv *env = png_closure->env;
	jbyte *bytes;
	jboolean isCopy;
	cairo_status_t status = CAIRO_STATUS_SUCCESS;

	jbyteArray array = (*env)->NewByteArray (env, length);
	if (!array)
		return CAIRO_STATUS_NO_MEMORY;

	bytes = (*env)->GetByteArrayElements (env, array, &isCopy);
	if (!bytes) {
		status = CAIRO_STATUS_NO_MEMORY;
		goto cleanup;
	}

	memcpy (bytes, data, length);

	(*env)->ReleaseByteArrayElements(env, array, bytes, 0);

	(*env)->CallVoidMethod(env,
			       png_closure->ostream, png_closure->methodID,
			       array);

	if ((*env)->ExceptionOccurred(env))
		status = CAIRO_STATUS_WRITE_ERROR;

 cleanup:
	/* need to explicitely Delete since we create objects in a loop
	 * without returning control to Java/
	 */
	(*env)->DeleteLocalRef (env, array);
	
	return status;
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_write_to_png_stream
 * Signature: (Lorg/gnu/glib/Handle;Ljava/io/OutputStream;)I
 */
JNIEXPORT jint JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1write_1to_1png_1stream
  (JNIEnv *env, jclass cls, jobject obj, jobject ostream)
{
	cairo_surface_t *surface;
	char *png_g;
	cairo_status_t status;
	write_to_png_closure_t png_closure;
	jclass ostreamClass;

	surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	
	png_closure.env = env;
	png_closure.ostream = ostream;
	
	ostreamClass = (*env)->FindClass(env, "java/io/OutputStream");
	if (!ostreamClass)
		return CAIRO_STATUS_WRITE_ERROR;

	png_closure.methodID = (*env)->GetMethodID(env, ostreamClass, "write", "([B)V");
	if (!png_closure.methodID)
		return CAIRO_STATUS_WRITE_ERROR;

	return cairo_surface_write_to_png_stream(surface, write_to_png_func, &png_closure);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_set_device_offset
 * Signature: (Lorg/gnu/glib/Handle;DD)V
 */
JNIEXPORT void JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1set_1device_1offset
  (JNIEnv *env, jclass cls, jobject obj, jdouble xoff, jdouble yoff)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_surface_set_device_offset(surface, xoff, yoff);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_get_font_options
 * Signature: (Lorg/gnu/glib/Handle;)Lorg/gnu/glib/Handle;
 */
JNIEXPORT jobject JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1get_1font_1options
  (JNIEnv *env, jclass cls, jobject obj)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_font_options_t *fo;
	cairo_surface_get_font_options(surface, fo);
	return getHandleFromPointer(env, fo);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_flush
 * Signature: (Lorg/gnu/glib/Handle;)V
 */
JNIEXPORT void JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1flush
  (JNIEnv *env, jclass cls, jobject obj)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_surface_flush(surface);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_mark_dirty
 * Signature: (Lorg/gnu/glib/Handle;)V
 */
JNIEXPORT void JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1mark_1dirty
  (JNIEnv *env, jclass cls, jobject obj)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_surface_mark_dirty(surface);
}

/*
 * Class:     org_freedesktop_cairo_Surface
 * Method:    cairo_surface_mark_dirty_rectangle
 * Signature: (Lorg/gnu/glib/Handle;IIII)V
 */
JNIEXPORT void JNICALL Java_org_freedesktop_cairo_Surface_cairo_1surface_1mark_1dirty_1rectangle
  (JNIEnv *env, jclass cls, jobject obj, jint x, jint y, jint w, jint h)
{
	cairo_surface_t *surface = (cairo_surface_t*)getPointerFromHandle(env, obj);
	cairo_surface_mark_dirty_rectangle(surface, x, y, w, h);
}


#ifdef __cplusplus
}
#endif
#endif
